package com.digiwin.mobile.mobileuibot.mongodb.uc;

import com.digiwin.mobile.mobileuibot.common.context.AppContext;
import com.digiwin.mobile.mobileuibot.mongodb.designerconfig.MobileDesignerConfig;
import com.digiwin.mobile.mobileuibot.openapi.designer.enums.DataVersionEnum;
import com.digiwin.mobile.mobileuibot.task.filter.TaskFilter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * <p>功能描述：操作 MongoDB，获取 uc_mobileDesignerConfig 数据</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: UcMobileDesignerConfigRepository.java
 * @Author: wangjwc
 * @Date: created at 2025/2/8 13:51
 */
@Component
public class UcMobileDesignerConfigRepository {

    @Autowired
    private MongoTemplate mongoTemplate;
    @Autowired
    @Qualifier("secondMongoTemplate")
    private MongoTemplate secondMongoTemplate;

    public Boolean isExsitTmActivityIdOfUcAppCode(String tmActivityId, String type, String version, Set<String> ucAppCodeList) {
        return this.findOne(tmActivityId, type, version, ucAppCodeList) != null;
    }

    private UcMobileDesignerConfig findOne(String tmActivityId, String type, String version, Set<String> ucAppCodeList) {
        // 查询指定字段
        Query queryAll = new Query();
        queryAll.addCriteria(Criteria.where("type").is(type))
                .addCriteria(Criteria.where("version").is(version))
                .fields().exclude("data").exclude("rules");
        List<UcMobileDesignerConfig> list = mongoTemplate.find(queryAll, UcMobileDesignerConfig.class)
                .stream()
                .filter(config -> {
                    // 过滤在ucAppCodeList中的应用
                    if (CollectionUtils.isEmpty(ucAppCodeList)) {
                        return true;
                    }
                    return ucAppCodeList.contains(config.getUcAppCode());
                })
                // 过滤当前环境下的数据
                .filter(config -> config.validWithAppEnvType(AppContext.getEnvType()))
                .collect(Collectors.toList());
        if (CollectionUtils.isEmpty(list)) {
            return null;
        }
        String newTmActivityId = null;
        String ucAppCode = null;
        for (UcMobileDesignerConfig designerConfig : list) {
            boolean match = TaskFilter.match(tmActivityId, designerConfig.getTmActivityId(), designerConfig.getTmActivityIdMatch());
            if (match) {
                newTmActivityId = designerConfig.getTmActivityId();
                ucAppCode = designerConfig.getUcAppCode();
                break;
            }
        }
        if (newTmActivityId == null) {
            return null;
        }
        Query query = new Query();
        query.addCriteria(Criteria.where("type").is(type))
                .addCriteria(Criteria.where("tmActivityId").is(newTmActivityId))
                .addCriteria(Criteria.where("version").is(version))
                .addCriteria(Criteria.where("ucAppCode").is(ucAppCode));
        UcMobileDesignerConfig findOne = this.mongoTemplate.findOne(query, UcMobileDesignerConfig.class);
        return Optional.ofNullable(findOne).orElse(new UcMobileDesignerConfig()).validWithAppEnvType(AppContext.getEnvType()) ? findOne : null;
    }

    public UcMobileDesignerConfig selectOneDsl(String tmActivityId, String type, Set<String> ucAppCodeList) {
        // wjw FIXME: 2025/2/13 因本地DSL配置是手动插入的，故数据都一样，取任意version版本数据即可
        return this.findOne(tmActivityId, type, DataVersionEnum.V1_0.getVersion(), ucAppCodeList);
    }

    public List<UcMobileDesignerConfig> findByType(String type, String version, Set<String> ucAppCodeList) {
        Query query = new Query();
        query.addCriteria(Criteria.where("type").is(type))
                .addCriteria(Criteria.where("version").is(version))
                .addCriteria(Criteria.where("ucAppCode").in(ucAppCodeList));

        return this.mongoTemplate.find(query, UcMobileDesignerConfig.class)
                .stream().filter(config -> config.validWithAppEnvType(AppContext.getEnvType()))
                .collect(Collectors.toList());
    }

    public void deleteByCode(String appCode, String ucAppCode, String version) {
        Query query = new Query();
        query.addCriteria(Criteria.where("application").is(appCode))
                .addCriteria(Criteria.where("ucAppCode").is(ucAppCode))
                .addCriteria(Criteria.where("version").is(version));
        // AB库同时操作
        mongoTemplate.findAllAndRemove(query, UcMobileDesignerConfig.class);
        secondMongoTemplate.findAllAndRemove(query, UcMobileDesignerConfig.class);
    }

    public List<UcMobileDesignerConfig> selectByCode(String appCode, String ucAppCode, String version) {
        Query query = new Query();
        query.addCriteria(Criteria.where("application").is(appCode))
                .addCriteria(Criteria.where("ucAppCode").is(ucAppCode))
                .addCriteria(Criteria.where("version").is(version));
        List<UcMobileDesignerConfig> list = mongoTemplate.find(query, UcMobileDesignerConfig.class);
        return CollectionUtils.isEmpty(list) ? new ArrayList<>() : list;
    }

    public void batchInsert(List<UcMobileDesignerConfig> mobileDesignerConfigs) {
        if (CollectionUtils.isEmpty(mobileDesignerConfigs)) {
            return;
        }
        // AB库同时操作
        mongoTemplate.insertAll(mobileDesignerConfigs);
        secondMongoTemplate.insertAll(mobileDesignerConfigs);
    }

    public void updateData(MobileDesignerConfig config) {
        if (config == null || !StringUtils.hasLength(config.getTmActivityId())
                || config.getData() == null || !StringUtils.hasLength(config.getApplication())) {
            return;
        }
        // 查询条件type不能加，因为任务当责者和执行者此接口保存传递同一个类型
        Query query = new Query()
                .addCriteria(Criteria.where("tmActivityId").is(config.getTmActivityId()))
                // 个案应用
                .addCriteria(Criteria.where("ucAppCode").is(config.getApplication()));
        // 查询个案应用下作业是否存在数据
        if (this.mongoTemplate.findOne(query, UcMobileDesignerConfig.class) == null) {
            // 不存在，返回
            return;
        }
        Update update = new Update().set("data", config.getData());
        // AB库同时操作
        this.mongoTemplate.updateMulti(query, update, UcMobileDesignerConfig.class);
        this.secondMongoTemplate.updateMulti(query, update, UcMobileDesignerConfig.class);
    }
}