package com.digiwin.mobile.mobileuibot.myteam;

import com.digiwin.mobile.mobileuibot.api.ApiRawData;
import com.digiwin.mobile.mobileuibot.api.ApiRequest;
import com.digiwin.mobile.mobileuibot.common.context.AppContext;
import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.core.common.BackgroundColorEnum;
import com.digiwin.mobile.mobileuibot.core.component.BaseMobileComponent;
import com.digiwin.mobile.mobileuibot.core.component.BaseMobileComponentWrapper;
import com.digiwin.mobile.mobileuibot.core.component.MobileRenderMetaData;
import com.digiwin.mobile.mobileuibot.core.component.list.card.*;
import com.digiwin.mobile.mobileuibot.core.component.title.teamtitle.TeamTitle;
import com.digiwin.mobile.mobileuibot.core.pagesetting.PageSetting;
import com.digiwin.mobile.mobileuibot.core.strategy.modelbuild.UiBotModelBuildStrategy;
import com.digiwin.mobile.mobileuibot.locale.service.LocaleService;
import com.digiwin.mobile.mobileuibot.model.db1.UserProjectFilter;
import com.digiwin.mobile.mobileuibot.mysql.service.UserProjectFilterService;
import com.digiwin.mobile.mobileuibot.project.filter.ProjectFilterService;
import com.digiwin.mobile.mobileuibot.project.service.ProjectService;
import com.digiwin.mobile.mobileuibot.project.service.impl.ProjectServiceImpl;
import com.digiwin.mobile.mobileuibot.proxy.atmc.model.DigiwinAtmcPersonnelStructure;
import com.digiwin.mobile.mobileuibot.proxy.atmc.model.DigiwinAtmcProject;
import com.digiwin.mobile.mobileuibot.proxy.atmc.model.DigiwinEmp;
import com.digiwin.mobile.mobileuibot.proxy.atmc.service.DigiwinAtmcProxyService;
import com.digiwin.mobile.mobileuibot.proxy.audc.model.DigiwinAtmcGroupRecord;
import com.digiwin.mobile.mobileuibot.proxy.audc.service.DigiwinAudcProxyService;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.UiBotModel;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.UiBotPageData;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.UiBotRenderData;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.layout.UiBotLayout;
import org.apache.commons.lang3.BooleanUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import java.io.IOException;
import java.util.*;
import java.util.stream.Collectors;


/**
 * 团队项目卡列表页
 *
 * @author 吴阳
 * @date 20230103
 */
@Component("uiBotModelBuildTeamProjectStrategy")
public class UiBotModelBuildTeamProjectStrategy implements UiBotModelBuildStrategy {

    private static final Logger logger = LoggerFactory.getLogger(UiBotModelBuildTeamProjectStrategy.class);

    /**
     * teamMember-团队成员
     */
    private static final String GROUP_TYPE_TEAM_MEMBER = "teamMember";
    /**
     * code-任务类型/团队类型
     */
    private static final String GROUP_TYPE_CODE = "code";

    @Autowired
    private DigiwinAtmcProxyService digiwinAtmcProxyService;

    @Autowired
    private ProjectFilterService projectFilterService;

    @Autowired
    private ProjectService projectService;

    @Autowired
    private DigiwinAudcProxyService digiwinAudcProxyService;

    @Autowired
    private LocaleService localeService;

    @Autowired
    private UserProjectFilterService userProjectFilterService;

    @Override
    public UiBotModel buildModel(PageSetting pageSetting, ApiRequest apiRequest) throws IOException {
        UiBotModel uiBotModel = null;
        Boolean useMockData = AppContext.getUseMockData();
        if (!useMockData) {
            uiBotModel = this.doBefore(pageSetting, apiRequest);
        } else {
            uiBotModel = this.doMockData(apiRequest);
        }
        return uiBotModel;
    }

    private UiBotModel doBefore(PageSetting pageSetting, ApiRequest apiRequest) {
        if (Optional.ofNullable(apiRequest.getRawData().get("groupType")).isPresent()) {
            // 防呆 -- APP低版本走原有逻辑
            return this.doActually(pageSetting, apiRequest);
        } else {
            return this.doActuallyV2(pageSetting, apiRequest);
        }

    }

    private UiBotModel doActually(PageSetting pageSetting, ApiRequest apiRequest) {
        String locale = apiRequest.getLocale();
        String iamUserToken = apiRequest.getIamUserToken();
        String tenantId = apiRequest.getTenantId();
        String userId = apiRequest.getUserId();
        ApiRawData rawData = apiRequest.getRawData();
        String groupType = (String) rawData.get("groupType");
        Map<String, Object> params = new HashMap<>(3);
        params.put("groupType", groupType);
        if (GROUP_TYPE_TEAM_MEMBER.equalsIgnoreCase(groupType)) {
            params.put("userId", rawData.get("userId"));
        } else if (GROUP_TYPE_CODE.equalsIgnoreCase(groupType)) {
            params.put("projectDefCode", rawData.get("projectDefCode"));
        }
        List<DigiwinAtmcProject> teamProjects = digiwinAtmcProxyService.getDepartProjectGroupList(
                iamUserToken, tenantId, locale, params);
        ProjectServiceImpl projectServiceImpl = new ProjectServiceImpl();
        List<ProjectServiceImpl.ProjectAndDoingTask> filteredProjectAndDoingTaskList = projectServiceImpl.convertProjectAndDoingTasks(teamProjects);
        List<CardCategory> categories = CardCategory.getByProjects(teamProjects,
                projectFilterService, projectServiceImpl, filteredProjectAndDoingTaskList, locale, iamUserToken, userId, tenantId);

        // 项目名称(简体：首字首字母升序，繁体：首字笔画升序)->查阅状态(未读优先)->项目时间(按结束时间升序)
        for (CardCategory cardCategory : categories) {
            cardCategory.setCardList(projectService.sortCardList(cardCategory.getCardList(), locale));
        }

        CardList cardList = new CardList();
        cardList.setCategoryList(categories);
        UiBotModel uiBotModel = pageSetting.getPageModel();
        List<UiBotLayout> layoutList = uiBotModel.getLayout();
        UiBotPageData pageData = uiBotModel.getPageData();
        for (UiBotLayout uiBotLayout : layoutList) {
            String schema = uiBotLayout.getSchema();
            switch (uiBotLayout.getType()) {
                case TeamTitle.COMPONENT_TYPE:
                    pageData.put(schema, JsonUtil.objectToJavaObject(rawData.get(schema), TeamTitle.class));
                    break;
                case CardList.COMPONENT_TYPE:
                    pageData.put(schema, cardList);
                    break;
                default:
                    break;
            }
        }
        return uiBotModel;
    }

    private UiBotModel doActuallyV2(PageSetting pageSetting, ApiRequest apiRequest) {
        UiBotModel uiBotModel = UiBotModel.emptyUiBotModel();
        uiBotModel.setPageBackgroudColor(BackgroundColorEnum.APP_MAIN_GREY.getValue());
        if (ObjectUtils.isEmpty(uiBotModel.getRenderData())) {
            uiBotModel.setRenderData(UiBotRenderData.createEmptyRenderData());
        }
        // 构建CardList组件
        uiBotModel.getRenderData().addWrapperedComponentToBody(this.buildCardListWrapper(apiRequest));
        uiBotModel.setRenderVersion(MobileRenderMetaData.RENDER_VERSION_2);
        return uiBotModel;
    }

    private BaseMobileComponentWrapper<BaseMobileComponent> buildCardListWrapper(ApiRequest apiRequest) {
        // 团队项目中的团队成员树结构及相关信息
        DigiwinAtmcPersonnelStructure personnelStructure = digiwinAtmcProxyService.getPersonnelStructure(apiRequest.getIamUserToken(), apiRequest.getTenantId(), apiRequest.getLocale());
        List<DigiwinEmp> defaultSelectEmpList = Optional.ofNullable(personnelStructure).map(DigiwinAtmcPersonnelStructure::getAllChildrenDigiwinEmp).orElse(new ArrayList<>());
        //获取用户是否保存过筛选条件
        UserProjectFilter projectFilter = userProjectFilterService.selectByParams(apiRequest.getUserId(), apiRequest.getTenantId(), PageFilterCodeEnum.TEAM_PROJECTS.getValue());

        CardList cardList = new CardList();
        cardList.setCardList(this.buildCardList(apiRequest, defaultSelectEmpList));
        cardList.setSearchUrl("/mobile/v1/project/search");
        cardList.setSearchIdList(this.buildSearchIdList(apiRequest, projectFilter, defaultSelectEmpList));
        cardList.setQueryParam(projectFilter == null ? "" : projectFilter.getQueryParam());
        cardList.setGroupFilter(this.buildGroupFilter(apiRequest, projectFilter));
        cardList.setPersonnelStructure(personnelStructure);
        return new BaseMobileComponentWrapper<>(cardList, "DW_" + CardList.COMPONENT_TYPE);
    }

    private List<String> buildSearchIdList(ApiRequest apiRequest, UserProjectFilter projectFilter, List<DigiwinEmp> defaultSelectEmpList) {
        if (projectFilter == null || !StringUtils.hasLength(projectFilter.getQueryParam()) || CollectionUtils.isEmpty(defaultSelectEmpList)) {
            return new ArrayList<>();
        }
        //再次查询新的筛选的关键字数据
        return this.digiwinAtmcProxyService.teamProjectSearch(projectFilter.getQueryParam(), defaultSelectEmpList,
                apiRequest.getIamUserToken(), apiRequest.getTenantId(), apiRequest.getLocale(), apiRequest.getClientId());
    }

    private CardListGroupFilter buildGroupFilter(ApiRequest apiRequest, UserProjectFilter projectFilter) {
        List<DigiwinAtmcGroupRecord> groupRecordList = digiwinAudcProxyService.groupRecord(apiRequest.getLocale(),
                apiRequest.getIamUserToken(), apiRequest.getTenantId(), PageFilterCodeEnum.TEAM_PROJECTS.getValue());
        if (CollectionUtils.isEmpty(groupRecordList)) {
            return null;
        }
        // 用户自定义配置分组数据
        String groupFilter = Optional.ofNullable(projectFilter).map(UserProjectFilter::getGroupFilter).orElse(null);
        UserProjectFilterGroupFilter customGroup = StringUtils.hasLength(groupFilter) ? JsonUtil.jsonStringToObject(groupFilter, UserProjectFilterGroupFilter.class) : null;
        // 用户自定义常规分组数据--单笔
        List<CardListGroupOption> customNormalGroup = Optional.ofNullable(customGroup).map(UserProjectFilterGroupFilter::getNormal).orElse(null);
//        // 用户自定义高级分组数据
//        List<CardListGroupOption> customSeniorGroup = Optional.ofNullable(customGroup).map(UserProjectFilterGroupFilter::getSenior).orElse(null);
        // 常规分组
        boolean existDefaultEffective = groupRecordList.stream().anyMatch(r -> BooleanUtils.isTrue(r.getDefaultEffective()));
        List<CardListGroupOption> normalGroup = groupRecordList.stream().map(item -> {
            CardListGroupOption cardListGroupFilterOption = new CardListGroupOption();
            cardListGroupFilterOption.setFilterOptionId(item.getCode());
            cardListGroupFilterOption.setFilterOptionName(localeService.getLanguageValue(apiRequest.getLocale(), item.getName()));
            cardListGroupFilterOption.setDefaultEffective(item.getDefaultEffective());
            cardListGroupFilterOption.setSort("asc");
            // 不存在预设选项 -- 默认设置不分组为预设选项
            if (!existDefaultEffective && "noGrouping".equalsIgnoreCase(item.getCode())) {
                cardListGroupFilterOption.setDefaultEffective(true);
            }

            if (CollectionUtils.isEmpty(customNormalGroup)) {
                // 未配置用户自定义常规分组信息 -- 第一次操作分组
                if (existDefaultEffective) {
                    // 存在预设分组
                    if (BooleanUtils.isTrue(item.getDefaultEffective())) {
                        cardListGroupFilterOption.setIsSelect(true);
                        cardListGroupFilterOption.setSort("asc");
                    }
                } else {
                    // 不存在预设分组 -- 默认设置不分组
                    if ("noGrouping".equalsIgnoreCase(item.getCode())) {
                        cardListGroupFilterOption.setIsSelect(true);
                        cardListGroupFilterOption.setSort("asc");
                    }
                }

                cardListGroupFilterOption.convertOptionConfigs(localeService, apiRequest.getLocale(), item.getConfigs(), null);
            } else {
                // 配置用户自定义常规分组信息
                customNormalGroup.forEach(r -> {
                    if (item.getCode().equals(r.getFilterOptionId())) {
                        // 选中
                        cardListGroupFilterOption.setIsSelect(true);
                        cardListGroupFilterOption.setSort(r.getSort());
                    }
                });

                // 因只有团队项目中按项目时间分组才有config
                CardListGroupOptionConfigs customConfigs = customNormalGroup.stream().filter(r -> Objects.nonNull(r.getOptionConfigs())).findFirst().map(CardListGroupOption::getOptionConfigs).orElse(null);
                cardListGroupFilterOption.convertOptionConfigs(localeService, apiRequest.getLocale(), item.getConfigs(), customConfigs);
            }

            // type枚举 system：系统分组标签，customer：自定义分组标签
            cardListGroupFilterOption.setType(item.getType());
            return cardListGroupFilterOption;
        }).collect(Collectors.toList());

        CardListGroupFilter cardListGroupFilter = new CardListGroupFilter();
        cardListGroupFilter.setModel(customGroup == null ? GroupFilterModelEnum.NORMAL.getValue() : customGroup.getModel());
        cardListGroupFilter.setFilterOptionList(normalGroup);
        return cardListGroupFilter;
    }

    private List<Card> buildCardList(ApiRequest apiRequest, List<DigiwinEmp> defaultSelectEmpList) {
        String locale = apiRequest.getLocale();
        String iamUserToken = apiRequest.getIamUserToken();
        String tenantId = apiRequest.getTenantId();
        if (CollectionUtils.isEmpty(defaultSelectEmpList)) {
            return new ArrayList<>();
        }
        // 获取团队成员的项目
        List<DigiwinAtmcProject> teamProjects = digiwinAtmcProxyService.selectEmpProject(defaultSelectEmpList, iamUserToken, locale, tenantId);
        if (CollectionUtils.isEmpty(teamProjects)) {
            return new ArrayList<>();
        }
        // APP端过滤dataFrom=PTM的项目
        teamProjects.removeIf(r -> !"PTM".equals(r.getDataFrom()));
        return teamProjects.stream().map(project -> {
            if (projectFilterService.isCenterConsoleProject(project, tenantId)) {
                return projectService.handleCenterConsoleProject(project, iamUserToken, tenantId, locale, true);
            } else if (projectFilterService.isStandardProject(project, tenantId)) {
                return projectService.handleStandardProject(project, iamUserToken, locale, tenantId, true);
            } else if (projectFilterService.isSystemCloudHouseKeeper(project, tenantId)) {
                return projectService.handleSystemCloudKeeperProject(project, iamUserToken, tenantId, locale, true);
            } else {
                return projectService.handleCurrentlyNotSupportedProject(project, iamUserToken, locale, tenantId, true);
            }
        }).collect(Collectors.toList());
    }

    private UiBotModel doMockData(ApiRequest apiRequest) {
        return null;
    }
}
