package com.digiwin.mobile.mobileuibot.notification.model;

import com.digiwin.mobile.mobileuibot.common.datetime.DateTimeUtil;
import com.digiwin.mobile.mobileuibot.common.localization.LocaleUtil;
import com.digiwin.mobile.mobileuibot.core.pagesetting.PageSettingIdPresetEnum;
import com.digiwin.mobile.mobileuibot.locale.service.LocaleService;
import com.digiwin.mobile.mobileuibot.notification.enums.*;
import com.digiwin.mobile.mobileuibot.proxy.aim.model.DigiwinAimMessage;
import com.digiwin.mobile.mobileuibot.proxy.aim.model.DigiwinAimMessageBoardCommenter;
import com.digiwin.mobile.mobileuibot.proxy.aim.model.DigiwinAimMessageSummary;
import com.digiwin.mobile.mobileuibot.proxy.atmc.model.DigiwinAtmcBacklog;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.Serializable;
import java.time.LocalDateTime;
import java.util.List;

/**
 * <p>功能描述：通知分类摘要的数据模型</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: NotificationSummary
 * @Author: Zaregoto
 * @Date: 2021/4/21 15:25
 */
public class NotificationSummary implements Serializable {
    private static final long serialVersionUID = 3253780939513502434L;

    private static final Logger logger = LoggerFactory.getLogger(NotificationSummary.class);

    private NotificationSummaryBadge badge;
    private String categoryId;
    private String categoryName;
    private String categoryNameDesc;
    private String jumpPageId;
    private String jumpPageTitle;
    private NotificationSummarySubject latestNotification;

    @Deprecated
    public static NotificationSummary create(String locale, DigiwinAimMessageSummary digiwinAimMessageSummary) {
        NotificationSummary notificationSummary = new NotificationSummary();
        // 通知分类设置
        String pcSubTypeCategory = digiwinAimMessageSummary.getSubTypeCategory();
        if (pcSubTypeCategory.equals(AthenaPcNotificationCategoryEnum.PROXY_REMIND.toString()) ||
                pcSubTypeCategory.equals(AthenaPcNotificationCategoryEnum.ATHENA_REMIND.toString()) ||
                pcSubTypeCategory.equals(AthenaPcNotificationCategoryEnum.EXCEPTION_OR_OVERDUE.toString())) {
            // Athena助手
            notificationSummary.setCategoryId(AthenaMobileNotificationCategoryEnum.MOBILE_ATHENA_ASSISTANT
                    .toString());
            notificationSummary.setCategoryName(
                    LocaleUtil.getMobileTextByKey(locale, AthenaMobileNotificationCategoryEnum.MOBILE_ATHENA_ASSISTANT.toString()));
        } else if (pcSubTypeCategory.equals(AthenaPcNotificationCategoryEnum.INTERNAL_REMIND.toString())) {
            // 内部个人通知单独处理。在createPersonal方法内
            return null;
        } else {
            // 其他分类放在移动端系统通知中
            notificationSummary.setCategoryId(AthenaMobileNotificationCategoryEnum.MOBILE_SYSTEM.toString());
            notificationSummary.setCategoryName(
                    LocaleUtil.getMobileTextByKey(locale, AthenaMobileNotificationCategoryEnum.MOBILE_SYSTEM.toString()));
        }
        notificationSummary.setBadge(NotificationSummaryBadge.create(digiwinAimMessageSummary, notificationSummary.getCategoryId()));
        notificationSummary.setJumpPageId(notificationSummary.getCategoryId());
        notificationSummary.setJumpPageTitle(notificationSummary.getCategoryName());
        notificationSummary.setLatestNotification(NotificationSummarySubject.create(locale, digiwinAimMessageSummary));

        return notificationSummary;
    }

    @Deprecated
    public static NotificationSummary create(String locale, List<DigiwinAtmcBacklog> digiwinAtmcBacklogList) {
        NotificationSummary notificationSummary = new NotificationSummary();
        // 计算今日待办数量=已逾期数量+完成日期等于今天的数量，且完成日期在今天的不能包括已逾期的，会重复计算。
        Integer overdueCount = (int) digiwinAtmcBacklogList.stream()
                .filter(backlog -> backlog.getOverdue().equals(Boolean.TRUE)).count();
        Integer todayEndExceptOverdueCount = (int) digiwinAtmcBacklogList.stream()
                .filter(backlog -> DateTimeUtil.isTodayTime(backlog.getEndTime()).equals(Boolean.TRUE))
                .filter(backlog -> !backlog.getOverdue().equals(Boolean.TRUE)).count();
        notificationSummary.setBadge(NotificationSummaryBadge.create(overdueCount + todayEndExceptOverdueCount));
        notificationSummary.setCategoryId(AthenaMobileNotificationCategoryEnum.MOBILE_MY_TODO.toString());
        notificationSummary.setCategoryName(
                LocaleUtil.getMobileTextByKey(locale, AthenaMobileNotificationCategoryEnum.MOBILE_MY_TODO.toString()));
        notificationSummary.setJumpPageId(notificationSummary.getCategoryId());
        notificationSummary.setJumpPageTitle(notificationSummary.getCategoryName());
        notificationSummary.setLatestNotification(NotificationSummarySubject.create(locale, digiwinAtmcBacklogList));

        return notificationSummary;
    }

    @Deprecated
    public static NotificationSummary createMyProject(String locale) {
        NotificationSummary notificationSummary = new NotificationSummary();
        notificationSummary.setCategoryId(AthenaMobileNotificationCategoryEnum.MOBILE_MY_PROJECT.toString());
        notificationSummary.setCategoryName(
                LocaleUtil.getMobileTextByKey(locale, AthenaMobileNotificationCategoryEnum.MOBILE_MY_PROJECT.toString())
        );
        String content = LocaleUtil.getMobileTextByKey(locale, "ATHENA_MY_PROJECT_PROMPT");
        notificationSummary.setJumpPageId(PageSettingIdPresetEnum.MOBILE_ATHENA_PROJECT.toString());
        notificationSummary.setJumpPageTitle(
                LocaleUtil.getMobileTextByKey(locale, PageSettingIdPresetEnum.MOBILE_ATHENA_PROJECT.toString())
        );
        notificationSummary.setLatestNotification(new NotificationSummarySubject(content));

        return notificationSummary;
    }

    /**
     * 从Athena PC端内部通知中，创建移动端个人消息
     * 2021/11/5 因产品需求暂时关闭内部通知消息的显示，故标记为@Deprecated
     *
     * @param locale
     * @param commenter
     * @param digiwinAimMessages
     * @return
     */
    @Deprecated
    public static NotificationSummary createPersonal(String locale,
                                                     DigiwinAimMessageBoardCommenter commenter, List<DigiwinAimMessage> digiwinAimMessages) {
        NotificationSummary notificationSummary = new NotificationSummary();
        // 未读消息角标
        Long unreadCount = digiwinAimMessages.stream().filter(digiwinAimMessage ->
                        digiwinAimMessage.getState().equals(AthenaNotificationReadStateEnum.UNREAD.getState()))
                .count();
        notificationSummary.setBadge(new NotificationSummaryBadge(unreadCount.intValue(),
                BadgeRemoveRuleEnum.REMOVE_AFTER_TAP.getType(), BadgeShowTypeEnum.SHOW_IN_NUMBER.getType()));
        // 消息分类使用固定枚举+userId
        notificationSummary.setCategoryId(
                AthenaMobileNotificationCategoryEnum.MOBILE_INTERNAL_PERSONAL.toString() + commenter.getUserId());
        notificationSummary.setCategoryName(commenter.getUserName());
        notificationSummary.setJumpPageId(AthenaMobileNotificationCategoryEnum.MOBILE_INTERNAL_PERSONAL.toString());
        notificationSummary.setJumpPageTitle(commenter.getUserName());
        notificationSummary.setLatestNotification(NotificationSummarySubject.createPersonal(locale, digiwinAimMessages.get(0)));

        return notificationSummary;
    }

    public static NotificationSummary createAthenaWelcomeSummary(String locale) {
        NotificationSummary notificationSummary = new NotificationSummary();
        notificationSummary.setBadge(new NotificationSummaryBadge(
                0, BadgeRemoveRuleEnum.REMOVE_AFTER_TAP.getType(), BadgeShowTypeEnum.SHOW_IN_NUMBER.getType()));
        notificationSummary.setCategoryId(AthenaMobileNotificationCategoryEnum.MOBILE_ATHENA_ASSISTANT.toString());
        notificationSummary.setCategoryName(
                LocaleUtil.getMobileTextByKey(locale, AthenaMobileNotificationCategoryEnum.MOBILE_ATHENA_ASSISTANT.toString())
        );
        String content = LocaleUtil.getMobileTextByKey(locale, "WELCOME_TO_ATHENA");
        notificationSummary.setJumpPageId(notificationSummary.getCategoryId());
        notificationSummary.setJumpPageTitle(notificationSummary.getCategoryName());
        notificationSummary.setLatestNotification(new NotificationSummarySubject(content));

        return notificationSummary;
    }

    public static NotificationSummary create(String locale, DigiwinAimMessage aimMessageSummary, Integer unRead, LocaleService localeService) {
        NotificationSummary notificationSummary = new NotificationSummary();
        // 通知分类设置
        String pcSubTypeCategory = aimMessageSummary.getSubTypeCategory();
        // Athena助手 更名为 工作提醒 by wuyang20220318 其他所以通知全部放到工作提醒中
        notificationSummary.setCategoryId(AthenaMobileNotificationCategoryEnum.MOBILE_ATHENA_ASSISTANT.toString());
        notificationSummary.setCategoryName(
                LocaleUtil.getMobileTextByKey(locale, AthenaMobileNotificationCategoryEnum.MOBILE_ATHENA_ASSISTANT.toString()));
        notificationSummary.setBadge(NotificationSummaryBadge.create(aimMessageSummary, notificationSummary.getCategoryId(), unRead));
        notificationSummary.setJumpPageId(notificationSummary.getCategoryId());
        notificationSummary.setJumpPageTitle(notificationSummary.getCategoryName());
        notificationSummary.setLatestNotification(NotificationSummarySubject.create(locale, aimMessageSummary, localeService));

        return notificationSummary;
    }

    /**
     * 与另一个相同categoryId的通知摘要合并，如果categoryId不同，则不做处理，直接返回当前对象。
     * 合并需要做两个事：
     * 1（废弃，现已经不需要进行分类）. 角标数量相加（其他角标设置比如消除规则、展示规则不需要调整，因为分类相同，设置一定相同）
     * 2. 最新一条通知选择时间最新的
     * 处理完后，返回这个新的通知摘要对象。
     *
     * @param notificationSummary
     * @return
     */
    public NotificationSummary merge(final NotificationSummary notificationSummary) {
        if (!notificationSummary.getCategoryId().equals(this.getCategoryId())) {
            logger.debug("categoryIds are different, we will do noting and just return....");
            return this;
        }
        NotificationSummarySubject currentLatestNotification = this.getLatestNotification();
        LocalDateTime currentLatest = DateTimeUtil.parseUseDefaultPattern(currentLatestNotification.getPublishTime());
        LocalDateTime comparedLatest = DateTimeUtil.parseUseDefaultPattern(notificationSummary.getLatestNotification().getPublishTime());
        if (currentLatest.isBefore(comparedLatest)) {
            this.setLatestNotification(notificationSummary.getLatestNotification());
        }
        return this;
    }

    public NotificationSummary() {
    }

    public NotificationSummaryBadge getBadge() {
        return badge;
    }

    public void setBadge(NotificationSummaryBadge badge) {
        this.badge = badge;
    }

    public String getCategoryId() {
        return categoryId;
    }

    public void setCategoryId(String categoryId) {
        this.categoryId = categoryId;
    }

    public String getCategoryName() {
        return categoryName;
    }

    public void setCategoryName(String categoryName) {
        this.categoryName = categoryName;
    }

    public String getCategoryNameDesc() {
        return categoryNameDesc;
    }

    public void setCategoryNameDesc(String categoryNameDesc) {
        this.categoryNameDesc = categoryNameDesc;
    }

    public String getJumpPageId() {
        return jumpPageId;
    }

    public void setJumpPageId(String jumpPageId) {
        this.jumpPageId = jumpPageId;
    }

    public String getJumpPageTitle() {
        return jumpPageTitle;
    }

    public void setJumpPageTitle(String jumpPageTitle) {
        this.jumpPageTitle = jumpPageTitle;
    }

    public NotificationSummarySubject getLatestNotification() {
        return latestNotification;
    }

    public void setLatestNotification(NotificationSummarySubject latestNotification) {
        this.latestNotification = latestNotification;
    }
}
