package com.digiwin.mobile.mobileuibot.notification.model;

import com.digiwin.mobile.mobileuibot.common.localization.LocaleUtil;
import com.digiwin.mobile.mobileuibot.common.string.StringUtil;
import com.digiwin.mobile.mobileuibot.locale.service.LocaleService;
import com.digiwin.mobile.mobileuibot.notification.enums.AthenaMessageTypeEnum;
import com.digiwin.mobile.mobileuibot.notification.enums.AthenaNotificationTypeEnum;
import com.digiwin.mobile.mobileuibot.proxy.aim.model.DigiwinAimMessage;
import com.digiwin.mobile.mobileuibot.proxy.aim.model.DigiwinAimMessageContent;
import com.digiwin.mobile.mobileuibot.proxy.aim.model.DigiwinAimMessageSummary;
import com.digiwin.mobile.mobileuibot.proxy.atmc.model.DigiwinAtmcBacklog;
import com.digiwin.mobile.mobileuibot.task.common.TaskStringUtil;

import java.io.Serializable;
import java.util.List;

/**
 * <p>功能描述：通知摘要中的通知主体数据模型（主要展示最新一条通知的内容）</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: NotificationSummarySubject
 * @Author: Zaregoto
 * @Date: 2021/4/21 15:31
 */
public class NotificationSummarySubject implements Serializable {
    private static final long serialVersionUID = 829508695582178904L;

    private String content;
    private NotificationLabel label;
    private String publishTime;
    /**
     * DigiwinAimMessage的消息ID
     */
    private String gid;

    public static NotificationSummarySubject create(String locale, DigiwinAimMessageSummary digiwinAimMessageSummary) {
        NotificationSummarySubject summarySubject = new NotificationSummarySubject();
        // PC端消息中心的分类展示页面中，最新一条消息的标题是取的message.title，内容是message.content.msg
        summarySubject.setContent(NotificationSummarySubject.calculateWithTitleAndContent(
                digiwinAimMessageSummary.getMessage().getTitle(),
                digiwinAimMessageSummary.getMessage().getContent().getMsg()));
        summarySubject.setLabel(NotificationLabel.create(locale, digiwinAimMessageSummary));
        if (digiwinAimMessageSummary.getMessage().getContent().getStartTime().contains("T")) {
            summarySubject.setPublishTime(digiwinAimMessageSummary.getMessage().getCreateDate());
        } else {
            summarySubject.setPublishTime(digiwinAimMessageSummary.getMessage().getSendDate());
        }
        summarySubject.setGid(digiwinAimMessageSummary.getMessage().getGid());
        return summarySubject;
    }

    public static NotificationSummarySubject create(String locale, List<DigiwinAtmcBacklog> digiwinAtmcBacklogList) {
        NotificationSummarySubject summarySubject = new NotificationSummarySubject();
        /**
         * 2021年4月21日 第1次迭代。
         * 这次没有逾期任务的提醒，也没有新增任务的提醒，所以首页上的数字永远显示的是汇总信息。首页内容的文案：
         * 1.有逾期任务+有无今日任务，显示「其中包含N个逾期任务，请尽快处理💪」
         * 2.无逾期任务+有今日任务时，显示「请及时处理💪」
         * 3.无逾期任务+无今日任务时，不显示「今日待办」模块（这种情况不会用到本方法）
         *
         * 2021年6月10日 第3次迭代
         * 无任务时，也会展示「今日待办」模块
         */
        long overdueBacklogCount = digiwinAtmcBacklogList.stream().filter(
                digiwinAtmcBacklog -> digiwinAtmcBacklog.getOverdue().equals(true)).count();
        summarySubject.setContent(TaskStringUtil.getTaskInNotificationSummaryContent(locale, overdueBacklogCount));
        summarySubject.setLabel(null);
        summarySubject.setPublishTime("");

        return summarySubject;
    }

    /**
     * 根据最新一条DigiwinAimMessage生成个人通知消息摘要对象
     *
     * @param locale
     * @param digiwinAimMessage
     * @return
     */
    public static NotificationSummarySubject createPersonal(String locale, DigiwinAimMessage digiwinAimMessage) {
        NotificationSummarySubject summarySubject = new NotificationSummarySubject();
        summarySubject.setContent(digiwinAimMessage.getContent().getMessageBoard().get(0).getComment());
        summarySubject.setPublishTime(digiwinAimMessage.getSendDate());

        return summarySubject;
    }

    public static String calculateWithTitleAndContent(String title, String content) {
        // TODO 消息内容展示的规则考虑是否考虑再通用化，跳出当前这个类
        String result;
        if (title != null) {
            if (title.trim().length() > 0 && (content == null || content.trim().length() == 0)) {
                result = title;
            } else if (title.trim().length() == 0 && (content != null && content.trim().length() > 0)) {
                result = content;
            } else if (title.trim().length() > 0 && (content != null && content.trim().length() > 0)) {
                result = title + " " + content;
            } else {
                result = "";
            }
        } else {
            result = "";
        }

        return result;
    }

    public NotificationSummarySubject() {
    }

    public NotificationSummarySubject(String content) {
        this.content = content;
    }

    public NotificationSummarySubject(String content, NotificationLabel label, String publishTime) {
        this.content = content;
        this.label = label;
        this.publishTime = publishTime;
    }

    /**
     * 创建APP消息页上工作提醒分类对象，含最新一条消息
     * 该方法内逻辑要和以下方法保持一致：
     * com.digiwin.mobile.mobileuibot.notification.model.NotificationDetail#create(java.lang.String, com.digiwin.mobile.mobileuibot.proxy.aim.model.DigiwinAimMessage, com.digiwin.mobile.mobileuibot.notification.model.NotificationDetailPersonalCard, java.lang.String, com.digiwin.mobile.mobileuibot.proxy.atmc.model.DigiwinAtmcBacklogDetail, java.util.List, java.lang.String)
     *
     * @param locale            语言别
     * @param aimMessageSummary 平台AIM消息中心最新一条消息摘要
     * @param localeService
     * @return
     */
    public static NotificationSummarySubject create(String locale, DigiwinAimMessage aimMessageSummary, LocaleService localeService) {
        NotificationSummarySubject summarySubject = new NotificationSummarySubject();
        // 2023.04.04 增加 type=EOC 构建消息参数
        if (AthenaMessageTypeEnum.EOC.getType().equals(aimMessageSummary.getType())) {
            // EOC提醒
            summarySubject.setContent(localeService.getLanguageValue(locale, "云控制台尚有重要设定未完成，将导致雅典娜发成不可预期的异常，请关注~"));
        } else if (AthenaMessageTypeEnum.REMINDER.getType().equals(aimMessageSummary.getType())) {
            // 提醒
            String str = "";
            if (aimMessageSummary.getContent() != null) {
                DigiwinAimMessageContent content = aimMessageSummary.getContent();
                str = localeService.getLanguageValue(locale, "您的任务") + content.getTaskName()
                        + localeService.getLanguageValue(locale, "即将逾期") + "，"
                        + localeService.getLanguageValue(locale, "所属项目") + content.getProjectName()
                        + "，" + localeService.getLanguageValue(locale, "项目发起人") + content.getPersonInChargeName();
            }
            summarySubject.setContent(str);
        } else if (AthenaMessageTypeEnum.BASE_DATA_ENTRY.getType().equals(aimMessageSummary.getType())) {
            // 数据录入
            String str = "";
            if (aimMessageSummary.getContent() != null) {
                //异常数量
                Integer failedNum = aimMessageSummary.getContent().getFailedNum();
                //成功数量
                Integer succeededNumN = aimMessageSummary.getContent().getSucceededNum();
                //进行中数量
                Integer processingNum = aimMessageSummary.getContent().getProcessingNum();
                if (aimMessageSummary.getContent().getActivityName() != null) {
                    if ("en_US".equals(locale)) {
                        str = aimMessageSummary.getContent().getActivityName().getEnUSName();
                    } else if ("zh_TW".equals(locale)) {
                        str = aimMessageSummary.getContent().getActivityName().getZhTWName();
                    } else {
                        str = aimMessageSummary.getContent().getActivityName().getDefaultName();
                    }
                }
                if (AthenaNotificationTypeEnum.EXCEPTION.toString().equals(aimMessageSummary.getCategory())) {
                    str += " " + LocaleUtil.getMobileTextByKey(locale, "导入异常，总数") + (failedNum + succeededNumN + processingNum) + LocaleUtil.getMobileTextByKey(locale, "笔") + "," + LocaleUtil.getMobileTextByKey(locale, "异常") + failedNum + LocaleUtil.getMobileTextByKey(locale, "笔") + "。";
                } else if (AthenaNotificationTypeEnum.OTHER.toString().equals(aimMessageSummary.getCategory())) {
                    str += " " + LocaleUtil.getMobileTextByKey(locale, "导入成功，总数") + (failedNum + succeededNumN + processingNum) + LocaleUtil.getMobileTextByKey(locale, "笔") + "。";
                }
            }
            summarySubject.setContent(str);
        } else if (AthenaMessageTypeEnum.AGILE_DATA_DETAIL.getType().equals(aimMessageSummary.getType())) {
            // 敏捷数据
            summarySubject.setContent(NotificationSummarySubject.calculateWithTitleAndContent(
                    aimMessageSummary.getTitle(),
                    aimMessageSummary.getContent().getTitle()));
        } else if (AthenaMessageTypeEnum.TEXT.getType().equals(aimMessageSummary.getType())) {
            // 智能助理
            summarySubject.setContent(NotificationSummarySubject.calculateWithTitleAndContent(
                    aimMessageSummary.getTitle(),
                    aimMessageSummary.getContent().getMsg()));
        } else {
            // PC端消息中心的分类展示页面中，最新一条消息的标题是取的message.title，内容是message.content.msg
            String title = aimMessageSummary.getTitle();
            if (StringUtil.isEmpty(title)) {
                if (aimMessageSummary.getContent() != null) {
                    title = aimMessageSummary.getContent().getTitle();
                }
            }
            summarySubject.setContent(NotificationSummarySubject.calculateWithTitleAndContent(
                    title,
                    aimMessageSummary.getContent().getMsg()));
        }
        summarySubject.setLabel(NotificationLabel.create(locale, aimMessageSummary));
        if (aimMessageSummary.getContent() != null && aimMessageSummary.getContent().getStartTime() != null && aimMessageSummary.getContent().getStartTime().contains("T")) {
            summarySubject.setPublishTime(aimMessageSummary.getCreateDate());
        } else {
            summarySubject.setPublishTime(StringUtil.isEmpty(aimMessageSummary.getSendDate()) ? aimMessageSummary.getCreateDate() : aimMessageSummary.getSendDate());
        }
        summarySubject.setGid(aimMessageSummary.getGid());
        return summarySubject;
    }

    public String getContent() {
        return content;
    }

    public void setContent(String content) {
        this.content = content;
    }

    public NotificationLabel getLabel() {
        return label;
    }

    public void setLabel(NotificationLabel label) {
        this.label = label;
    }

    public String getPublishTime() {
        return publishTime;
    }

    public void setPublishTime(String publishTime) {
        this.publishTime = publishTime;
    }

    public String getGid() {
        return gid;
    }

    public void setGid(String gid) {
        this.gid = gid;
    }
}
