package com.digiwin.mobile.mobileuibot.notification.service.impl;

import com.alibaba.fastjson.JSONArray;
import com.digiwin.mobile.mobileuibot.common.calculate.UUIDUtil;
import com.digiwin.mobile.mobileuibot.common.context.AppContext;
import com.digiwin.mobile.mobileuibot.common.datetime.DateTimeUtil;
import com.digiwin.mobile.mobileuibot.common.exception.ServiceException;
import com.digiwin.mobile.mobileuibot.common.file.FileUtil;
import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.common.string.StringUtil;
import com.digiwin.mobile.mobileuibot.config.SysEnvConfig;
import com.digiwin.mobile.mobileuibot.locale.service.LocaleService;
import com.digiwin.mobile.mobileuibot.notification.enums.AthenaMobileNotificationCategoryEnum;
import com.digiwin.mobile.mobileuibot.notification.model.NotificationDetail;
import com.digiwin.mobile.mobileuibot.notification.model.NotificationRemoveResp;
import com.digiwin.mobile.mobileuibot.notification.model.NotificationSummary;
import com.digiwin.mobile.mobileuibot.notification.model.PushNotification;
import com.digiwin.mobile.mobileuibot.notification.service.NotificationService;
import com.digiwin.mobile.mobileuibot.notification.strategy.NotificationDetailStrategyFactory;
import com.digiwin.mobile.mobileuibot.proxy.aim.model.DigiwinAimMessage;
import com.digiwin.mobile.mobileuibot.proxy.aim.model.DigiwinAimNewMessageSummary;
import com.digiwin.mobile.mobileuibot.proxy.aim.service.DigiwinAimProxyService;
import com.digiwin.mobile.mobileuibot.push.jpush.PushConfig;
import com.digiwin.mobile.mobileuibot.push.jpush.model.Notification;
import com.digiwin.mobile.mobileuibot.push.jpush.service.PushSupplierEnable;
import lombok.SneakyThrows;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.io.IOException;
import java.sql.Timestamp;
import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;

/**
 * <p>功能描述：</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: NotificationServiceImpl
 * @Author: Zaregoto
 * @Date: 2021/4/21 19:10
 */
@Service("notificationService")
public class NotificationServiceImpl implements NotificationService {

    private static final Logger logger = LoggerFactory.getLogger(NotificationServiceImpl.class);

    /**
     * 消息已经读取
     */
    public static final Integer MESSAGE_READ = 1;
    /**
     * 消息未读取
     */
    public static final Integer MESSAGE_NOT_READ = 0;
    /**
     * 系统通知
     */
    public static final Integer NOTIFICATION_TYPE_SYSTEM = 1;
    public static final String CATEGORY_KEY_SYSTEM = "SYSTEM";

    @Autowired
    private DigiwinAimProxyService digiwinAimProxyService;

    @Autowired
    private PushSupplierEnable jPushService;

    @Autowired
    private SysEnvConfig sysEnvConfig;

    @Resource(name = "defaultThreadPool")
    private ThreadPoolTaskExecutor threadPoolTaskExecutor;

    @Autowired
    private NotificationDetailStrategyFactory notificationDetailStrategyFactory;

    @Autowired
    private LocaleService localeService;

    @Override
    public List<NotificationSummary> listNotificationSummaries(final String clientId,
                                                               final String iamUserToken, final String locale, Integer pageNum, Integer pageSize) throws IOException {

        List<NotificationSummary> resultSummaries;
        Boolean useMockData = AppContext.getUseMockData();
        if (!useMockData) {
            resultSummaries = this.doActually(clientId, iamUserToken, locale, pageNum, pageSize);
        } else {
            resultSummaries = this.doMockData(locale);
        }
        return resultSummaries;
    }

    @NotNull
    @SneakyThrows
    private List<NotificationSummary> doActually(String clientId, String iamUserToken,
                                                 String locale, Integer pageNum, Integer pageSize) {
        List<DigiwinAimMessage> aimMessageSummaries = null;
        Integer unRead = 0;

        //使用Future方式执行多任务
        List<Future<?>> futureList = new ArrayList<>();
        unRead = this.digiwinAimProxyService.countMessageUnRead(locale, iamUserToken, 0);
        Future<DigiwinAimNewMessageSummary> futureAimMessageSummaries = this.threadPoolTaskExecutor.submit(() ->
                this.digiwinAimProxyService.listMessageSummary(locale, iamUserToken, pageNum, pageSize));
        futureList.add(futureAimMessageSummaries);
        //查询任务执行的结果
        try {
            for (Future<?> future : futureList) {
                // CPU高速轮询：每个future都并发轮循，判断完成状态然后获取结果，
                // 这一行，是本实现方案的精髓所在。即有10个future在高速轮询，完成一个future的获取结果，就关闭一个轮询
                while (true) {
                    // 获取future成功完成状态，如果想要限制每个任务的超时时间，取消本行的状态判断+future.get(1000*1, TimeUnit.MILLISECONDS)+catch超时异常使用即可。
                    if (future.isDone() && !future.isCancelled()) {
                        //获取结果
                        Object obj = future.get();
                        if (obj instanceof DigiwinAimNewMessageSummary) {
                            aimMessageSummaries = ((DigiwinAimNewMessageSummary) obj).getContent();
                        }
                        break;//当前future获取结果完毕，跳出while
                    } else {
                        //每次轮询休息1毫秒（CPU纳秒级），避免CPU高速轮循耗空CPU---》新手别忘记这个
                        Thread.sleep(1);
                    }
                }
            }
        } catch (InterruptedException | ExecutionException e) {
            logger.error(e.getLocalizedMessage());
            e.printStackTrace();
            String message = e.getMessage();
            if (null != e.getCause()) {
                message = e.getCause().getMessage();
            }
            throw new ServiceException(message);
        } finally {

        }

        if (aimMessageSummaries == null) {
            aimMessageSummaries = Collections.emptyList();
        }

        List<NotificationSummary> resultSummaries = new ArrayList<>();
        /**
         * 1.根据规则，将PC端通知分类与最新消息先转换为移动端通知分类与最新消息
         * 2.合并相同分类：角标未读数量求和，并选择通知发送时间最新的
         * 3.依通知发送时间将已有分类倒序排列
         * 4.将内部通知转成多个人的个人通知
         * 2021/11/5 正式区，内部通知暂时关闭，首页不会出现这类新消息>备注：之前迭代2做过的PC发起一个通知留言，移动会收到这个消息；内部通知接入IM对话这个需求，PC端迭代21已提了该需求，待PC完成之后移动再对应处理
         */
        List<NotificationSummary> tempNotificationSummaries = new ArrayList<>();
        for (DigiwinAimMessage aimMessageSummary : aimMessageSummaries) {
            //由于行事历通知也放入到内部通知了，故为了保持通知条数一致性，内部通知打开
//            if (aimMessageSummary.equals(AthenaPcNotificationCategoryEnum.INTERNAL_REMIND.toString())) {
            // 内部通知暂时关闭。
//            } else {
            tempNotificationSummaries.add(NotificationSummary.create(locale, aimMessageSummary, unRead, localeService));
//            }
        }

        Map<String, Integer> categoryIdIndexRecord = new HashMap<>();
        for (NotificationSummary tempNotificationSummary : tempNotificationSummaries) {
            String categoryId = tempNotificationSummary.getCategoryId();
            if (!categoryIdIndexRecord.containsKey(categoryId)) {
                categoryIdIndexRecord.put(categoryId, resultSummaries.size());
                resultSummaries.add(tempNotificationSummary);
            } else {
                // 已经有相同分类ID的进行合并，合并操作在模型内实现
                Integer existedIndex = categoryIdIndexRecord.get(categoryId);
                NotificationSummary existedSummary = resultSummaries.get(existedIndex);
                existedSummary.merge(tempNotificationSummary);
            }
        }
        // 按通知送达最新时间倒序，并将系统通知放在最下方
        resultSummaries.sort((o1, o2) -> {
            LocalDateTime o1NotificationSendDateTime = DateTimeUtil.parseUseDefaultPattern(o1.getLatestNotification().getPublishTime());
            LocalDateTime o2NotificationSendDateTime = DateTimeUtil.parseUseDefaultPattern(o2.getLatestNotification().getPublishTime());
            if (o1NotificationSendDateTime.isBefore(o2NotificationSendDateTime)) {
                return 1;
            } else if (o1NotificationSendDateTime.isEqual(o2NotificationSendDateTime)) {
                return 0;
            } else {
                return -1;
            }
        });
        resultSummaries.sort((o1, o2) -> {
            String o1CategoryId = o1.getCategoryId();
            String o2CategoryId = o2.getCategoryId();
            if (o1CategoryId.equals(AthenaMobileNotificationCategoryEnum.MOBILE_SYSTEM.toString()) &&
                    !o2CategoryId.equals(AthenaMobileNotificationCategoryEnum.MOBILE_SYSTEM.toString())) {
                return 1;
            } else if (!o1CategoryId.equals(AthenaMobileNotificationCategoryEnum.MOBILE_SYSTEM.toString()) &&
                    o2CategoryId.equals(AthenaMobileNotificationCategoryEnum.MOBILE_SYSTEM.toString())) {
                return -1;
            } else {
                return 0;
            }
        });

        // 无任何消息时提供欢迎信息
        if (resultSummaries.isEmpty()) {
            resultSummaries.add(NotificationSummary.createAthenaWelcomeSummary(locale));
        }
        return resultSummaries;
    }

    private List<NotificationSummary> doMockData(String locale) {
        try {
            Thread.sleep(1000L);
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
        String data = FileUtil.readText("static/appstore-use/notification-sort-list.json");
        List<NotificationSummary> notificationSummaryList = Collections.emptyList();
        if (null != data && !data.isEmpty()) {
            notificationSummaryList = JSONArray.parseArray(data, NotificationSummary.class);
        }
        return notificationSummaryList;
    }

    @Override
    public List<NotificationDetail> listNotificationDetailsByCategoryId(final String userId, final String clientId,
                                                                        final String iamUserToken, final String tenantId,
                                                                        final String locale, final String mobileNotificationCategoryId,
                                                                        final String dataId, Integer pageNum, Integer pageSize) throws IOException {
        if (pageNum == null) {
            pageNum = 0;
        }
        if (mobileNotificationCategoryId.equals(AthenaMobileNotificationCategoryEnum.MOBILE_ATHENA_ASSISTANT.toString())
                || mobileNotificationCategoryId.equals(AthenaMobileNotificationCategoryEnum.MOBILE_SYSTEM.toString())) {
            return this.notificationDetailStrategyFactory.get("notificationDetailChattingStrategy")
                    .listNotificationDetails(userId, clientId, locale, iamUserToken, tenantId,
                            mobileNotificationCategoryId, dataId, pageNum, pageSize);
        } else {
            return null;
        }
    }

    @Override
    public Integer pushNotificationToUserId(String outerUserId, String outerTenantId, PushNotification push, String creatorName, List label) {
        logger.error("+++ pushNotificationToUserId,toUserId:" + outerUserId + ";toTenant:" +
                outerTenantId + ";pushNotification:" + (null == push ? "推送内容为空" :
                JsonUtil.javaObjectToJsonString(push)));
        if (push == null) {
            return 0;
        }
        Notification notification = buildNotification(outerUserId, outerTenantId, push, creatorName, label);
        //塞入mysql和mongodb数据

        //开始推送
        PushConfig pushConfig = new PushConfig();
        pushConfig.setType(sysEnvConfig.getType());
        pushConfig.setAppkey(sysEnvConfig.getJpushAppKey());
        pushConfig.setMasterSecret(sysEnvConfig.getJpushMasterSecret());
        pushConfig.setJpushAndroidIntentComponent(sysEnvConfig.getJpushAndroidIntentComponent());
        Boolean isSuccess = jPushService.pushAlert(notification, pushConfig);
        return isSuccess ? 1 : 0;
    }

    private Notification buildNotification(String outerUserId, String outerTenantId, PushNotification push,
                                           String creatorName, List label) {
        Notification notification = new Notification();
        String uuid = UUIDUtil.getUuid();
        notification.setNotificationID(uuid);
        notification.setUserId(outerUserId);
        notification.setTenantId(outerTenantId);
        notification.setNotificationTime(new Timestamp(System.currentTimeMillis()));
        notification.setIsRead(MESSAGE_NOT_READ);
        Map contentMap = new HashMap(8);
        contentMap.put("content", push.getContent());
        contentMap.put("frompeople", creatorName);
        contentMap.put("label", label);
        contentMap.put("title", push.getTitle());
        if (StringUtil.isNotEmpty(push.getUrl())) {
            String detailUrl = push.getUrl();
            contentMap.put("detailUrl", detailUrl);
        } else {
            contentMap.put("detailUrl", "");
        }
        if (push.getDetail() != null) {
            contentMap.put("detail", push.getDetail());
        }
        if (StringUtil.isNotEmpty(push.getTenantId())) {
            contentMap.put("tenantId", push.getTenantId());
        }
        contentMap.put("apnsProduction", push.getApnsProduction());
        notification.setNotificationContent(JsonUtil.javaObjectToJsonString(contentMap));
        notification.setNotificationType(NOTIFICATION_TYPE_SYSTEM);
        notification.setTitle(push.getTitle());
        notification.setSourceType(CATEGORY_KEY_SYSTEM);
        notification.setContentID(uuid);
        notification.setMobileType(0);
        notification.setApnsProduction(true);
        return notification;
    }


    @Override
    public NotificationRemoveResp oneClickRemove() {
        NotificationRemoveResp resp = new NotificationRemoveResp();
        Integer removeNum = digiwinAimProxyService.oneClickRemove();
        resp.setRemoveNum(removeNum);
        if (removeNum == null || removeNum == 0) {
        resp.setMessage(localeService.getLanguageValue("暂无可清除的工作提醒"));
        } else if (removeNum < 1000) {
            resp.setMessage(localeService.getLanguageValue("所有消息清除完成"));
        } else {
            resp.setMessage(localeService.getLanguageValue("本次完成清除1000条"));
        }
        return resp;
    }
}
