package com.digiwin.mobile.mobileuibot.openapi.service.impl;

import com.digiwin.mobile.mobileuibot.core.pagetaskrelation.PageTaskRelation;
import com.digiwin.mobile.mobileuibot.core.pagetaskrelation.PageTaskRelationService;
import com.digiwin.mobile.mobileuibot.mongodb.designerconfig.MobileDesignerConfig;
import com.digiwin.mobile.mobileuibot.mongodb.designerconfig.MobileDesignerConfigRepository;
import com.digiwin.mobile.mobileuibot.mongodb.designerconfig.common.MobileDesignerConfigTypeEnum;
import com.digiwin.mobile.mobileuibot.mongodb.uc.*;
import com.digiwin.mobile.mobileuibot.openapi.designer.enums.DataVersionEnum;
import com.digiwin.mobile.mobileuibot.openapi.service.ActivityFilterService;
import com.digiwin.mobile.mobileuibot.proxy.knowledgemaps.service.DigiwinKnowledgeMapsProxyService;
import com.digiwin.mobile.mobileuibot.task.filter.TaskFilter;
import com.digiwin.mobile.mobileuibot.task.filter.TaskFilterRepository;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.BooleanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * <p>功能描述：设计器作业过滤服务接口实现类</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: ActivityFilterServiceImpl.java
 * @Author: wangjwc
 * @Date: created at 2025/2/12 11:41
 */
@Component
@Slf4j
public class ActivityFilterServiceImpl implements ActivityFilterService {

    @Autowired
    private TaskFilterRepository taskFilterRepository;
    @Autowired
    private PageTaskRelationService pageTaskRelationService;
    @Autowired
    private MobileDesignerConfigRepository mobileDesignerConfigRepository;

    @Autowired
    private UcTaskFilterRepository ucTaskFilterRepository;
    @Autowired
    private UcPageTaskRelationRepository ucPageTaskRelationRepository;
    @Autowired
    private UcMobileDesignerConfigRepository ucMobileDesignerConfigRepository;
    @Autowired
    private UcTenantRelationRepository ucTenantRelationRepository;

    @Autowired
    private DigiwinKnowledgeMapsProxyService digiwinKnowledgeMapsProxyService;

    @Override
    public String tenantVersion(String iamUserToken, String tenantId, String locale) {
        String tenantVersion = digiwinKnowledgeMapsProxyService.tenantVersion(iamUserToken, tenantId, locale);
        // 租户版本默认为V2.0版本，防止接口异常，正式客户返回V1.0版本数据
        if (StringUtils.hasLength(tenantVersion)) {
            if (DataVersionEnum.V1_0.getVersion().equals(tenantVersion)) {
                tenantVersion = DataVersionEnum.V1_0.getVersion();
            } else {
                tenantVersion = DataVersionEnum.V2_0.getVersion();
            }
        } else {
            tenantVersion = DataVersionEnum.V2_0.getVersion();
        }
        return tenantVersion;
    }

    @Override
    public List<TaskFilter> listValidInAthenaOfTaskFilter(String tenantId, String version) {
        // step 1: 确认个案下uc_tenantRelation表对应version版本的数据是否存在
        List<UcTenantRelation> tenantRelations = ucTenantRelationRepository.selectByTenantId(tenantId, version);
        if (CollectionUtils.isEmpty(tenantRelations)) {
            // 不存在：标准租户
            // step 2: 查询标准下任务的taskFilter表对应version版本的数据
            return taskFilterRepository.listValidInAthena(version);
        } else {
            // 存在：个案租户
            // step 3: 因已经是个案租户(一个租户只能在同一个应用的一个个案中),过滤individualAll=true(是否开启任务定制)
            List<UcTenantRelation> openTaskCustom = tenantRelations.stream().filter(r -> BooleanUtils.isTrue(r.getIndividualAll())).collect(Collectors.toList());
            // step 4: 获取开启任务定制的个案下应用code列表
            Set<String> codeList = openTaskCustom.stream().map(UcTenantRelation::getAppCode).collect(Collectors.toSet());
            Set<String> ucCodeList = openTaskCustom.stream().map(UcTenantRelation::getUcAppCode).collect(Collectors.toSet());
            List<TaskFilter> ucTaskFilters = new ArrayList<>();
            if (!CollectionUtils.isEmpty(ucCodeList)) {
                // 开启任务定制
                // step 5: 查询个案下任务此ucCodeList下uc_taskFilter表对应version版本的数据
                ucTaskFilters = new ArrayList<>(ucTaskFilterRepository.listValidInAthena(version, ucCodeList));
            }
            // step 6: 查询标准下任务非此codeList下taskFilter表对应version版本的数据
            List<TaskFilter> stdTaskFilters = taskFilterRepository.listValidInAthena(version).stream()
                    .filter(taskFilter -> !codeList.contains(taskFilter.getAppCode())).collect(Collectors.toList());
            // step 7: 合并两个列表并返回
            return Stream.concat(stdTaskFilters.stream(), ucTaskFilters.stream()).collect(Collectors.toList());
        }
    }

    @Override
    public List<PageTaskRelation> listAthenaOfPageTaskRelations(String tenantId, String version) {
        // step 1: 确认个案下uc_tenantRelation表对应version版本的数据是否存在
        List<UcTenantRelation> tenantRelations = ucTenantRelationRepository.selectByTenantId(tenantId, version);
        if (CollectionUtils.isEmpty(tenantRelations)) {
            // 不存在：标准租户
            // step 2: 查询标准下任务的pageTaskRelation表对应version版本的数据
            return pageTaskRelationService.listAthenaPageTaskRelations(version);
        } else {
            // 存在：个案租户
            // step 3: 因已经是个案租户(一个租户只能在同一个应用的一个个案中),过滤individualAll=true(是否开启任务定制)
            List<UcTenantRelation> openTaskCustom = tenantRelations.stream().filter(r -> BooleanUtils.isTrue(r.getIndividualAll())).collect(Collectors.toList());
            // step 4: 获取开启任务定制的个案下应用code列表
            Set<String> codeList = openTaskCustom.stream().map(UcTenantRelation::getAppCode).collect(Collectors.toSet());
            Set<String> ucCodeList = openTaskCustom.stream().map(UcTenantRelation::getUcAppCode).collect(Collectors.toSet());
            List<PageTaskRelation> ucPageTaskRelations = new ArrayList<>();
            if (!CollectionUtils.isEmpty(ucCodeList)) {
                // 开启任务定制
                // step 5: 查询个案下任务此ucCodeList下uc_pageTaskRelation表对应version版本的数据
                ucPageTaskRelations = new ArrayList<>(ucPageTaskRelationRepository.listAthenaPageTaskRelations(version, ucCodeList));
            }
            // step 6: 查询标准下任务非此codeList下pageTaskRelation表对应version版本的数据
            List<PageTaskRelation> stdPageTaskRelations = pageTaskRelationService.listAthenaPageTaskRelations(version).stream()
                    .filter(pageTaskRelation -> !codeList.contains(pageTaskRelation.getAppCode())).collect(Collectors.toList());
            // step 7: 合并两个列表并返回
            return Stream.concat(stdPageTaskRelations.stream(), ucPageTaskRelations.stream()).collect(Collectors.toList());
        }
    }

    @Override
    public List<MobileDesignerConfig> findByTypeOfMobileDesignerConfig(String type, String tenantId, String version) {
        // step 1: 确认个案下uc_tenantRelation表对应version版本的数据是否存在
        List<UcTenantRelation> tenantRelations = ucTenantRelationRepository.selectByTenantId(tenantId, version);
        if (CollectionUtils.isEmpty(tenantRelations)) {
            // 不存在：标准租户
            // step 2: 查询标准下type类型的mobileDesignerConfig表对应version版本的数据
            return mobileDesignerConfigRepository.findByType(type, version, null);
        } else {
            // 存在：个案租户
            // step 2: 不同type类型处理逻辑
            if (MobileDesignerConfigTypeEnum.TASK.getValue().equals(type)
                    || MobileDesignerConfigTypeEnum.PROJECT.getValue().equals(type)
                    || MobileDesignerConfigTypeEnum.START_PROJECT.getValue().equals(type)) {
                // 任务/项目/发起项目类型
                // step 3: 因已经是个案租户(一个租户只能在同一个应用的一个个案中),过滤individualAll=true(是否开启任务定制)
                List<UcTenantRelation> openTaskCustom = tenantRelations.stream().filter(r -> BooleanUtils.isTrue(r.getIndividualAll())).collect(Collectors.toList());
                // step 4: 获取开启任务定制的个案下应用code列表
                Set<String> codeList = openTaskCustom.stream().map(UcTenantRelation::getAppCode).collect(Collectors.toSet());
                Set<String> ucCodeList = openTaskCustom.stream().map(UcTenantRelation::getUcAppCode).collect(Collectors.toSet());
                List<MobileDesignerConfig> ucMobileDesignerConfigs = new ArrayList<>();
                if (!CollectionUtils.isEmpty(ucCodeList)) {
                    // 开启任务定制
                    // step 5: 查询个案下任务/项目/发起项目类型此ucCodeList下uc_mobileDesignerConfig表对应version版本的数据
                    ucMobileDesignerConfigs = new ArrayList<>(ucMobileDesignerConfigRepository.findByType(type, version, ucCodeList));
                }
                // step 6: 查询标准下任务/项目/发起项目类型非此codeList下mobileDesignerConfig表对应version版本的数据
                List<MobileDesignerConfig> stdMobileDesignerConfigs = mobileDesignerConfigRepository.findByType(type, version, codeList);
                // step 7: 合并两个列表并返回
                return Stream.concat(stdMobileDesignerConfigs.stream(), ucMobileDesignerConfigs.stream()).collect(Collectors.toList());
            } else if (MobileDesignerConfigTypeEnum.DATA_ENTRY.getValue().equals(type)) {
                // 数据录入类型 -- 个案ucAppCode下作业与标准下作业取并集
                // step 3: 查询标准下数据录入类型的mobileDesignerConfig表对应version版本的数据
                List<MobileDesignerConfig> stdMobileDesignerConfigs = mobileDesignerConfigRepository.findByType(type, version, null);
                // step 4: 查询个案下数据录入类型此ucCodeList下uc_mobileDesignerConfig表对应version版本的数据
                Set<String> ucCodeList = tenantRelations.stream().map(UcTenantRelation::getUcAppCode).collect(Collectors.toSet());
                List<MobileDesignerConfig> ucMobileDesignerConfigs = new ArrayList<>(ucMobileDesignerConfigRepository.findByType(type, version, ucCodeList));
                // step 5: 合并两个列表并返回
                return Stream.concat(stdMobileDesignerConfigs.stream(), ucMobileDesignerConfigs.stream()).collect(Collectors.toList());
            } else {
                // 其他类型
                return new ArrayList<>();
            }
        }
    }

    @Override
    public Boolean isExsitTmActivityIdOfMobileDesignerConfig(String appCode, String tmActivityId, String type, String tenantId, String version) {
        // step 1: 确认个案下uc_tenantRelation表对应version版本的数据是否存在
        List<UcTenantRelation> tenantRelations = ucTenantRelationRepository.selectByTenantId(tenantId, version);
        if (CollectionUtils.isEmpty(tenantRelations)) {
            // 不存在：标准租户
            // step 2: 说明作业使用标准表配置, 查询标准下type类型对应version版本下数据是否存在
            return mobileDesignerConfigRepository.isExsitTmActivityId(tmActivityId, type, version);
        } else {
            // 存在：个案租户
            // step 2: 不同type类型处理逻辑
            if (MobileDesignerConfigTypeEnum.TASK.getValue().equals(type)
                    || MobileDesignerConfigTypeEnum.PROJECT.getValue().equals(type)
                    || MobileDesignerConfigTypeEnum.START_PROJECT.getValue().equals(type)) {
                // 任务/项目/发起项目类型
                // step 3: 因已经是个案租户(一个租户只能在同一个应用的一个个案中),查询对应个案应用配置
                UcTenantRelation ucTenantRelation = tenantRelations.stream().filter(t -> Objects.equals(t.getAppCode(), appCode)).findFirst().orElse(null);
                // step 4: 判断individualAll=true(是否开启任务定制)
                if (ucTenantRelation != null && BooleanUtils.isTrue(ucTenantRelation.getIndividualAll())) {
                    // 开启任务定制
                    // step 5: 说明作业使用个案表配置,查询个案下任务/项目/发起项目类型此ucCodeList下对应version版本下数据是否存在
                    return ucMobileDesignerConfigRepository.isExsitTmActivityIdOfUcAppCode(tmActivityId, type, version, Sets.newHashSet(ucTenantRelation.getUcAppCode()));
                } else {
                    // 关闭任务定制
                    // step 5: 说明作业使用标准表配置,查询标准下任务/项目/发起项目类型对应version版本下数据是否存在
                    return mobileDesignerConfigRepository.isExsitTmActivityId(tmActivityId, type, version);
                }
            } else if (MobileDesignerConfigTypeEnum.DATA_ENTRY.getValue().equals(type)) {
                // 数据录入类型
                // step 3: 因已经是个案租户(一个租户只能在同一个应用的一个个案中),查询对应个案应用code
                String ucAppCode = tenantRelations.stream().filter(t -> Objects.equals(t.getAppCode(), appCode)).findFirst().map(UcTenantRelation::getUcAppCode).orElse("");
                if (StringUtils.hasLength(ucAppCode)) {
                    // step 4: 存在个案，故先查个案表，再查标准表
                    Boolean uc = ucMobileDesignerConfigRepository.isExsitTmActivityIdOfUcAppCode(tmActivityId, type, version, Sets.newHashSet(ucAppCode));
                    if (BooleanUtils.isTrue(uc)) {
                        return uc;
                    }
                    return mobileDesignerConfigRepository.isExsitTmActivityId(tmActivityId, type, version);
                }
                // step 4: 不存在个案，故只查标准表
                return mobileDesignerConfigRepository.isExsitTmActivityId(tmActivityId, type, version);
            } else {
                // 其他类型
                return false;
            }
        }
    }

    @Override
    public MobileDesignerConfig selectOneData(String appCode, String tmActivityId, String type, String tenantId, String version) {
        // step 1: 确认个案下uc_tenantRelation表对应version版本的数据是否存在
        List<UcTenantRelation> tenantRelations = ucTenantRelationRepository.selectByTenantId(tenantId, version);
        if (CollectionUtils.isEmpty(tenantRelations)) {
            // 不存在：标准租户
            // step 2: 说明作业使用标准表配置
            return mobileDesignerConfigRepository.selectOneDsl(tmActivityId, type);
        } else {
            // 存在：个案租户
            // step 2: 不同type类型处理逻辑
            if (MobileDesignerConfigTypeEnum.TASK.getValue().equals(type)
                    || MobileDesignerConfigTypeEnum.PROJECT.getValue().equals(type)
                    || MobileDesignerConfigTypeEnum.START_PROJECT.getValue().equals(type)) {
                // 任务/项目/发起项目类型
                // step 3: 因已经是个案租户(一个租户只能在同一个应用的一个个案中),查询对应个案应用配置
                UcTenantRelation ucTenantRelation = tenantRelations.stream().filter(t -> Objects.equals(t.getAppCode(), appCode)).findFirst().orElse(null);
                // step 4: 判断individualAll=true(是否开启任务定制)
                if (ucTenantRelation != null && BooleanUtils.isTrue(ucTenantRelation.getIndividualAll())) {
                    // 开启任务定制
                    // step 5: 说明作业使用个案表配置
                    return ucMobileDesignerConfigRepository.selectOneDsl(tmActivityId, type, Sets.newHashSet(ucTenantRelation.getUcAppCode()));
                } else {
                    // 关闭任务定制
                    // step 5: 说明作业使用标准表配置
                    return mobileDesignerConfigRepository.selectOneDsl(tmActivityId, type);
                }
            } else if (MobileDesignerConfigTypeEnum.DATA_ENTRY.getValue().equals(type)) {
                // 数据录入类型
                // step 3: 因已经是个案租户(一个租户只能在同一个应用的一个个案中),查询对应个案应用code
                String ucAppCode = tenantRelations.stream().filter(t -> Objects.equals(t.getAppCode(), appCode)).findFirst().map(UcTenantRelation::getUcAppCode).orElse("");
                if (StringUtils.hasLength(ucAppCode)) {
                    // step 4: 存在个案，故先查个案表，再查标准表
                    UcMobileDesignerConfig ucMobileDesignerConfig = ucMobileDesignerConfigRepository.selectOneDsl(tmActivityId, type, Sets.newHashSet(ucAppCode));
                    if (ucMobileDesignerConfig != null) {
                        return ucMobileDesignerConfig;
                    }
                    return mobileDesignerConfigRepository.selectOneDsl(tmActivityId, type);
                }
                // step 4: 不存在个案，故只查标准表
                return mobileDesignerConfigRepository.selectOneDsl(tmActivityId, type);
            } else {
                // 其他类型
                return null;
            }
        }
    }
}
