package com.digiwin.mobile.mobileuibot.openapi.service.impl;

import com.digiwin.mobile.mobileuibot.common.context.AppContext;
import com.digiwin.mobile.mobileuibot.common.context.AppEnvTypeEnum;
import com.digiwin.mobile.mobileuibot.common.datetime.DateTimeUtil;
import com.digiwin.mobile.mobileuibot.common.string.StringUtil;
import com.digiwin.mobile.mobileuibot.core.common.EnvScope;
import com.digiwin.mobile.mobileuibot.core.pagesetting.PageSettingIdPresetEnum;
import com.digiwin.mobile.mobileuibot.core.pagetaskrelation.PageTaskRelation;
import com.digiwin.mobile.mobileuibot.core.pagetaskrelation.PageTaskRelationCategoryEnum;
import com.digiwin.mobile.mobileuibot.core.pagetaskrelation.PageTaskRelationService;
import com.digiwin.mobile.mobileuibot.mongodb.designerconfig.MobileDesignerConfig;
import com.digiwin.mobile.mobileuibot.mongodb.designerconfig.MobileDesignerConfigRepository;
import com.digiwin.mobile.mobileuibot.mongodb.designerconfig.common.MobileDesignerConfigTypeEnum;
import com.digiwin.mobile.mobileuibot.mongodb.uc.*;
import com.digiwin.mobile.mobileuibot.openapi.designer.enums.AppPublishOperateEnum;
import com.digiwin.mobile.mobileuibot.openapi.designer.enums.DataVersionEnum;
import com.digiwin.mobile.mobileuibot.openapi.designer.request.AppPublishVO;
import com.digiwin.mobile.mobileuibot.openapi.service.OpenApiService;
import com.digiwin.mobile.mobileuibot.task.filter.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.BooleanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * <p>功能描述：openapi 服务接口实现类</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: OpenApiServiceImpl.java
 * @Author: wangjwc
 * @Date: created at 2025/2/7 18:02
 */
@Component
@Slf4j
public class OpenApiServiceImpl implements OpenApiService {

    @Autowired
    private TaskFilterRepository taskFilterRepository;
    @Autowired
    private PageTaskRelationService pageTaskRelationService;
    @Autowired
    private MobileDesignerConfigRepository mobileDesignerConfigRepository;

    @Autowired
    private UcTaskFilterRepository ucTaskFilterRepository;
    @Autowired
    private UcPageTaskRelationRepository ucPageTaskRelationRepository;
    @Autowired
    private UcMobileDesignerConfigRepository ucMobileDesignerConfigRepository;
    @Autowired
    private UcTenantRelationRepository ucTenantRelationRepository;

    /**
     * 应用发版/切板
     *
     * @param publishVO 发布/切板参数
     */
    @Override
    public void publishOrSwitch(AppPublishVO publishVO) {
        if (AppPublishOperateEnum.PUBLISH.getOperate().equals(publishVO.getOperate())) {
            // 发版
            this.publish(publishVO);
        } else if (AppPublishOperateEnum.SWITCH.getOperate().equals(publishVO.getOperate())) {
            // 切版
            this.switchover(publishVO);
        } else {
            // 不支持的操作
            log.warn("Unsupported operation: {}", publishVO.getOperate());
            return;
        }
    }

    /**
     * 发版操作
     * @param publishVO 发版参数
     */
    private void publish(AppPublishVO publishVO) {
        if (BooleanUtils.isTrue(publishVO.getIndividualCase())) {
            // 个案发版
            this.ucPublish(publishVO);
        } else {
            // 标准发版
            this.stdPublish(publishVO);
        }
    }

    /**
     * 标准发版
     * @param publishVO 发版参数
     */
    private void stdPublish(AppPublishVO publishVO) {
        // step1: appCode 删除version=1.0版本作业
        this.deleteStdVersion(publishVO.getAppCode(), DataVersionEnum.V1_0.getVersion());

        // step2: taskList/projectList/startProjectList/dataEntryList 插入version=1.0版本作业
        // step3: appType区分 数据录入/模型驱动 -- 废弃
        String time = DateTimeUtil.getTodayTimeUseDefaultPattern();
        List<TaskFilter> taskFilters = new ArrayList<>();
        List<PageTaskRelation> pageTaskRelations = new ArrayList<>();
        List<MobileDesignerConfig> mobileDesignerConfigs = new ArrayList<>();
        publishVO.getTaskList().forEach(task -> {
            taskFilters.add(this.buildStdTaskFilter(new TaskFilter(), publishVO.getAppCode(), task.getTmActivityId(), time, DataVersionEnum.V1_0.getVersion()));
            pageTaskRelations.add(this.buildStdPageTaskRelation(new PageTaskRelation(), publishVO.getAppCode(), task.getTmActivityId(), time, DataVersionEnum.V1_0.getVersion()));
            mobileDesignerConfigs.add(this.buildStdMobileDesignerConfig(new MobileDesignerConfig(), publishVO.getAppCode(), task.getTmActivityId(), MobileDesignerConfigTypeEnum.TASK.getValue(), time, DataVersionEnum.V1_0.getVersion()));
        });
        publishVO.getProjectList().forEach(project -> mobileDesignerConfigs.add(this.buildStdMobileDesignerConfig(new MobileDesignerConfig(), publishVO.getAppCode(), project.getTmActivityId(), MobileDesignerConfigTypeEnum.PROJECT.getValue(), time, DataVersionEnum.V1_0.getVersion())));
        publishVO.getStartProjectList().forEach(startProject -> mobileDesignerConfigs.add(this.buildStdMobileDesignerConfig(new MobileDesignerConfig(), publishVO.getAppCode(), startProject.getTmActivityId(), MobileDesignerConfigTypeEnum.START_PROJECT.getValue(), time, DataVersionEnum.V1_0.getVersion())));
        // wjw TODO: 2025/2/8 数据录入类型全部统一为DATAENTRY
        publishVO.getDataEntryList().forEach(dataEntry -> mobileDesignerConfigs.add(this.buildStdMobileDesignerConfig(new MobileDesignerConfig(), publishVO.getAppCode(), dataEntry.getTmActivityId(), MobileDesignerConfigTypeEnum.DATA_ENTRY.getValue(), time, DataVersionEnum.V1_0.getVersion())));
        // step4: 批量插入数据
        this.batchInsertStdVersion(taskFilters, pageTaskRelations, mobileDesignerConfigs, DataVersionEnum.V1_0.getVersion());
    }

    /**
     * 个案发版
     * @param publishVO 切板参数
     */
    private void ucPublish(AppPublishVO publishVO) {
        // step1: appCode、ucAppCode 删除version=1.0版本作业
        this.deleteUcVersion(publishVO.getAppCode(), publishVO.getUcAppCode(), DataVersionEnum.V1_0.getVersion());

        // step2: 判断tenantIdList=空(个案租户为空表示删除所有作业，不重新插入作业)--return
        if (CollectionUtils.isEmpty(publishVO.getTenantIdList())) {
            log.info("Tenant ID list is empty, skipping insertion.");
            return;
        }
        // step3: 判断individualAll=true(是否开启任务定制)
        String time = DateTimeUtil.getTodayTimeUseDefaultPattern();
        List<UcTaskFilter> taskFilters = new ArrayList<>();
        List<UcPageTaskRelation> pageTaskRelations = new ArrayList<>();
        List<UcMobileDesignerConfig> mobileDesignerConfigs = new ArrayList<>();
        List<UcTenantRelation> tenantRelations = new ArrayList<>();
        if (BooleanUtils.isTrue(publishVO.getIndividualAll())) {
            // 开启任务定制
            // step4: taskList/projectList/startProjectList 插入version=1.0版本作业
            publishVO.getTaskList().forEach(task -> {
                taskFilters.add(this.buildUcTaskFilter(publishVO.getUcAppCode(), publishVO.getAppCode(), task.getTmActivityId(), time, DataVersionEnum.V1_0.getVersion()));
                pageTaskRelations.add(this.buildUcPageTaskRelation(publishVO.getUcAppCode(), publishVO.getAppCode(), task.getTmActivityId(), time, DataVersionEnum.V1_0.getVersion()));
                mobileDesignerConfigs.add(this.buildUcMobileDesignerConfig(publishVO.getUcAppCode(), publishVO.getAppCode(), task.getTmActivityId(), MobileDesignerConfigTypeEnum.TASK.getValue(), time, DataVersionEnum.V1_0.getVersion()));
            });
            publishVO.getProjectList().forEach(project -> mobileDesignerConfigs.add(this.buildUcMobileDesignerConfig(publishVO.getUcAppCode(), publishVO.getAppCode(), project.getTmActivityId(), MobileDesignerConfigTypeEnum.PROJECT.getValue(), time, DataVersionEnum.V1_0.getVersion())));
            publishVO.getStartProjectList().forEach(startProject -> mobileDesignerConfigs.add(this.buildUcMobileDesignerConfig(publishVO.getUcAppCode(), publishVO.getAppCode(), startProject.getTmActivityId(), MobileDesignerConfigTypeEnum.START_PROJECT.getValue(), time, DataVersionEnum.V1_0.getVersion())));
        }
        // step5: dataEntryList 插入version=1.0版本作业
        // wjw TODO: 2025/2/8 数据录入类型全部统一为DATAENTRY
        publishVO.getDataEntryList().forEach(dataEntry -> mobileDesignerConfigs.add(this.buildUcMobileDesignerConfig(publishVO.getUcAppCode(), publishVO.getAppCode(), dataEntry.getTmActivityId(), MobileDesignerConfigTypeEnum.DATA_ENTRY.getValue(), time, DataVersionEnum.V1_0.getVersion())));
        // step6: appType区分 数据录入/模型驱动 -- 废弃
        // step7: tenantIdList 插入个案租户表
        publishVO.getTenantIdList().forEach(tenantId -> tenantRelations.add(this.buildUcTenantRelation(publishVO.getUcAppCode(), publishVO.getAppCode(), tenantId, publishVO.getIndividualAll(), time, DataVersionEnum.V1_0.getVersion())));
        // step8: 批量插入数据
        this.batchInsertUcVersion(taskFilters, pageTaskRelations, mobileDesignerConfigs, tenantRelations);
    }

    /**
     * 切板操作
     * @param publishVO 切板参数
     */
    private void switchover(AppPublishVO publishVO) {
        if (BooleanUtils.isTrue(publishVO.getIndividualCase())) {
            // 个案切版
            this.ucSwitch(publishVO);
        } else {
            // 标准切版
            this.stdSwitch(publishVO);
        }
    }

    /**
     * 标准切板
     * @param publishVO 切板参数
     */
    private void stdSwitch(AppPublishVO publishVO) {
        // step1: appCode 删除version=2.0版本作业
        this.deleteStdVersion(publishVO.getAppCode(), DataVersionEnum.V2_0.getVersion());

        // step2: appCode 获取version=1.0版本作业复制插入version=2.0版本作业
        List<TaskFilter> taskFilters = taskFilterRepository.selectByCode(publishVO.getAppCode(), DataVersionEnum.V1_0.getVersion());
        List<PageTaskRelation> pageTaskRelations = pageTaskRelationService.selectByCode(publishVO.getAppCode(), DataVersionEnum.V1_0.getVersion());
        List<MobileDesignerConfig> mobileDesignerConfigs = mobileDesignerConfigRepository.selectByCode(publishVO.getAppCode(), DataVersionEnum.V1_0.getVersion());
        String time = DateTimeUtil.getTodayTimeUseDefaultPattern();
        taskFilters.forEach(taskFilter -> {
            taskFilter.setId(null);
            taskFilter.setCreateTime(time);
            taskFilter.setUpdateTime(time);
            taskFilter.setVersion(DataVersionEnum.V2_0.getVersion());
        });
        pageTaskRelations.forEach(pageTaskRelation -> {
            pageTaskRelation.setId(null);
            pageTaskRelation.setCreateTime(time);
            pageTaskRelation.setUpdateTime(time);
            pageTaskRelation.setVersion(DataVersionEnum.V2_0.getVersion());
        });
        mobileDesignerConfigs.forEach(mobileDesignerConfig -> {
            mobileDesignerConfig.setId(null);
            mobileDesignerConfig.setCreateTime(time);
            mobileDesignerConfig.setUpdateTime(time);
            mobileDesignerConfig.setVersion(DataVersionEnum.V2_0.getVersion());
        });
        // step3: 批量插入数据
        this.batchInsertStdVersion(taskFilters, pageTaskRelations, mobileDesignerConfigs, DataVersionEnum.V2_0.getVersion());
    }

    /**
     * 个案切板
     * @param publishVO 切板参数
     */
    private void ucSwitch(AppPublishVO publishVO) {
        // step1: appCode、ucAppCode 删除version=2.0版本作业
        this.deleteUcVersion(publishVO.getAppCode(), publishVO.getUcAppCode(), DataVersionEnum.V2_0.getVersion());

        // step2: appCode、ucAppCode 获取version=1.0版本作业复制插入version=2.0版本作业
        List<UcTaskFilter> taskFilters = ucTaskFilterRepository.selectByCode(publishVO.getAppCode(), publishVO.getUcAppCode(), DataVersionEnum.V1_0.getVersion());
        List<UcPageTaskRelation> pageTaskRelations = ucPageTaskRelationRepository.selectByCode(publishVO.getAppCode(), publishVO.getUcAppCode(), DataVersionEnum.V1_0.getVersion());
        List<UcMobileDesignerConfig> mobileDesignerConfigs = ucMobileDesignerConfigRepository.selectByCode(publishVO.getAppCode(), publishVO.getUcAppCode(), DataVersionEnum.V1_0.getVersion());
        List<UcTenantRelation> tenantRelations = ucTenantRelationRepository.selectByCode(publishVO.getAppCode(), publishVO.getUcAppCode(), DataVersionEnum.V1_0.getVersion());
        String time = DateTimeUtil.getTodayTimeUseDefaultPattern();
        taskFilters.forEach(taskFilter -> {
            taskFilter.setId(null);
            taskFilter.setCreateTime(time);
            taskFilter.setUpdateTime(time);
            taskFilter.setVersion(DataVersionEnum.V2_0.getVersion());
        });
        pageTaskRelations.forEach(pageTaskRelation -> {
            pageTaskRelation.setId(null);
            pageTaskRelation.setCreateTime(time);
            pageTaskRelation.setUpdateTime(time);
            pageTaskRelation.setVersion(DataVersionEnum.V2_0.getVersion());
        });
        mobileDesignerConfigs.forEach(mobileDesignerConfig -> {
            mobileDesignerConfig.setId(null);
            mobileDesignerConfig.setCreateTime(time);
            mobileDesignerConfig.setUpdateTime(time);
            mobileDesignerConfig.setVersion(DataVersionEnum.V2_0.getVersion());
        });
        tenantRelations.forEach(tenantRelation -> {
            tenantRelation.setId(null);
            tenantRelation.setCreateTime(time);
            tenantRelation.setUpdateTime(time);
            tenantRelation.setVersion(DataVersionEnum.V2_0.getVersion());
        });
        // step3: 批量插入数据
        this.batchInsertUcVersion(taskFilters, pageTaskRelations, mobileDesignerConfigs, tenantRelations);
    }

    private void deleteStdVersion(String appCode, String version) {
        taskFilterRepository.deleteByCode(appCode, version);
        pageTaskRelationService.deleteByCode(appCode, version);
        mobileDesignerConfigRepository.deleteByCode(appCode, version);
    }

    private void deleteUcVersion(String appCode, String ucAppCode, String version) {
        ucTaskFilterRepository.deleteByCode(appCode, ucAppCode, version);
        ucPageTaskRelationRepository.deleteByCode(appCode, ucAppCode, version);
        ucMobileDesignerConfigRepository.deleteByCode(appCode, ucAppCode, version);
        ucTenantRelationRepository.deleteByCode(appCode, ucAppCode, version);
    }

    private void batchInsertStdVersion(List<TaskFilter> taskFilters, List<PageTaskRelation> pageTaskRelations, List<MobileDesignerConfig> mobileDesignerConfigs, String version) {
        taskFilterRepository.batchInsert(taskFilters, version);
        pageTaskRelationService.batchInsert(pageTaskRelations);
        mobileDesignerConfigRepository.batchInsert(mobileDesignerConfigs);
    }

    private void batchInsertUcVersion(List<UcTaskFilter> taskFilters, List<UcPageTaskRelation> pageTaskRelations, List<UcMobileDesignerConfig> mobileDesignerConfigs, List<UcTenantRelation> tenantRelations) {
        ucTaskFilterRepository.batchInsert(taskFilters);
        ucPageTaskRelationRepository.batchInsert(pageTaskRelations);
        ucMobileDesignerConfigRepository.batchInsert(mobileDesignerConfigs);
        ucTenantRelationRepository.batchInsert(tenantRelations);
    }

    private <T extends TaskFilter> T buildStdTaskFilter(T taskFilter, String appCode, String tmActivityId, String time, String version) {
        taskFilter.setId(null);
        taskFilter.setSource(StringUtil.SOURCE_ATHENA);
        taskFilter.setCategory(TaskFilterCategoryEnum.TASK.getValue());
        TaskFilterParam param = new TaskFilterParam();
        param.setTmActivityId(tmActivityId);
        taskFilter.setParam(param);
        taskFilter.setParamOp("==");
        taskFilter.setValid(true);
        taskFilter.setEnvScopes(this.getEnvScopes());
        taskFilter.setFrom(TaskFilterFromEnum.DESIGNER.getValue());
        taskFilter.setCreateTime(time);
        taskFilter.setUpdateTime(time);
        taskFilter.setAppCode(appCode);
        taskFilter.setVersion(version);
        return taskFilter;
    }

    private <T extends PageTaskRelation> T buildStdPageTaskRelation(T pageTaskRelation, String appCode, String tmActivityId, String time, String version) {
        pageTaskRelation.setId(null);
        pageTaskRelation.setSource(StringUtil.SOURCE_ATHENA);
        pageTaskRelation.setPageId(PageSettingIdPresetEnum.MOBILE_ATHENA_UNIVERSAL_PARSING_TASK_DETAIL.name());
        pageTaskRelation.setCategory(PageTaskRelationCategoryEnum.DETAIL.getValue());
        TaskFilterParam param = new TaskFilterParam();
        param.setTmActivityId(tmActivityId);
        pageTaskRelation.setParam(param);
        pageTaskRelation.setEnvScopes(this.getEnvScopes());
        pageTaskRelation.setFrom(TaskFilterFromEnum.DESIGNER.getValue());
        pageTaskRelation.setCreateTime(time);
        pageTaskRelation.setUpdateTime(time);
        pageTaskRelation.setAppCode(appCode);
        pageTaskRelation.setVersion(version);
        return pageTaskRelation;
    }

    private <T extends MobileDesignerConfig> T buildStdMobileDesignerConfig(T mobileDesignerConfig, String appCode, String tmActivityId, String type, String time, String version) {
        mobileDesignerConfig.setId(null)
                .setApplication(appCode)
                .setTmTaskId(null)
                .setTmActivityId(tmActivityId)
                .setTmActivityIdMatch(null)
                .setTmPattern(null)
                .setTmCategory(null)
                .setType(type)
                .setEnvScopes(this.getEnvScopes())
                .setData(null)
                .setRules(null)
                .setCreateTime(time)
                .setUpdateTime(time)
                .setVersion(version);
        return mobileDesignerConfig;
    }

    private UcTaskFilter buildUcTaskFilter(String ucAppCode, String appCode, String tmActivityId, String time, String version) {
        UcTaskFilter taskFilter = this.buildStdTaskFilter(new UcTaskFilter(), appCode, tmActivityId, time, version);
        taskFilter.setUcAppCode(ucAppCode);
        return taskFilter;
    }

    private UcPageTaskRelation buildUcPageTaskRelation(String ucAppCode, String appCode, String tmActivityId, String time, String version) {
        UcPageTaskRelation pageTaskRelation = this.buildStdPageTaskRelation(new UcPageTaskRelation(), appCode, tmActivityId, time, version);
        pageTaskRelation.setUcAppCode(ucAppCode);
        return pageTaskRelation;
    }

    private UcMobileDesignerConfig buildUcMobileDesignerConfig(String ucAppCode, String appCode, String tmActivityId, String type, String time, String version) {
        return this.buildStdMobileDesignerConfig(new UcMobileDesignerConfig(), appCode, tmActivityId, type, time, version)
                .setUcAppCode(ucAppCode);
    }

    private UcTenantRelation buildUcTenantRelation(String ucAppCode, String appCode, String tenantId, Boolean individualAll, String time, String version) {
        return new UcTenantRelation().setId(null)
                .setAppCode(appCode)
                .setUcAppCode(ucAppCode)
                .setTenantId(tenantId)
                .setIndividualAll(individualAll)
                .setCreateTime(time)
                .setUpdateTime(time)
                .setVersion(version);
    }

    private List<EnvScope> getEnvScopes() {
        // 发版对应环境，开启对应环境开关
        List<EnvScope> list = new ArrayList<>();
        EnvScope prod = new EnvScope();
        prod.setType(AppEnvTypeEnum.PRODUCTION.getEnvType());
        prod.setValid(AppEnvTypeEnum.PRODUCTION.getEnvType().equals(AppContext.getEnvType()));
        list.add(prod);
        EnvScope test = new EnvScope();
        test.setType(AppEnvTypeEnum.TEST.getEnvType());
        test.setValid(AppEnvTypeEnum.TEST.getEnvType().equals(AppContext.getEnvType()));
        list.add(test);
        EnvScope dev = new EnvScope();
        dev.setType(AppEnvTypeEnum.DEVELOP.getEnvType());
        dev.setValid(AppEnvTypeEnum.DEVELOP.getEnvType().equals(AppContext.getEnvType()));
        list.add(dev);
        return list;
    }

    @Override
    public void updataData(MobileDesignerConfig config) {
        mobileDesignerConfigRepository.updataData(config);
        ucMobileDesignerConfigRepository.updateData(config);
    }
}
