package com.digiwin.mobile.mobileuibot.printer.builder.cpcl;

import com.digiwin.mobile.mobileuibot.printer.builder.zpl.ZPLPrinterCommandBuilder;
import com.digiwin.mobile.mobileuibot.printer.enums.PrinterCommandTypeEnum;
import com.digiwin.mobile.mobileuibot.printer.factory.PrinterCommandBuilder;
import com.digiwin.mobile.mobileuibot.printer.model.CommandRequired;
import com.digiwin.mobile.mobileuibot.printer.model.ReportData;
import com.digiwin.mobile.mobileuibot.printer.model.ReportDetail;
import com.google.common.collect.Lists;
import org.springframework.beans.factory.config.ConfigurableBeanFactory;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import javax.imageio.ImageIO;
import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;

/**
 * <p>功能描述：CPCL打印机指令生成器</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: CPCLPrinterCommandBuilder.java
 * @Author: wangjwc
 * @Date: created at 2024/9/23 11:11
 */
@Component
@Scope(ConfigurableBeanFactory.SCOPE_PROTOTYPE) // 原本是单例模式，改为多例模式，避免线程安全问题
public class CPCLPrinterCommandBuilder implements PrinterCommandBuilder {

    private byte[] graphBuffer = new byte[0];
    private double tcpLabelMaxHeightCm;
    private int tcpPrinterDpi = 203;
    private int copies;
    private int rotate;
    private int graphWidth;
    private int graphHeight;

    @Override
    public String getPrinterType() {
        return PrinterCommandTypeEnum.CPCL.name();
    }

    @Override
    public ReportDetail getReportFilePaths(ReportDetail report, String pdfUrl) {
        report.setBmpFilesPath(Lists.newArrayList(pdfUrl));
        report.setCopy(1);
        report.setHeight("");
        report.setWidth("");
        return report;
    }

    @Override
    public List<String> buildPrintCommand(List<String> imagePaths) throws Exception {
        if (CollectionUtils.isEmpty(imagePaths)) {
            return Lists.newArrayList();
        }
        ReportData reportData = new ReportData();
        reportData.setPdfUrls(imagePaths);
        return this.printerCommandString(reportData);
    }

    private List<CommandRequired> getCommandBytes(ReportDetail report, String pdfUrl) throws Exception {
        List<CommandRequired> reportBytesCopy = new ArrayList<>();
        report = this.getReportFilePaths(report, pdfUrl);

        for (String path : report.getBmpFilesPath()) {
            BufferedImage originalBitmap = ImageIO.read(Files.newInputStream(Paths.get(path)));
            originalBitmap = ZPLPrinterCommandBuilder.convertToGrayscale2(originalBitmap);

            if (report.getHeight().isEmpty()) {
                report.setHeight(String.format("%.2f", (double) originalBitmap.getHeight() / 203 * 25.4));
            }

            if (report.getWidth().isEmpty()) {
                report.setWidth(String.format("%.2f", (double) originalBitmap.getWidth() / 203 * 25.4));
            }

            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            ImageIO.write(originalBitmap, "png", baos);
            byte[] imageBytes = baos.toByteArray();

            reportBytesCopy.add(new CommandRequired(
                    report.getHeight(),
                    report.getWidth(),
                    imageBytes,
                    report.getCopy()
            ));
        }
        return reportBytesCopy;
    }

    private List<String> printerCommandString(ReportData reportData) throws Exception {
        List<String> result = new ArrayList<>();
        List<List<CommandRequired>> reportBytesCopy = getCommandBytes(reportData);

        if (!reportBytesCopy.isEmpty()) {
            for (List<CommandRequired> reportBytesData : reportBytesCopy) {
                for (CommandRequired reportInfo : reportBytesData) {
                    graphBuffer = reportInfo.getBytes();
                    tcpLabelMaxHeightCm = Double.parseDouble(reportInfo.getHeight()) / 10;
                    copies = reportInfo.getCopy();
                    rotate = reportData.getRotate();
                    result.add(getCPCLCommand());
                }
            }
        } else {
            throw new Exception("无法生成十六进制指令，没有标签信息");
        }

        return result;
    }

    private List<byte[]> printerCommandByte(ReportData reportData) {
        throw new UnsupportedOperationException();
    }

    private List<List<CommandRequired>> getCommandBytes(ReportData reportData) throws Exception {
        List<List<CommandRequired>> reportBytesCopy = new ArrayList<>();

        if (!reportData.getPdfUrls().isEmpty()) {
            for (String pdfUrl : reportData.getPdfUrls()) {
                reportBytesCopy.add(getCommandBytes(new ReportDetail(), pdfUrl));
            }
        } else {
            for (ReportDetail report : reportData.getData()) {
                reportBytesCopy.add(getCommandBytes(report, null));
            }
        }

        return reportBytesCopy;
    }

    private String getCPCLCommand() {
        String textHex = getBitmapHexData();
        int printHeight = (int) (tcpLabelMaxHeightCm / 2.54 * tcpPrinterDpi);

        if (printHeight < graphHeight) {
            printHeight = graphHeight;
        }

        return String.format("! %d %d %d %d %d\r\nROTATE %d\r\nEG %d %d %d %d %s\r\nFORM\r\nPRINT\r\n",
                0, //水平偏移量
                tcpPrinterDpi, //横向DPI
                tcpPrinterDpi, //纵向DPI
                printHeight, //标签最大像素高度=DPI*标签纸高度(英寸)
                copies, //份数
                rotate, //旋转角度
                rowRealBytesCount(), //图像的字节宽度
                graphHeight, //图像的像素高度
                0, //横向的开始位置
                0, //纵向的开始位置
                textHex);
    }

    private String getBitmapHexData() {
        StringBuilder sb = new StringBuilder();
        try (ByteArrayInputStream srcStream = new ByteArrayInputStream(graphBuffer)) {
            BufferedImage bmp = ImageIO.read(srcStream);
            graphWidth = bmp.getWidth();
            graphHeight = bmp.getHeight();

            int loopWidth = 8 - (bmp.getWidth() % 8);
            if (loopWidth == 8) loopWidth = bmp.getWidth();
            else loopWidth += bmp.getWidth();

            for (int y = 0; y < bmp.getHeight(); y++) {
                int bit = 128;
                int currentValue = 0;
                for (int x = 0; x < loopWidth; x++) {
                    int intensity;

                    if (x < bmp.getWidth()) {
                        int color = bmp.getRGB(x, y);
                        int red = (color >> 16) & 0xff;
                        int green = (color >> 8) & 0xff;
                        int blue = color & 0xff;
                        intensity = 255 - (red + green + blue) / 3;
                    } else {
                        intensity = 0;
                    }

                    if (intensity >= 128) currentValue |= bit;
                    bit = bit >> 1;
                    if (bit == 0) {
                        sb.append(String.format("%02X", currentValue));
                        bit = 128;
                        currentValue = 0;
                    }
                }
            }
        } catch (Exception ex) {
            throw new RuntimeException(ex);
        }

        return sb.toString();
    }

    private int rowRealBytesCount() {
        return (graphWidth % 8 > 0) ? (graphWidth / 8) + 1 : graphWidth / 8;
    }


    public static void main(String[] args) throws Exception {
        ReportData reportData = new ReportData();
        reportData.setPdfUrls(Lists.newArrayList("C:\\Users\\wjw\\Downloads\\UserBasis001.bmp"));
        List<String> strings = new CPCLPrinterCommandBuilder().printerCommandString(reportData);
        System.out.println(strings);
    }
}
