package com.digiwin.mobile.mobileuibot.printer.builder.escpos;

import com.digiwin.mobile.mobileuibot.printer.enums.PrinterCommandTypeEnum;
import com.digiwin.mobile.mobileuibot.printer.factory.PrinterCommandBuilder;
import com.digiwin.mobile.mobileuibot.printer.model.CommandRequired;
import com.digiwin.mobile.mobileuibot.printer.model.ReportData;
import com.digiwin.mobile.mobileuibot.printer.model.ReportDetail;
import com.google.common.collect.Lists;
import org.springframework.beans.factory.config.ConfigurableBeanFactory;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import javax.imageio.ImageIO;
import java.awt.*;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Base64;
import java.util.List;

/**
 * <p>功能描述：ESCPOS 打印机指令生成器</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: ESCPOSPrinterCommandBuilder.java
 * @Author: wangjwc
 * @Date: created at 2024/9/23 11:11
 */
@Component
@Scope(ConfigurableBeanFactory.SCOPE_PROTOTYPE) // 原本是单例模式，改为多例模式，避免线程安全问题
public class ESCPOSPrinterCommandBuilder implements PrinterCommandBuilder {

    @Override
    public String getPrinterType() {
        return PrinterCommandTypeEnum.ESC.name();
    }

    @Override
    public ReportDetail getReportFilePaths(ReportDetail report, String pdfUrl) {
        report.setBmpFilesPath(Lists.newArrayList(pdfUrl));
        report.setCopy(1);
        return report;
    }

    @Override
    public List<String> buildPrintCommand(List<String> imagePaths) throws Exception {
        if (CollectionUtils.isEmpty(imagePaths)) {
            return Lists.newArrayList();
        }
        ReportData reportData = new ReportData();
        reportData.setPdfUrls(imagePaths);
        return this.printerCommandString(reportData);
    }

    /**
     * 根据单个模板信息生成指令数据
     * 返回List是因为可能出现需要复制X份的情况
     */
    private List<CommandRequired> getCommandBytes(ReportDetail report, String pdfUrl) throws IOException {
        List<CommandRequired> reportBytesCopy = new ArrayList<>();
        report = this.getReportFilePaths(report, pdfUrl);

        for (String path : report.getBmpFilesPath()) {
            BufferedImage originalBitmap = ImageIO.read(new File(path));

            // 创建 CommandRequired 对象并添加到列表中
            reportBytesCopy.add(new CommandRequired(
                    copyBitmap(originalBitmap), report.getCopy(), path
            ));
        }

        return reportBytesCopy;
    }

    private List<String> printerCommandString(ReportData reportData) throws IOException {
        List<String> result = new ArrayList<>();
        List<List<CommandRequired>> reportBytesCopy = getCommandBytes(reportData);

        if (!reportBytesCopy.isEmpty()) {
            for (List<CommandRequired> reportBytesData : reportBytesCopy) {
                for (CommandRequired reportInfo : reportBytesData) {
                    byte[] commandByte = draw2PxPoint(reportInfo.getBitmap());

                    for (int i = 0; i < reportInfo.getCopy(); i++) {
                        result.add(Base64.getEncoder().encodeToString(commandByte));
                    }
                }
            }
        } else {
            throw new RuntimeException("无法生成十六进制指令，没有标签信息");
        }

        return result;
    }

    private List<byte[]> printerCommandByte(ReportData reportData) throws IOException {
        List<byte[]> result = new ArrayList<>();
        List<List<CommandRequired>> reportBytesCopy = getCommandBytes(reportData);

        if (!reportBytesCopy.isEmpty()) {
            for (List<CommandRequired> reportBytesData : reportBytesCopy) {
                for (CommandRequired reportInfo : reportBytesData) {
                    byte[] commandByte = draw2PxPoint(reportInfo.getBitmap());

                    for (int i = 0; i < reportInfo.getCopy(); i++) {
                        result.add(commandByte);
                    }
                }
            }
        } else {
            throw new RuntimeException("无法生成十六进制指令，没有标签信息");
        }

        return result;
    }

    /**
     * 根据批量模板信息生成打印指令数据
     */
    private List<List<CommandRequired>> getCommandBytes(ReportData reportData) throws IOException {
        List<List<CommandRequired>> reportBytesCopy = new ArrayList<>();
        if (!reportData.getPdfUrls().isEmpty()) {
            for (String pdfUrl : reportData.getPdfUrls()) {
                reportBytesCopy.add(getCommandBytes(new ReportDetail(), pdfUrl));
            }
        } else {
            for (ReportDetail report : reportData.getData()) {
                reportBytesCopy.add(getCommandBytes(report, ""));
            }
        }

        return reportBytesCopy;
    }

    /**
     * *************************************************************************
     * * 假设一个240*240的图片，分辨率设为24, 共分10行打印
     * * 每一行,是一个 240*24 的点阵, 每一列有24个点,存储在3个byte里面。
     * * 每个byte存储8个像素点信息。因为只有黑白两色，所以对应为1的位是黑色，对应为0的位是白色
     * **************************************************************************
     * 把一张 BufferedImage 图片转化为打印机可以打印的字节流
     */
    private byte[] draw2PxPoint(BufferedImage bmp) {
        //用来存储转换后的 bitmap 数据。为什么要再加1000，这是为了应对当图片高度无法
        //整除24时的情况。比如bitmap 分辨率为 240 * 250，占用 7500 byte，
        //但是实际上要存储11行数据，每一行需要 24 * 240 / 8 =720byte 的空间。再加上一些指令存储的开销，
        //所以多申请 1000byte 的空间是稳妥的，不然运行时会抛出数组访问越界的异常。
        List<Byte> data = new ArrayList<>();

        int n1 = bmp.getWidth() % 256;
        int n2 = bmp.getWidth() / 256;

        // 初始化打印机
        data.add((byte) 0x1B);
        data.add((byte) 0x40);
        // 设置文本对齐方式
        data.add((byte) 0x1B);
        data.add((byte) 0x61);
        data.add((byte) 1);
        // 设置行距为0的指令
        data.add((byte) 0x1B);
        data.add((byte) 0x33);
        data.add((byte) 0x00);

        // 逐行打印
        for (int j = 0; j < bmp.getHeight() / 24f; j++) {
            // 打印图片的指令
            data.add((byte) 0x1B);
            data.add((byte) 0x2A);
            data.add((byte) 33);
            data.add((byte) n1); // nL
            data.add((byte) n2); // nH

            // 对于每一行，逐列打印
            for (int i = 0; i < bmp.getWidth(); i++) {
                // 每一列24个像素点，分为3个字节存储
                for (int m = 0; m < 3; m++) {
                    byte byteData = 0;
                    // 每个字节表示8个像素点，0表示白色，1表示黑色
                    for (int n = 0; n < 8; n++) {
                        byte b = px2Byte(i, (j * 24) + (m * 8) + n, bmp);
                        byteData += (byte) (byteData + b);
                    }

                    data.add(byteData);
                }
            }
            data.add((byte) 10); // 换行
        }

        // 打印并回车
        data.add((byte) 0x0c);

        // 将 List<Byte> 转换为 byte[]
        byte[] byteArray = new byte[data.size()];
        for (int i = 0; i < data.size(); i++) {
            byteArray[i] = data.get(i);
        }

        return byteArray;
    }

    /**
     * 灰度图片黑白化，黑色是1，白色是0
     */
    private byte px2Byte(int x, int y, BufferedImage bit) {
        if (x < bit.getWidth() && y < bit.getHeight()) {
            int pixel = bit.getRGB(x, y);
            int red = (pixel >> 16) & 0xFF;
            int green = (pixel >> 8) & 0xFF;
            int blue = pixel & 0xFF;

            // 简单判断黑白
            return (red == 255 && green == 255 && blue == 255) ? (byte) 0 : (byte) 1;
        }

        return 0;
    }

    private BufferedImage copyBitmap(BufferedImage original) {
        BufferedImage copy = new BufferedImage(original.getWidth(), original.getHeight(), original.getType());
        Graphics g = copy.createGraphics();
        g.drawImage(original, 0, 0, null);
        g.dispose();
        return copy;
    }

    private byte[] mergeByteArrays(List<byte[]> arrays) {
        int totalLength = arrays.stream().mapToInt(arr -> arr.length).sum();
        byte[] merged = new byte[totalLength];
        int currentPos = 0;
        for (byte[] array : arrays) {
            System.arraycopy(array, 0, merged, currentPos, array.length);
            currentPos += array.length;
        }
        return merged;
    }

    public static void main(String[] args) throws Exception {
        ReportData reportData = new ReportData();
        reportData.setPdfUrls(Lists.newArrayList("C:\\Users\\wjw\\Downloads\\UserBasis001.bmp"));
        List<String> strings = new ESCPOSPrinterCommandBuilder().printerCommandString(reportData);
        System.out.println(strings);
    }
}
