package com.digiwin.mobile.mobileuibot.printer.builder.tspl;

import com.digiwin.mobile.mobileuibot.printer.enums.PrinterCommandTypeEnum;
import com.digiwin.mobile.mobileuibot.printer.factory.PrinterCommandBuilder;
import com.digiwin.mobile.mobileuibot.printer.model.CommandRequired;
import com.digiwin.mobile.mobileuibot.printer.model.ReportData;
import com.digiwin.mobile.mobileuibot.printer.model.ReportDetail;
import com.google.common.collect.Lists;
import org.springframework.beans.factory.config.ConfigurableBeanFactory;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import javax.imageio.ImageIO;
import java.awt.*;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.List;

/**
 * <p>功能描述：TSPL打印机指令生成器</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: TSPLPrinterCommandBuilder.java
 * @Author: wangjwc
 * @Date: created at 2024/9/23 11:11
 */
@Component
@Scope(ConfigurableBeanFactory.SCOPE_PROTOTYPE) // 原本是单例模式，改为多例模式，避免线程安全问题
public class TSPLPrinterCommandBuilder implements PrinterCommandBuilder {

    @Override
    public String getPrinterType() {
        return PrinterCommandTypeEnum.TSPL.name();
    }

    @Override
    public ReportDetail getReportFilePaths(ReportDetail report, String pdfUrl) {
        report.setBmpFilesPath(Lists.newArrayList(pdfUrl));
        report.setCopy(1);
        return report;
    }

    @Override
    public List<String> buildPrintCommand(List<String> imagePaths) throws Exception {
        if (CollectionUtils.isEmpty(imagePaths)) {
            return Lists.newArrayList();
        }
        ReportData reportData = new ReportData();
        reportData.setPdfUrls(imagePaths);
        return this.printerCommandString(reportData);
    }

    private List<CommandRequired> getCommandBytes(ReportDetail report, String pdfUrl) throws IOException {
        List<CommandRequired> reportBytesCopy = new ArrayList<>();
        report = this.getReportFilePaths(report, pdfUrl);

        for (String path : report.getBmpFilesPath()) {
            BufferedImage originalBitmap = ImageIO.read(new File(path));
            reportBytesCopy.add(new CommandRequired(copyBitmap(originalBitmap), report.getHeight(), report.getWidth(), report.getCopy()));
        }

        return reportBytesCopy;
    }

    /**
     * 根据批量模板信息生成打印指令数据
     */
    private List<String> printerCommandString(ReportData reportData) throws IOException {
        List<String> result = new ArrayList<>();
        List<List<CommandRequired>> reportBytesCopy = getCommandBytes(reportData);
        if (!reportBytesCopy.isEmpty()) {
            for (List<CommandRequired> reportBytesData : reportBytesCopy) {
                for (CommandRequired reportInfo : reportBytesData) {
                    BufferedImage sourceBmp = reportInfo.getBitmap();
                    byte[] commandByte = getBitmapBytes(sourceBmp, reportInfo.getWidth(), reportInfo.getHeight());

                    for (int i = 0; i < reportInfo.getCopy(); i++) {
                        result.add(Base64.getEncoder().encodeToString(commandByte));
                    }

                    sourceBmp.flush(); // 释放资源
                }
            }
        } else {
            throw new RuntimeException("无法生成十六进制指令，没有标签信息");
        }

        return result;
    }

    private List<byte[]> printerCommandByte(ReportData reportData) throws IOException {
        List<byte[]> result = new ArrayList<>();
        List<List<CommandRequired>> reportBytesCopy = getCommandBytes(reportData);

        if (!reportBytesCopy.isEmpty()) {
            for (List<CommandRequired> reportBytesData : reportBytesCopy) {
                for (CommandRequired reportInfo : reportBytesData) {
                    BufferedImage sourceBmp = reportInfo.getBitmap();
                    byte[] commandByte = getBitmapBytes(sourceBmp, reportInfo.getWidth(), reportInfo.getHeight());

                    for (int i = 0; i < reportInfo.getCopy(); i++) {
                        result.add(commandByte);
                    }

                    sourceBmp.flush(); // 释放资源
                }
            }
        } else {
            throw new RuntimeException("无法生成十六进制指令，没有标签信息");
        }

        return result;
    }

    private List<List<CommandRequired>> getCommandBytes(ReportData reportData) throws IOException {
        List<List<CommandRequired>> reportBytesCopy = new ArrayList<>();
        if (!reportData.getPdfUrls().isEmpty()) {
            for (String pdfUrl : reportData.getPdfUrls()) {
                reportBytesCopy.add(getCommandBytes(new ReportDetail(), pdfUrl));
            }
        } else {
            for (ReportDetail report : reportData.getData()) {
                reportBytesCopy.add(getCommandBytes(report, ""));
            }
        }

        return reportBytesCopy;
    }

    /**
     * 获取TSPL指令字节数组
     */
    private byte[] getBitmapBytes(BufferedImage getImage, String width, String height) {
        byte[] bmpData;
        byte[] sendBmpData;
        int k = 0;
        // 实际发送的宽
        int sendWidth = getImage.getWidth();
        // 实际发送的高
        int sendHeight = getImage.getHeight();

        // 调整高度和宽度为8的倍数
        if (sendHeight % 8 != 0) {
            sendHeight += 8 - (sendHeight % 8);
        }
        if (sendWidth % 8 != 0) {
            sendWidth += 8 - (sendWidth % 8);
        }

        bmpData = new byte[sendWidth * sendHeight / 8];
        sendBmpData = new byte[bmpData.length + 1000];
        // 0xFF为全白
        Arrays.fill(bmpData, (byte) 0xFF);

        // 求灰度平均值
        double redSum = 0, greenSum = 0, blueSum = 0;
        double total = sendWidth * sendHeight;
        for (int i = 0; i < getImage.getWidth(); i++) {
            for (int j = 0; j < getImage.getHeight(); j++) {
                // 获取每个点颜色
                Color color = new Color(getImage.getRGB(i, j));
                redSum += color.getAlpha();
                greenSum += color.getGreen();
                blueSum += color.getBlue();
            }
        }

        int meanR = (int) (redSum / total);
        int meanG = (int) (greenSum / total);
        int meanB = (int) (blueSum / total);

        // 处理图像并生成位图数据
        for (int j = 0; j < getImage.getHeight(); j++) {
            for (int i = 0; i < getImage.getWidth(); i++) {
                // 获取每个点颜色
                Color color = new Color(getImage.getRGB(i, j));
                if (color.getRed() * 0.299 + color.getGreen() * 0.587 + color.getBlue() * 0.114 <
                        meanR * 0.299 + meanG * 0.587 + meanB * 0.114) {
                    // 颜色转灰度(可调 0-255)
                    // 位操作
                    bmpData[j * sendWidth / 8 + i / 8] &= (byte) ~(1 << (7 - (i % 8)));
                }
            }
        }

        // 拼装打印指令 --- CLS + 换行
        sendBmpData[k++] = 0x43;
        sendBmpData[k++] = 0x4C;
        sendBmpData[k++] = 0x53;
        sendBmpData[k++] = 0x0A;

        // SIZE 指令 --- size
        sendBmpData[k++] = 0x53;
        sendBmpData[k++] = 0x49;
        sendBmpData[k++] = 0x5A;
        sendBmpData[k++] = 0x45;
        // 空格
        sendBmpData[k++] = 0x20;
        // 宽度314/203*25.4~40 --- 按203DPI计算 纸张宽度（mm）= 图像宽度（像素） / DPI * 25.4 --- 向上取整
//        sendBmpData[k++] = 0x34;
//        sendBmpData[k++] = 0x30;
        if (width == null || width.isEmpty()) {
            width = String.valueOf((int) Math.ceil((double) getImage.getWidth() / 203 * 25.4));
        }
        byte[] widthData = convertStringToHex(width);
        System.arraycopy(widthData, 0, sendBmpData, k, widthData.length);
        k += widthData.length;

        // 空格
        sendBmpData[k++] = 0x20;
        // 单位mm
        sendBmpData[k++] = 0x6D;
        sendBmpData[k++] = 0x6D;
        // 英文逗号
        sendBmpData[k++] = 0x2C;
        // 高度472/203*25.4~60 --- 按203DPI计算 纸张高度（mm）= 图像高度（像素） / DPI * 25.4 --- 向上取整
//        sendBmpData[k++] = 0x36;
//        sendBmpData[k++] = 0x30;
        if (height == null || height.isEmpty()) {
            height = String.valueOf((int) Math.ceil((double) getImage.getHeight() / 203 * 25.4));
        }
        byte[] heightData = convertStringToHex(height);
        System.arraycopy(heightData, 0, sendBmpData, k, heightData.length);
        k += heightData.length;

        // 空格
        sendBmpData[k++] = 0x20;
        // 单位mm
        sendBmpData[k++] = 0x6D;
        sendBmpData[k++] = 0x6D;
        // 换行
        sendBmpData[k++] = 0x0A;

        // BITMAP指令（BITMAP 0,0,%d,%d 0）
        sendBmpData[k++] = 0x42;
        sendBmpData[k++] = 0x49;
        sendBmpData[k++] = 0x54;
        sendBmpData[k++] = 0x4D;
        sendBmpData[k++] = 0x41;
        sendBmpData[k++] = 0x50;
        // 空格
        sendBmpData[k++] = 0x20;
        // 固定值 --- 0,0,
        sendBmpData[k++] = 0x30;
        sendBmpData[k++] = 0x2C;
        sendBmpData[k++] = 0x30;
        sendBmpData[k++] = 0x2C;
        // 宽度字节数314/8+1~40 -- (graphWidth % 8 > 0) ? (graphWidth / 8) + 1 : (graphWidth / 8)
//        sendBmpData[k++] = 0x34;
//        sendBmpData[k++] = 0x30;
        byte[] wdata = convertStringToHex(String.valueOf((getImage.getWidth() % 8 > 0) ? (getImage.getWidth() / 8) + 1 : (getImage.getWidth() / 8)));
        System.arraycopy(wdata, 0, sendBmpData, k, wdata.length);
        k += wdata.length;

        // 英文逗号
        sendBmpData[k++] = 0x2C;
        // 高度472
//        sendBmpData[k++] = 0x34;
//        sendBmpData[k++] = 0x37;
//        sendBmpData[k++] = 0x32;
        byte[] hdata = convertStringToHex(String.valueOf(getImage.getHeight()));
        System.arraycopy(hdata, 0, sendBmpData, k, hdata.length);
        k += hdata.length;

        // 固定值 --- ,0,
        sendBmpData[k++] = 0x2C;
        sendBmpData[k++] = 0x30;
        sendBmpData[k++] = 0x2C;

        // BMP数据拼装
        System.arraycopy(bmpData, 0, sendBmpData, k, bmpData.length);
        k += bmpData.length;

        // 结束符 --- PRINT 1
        sendBmpData[k++] = 0x0A;
        sendBmpData[k++] = 0x50;
        sendBmpData[k++] = 0x52;
        sendBmpData[k++] = 0x49;
        sendBmpData[k++] = 0x4E;
        sendBmpData[k++] = 0x54;
        // 空格
        sendBmpData[k++] = 0x20;
        sendBmpData[k++] = 0x31;
        // 换行
        sendBmpData[k++] = 0x0A;

        return sendBmpData;
    }

    private void memset(byte[] buf, byte val, int size) {
        for (int i = 0; i < size; i++) {
            buf[i] = val;
        }
    }

    private byte[] convertStringToHex(String input) {
        byte[] hexValues = new byte[input.length()];
        for (int i = 0; i < input.length(); i++) {
            hexValues[i] = (byte) input.charAt(i);
        }
        return hexValues;
    }

    private BufferedImage copyBitmap(BufferedImage original) {
        BufferedImage copy = new BufferedImage(original.getWidth(), original.getHeight(), original.getType());
        Graphics g = copy.createGraphics();
        g.drawImage(original, 0, 0, null);
        g.dispose();
        return copy;
    }

    public static void main(String[] args) throws Exception {
        ReportData reportData = new ReportData();
        reportData.setPdfUrls(Lists.newArrayList("C:\\Users\\wjw\\Downloads\\UserBasis001.bmp"));
//        reportData.setPdfUrls(Lists.newArrayList("D:\\code\\dj\\mobile-uibot-copy\\temp\\pdf_to_bmp\\b187fe47-ab9b-4b13-b21b-cfd7b169d841_3.bmp"));
//        reportData.setPdfUrls(Lists.newArrayList("C:\\Users\\wjw\\Downloads\\20241104103937.png"));
        List<String> strings = new TSPLPrinterCommandBuilder().printerCommandString(reportData);
        System.out.println(strings);
    }
}
