package com.digiwin.mobile.mobileuibot.printer.builder.zpl;

import com.digiwin.mobile.mobileuibot.printer.enums.PrinterCommandTypeEnum;
import com.digiwin.mobile.mobileuibot.printer.factory.PrinterCommandBuilder;
import com.digiwin.mobile.mobileuibot.printer.model.CommandRequired;
import com.digiwin.mobile.mobileuibot.printer.model.ReportData;
import com.digiwin.mobile.mobileuibot.printer.model.ReportDetail;
import com.google.common.collect.Lists;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.config.ConfigurableBeanFactory;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import javax.imageio.ImageIO;
import javax.xml.bind.DatatypeConverter;
import java.awt.*;
import java.awt.image.BufferedImage;
import java.awt.image.DataBufferByte;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * <p>功能描述：ZPL打印机指令生成器</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: ZPLPrinterCommandBuilder.java
 * @Author: wangjwc
 * @Date: created at 2024/9/23 11:09
 */
@Slf4j
@Component
@Scope(ConfigurableBeanFactory.SCOPE_PROTOTYPE) // 原本是单例模式，改为多例模式，避免线程安全问题
public class ZPLPrinterCommandBuilder implements PrinterCommandBuilder {

    // 定义私有字段
    private static List<KeyValuePair<Character, Integer>> compressDictionary = new ArrayList<>();

    private byte[] graphBuffer;
    private int graphWidth;
    private int graphHeight;
    private int graphWidthByte;
    private int graphHeightDpi;

    public ZPLPrinterCommandBuilder() {
        initCompressCode();
    }

    @Override
    public String getPrinterType() {
        return PrinterCommandTypeEnum.ZPL.name();
    }

    @Override
    public ReportDetail getReportFilePaths(ReportDetail report, String pdfUrl) {
        report.setBmpFilesPath(Lists.newArrayList(pdfUrl));
        report.setCopy(1);
        return report;
    }

    @Override
    public List<String> buildPrintCommand(List<String> imagePaths) throws Exception {
        if (CollectionUtils.isEmpty(imagePaths)) {
            return Lists.newArrayList();
        }
        ReportData reportData = new ReportData();
        reportData.setPdfUrls(imagePaths);
        return this.printerCommandString(reportData);
    }

    private List<CommandRequired> getCommandBytes(ReportDetail report, String pdfUrl) {
        List<CommandRequired> reportBytesCopy = new ArrayList<>();
        report = this.getReportFilePaths(report, pdfUrl);

        for (String path : report.getBmpFilesPath()) {
            try (ByteArrayOutputStream stream = new ByteArrayOutputStream()) {
                BufferedImage sourceBmp = ImageIO.read(new File(path));
                sourceBmp = convertToGrayscale2(sourceBmp);

                ImageIO.write(sourceBmp, "BMP", stream);

                reportBytesCopy.add(new CommandRequired(stream.toByteArray(), report.getCopy()));

                sourceBmp.flush();
            } catch (IOException e) {
                log.error("生成ZPL指令失败", e);
            }
        }

        return reportBytesCopy;
    }

    private List<String> printerCommandString(ReportData reportData) {
        List<String> result = new ArrayList<>();
        List<List<CommandRequired>> reportBytesCopy = getCommandBytes(reportData);
        if (!reportBytesCopy.isEmpty()) {
            for (List<CommandRequired> reportBytesData : reportBytesCopy) {
                for (CommandRequired reportInfo : reportBytesData) {
                    Map<String, Object> map = bmpToZplStr(reportInfo.getBytes());
                    result.add("^XA\n" +
                            "^LH0,0^LL" + graphHeight + "^PW" + graphWidth + "\n" +
                            "~DGR:Temp0.GRF," + map.get("totalBytes") + "," + map.get("rowBytes") + "," + map.get("textBitmap") + "\n" +
                            "^FO30,0^XGR:Temp0.GRF,1,1^FS\n" +
                            "^PQ" + reportInfo.getCopy() + "\n" +
                            "^XZ");
                }
            }
        } else {
            throw new RuntimeException("无法生成十六进制指令，没有标签信息");
        }

        return result;
    }

    private List<byte[]> printerCommandByte(ReportData reportData) {
        throw new UnsupportedOperationException("Not implemented");
    }

    private List<List<CommandRequired>> getCommandBytes(ReportData reportData) {
        List<List<CommandRequired>> reportBytesCopy = new ArrayList<>();
        if (!reportData.getPdfUrls().isEmpty()) {
            for (String pdfUrl : reportData.getPdfUrls()) {
                reportBytesCopy.add(getCommandBytes(new ReportDetail(), pdfUrl));
            }
        } else {
            for (ReportDetail report : reportData.getData()) {
                reportBytesCopy.add(getCommandBytes(report, ""));
            }
        }

        return reportBytesCopy;
    }

    private int getRowSize() {
        return ((graphWidth + 31) >> 5) << 2;
    }

    private int getRowRealBytesCount() {
        return (graphWidth % 8 > 0) ? (graphWidth / 8) + 1 : graphWidth / 8;
    }

    /**
     * 位图转ZPL指令字符串
     */
    private Map<String, Object> bmpToZplStr(byte[] bitmap) {
        try {
            graphBuffer = bitmap;
            byte[] bmpData = getBitmapData();
            String textHex = DatatypeConverter.printHexBinary(bmpData).replace("-", "");

            String textBitmap = compressLZ77(textHex);
            int totalBytes = graphHeight * getRowRealBytesCount();
            int rowBytes = getRowRealBytesCount();
            Map<String, Object> map = new HashMap<>();
            map.put("textBitmap", textBitmap);
            map.put("totalBytes", totalBytes);
            map.put("rowBytes", rowBytes);
            return map;
        } catch (Exception ex) {
            throw new RuntimeException(ex);
        }
    }


    /**
     * BufferedImage 获取单色位图数据(1bpp)，不含文件头、信息头、调色板三类数据 返回 bufferedimage
     */
    public static BufferedImage convertToGrayscale2(BufferedImage original) {
        int width = original.getWidth();
        int height = original.getHeight();

        // 创建一个新的 BufferedImage，使用 TYPE_BYTE_BINARY 表示1bpp图像
        BufferedImage monoImage = new BufferedImage(width, height, BufferedImage.TYPE_BYTE_BINARY);

        // 遍历原始图像，去创建单色位图
        for (int y = 0; y < height; y++) {
            for (int x = 0; x < width; x++) {
                Color color = new Color(original.getRGB(x, y));
                // 计算灰度值
                int grayValue = (int) (0.299 * color.getRed() + 0.587 * color.getGreen() + 0.114 * color.getBlue());
                // 根据灰度值判断是黑色还是白色
                // 可以调整阈值
                boolean isBlack = grayValue < 128;
                // 设置黑白像素到单色位图中
                monoImage.setRGB(x, y, isBlack ? Color.BLACK.getRGB() : Color.WHITE.getRGB());
            }
        }

        return monoImage;
    }

    /**
     * 获取单色位图数据
     */
    private BufferedImage convertToGrayscale(BufferedImage pimage) {
        BufferedImage source = null;

        if (pimage.getType() != BufferedImage.TYPE_3BYTE_BGR) {
            source = new BufferedImage(pimage.getWidth(), pimage.getHeight(), BufferedImage.TYPE_3BYTE_BGR);
            Graphics g = source.getGraphics();
            g.drawImage(pimage, 0, 0, null);
            g.dispose();
        } else {
            source = pimage;
        }

        byte[] sourceBuffer = ((DataBufferByte) source.getRaster().getDataBuffer()).getData();
        BufferedImage destination = new BufferedImage(source.getWidth(), source.getHeight(), BufferedImage.TYPE_BYTE_BINARY);
        byte[] destinationBuffer = ((DataBufferByte) destination.getRaster().getDataBuffer()).getData();

        int sourceIndex = 0;
        int destinationIndex = 0;
        int pixelTotal = 0;
        byte destinationValue = 0;
        int pixelValue = 128;
        int height = source.getHeight();
        int width = source.getWidth();
        int threshold = 500;

        for (int y = 0; y < height; y++) {
            sourceIndex = y * source.getWidth();
            destinationIndex = y * destination.getWidth();
            destinationValue = 0;
            pixelValue = 128;

            for (int x = 0; x < width; x++) {
                pixelTotal = sourceBuffer[sourceIndex + 1] + sourceBuffer[sourceIndex + 2] + sourceBuffer[sourceIndex + 3];
                if (pixelTotal > threshold) {
                    destinationValue += (byte) pixelValue;
                }

                if (pixelValue == 1) {
                    if (destinationIndex < destinationBuffer.length) {
                        destinationBuffer[destinationIndex] = destinationValue;
                    }
                    destinationIndex++;
                    destinationValue = 0;
                    pixelValue = 128;
                } else {
                    pixelValue >>= 1;
                }

                sourceIndex += 4;
            }

            if (pixelValue != 128) {
                if (destinationIndex < destinationBuffer.length) {
                    destinationBuffer[destinationIndex] = destinationValue;
                }
            }
        }

        if (source != pimage) {
            source.flush();
        }

        return destination;
    }

    /**
     * 获取单色位图数据(1bpp)，不含文件头、信息头、调色板三类数据。
     */
    private byte[] getBitmapData() {
        ByteArrayInputStream srcStream = null;
        ByteArrayOutputStream dstStream = null;
        BufferedImage srcBmp = null;
        BufferedImage dstBmp = null;
        byte[] srcBuffer = null;
        byte[] dstBuffer = null;
        byte[] result = null;
        try {
            srcStream = new ByteArrayInputStream(graphBuffer);
            srcBmp = ImageIO.read(srcStream);

            graphWidth = srcBmp.getWidth();
            graphHeight = srcBmp.getHeight();

            graphHeightDpi = srcBmp.getHeight();
            graphWidthByte = srcBmp.getWidth();

            dstBmp = convertToGrayscale2(srcBmp);
            dstStream = new ByteArrayOutputStream();
            ImageIO.write(dstBmp, "BMP", dstStream);
            dstBuffer = dstStream.toByteArray();

            result = dstBuffer;

            // 使用 ByteBuffer 读取指定位置的 4 个字节并转换为 int
            ByteBuffer buffer = ByteBuffer.wrap(dstBuffer);
            // 假设 BMP 文件是小端序
            buffer.order(ByteOrder.LITTLE_ENDIAN);
            int bfOffBits = buffer.getInt(10);
            result = new byte[graphHeight * getRowRealBytesCount()];

            for (int i = 0; i < graphHeight; i++) {
                int sindex = bfOffBits + ((graphHeight - 1 - i) * getRowSize());
                int dindex = i * getRowRealBytesCount();
                System.arraycopy(dstBuffer, sindex, result, dindex, getRowRealBytesCount());
            }

            for (int i = 0; i < result.length; i++) {
                result[i] ^= 0xFF;
            }
        } catch (Exception ex) {
            throw new RuntimeException(ex.getMessage(), ex);
        } finally {
            if (srcStream != null) {
                try {
                    srcStream.close();
                } catch (IOException e) {
                    log.error("关闭源流时发生异常", e);
                }
            }

            if (dstStream != null) {
                try {
                    dstStream.close();
                } catch (IOException e) {
                    log.error("关闭目标流时发生异常", e);
                }
            }

            if (srcBmp != null) {
                srcBmp.flush();
            }

            if (dstBmp != null) {
                dstBmp.flush();
            }
        }

        return result;
    }

    /**
     * LZ77图像字节流压缩方法
     */
    private String compressLZ77(String text) {
        // 将转成16进制的文本进行压缩
        String result = "";
        char[] arrChar = text.toCharArray();
        int count = 1;
        for (int i = 1; i < text.length(); i++) {
            if (arrChar[i - 1] == arrChar[i]) {
                count++;
            } else {
                result += convertNumber(count) + arrChar[i - 1];
                count = 1;
            }

            if (i == text.length() - 1) {
                result += convertNumber(count) + arrChar[i];
            }
        }

        return result;
    }

    private String decompressLZ77(String text) {
        String result = "";
        char[] arrChar = text.toCharArray();
        int count = 0;
        for (int i = 0; i < arrChar.length; i++) {
            if (isHexChar(arrChar[i])) {
                result += new String(new char[]{arrChar[i]}, 0, count == 0 ? 1 : count);
                count = 0;
            } else {
                int value = getCompressValue(arrChar[i]);
                count += value;
            }
        }

        return result;
    }

    private int getCompressValue(char c) {
        int result = 0;
        for (KeyValuePair<Character, Integer> entry : compressDictionary) {
            if (c == entry.getKey()) {
                result = entry.getValue();
            }
        }

        return result;
    }

    private boolean isHexChar(char c) {
        return (c >= '0' && c <= '9') || (c >= 'A' && c <= 'F') || (c >= 'a' && c <= 'f');
    }

    /**
     * 将连续的数字转换成LZ77压缩代码，如000可用I0表示。
     */
    private String convertNumber(int count) {
        String result = "";
        if (count > 1) {
            while (count > 0) {
                for (int i = compressDictionary.size() - 1; i >= 0; i--) {
                    if (count >= compressDictionary.get(i).getValue()) {
                        result += compressDictionary.get(i).getKey();
                        count -= compressDictionary.get(i).getValue();
                        break;
                    }
                }
            }
        }

        return result;
    }

    /**
     * G H I J K L M N O P Q R S T U V W X Y        对应1,2,3,4……18,19。
     * g h i j k l m n o p q r s t u v w x y z      对应20,40,60,80……340,360,380,400。
     */
    private void initCompressCode() {
        compressDictionary = new ArrayList<>();
        for (int i = 0; i < 19; i++) {
            compressDictionary.add(new KeyValuePair<>((char) (71 + i), i + 1));
        }

        for (int i = 0; i < 20; i++) {
            compressDictionary.add(new KeyValuePair<>((char) (103 + i), (i + 1) * 20));
        }
    }

    private String byteArrayToHexString(byte[] bytes) {
        StringBuilder result = new StringBuilder();
        for (byte b : bytes) {
            result.append(String.format("%02X", b));
        }
        return result.toString();
    }

    @Data
    static
    class KeyValuePair<K, V> {
        private K key;
        private V value;

        public KeyValuePair(K key, V value) {
            this.key = key;
            this.value = value;
        }
    }

    public static void main(String[] args) {
        ReportData reportData = new ReportData();
        reportData.setPdfUrls(Lists.newArrayList("C:\\Users\\wjw\\Pictures\\lQLPDhsn8uDYAoPNAcLNAcKwqnyeWOJXelYCEiAOSED0AA_450_450.png"));
        List<String> strings = new ZPLPrinterCommandBuilder().printerCommandString(reportData);
        System.out.println(strings);
    }
}


