package com.digiwin.mobile.mobileuibot.printer.service.impl;

import com.digiwin.mobile.mobileuibot.common.context.SpringContextHolder;
import com.digiwin.mobile.mobileuibot.common.http.model.AthenaBasicHttpRequestHead;
import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.common.string.StringUtil;
import com.digiwin.mobile.mobileuibot.designer.uibot.service.UiBotDesignerService;
import com.digiwin.mobile.mobileuibot.printer.enums.PrinterTemplatePatternEnum;
import com.digiwin.mobile.mobileuibot.printer.factory.PrinterCommandBuilder;
import com.digiwin.mobile.mobileuibot.printer.factory.PrinterCommandFactory;
import com.digiwin.mobile.mobileuibot.printer.request.ApiRequestCommand;
import com.digiwin.mobile.mobileuibot.printer.request.ApiRequestCommandByImage;
import com.digiwin.mobile.mobileuibot.printer.request.ApiRequestTemplate;
import com.digiwin.mobile.mobileuibot.printer.response.CommandVO;
import com.digiwin.mobile.mobileuibot.printer.response.PrintTemplateVO;
import com.digiwin.mobile.mobileuibot.printer.service.AbiReportService;
import com.digiwin.mobile.mobileuibot.printer.service.PrinterService;
import com.digiwin.mobile.mobileuibot.printer.utils.PdfToImageUtils;
import com.digiwin.mobile.mobileuibot.proxy.DigiwinAthenaApiRequest;
import com.digiwin.mobile.mobileuibot.proxy.atmc.model.DigiwinAtmcPrintTemplate;
import com.digiwin.mobile.mobileuibot.proxy.atmc.service.DigiwinAtmcProxyService;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.PcUiBotSubmitAction;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.PcUiBotSubmitExecuteContext;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.UiBotModel;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.text.SimpleDateFormat;
import java.util.*;
import java.util.stream.Collectors;

/**
 * <p>功能描述：打印服务实现类</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: PrinterServiceImpl.java
 * @Author: wangjwc
 * @Date: created at 2024/9/19 10:20
 */
@Component
@Slf4j
public class PrinterServiceImpl implements PrinterService {

    @Autowired
    private PrinterCommandFactory printerCommandFactory;

    @Autowired
    private AbiReportService abiReportService;

    @Autowired
    private DigiwinAtmcProxyService digiwinAtmcProxyService;

    @Override
    public CommandVO getCommand(ApiRequestCommand request) throws Exception {
        request.setDataId(request.getRawData().getString("printTemplateCode"));
        Class<? extends PrinterCommandBuilder> builder = printerCommandFactory.getBuilder(request.getCommandType());
        CommandVO commandVO = new CommandVO();
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        commandVO.setCurrentTime(sdf.format(new Date()));
        //先获取uibotModel,防止调用两遍
        UiBotModel uibotModel = abiReportService.getUibotModel(request);
        commandVO.setFileName(Optional.ofNullable(uibotModel.getRawUiBotPageData()).map(g -> g.get("UIBOT__activityName")).map(StringUtil::valueOf).orElse(""));
        if (builder == null) {
            log.error("无法识别的指令格式");
            return commandVO;
        }
        // 通过abi报表服务获取PDF
        String pdfUrl = abiReportService.getExportPdfUrl(request, uibotModel);
        log.info("PDF URL：{}", pdfUrl);
        if (StringUtils.isEmpty(pdfUrl)) {
            log.error(commandVO.toString());
            return commandVO;
        }
        // pdf转图片
        List<String> imagePaths = PdfToImageUtils.convertPdfUrlToBmp(pdfUrl);
        log.info("图片路径：{}", imagePaths);
        // 图片转对应的指令
        List<String> commands = SpringContextHolder.getBean(builder).buildPrintCommand(imagePaths);
        // 注意：删除临时图片
        PdfToImageUtils.deleteFiles(imagePaths);
        //新增Url、文件名称、当前时间字段给前端
        commandVO.setCommands(commands);
        commandVO.setPdfUrl(pdfUrl);
        return commandVO;
    }

    @Override
    public CommandVO getCommand2(ApiRequestCommand request) throws Exception {
        Class<? extends PrinterCommandBuilder> builder = printerCommandFactory.getBuilder(request.getCommandType());
        CommandVO commandVO = new CommandVO();
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        commandVO.setCurrentTime(sdf.format(new Date()));
        commandVO.setFileName(Optional.ofNullable(request.getPrintTemplate()).map(PrintTemplateVO::getName).orElse(""));
        if (builder == null) {
            log.error("无法识别的指令格式");
            return commandVO;
        }
        // 通过abi报表服务获取PDF
        String pdfUrl = abiReportService.getExportPdfUrl(request);
        log.info("PDF URL：{}", pdfUrl);
        if (StringUtils.isEmpty(pdfUrl)) {
            log.error(commandVO.toString());
            return commandVO;
        }
        // pdf转图片
        List<String> imagePaths = PdfToImageUtils.convertPdfUrlToBmp(pdfUrl);
        log.info("图片路径：{}", imagePaths);
        // 图片转对应的指令
        List<String> commands = SpringContextHolder.getBean(builder).buildPrintCommand(imagePaths);
        // 注意：删除临时图片
        PdfToImageUtils.deleteFiles(imagePaths);
        //新增Url、文件名称、当前时间字段给前端
        commandVO.setCommands(commands);
        commandVO.setPdfUrl(pdfUrl);
        return commandVO;
    }

    @Override
    public CommandVO getCommandByImage(ApiRequestCommandByImage request) throws Exception {
        return null;
    }

    @Override
    public List<PrintTemplateVO> templateList(ApiRequestTemplate request) {
        if (BooleanUtils.isNotTrue(request.getRawData().getBoolean("isNewVersion"))) {
            // 老版本，直接返回空列表,前端选择空模板后调用打印接口(因老版本不使用模板，所以此处返回空对象)
            return Lists.newArrayList(new PrintTemplateVO());
        }
        PcUiBotSubmitAction submitAction = JsonUtil.objectToJavaObject(request.getRawData().get("action"), PcUiBotSubmitAction.class);
        if (submitAction == null) {
            return null;
        }
        PcUiBotSubmitExecuteContext executeContext = submitAction.getExecuteContext();
        if (executeContext == null) {
            return null;
        }
        Map<String, String> uriVariables = new HashMap<>();
        // 应用code
        uriVariables.put("appCode", executeContext.getApplication());
        // 作业code
        uriVariables.put("activityId", executeContext.getTmActivityId());
        // 作业类型,BUSINESS_PERFORM(任务);BUSINESS_CHARGE(项目);DATA_ENTRY_PERFORM(基础资料)
        uriVariables.put("pattern", this.buildPattern(executeContext));
        // 当前操作对象id(唯一标识（按钮、操作）) -- PC取ID，移动取componentID
        uriVariables.put("targetId", request.getRawData().getString("targetId"));
        // 客户端类型 APP or PC
        uriVariables.put("client", "APP");
        DigiwinAthenaApiRequest athenaApiRequest =
                DigiwinAthenaApiRequest.builder()
                        .requestHead(new AthenaBasicHttpRequestHead("mobileplatform",
                                request.getLocale(), request.getIamUserToken(),
                                request.getTenantId()))
                        .uriVariables(uriVariables)
                        .build();
        List<DigiwinAtmcPrintTemplate> templateList = this.digiwinAtmcProxyService.getPrintList(athenaApiRequest);
        return templateList.stream().map(PrintTemplateVO::convert).collect(Collectors.toList());
    }

    private String buildPattern(PcUiBotSubmitExecuteContext executeContext) {
        if (executeContext == null) {
            return null;
        }
        if (UiBotDesignerService.PAGECODE_TASK_DETAIL.equals(executeContext.getPageCode())) {
            return PrinterTemplatePatternEnum.BUSINESS_PERFORM.name();
        }
        if (UiBotDesignerService.PAGECODE_PROJECT_DETAIL.equals(executeContext.getPageCode())) {
            return PrinterTemplatePatternEnum.BUSINESS_CHARGE.name();
        }
        return PrinterTemplatePatternEnum.DATA_ENTRY_PERFORM.name();
    }
}
