package com.digiwin.mobile.mobileuibot.printer.utils;

import com.digiwin.mobile.mobileuibot.common.calculate.UUIDUtil;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.pdfbox.pdmodel.PDDocument;
import org.apache.pdfbox.rendering.ImageType;
import org.apache.pdfbox.rendering.PDFRenderer;
import org.springframework.core.io.UrlResource;
import org.springframework.util.CollectionUtils;

import javax.imageio.ImageIO;
import java.awt.image.BufferedImage;
import java.io.*;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;

/**
 * <p>功能描述：pdf转image工具类</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: PdfToImageUtils.java
 * @Author: wangjwc
 * @Date: created at 2024/9/23 15:00
 */
@Slf4j
public class PdfToImageUtils {

    // pdf 转 bmp 临时目录
    private static final String PDF_TO_IMAGE_TEMP_DIR =
            System.getProperty("user.dir") + File.separator
                    + "temp" + File.separator
                    + "pdf_to_bmp" + File.separator;

    static {
        // 创建临时目录
        createDirectoryIfNotExists(PDF_TO_IMAGE_TEMP_DIR);
    }

    /**
     * pdf 转 bmp
     */
    public static List<String> convertPdfUrlToBmp(String pdfUrl) {
        if (StringUtils.isEmpty(pdfUrl)) {
            return new ArrayList<>();
        }
        // 名称前缀
        String prefix = UUIDUtil.getUuid() + "_";
        String tempPdfPath = PDF_TO_IMAGE_TEMP_DIR + prefix + ".pdf";
        // 下载pdf文件
        downloadFile(pdfUrl, tempPdfPath);
        List<String> images = pdfPathToBmp(tempPdfPath, PDF_TO_IMAGE_TEMP_DIR + prefix);
        // 删除临时pdf文件
        deleteFile(tempPdfPath);
        return images;
    }

    private static List<String> pdfUrlToBmp(String pdfInputUrl, String imageOutputPath) {
        List<String> images = new ArrayList<>();
        BufferedInputStream inputStream = null;
        PDDocument document = null;
        try {
            inputStream = new BufferedInputStream(new UrlResource(pdfInputUrl).getInputStream());
            // 加载PDF文档
            document = PDDocument.load(inputStream);
            // 创建PDF渲染器
            PDFRenderer pdfRenderer = new PDFRenderer(document);
            // 获取文档页数
            int pageCount = document.getNumberOfPages();
            // 渲染每一页为图片
            for (int page = 0; page < pageCount; ++page) {
                // DPI: dpi越大，图片越清晰，占用空间也越大，转换时间也越长
                BufferedImage bim = pdfRenderer.renderImageWithDPI(page, 256, ImageType.GRAY);

                String imagePath = imageOutputPath + (page + 1) + ".bmp";
                ImageIO.write(bim, "bmp", new File(imagePath));
                images.add(imagePath);
            }
        } catch (IOException e) {
            log.error("pdfurl转bmp失败", e);
        } finally {
            if (inputStream != null) {
                try {
                    inputStream.close();
                } catch (IOException e) {
                    log.error("inputStream关闭失败", e);
                }
            }
            if (document != null) {
                try {
                    document.close();
                } catch (IOException e) {
                    log.error("document关闭失败", e);
                }
            }
        }
        return images;
    }

    private static List<String> pdfPathToBmp(String pdfInputPath, String imageOutputPath) {
        List<String> images = new ArrayList<>();
        // 加载PDF文档
        try (PDDocument document = PDDocument.load(new File(pdfInputPath))) {
            // 创建PDF渲染器
            PDFRenderer pdfRenderer = new PDFRenderer(document);
            // 获取文档页数
            int pageCount = document.getNumberOfPages();
            // 渲染每一页为图片
            for (int page = 0; page < pageCount; ++page) {
                // DPI: dpi越大，图片越清晰，占用空间也越大，转换时间也越长
                BufferedImage bim = pdfRenderer.renderImageWithDPI(page, 256, ImageType.GRAY);

                String imagePath = imageOutputPath + (page + 1) + ".bmp";
                ImageIO.write(bim, "bmp", new File(imagePath));
                images.add(imagePath);
            }
        } catch (IOException e) {
            log.error("pdfpath转bmp失败", e);
        }
        return images;
    }

    public static void deleteFile(String path) {
        deleteFiles(Lists.newArrayList(path));
    }

    public static void deleteFiles(List<String> imagePaths) {
        if (CollectionUtils.isEmpty(imagePaths)) {
            return;
        }
        for (String imagePath : imagePaths) {
            try {
                Files.deleteIfExists(Paths.get(imagePath));
            } catch (IOException e) {
                log.error("删除图片失败", e);
            }
        }
    }

    // 创建文件夹如果不存在
    public static void createDirectoryIfNotExists(String dirPath) {
        if (dirPath == null || dirPath.isEmpty()) {
            throw new IllegalArgumentException("目录路径不能为空");
        }
        File directory = new File(dirPath);
        try {
            if (directory.mkdirs()) {
                log.debug("目录创建成功: " + dirPath);
            } else {
                log.debug("目录已存在或无法创建: " + dirPath);
            }
        } catch (SecurityException e) {
            log.error("权限不足，无法创建目录", e);
        }
    }

    public static void downloadFile(String fileUrl, String tempPath) {
        File file = new File(tempPath);
        try (CloseableHttpClient httpClient = HttpClients.createDefault();
             FileOutputStream fos = new FileOutputStream(file)) {
            HttpGet httpGet = new HttpGet(new URIBuilder(fileUrl).build().toString());
            httpClient.execute(httpGet, httpResponse -> {
                InputStream inputStream = httpResponse.getEntity().getContent();
                file.createNewFile();
                int read;
                byte[] bytes = new byte[1024];
                while ((read = inputStream.read(bytes)) != -1) {
                    fos.write(bytes, 0, read);
                }
                fos.flush();
                return null;
            });
        } catch (Exception e) {
            log.error("下载文件失败", e);
        }
    }

    public static void main(String[] args) {
//        String pdfInputPath = "C:\\Users\\wjw\\Downloads\\生产标签-凭证.pdf";
//        // 图片名称前缀
//        String imageNamePrefix = UUIDUtil.getUuid() + "_";
//        List<String> strings = pdfPathToBmp(pdfInputPath, PDF_TO_IMAGE_TEMP_DIR + imageNamePrefix);
////        deleteImages(strings);


        String pdfUrl = "http://digiwinabi-test.apps.digiwincloud.com.cn/abi/eanalysemgr/httpdownload.do?cacheid=5095c0df93f94f49868cf0ba554aa6b3&resid=EANA$2$17$1$93943b570dfc46c78fea79c96fa16d01$4bfbb8fce75447a9b700721c0f107c08";

        String prefix = UUIDUtil.getUuid() + "_";
        String tempPdfPath = PDF_TO_IMAGE_TEMP_DIR + prefix + ".pdf";

        downloadFile(pdfUrl, tempPdfPath);

        List<String> strings = pdfPathToBmp(tempPdfPath, PDF_TO_IMAGE_TEMP_DIR + prefix);
        deleteFiles(strings);
        deleteFile(tempPdfPath);
        System.out.println(strings);
    }
}
