package com.digiwin.mobile.mobileuibot.project.common;

import com.digiwin.mobile.mobileuibot.common.context.SpringContextHolder;
import com.digiwin.mobile.mobileuibot.common.datetime.DateTimeUtil;
import com.digiwin.mobile.mobileuibot.common.localization.LocaleUtil;
import com.digiwin.mobile.mobileuibot.locale.service.LocaleService;
import com.digiwin.mobile.mobileuibot.proxy.atdm.model.pcc.PccProjectInfo;
import com.digiwin.mobile.mobileuibot.proxy.atmc.model.DigiwinAtmcPanel;
import com.digiwin.mobile.mobileuibot.proxy.atmc.model.DigiwinAtmcPanelTask;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import java.time.LocalDate;
import java.util.Locale;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * <p>功能描述：和项目相关字符串的工具类</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: ProjectStringUtil
 * @Author: Zaregoto
 * @Date: 2021/5/13 15:50
 */
public final class ProjectStringUtil {

    /**
     * 项目号pattern
     */
    private static Pattern PROJECT_NO_PATTERN = Pattern.compile("(\\[).*?(])");

    public static final String PCC_OLD_PROJECT_TYPE = "task_Project_Center_Console";
    public static final String PCC_OLD_PROJECT_TYPE_PROGRESS_TRACK_ACTIVITY = "progressTrack";
    public static final String PCC_NEW_PROJECT_TYPE = "projectCenterConsole_userProject";
    public static final String KSC_PROJECT_TYPE = "systemCloudHouseKeeper_userProject";
    public static final String PCC_NEW_PROJECT_TYPE_PROGRESS_TRACK_ACTIVITY = "progressTrack_DTD";
    public static final String KSC_PROJECT_TYPE_PROGRESS_TRACK_ACTIVITY = "sch_progressTrack";

    //编辑计划
    public static final String KSC_PROJECT_TYPE_PROGRESS_MAINTENANC_ACTIVITY = "sch_maintenanceProject";


    public static String
    getProjectNameOnCard(String locale, String startTime, String endTime, String projectName, boolean isPccTask) {
        String[] projectNameSpilt = projectNameSpilt(projectName);
        if (isPccTask && projectNameSpilt.length == 2) {
//            // 文字格式：「项目：」+「项目号」+「项目名」
//            String contentTemplate = "%s: %s";
//            return String.format(
//                    contentTemplate, LocaleUtil.getMobileTextByKey(locale, "项目"),
//                    projectNameSpilt[1] + projectNameSpilt[0]
//            ).trim();
            return projectNameSpilt[0] + " " + projectNameSpilt[1];
        }
        // 文字格式：「项目：」+「项目时距」+「项目名称」
        String contentTemplate = "%s: %s %s";
        return String.format(
                contentTemplate, LocaleUtil.getMobileTextByKey(locale, "项目"),
                DateTimeUtil.getTimeDistanceFromStartToEnd(
                        startTime, endTime),
                projectName
        ).trim();
    }

    /**
     * 截取项目名 获取项目号
     *
     * @param projectName
     * @return
     */
    private static String[] projectNameSpilt(String projectName) {
        if (!StringUtils.hasLength(projectName)) {
            return new String[0];
        }
        Matcher matcher = PROJECT_NO_PATTERN.matcher(projectName);
        String projectNo = "";
        while (matcher.find()) {
            projectNo = matcher.group();
        }
        return StringUtils.hasLength(projectNo) ?
                new String[]{projectName.substring(0, projectName.indexOf(projectNo)), projectNo} :
                new String[0];
    }

    /**
     * 获取项目中控台应用的项目详情时距描述文字
     *
     * @param locale
     * @param projectInfo 项目中控台项目详情信息
     * @return
     */
    public static String getProjectCCProjectDateDistanceString(String locale,
                                                               PccProjectInfo projectInfo) {
        // 文字格式：「时距：」+「项目时距」
        String contentTemplate = "%s: %s";
        return String.format(
                contentTemplate, LocaleUtil.getMobileTextByKey(locale, "时距"),
                DateTimeUtil.getDateDistanceFromStartToEnd(
                        projectInfo.getPlanStartDate(), projectInfo.getPlanFinishDate())
        ).trim();
    }
    public static String getProjectDateDistanceString(String locale, DigiwinAtmcPanelTask panelTask) {
        // 文字格式：「时距：」+「项目时距」
        String contentTemplate = "%s: %s";
        return String.format(
                contentTemplate, LocaleUtil.getMobileTextByKey(locale, "时距"),
                DateTimeUtil.getTimeDistanceFromStartToEnd(
                        panelTask.getStartTime(), panelTask.getEndTime())
        ).trim();
    }

    public static String getProjectBuildTypeByDetailPatternCategory(String tmTaskId, String tmActivityId) {
        Assert.notNull(tmTaskId, "tmTaskId must not null");
        Assert.notNull(tmActivityId, "tmActivityId must not null");
        return tmTaskId + "_" + tmActivityId;
    }

    public static String getProjectCCStatusString(String locale, String statusValue) {
        Assert.notNull(statusValue, "statusValue must not null");
        LocaleService localeService = SpringContextHolder.getBean(LocaleService.class);
        if ("10".equalsIgnoreCase(statusValue)) {
            return localeService.getLanguageValue(locale, "未开始");
        } else if ("20".equalsIgnoreCase(statusValue)) {
            return localeService.getLanguageValue(locale, "签核中");
        } else if ("30".equalsIgnoreCase(statusValue)) {
            return localeService.getLanguageValue(locale, "进行中");
        } else if ("40".equalsIgnoreCase(statusValue)) {
            return localeService.getLanguageValue(locale, "已结案");
        } else if ("50".equalsIgnoreCase(statusValue)) {
            return localeService.getLanguageValue(locale, "暂停");
        } else if ("60".equalsIgnoreCase(statusValue)) {
            return localeService.getLanguageValue(locale, "指定结案");
        } else {
            return localeService.getLanguageValue(locale, "未发起");
        }
    }

    public static String getProjectCCPropertyString(String locale, String propertyValue) {
        Assert.notNull(propertyValue, "propertyValue must not null");
        if ("10".equalsIgnoreCase(propertyValue)) {
            return LocaleUtil.getMobileTextByKey(locale, "潜在");
        } else {
            return LocaleUtil.getMobileTextByKey(locale, "正式");
        }
    }

    /**
     * 获取项目来源说明文字
     * <p>当语言为中文时，显示以下内容：</p>
     * <p>来源：由 {{ ids }} 等 {{ n }} 张{{sourceName}}组成</p>
     * <p>当语言为英文时，显示以下内容：</p>
     * <p>Source：There are {{ n }} {{sourceName}} such as {{ ids }}</p>
     *
     * @param locale    语言别
     * @param atmcPanel 可视化看板对象
     * @return 项目来源说明
     */
    public static String getProjectSourceInfoString(String locale, DigiwinAtmcPanel atmcPanel) {
        // 文字格式：「来源：」+「项目来源及数量」
        String contentTemplate = "%s: %s";
        if (Locale.US.toString().equals(locale)) {
            return String.format(
                    contentTemplate,
                    // pc有"来源"的多语言
                    LocaleUtil.getMobileTextByKey(locale, "来源"),
                    String.format(LocaleUtil.getMobileTextByKey(locale, "项目来源说明"),
                            Optional.ofNullable(atmcPanel)
                                    .map(DigiwinAtmcPanel::getTask)
                                    .map(DigiwinAtmcPanelTask::getSourceCount).orElse(0),
                            Optional.ofNullable(atmcPanel)
                                    .map(DigiwinAtmcPanel::getTask)
                                    .map(DigiwinAtmcPanelTask::getSourceName).orElse(""),
                            Optional.ofNullable(atmcPanel)
                                    .map(DigiwinAtmcPanel::getTask)
                                    .map(DigiwinAtmcPanelTask::getSourceIds).orElse("")
                    )
            ).trim();
        } else {
            return String.format(
                    contentTemplate,
                    // pc有"来源"的多语言
                    LocaleUtil.getMobileTextByKey(locale, "来源"),
                    String.format(LocaleUtil.getMobileTextByKey(locale, "项目来源说明"),
                            Optional.ofNullable(atmcPanel)
                                    .map(DigiwinAtmcPanel::getTask)
                                    .map(DigiwinAtmcPanelTask::getSourceIds).orElse(""),
                            Optional.ofNullable(atmcPanel)
                                    .map(DigiwinAtmcPanel::getTask)
                                    .map(DigiwinAtmcPanelTask::getSourceCount).orElse(0),
                            Optional.ofNullable(atmcPanel)
                                    .map(DigiwinAtmcPanel::getTask)
                                    .map(DigiwinAtmcPanelTask::getSourceName).orElse("")
                    )
            ).trim();
        }
    }

    public static String getProjectSourceInfoString(String locale, DigiwinAtmcPanelTask panelTask) {
        LocaleService localeService = SpringContextHolder.getBean(LocaleService.class);
        // 文字格式：「来源：」+「项目来源及数量」
        return localeService.getLanguageValue(locale, "来源：") + panelTask.getSourceIds()
                + localeService.getLanguageValue(locale, ",共包括1个分公司共") + panelTask.getSourceCount()
                + localeService.getLanguageValue(locale, "张") + panelTask.getSourceName()
                + localeService.getLanguageValue(locale, "组成");
    }

    /**
     * 获取项目发起方信息
     *
     * @param locale
     * @param atmcPanel
     * @return
     */
    public static String getProjectInitiatorInfoString(String locale, DigiwinAtmcPanel atmcPanel) {
        // 文字格式：「发起方：」+「YYYY.MM.DD」+「由xxx发起当前项目」
        String contentTemplate = "%s: %s %s";
        LocalDate initiateDate = DateTimeUtil.parseUseDefaultPattern(atmcPanel.getTask().getCreateTime()).toLocalDate();
        return String.format(contentTemplate,
                LocaleUtil.getMobileTextByKey(locale, "发起方"),
                String.format("%s.%s.%s",
                        initiateDate.getYear(),
                        DateTimeUtil.getFullMonth(initiateDate.getMonthValue()),
                        DateTimeUtil.getFullDate(initiateDate.getDayOfMonth())
                ),
                // pc有"由Athena发起当前项目"的多语言
                LocaleUtil.getMobileTextByKey(locale, "由Athena发起当前项目")
        );
    }

    public static String getProjectInitiatorInfoString(String locale, DigiwinAtmcPanelTask panelTask) {
        // 文字格式：「发起方：」+「YYYY.MM.DD」+「由xxx发起当前项目」
        String contentTemplate = "%s: %s %s";
        LocalDate initiateDate = DateTimeUtil.parseUseDefaultPattern(panelTask.getCreateTime()).toLocalDate();
        return String.format(contentTemplate,
                LocaleUtil.getMobileTextByKey(locale, "发起方"),
                String.format("%s.%s.%s",
                        initiateDate.getYear(),
                        DateTimeUtil.getFullMonth(initiateDate.getMonthValue()),
                        DateTimeUtil.getFullDate(initiateDate.getDayOfMonth())
                ),
                // pc有"由Athena发起当前项目"的多语言
                LocaleUtil.getMobileTextByKey(locale, "由Athena发起当前项目")
        );
    }
}