package com.digiwin.mobile.mobileuibot.project.filter;

import com.digiwin.mobile.mobileuibot.common.context.AppContext;
import com.digiwin.mobile.mobileuibot.common.datetime.DateTimeUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author zhangjj
 * @date 2022/7/29 15:12
 */
@Component
public class ProjectFilterRepositoryImpl implements ProjectFilterRepository {

    public static final String COLLECTION_NAME = "projectFilter";

    @Autowired
    private MongoTemplate mongoTemplate;

    @Autowired
    @Qualifier("secondMongoTemplate")
    private MongoTemplate secondMongoTemplate;

    @Override
    @Cacheable(cacheNames = "projectFilterList", cacheManager = "systemCacheManager")
    public List<ProjectFilter> listProjectFilter() {
        Query query = new Query();
        query.addCriteria(Criteria.where("valid").is(true));
        List<ProjectFilter> projectFilterList = mongoTemplate.find(query, ProjectFilter.class);
        //过滤环境
        projectFilterList = projectFilterList.stream()
                .filter(projectFilter -> projectFilter.validWithAppEnvType(AppContext.getEnvType()))
                .collect(Collectors.toList());
        return projectFilterList;
    }

    @Override
    public List<ProjectFilter> upsertAll(List<ProjectFilter> projectFilterList) {
        List<ProjectFilter> filterList = new ArrayList<>();
        if (CollectionUtils.isEmpty(projectFilterList)) {
            return filterList;
        }
        for (ProjectFilter projectFilter : projectFilterList) {
            filterList.add(upsert(projectFilter));
        }
        return filterList;
    }

    @Override
    public ProjectFilter upsert(ProjectFilter projectFilter) {
        if (!StringUtils.hasLength(projectFilter.getTenantId())) {
            //租户没值时统一处理为null
            projectFilter.setTenantId(null);
        }
        Query query = new Query();
        query.addCriteria(Criteria.where("appCode").is(projectFilter.getAppCode()))
                .addCriteria(Criteria.where("tenantId").is(projectFilter.getTenantId()))
                .addCriteria(Criteria.where("content.tmTaskId").is(projectFilter.getContent().getTmTaskId()));
        Update update = new Update();
        update.setOnInsert("createTime", DateTimeUtil.getTodayTimeUseDefaultPattern());
        update.set("updateTime", DateTimeUtil.getTodayTimeUseDefaultPattern());
        update.set("companyId", projectFilter.getCompanyId());
        update.set("systemId", projectFilter.getSystemId());
        update.set("tenantId", projectFilter.getTenantId());
        update.set("appCode", projectFilter.getAppCode());
        update.set("content", projectFilter.getContent());
        update.set("valid", projectFilter.getValid());
        update.set("envScopes", projectFilter.getEnvScopes());
        mongoTemplate.upsert(query, update, ProjectFilter.class);
        secondMongoTemplate.upsert(query, update, ProjectFilter.class);
        return mongoTemplate.findOne(query, ProjectFilter.class);
    }

    @Override
    public List<ProjectFilter> listProjectFilter(String appCode, String tenantId) {
        Query query = new Query();
        query.addCriteria(Criteria.where("appCode").is(appCode));
        if (StringUtils.hasLength(tenantId)) {
            query.addCriteria(Criteria.where("tenantId").is(tenantId));
        }
        return mongoTemplate.find(query, ProjectFilter.class);
    }

    @Override
    public List<ProjectFilter> deleteProjectFilter(String appCode, String tenantId, String tmTaskId) {
        Query query = new Query();
        query.addCriteria(Criteria.where("appCode").is(appCode));
        if (StringUtils.hasLength(tenantId)) {
            query.addCriteria(Criteria.where("tenantId").is(tenantId));
        }
        if (StringUtils.hasLength(tmTaskId)) {
            query.addCriteria(Criteria.where("content.tmTaskId").is(tmTaskId));
        }
        secondMongoTemplate.findAllAndRemove(query, ProjectFilter.class, COLLECTION_NAME);
        return mongoTemplate.findAllAndRemove(query, ProjectFilter.class, COLLECTION_NAME);
    }
}
