package com.digiwin.mobile.mobileuibot.project.service.impl;

import com.digiwin.mobile.mobileuibot.api.ApiRequest;
import com.digiwin.mobile.mobileuibot.common.context.AppContext;
import com.digiwin.mobile.mobileuibot.common.datetime.DateTimeUtil;
import com.digiwin.mobile.mobileuibot.common.localization.LocaleUtil;
import com.digiwin.mobile.mobileuibot.common.map.MapUtil;
import com.digiwin.mobile.mobileuibot.common.math.MathUtil;
import com.digiwin.mobile.mobileuibot.core.component.action.Action;
import com.digiwin.mobile.mobileuibot.core.component.action.ActionTypeEnum;
import com.digiwin.mobile.mobileuibot.core.component.basic.Label;
import com.digiwin.mobile.mobileuibot.core.component.basic.LabelTypeEnum;
import com.digiwin.mobile.mobileuibot.core.component.layout.app.AppFunctionTouchable;
import com.digiwin.mobile.mobileuibot.core.component.layout.app.entrance.AppEntranceAccessRecord;
import com.digiwin.mobile.mobileuibot.core.component.list.card.*;
import com.digiwin.mobile.mobileuibot.core.component.progress.ProgressRate;
import com.digiwin.mobile.mobileuibot.core.component.progress.ProgressRateTypeEnum;
import com.digiwin.mobile.mobileuibot.core.pagesetting.PageSettingIdPresetEnum;
import com.digiwin.mobile.mobileuibot.locale.service.LocaleService;
import com.digiwin.mobile.mobileuibot.mongodb.designerconfig.MobileDesignerConfig;
import com.digiwin.mobile.mobileuibot.mongodb.designerconfig.common.MobileDesignerConfigTypeEnum;
import com.digiwin.mobile.mobileuibot.openapi.service.ActivityFilterService;
import com.digiwin.mobile.mobileuibot.project.common.ProjectConstant;
import com.digiwin.mobile.mobileuibot.project.common.ProjectStringUtil;
import com.digiwin.mobile.mobileuibot.project.filter.ProjectFilterService;
import com.digiwin.mobile.mobileuibot.project.model.LaunchableProject;
import com.digiwin.mobile.mobileuibot.project.model.TaskAbnormalReason;
import com.digiwin.mobile.mobileuibot.project.service.ProjectService;
import com.digiwin.mobile.mobileuibot.proxy.atmc.model.*;
import com.digiwin.mobile.mobileuibot.proxy.atmc.service.DigiwinAtmcProxyService;
import com.digiwin.mobile.mobileuibot.proxy.audc.model.DigiwinAudcActivityAccessible;
import com.digiwin.mobile.mobileuibot.proxy.audc.service.DigiwinAudcProxyService;
import com.digiwin.mobile.mobileuibot.proxy.eoc.model.EocPerson;
import com.digiwin.mobile.mobileuibot.proxy.eoc.model.EocPersonResponse;
import com.digiwin.mobile.mobileuibot.proxy.eoc.service.DigiwinEocProxyService;
import com.digiwin.mobile.mobileuibot.proxy.knowledgemaps.model.KnowledgeMapsProject;
import com.digiwin.mobile.mobileuibot.proxy.knowledgemaps.service.DigiwinKnowledgeMapsProxyService;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.UiBotPageData;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.layout.UiBotLayout;
import com.digiwin.mobile.mobileuibot.task.filter.TaskFilter;
import com.google.common.collect.Lists;
import lombok.Data;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.annotation.PostConstruct;
import java.io.IOException;
import java.text.Collator;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

import static com.digiwin.mobile.mobileuibot.pattern.bean.PatternDetail.PATTERN_TYPE_PROJECT;

/**
 * <p>功能描述：项目领域servcie实现类</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: ProjectServiceImpl
 * @Author: Zaregoto
 * @Date: 2021/4/29 11:29
 */
@Service("projectService")
public class ProjectServiceImpl implements ProjectService {
    private static final Logger logger = LoggerFactory.getLogger(ProjectServiceImpl.class);

    /**
     * PCC项目移动端标识
     */
    public static final String PCC_APP_CODE = "PCC";
    /**
     * KSC项目移动端标识
     */
    public static final String KSC_APP_CODE = "KSC";

    private final Map<String, String> codeSuffixMap = new HashMap<>();

    @PostConstruct
    private void init() {
        //项目预算编制
        codeSuffixMap.put("task_Project_Budget", "LaunchBudgetProject");
        //项目中控台
        codeSuffixMap.put("task_Project_Center_Console", "LaunchSpecialProject");
        //项目备案
        codeSuffixMap.put("valuationInformationMaintenance_mainproject", "itemInformation_create");
        //问题快反
        codeSuffixMap.put("task_question_answer_mainline", "athena_questionAnswer_create");
        //除外回报
        codeSuffixMap.put("excluded_return_mainproject", "LaunchExcludedProject");
        //智慧检料发起
        codeSuffixMap.put(ProjectConstant.INTELLIGENT_MATERIAL_INSPECTION_PROJECT_CODE, ProjectConstant.INTELLIGENT_MATERIAL_INSPECTION_TM_ACTIVITY_ID);
        //收货排程修改 TODO 展示从项目发起入口隐藏
//        codeSuffixMap.put("transport_start", "transport_change_start");
        //手动发起出厂物流
        codeSuffixMap.put("manual_freightage_start", "manual_freightage_start");
        codeSuffixMap.put("scan_freightage_start", "scan_freightage_start");
    }

    /**
     * 体系云项目卡按创建时间降序排序-指定租户id（72002320，72002320_001）
     *
     * 002大陆测试区租户
     * PCCcloundProd 微软预生产租户
     */
    private static final List<String> NEED_SORTCARD_BY_TENANTID_LIST = Lists.newArrayList(
            "72002320", "72002320_001", "002", "PCCcloundProd");

    @Autowired
    private DigiwinAtmcProxyService digiwinAtmcProxyService;

    @Autowired
    private DigiwinKnowledgeMapsProxyService digiwinKnowledgemapsProxyService;

    @Autowired
    private DigiwinAudcProxyService audcProxyService;

    @Autowired
    private ProjectFilterService projectFilterService;

    @Autowired
    private DigiwinEocProxyService eocProxyService;

    @Autowired
    private ActivityFilterService activityFilterService;

    @Autowired
    private LocaleService localeService;

    @Override
    public TaskAbnormalReason getTaskAbnormalReason(
            String locale, DigiwinAtmcProjectCheckTaskResult digiwinAtmcProjectCheckTaskResult) {
        if (null == digiwinAtmcProjectCheckTaskResult ||
                null == digiwinAtmcProjectCheckTaskResult.getCheckItems() ||
                digiwinAtmcProjectCheckTaskResult.getCheckItems().isEmpty()) {
            return new TaskAbnormalReason();
        }
        String itemUnitText = LocaleUtil.getMobileTextByKey(locale, "task-report-项");
        TaskAbnormalReason taskAbnormalReason = new TaskAbnormalReason();
        for (DigiwinAtmcCheckTaskResultItem item : digiwinAtmcProjectCheckTaskResult.getCheckItems()) {
            taskAbnormalReason.addAbnormalMsg(item.getValue() + itemUnitText + " " + item.getItem());
        }

        return taskAbnormalReason;
    }

    @Override
    public CardList getCardList(ApiRequest apiRequest, boolean needSearchAndFilter) {
        String locale = apiRequest.getLocale();
        CardList cardList = new CardList();
        List<CardCategory> categoryList = getProjectList(apiRequest);
        cardList.setCategoryList(categoryList);
        List<CardListFilter> filterList = new ArrayList<>();
        //搜索
        if (needSearchAndFilter) {
            cardList.setSearchUrl("/mobile/v1/project/search");
            cardList.setFilterList(filterList);
        }
        //筛选项目类型
        CardListFilter projectTypeFilter = new CardListFilter();
        projectTypeFilter.setFilterId(DigiwinAtmcProject.PROJECT_TYPE);
        projectTypeFilter.setFilterName(LocaleUtil.getMobileTextByKey(locale, "项目类型"));
        Set<CardListOption> projectTypeFilterOptionSet = new HashSet<>();
        if (!CollectionUtils.isEmpty(categoryList)) {
            categoryList.forEach(cardCategory -> {
                List<Card> cards = cardCategory.getCardList();
                if (!CollectionUtils.isEmpty(cards)) {
                    cards.forEach(card -> {
                        List<CardListFilter> filterContentList = card.getFilterContentList();
                        filterContentList.forEach(filter -> {
                            if (DigiwinAtmcProject.PROJECT_TYPE.equalsIgnoreCase(filter.getFilterId())) {
                                projectTypeFilterOptionSet.addAll(filter.getFilterOptionList());
                            }
                        });
                    });

                    //体系云特定租户的项目卡按创建时间降序排序
                    if (this.validIsMatchTenantId(apiRequest.getTenantId())) {
                        cards = this.sortCardListByCreateTime(cards);
                    } else {
                        cards = sortCardList(cards, locale);
                    }
                    cardCategory.setCardList(cards);
                }
            });
        }
        List<CardListOption> projectTypeFilterOptionList = new ArrayList<>();
        projectTypeFilterOptionList.addAll(projectTypeFilterOptionSet);
        projectTypeFilter.setFilterOptionList(projectTypeFilterOptionList);
        filterList.add(projectTypeFilter);
        //筛选项目状态
        CardListFilter projectStatusFilter = new CardListFilter();
        projectStatusFilter.setFilterId(DigiwinAtmcProject.PROJECT_STATUS);
        projectStatusFilter.setFilterName(LocaleUtil.getMobileTextByKey(locale, "项目状态"));
        List<CardListOption> projectStatusFilterOptionList = new ArrayList<>();
        projectStatusFilter.setFilterOptionList(projectStatusFilterOptionList);
        projectStatusFilterOptionList.add(new CardListOption(DigiwinAtmcProject.PROJECT_STATUS_ABNORMAL,
                LocaleUtil.getMobileTextByKey(locale, "异常") + "/" + LocaleUtil.getMobileTextByKey(locale, "逾期")));
        projectStatusFilterOptionList.add(new CardListOption(DigiwinAtmcProject.PROJECT_STATUS_NORMAL,
                LocaleUtil.getMobileTextByKey(locale, "正常进行")));
        projectStatusFilterOptionList.add(new CardListOption(DigiwinAtmcProject.PROJECT_STATUS_URGENT,
                LocaleUtil.getMobileTextByKey(locale, "紧急")));
        projectStatusFilterOptionList.add(new CardListOption(DigiwinAtmcProject.PROJECT_STATUS_NO_TASK,
                LocaleUtil.getMobileTextByKey(locale, "暂无任务")));
        projectStatusFilterOptionList.add(new CardListOption(DigiwinAtmcProject.PROJECT_STATUS_SHARE,
                LocaleUtil.getMobileTextByKey(locale, "他人分享")));
        filterList.add(projectStatusFilter);
        return cardList;
    }

    /**
     * 获取项目
     * 2021年6月迭代只支持类型为项目中控台，且未结束的项目
     * 2021年11月迭代新增支持类型为设计变更的项目
     *
     * @param apiRequest
     * @return
     */
    @Override
    public List<CardCategory> getProjectList(ApiRequest apiRequest) {

        String locale = apiRequest.getLocale();
        String iamUserToken = apiRequest.getIamUserToken();
        String tenantId = apiRequest.getTenantId();
        List<String> dataIds = apiRequest.getJumpDataIds();
        List<CardCategory> cardCategoryList = new ArrayList<>();
        //逾期/异常卡片列表
        CardCategory abnormalCardCategory = new CardCategory();
        abnormalCardCategory.setCategoryName(LocaleUtil.getMobileTextByKey(apiRequest.getLocale(), "逾期/异常"));
        List<Card> abnormalCardList = new ArrayList<>();
        abnormalCardCategory.setCardList(abnormalCardList);
        //正常卡片列表
        CardCategory normalCardCategory = new CardCategory();
        normalCardCategory.setCategoryName(LocaleUtil.getMobileTextByKey(apiRequest.getLocale(), "正常"));
        List<Card> normalCardList = new ArrayList<>();
        normalCardCategory.setCardList(normalCardList);
        try {
            List<DigiwinAtmcProject> allProjectList = new ArrayList<>();
            List<DigiwinAtmcProject> jumpProjectList = new ArrayList<>();
            List<DigiwinAtmcProject> myProjectList = digiwinAtmcProxyService.listMyProject(
                    apiRequest.getClientId(),
                    apiRequest.getLocale(),
                    apiRequest.getIamUserToken(),
                    apiRequest.getTenantId());
            allProjectList.addAll(myProjectList);
            if (!CollectionUtils.isEmpty(dataIds)) {
                List<DigiwinEmp> digiwinEmps = new ArrayList<>();
                //得到当前团队所有人
                EocPersonResponse eocPersonResponse =
                        eocProxyService.getPersonList(iamUserToken,
                                9999, 1,
                                "");
                if (eocPersonResponse != null && eocPersonResponse.getList() != null && eocPersonResponse.getList().size() > 0) {
                    for (EocPerson eocPerson : eocPersonResponse.getList()) {
                        if (eocPerson.getUserId() == null) {
                            continue;
                        }
                        DigiwinEmp digiwinEmp = new DigiwinEmp();
                        digiwinEmp.setUserId(eocPerson.getUserId());
                        digiwinEmp.setTitle(eocPerson.getUserName());
                        digiwinEmp.setType("employee");
                        digiwinEmps.add(digiwinEmp);
                    }
                }
                List<DigiwinAtmcProject> allTeamProjectList = digiwinAtmcProxyService
                        .selectEmpProject(digiwinEmps, iamUserToken, locale, tenantId);
                allProjectList.addAll(allTeamProjectList);
                /**
                 * 筛选出支持的项目卡
                 */
                List<DigiwinAtmcProject> supportedProjectList = allProjectList.stream()
                        .filter(project -> projectFilterService.isSupportedProjectByDigiwinAtmcProject(project, tenantId))
                        .collect(Collectors.toList());
                if (PCC_APP_CODE.equalsIgnoreCase(apiRequest.getAppCode())
                        || KSC_APP_CODE.equalsIgnoreCase(apiRequest.getAppCode())
                ) {
                    List<DigiwinAtmcProject> pccProjectList = supportedProjectList.stream()
                            .filter(project -> projectFilterService.isSupportedProjectByDigiwinAtmcProject(project, tenantId))
                            .collect(Collectors.toList());
                    for (DigiwinAtmcProject digiwinAtmcProject : pccProjectList) {
                        int start = digiwinAtmcProject.getName().trim().lastIndexOf("[");
                        if (start < 0) {
                            continue;
                        }
                        int end = digiwinAtmcProject.getName().trim().lastIndexOf("]");
                        String dataId = digiwinAtmcProject.getName().substring(start + 1, end);
                        if (dataIds.contains(dataId)) {
                            jumpProjectList.add(digiwinAtmcProject);
                            dataIds.remove(dataId);
                        }
                        if (dataIds.size() == 0) {
                            break;
                        }
                    }
                }

            } else {
                jumpProjectList = allProjectList;
            }

            // 筛选低代码配置的项目卡
            String tenantVersion = activityFilterService.tenantVersion(iamUserToken, tenantId, locale);
            Set<String> tmActivityIds = activityFilterService.findByTypeOfMobileDesignerConfig(MobileDesignerConfigTypeEnum.PROJECT.getValue(), tenantId, tenantVersion)
                    .stream().map(MobileDesignerConfig::getTmActivityId).collect(Collectors.toSet());
            List<DigiwinAtmcProject> designerProjectList = jumpProjectList.stream()
                    .filter(project -> {
                        List<DigiwinAtmcTask> tasks = project.getTasks();
                        if (CollectionUtils.isEmpty(tasks)) {
                            return false;
                        }
                        Set<String> projectActivityIds = project.getTasks().stream().map(DigiwinAtmcTask::getTmActivityId).collect(Collectors.toSet());
                        return tmActivityIds.stream().anyMatch(projectActivityIds::contains);
                    }).collect(Collectors.toList());
            designerProjectList.forEach(project -> {
                this.buildDesignerProject(project, normalCardList, abnormalCardList, iamUserToken, tenantId, locale);
            });

            // 移除低代码配置的项目卡
            jumpProjectList.removeAll(designerProjectList);
            List<ProjectAndDoingTask> filteredProjectAndDoingTaskList = this.getFilteredProjectAndDoingTasks(jumpProjectList, tenantId);
            if (!filteredProjectAndDoingTaskList.isEmpty()) {
                for (ProjectAndDoingTask projectAndDoingTask : filteredProjectAndDoingTaskList) {
                    DigiwinAtmcProject project = projectAndDoingTask.getProject();
                    DigiwinAtmcTask doingTask = projectAndDoingTask.getTask();

                    if (projectFilterService.isCenterConsoleProject(project, tenantId)) {
                        this.handleCenterConsoleProject(project, doingTask, normalCardList, abnormalCardList, iamUserToken, tenantId, locale, false);
                    } else if (projectFilterService.isSystemCloudHouseKeeper(project, tenantId)) {
                        this.handleSystemCloudKeeperProject(project, doingTask, normalCardList, abnormalCardList, iamUserToken, tenantId, locale, false);
                    } else if (projectFilterService.isStandardProject(project, tenantId)) {
                        this.handleStandardProject(project, normalCardList, abnormalCardList, iamUserToken, locale, tenantId, false);
                    } else {
                        this.handleCurrentlyNotSupportedProject(project, normalCardList, abnormalCardList, iamUserToken, locale, tenantId, false);
                    }
                }
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        if (!abnormalCardList.isEmpty()) {
            cardCategoryList.add(abnormalCardCategory);
        }
        if (!normalCardList.isEmpty()) {
            cardCategoryList.add(normalCardCategory);
        }
        return cardCategoryList;
    }

    private void buildDesignerProject(DigiwinAtmcProject project, List<Card> normalCardList, List<Card> abnormalCardList, String iamUserToken, String tenantId, String locale) {
        Card card = new Card();
        card.setId(project.getId().toString());
        card.setIsTeamProject(false);
        card.setName(project.getName());
        card.setStartTime(project.getStartTime());
        card.setEndTime(project.getEndTime());
        card.setCreateTime(project.getCreateTime());
        card.setImportance(project.getImportance());
        List<CardListFilter> filterContentList = new ArrayList<>();
        filterContentList.add(project.getProjectType(locale));
        filterContentList.add(project.getProjectStatus(locale));
        card.setFilterContentList(filterContentList);
        Map<String, Object> rawData = new HashMap<>(1);
        if (project.getPinning() != null && project.getPinning()) {
            //被分享的项目卡
            String text = String.format(LocaleUtil.getMobileTextByKey(locale, "来自%s的分享"), project.getPersonInChargeName());
            card.setBottomTipText(text);
            rawData.put("isShare", true);
        } else {
            rawData.put("isShare", false);
        }
        if (project.getReadCount() != null && project.getReadCount() == 0) {
            //未读，加标签NEW
            Label label = Label.create("NEW", LabelTypeEnum.NEW.getType());
            List<Label> labelList = new ArrayList<>();
            labelList.add(label);
            card.setLabelList(labelList);
        }

        Action action = new Action(project.getId() + "",
                PageSettingIdPresetEnum.MOBILE_ATHENA_PROJECT_V2_PAGE.toString(),
                "", ActionTypeEnum.OPEN_NEW_PAGE.getValue(),
                "");
        action.setRawData(rawData);
        card.setAction(action);

        ChildCard childCard = new ChildCard();
        card.setChildCard(childCard);
        List<DigiwinAtmcTask> taskList = project.getTasks();
        //逾期/异常任务
        List<DigiwinAtmcTask> abnormalTaskList = new ArrayList<>();
        if (CollectionUtils.isEmpty(taskList)) {
            //无任务进行中
            card.setType(CardTypeEnum.GENERAL.getValue());
            childCard.setName(LocaleUtil.getMobileTextByKey(locale, "暂无任务进行中"));
            childCard.setType(CardTypeEnum.GENERAL.getValue());
            normalCardList.add(card);
            return;
        } else {
            abnormalTaskList = taskList.stream().filter(
                    task -> (Optional.ofNullable(task.getException()).orElse(false))
                            || (Optional.ofNullable(task.getOverdue()).orElse(false))
            ).collect(Collectors.toList());
        }
        //无逾期/异常任务
        if (abnormalTaskList.isEmpty()) {
            card.setType(CardTypeEnum.NORMAL.getValue());
            childCard.setType(CardTypeEnum.NORMAL.getValue());
            normalCardList.add(card);
            if (taskList.size() == 1) {
                //无逾期/异常任务,且只有一个任务进行中--显示任务名称
                childCard.setName(taskList.get(0).getName());
            } else {
                //无逾期/异常任务,且有多个任务进行中--显示-共xx个任务正在进行中
                childCard.setName(String.format(LocaleUtil.getMobileTextByKey(locale, "多个任务正在进行中"), taskList.size()));
            }
            return;
        }
        //有逾期/异常任务
        card.setType(CardTypeEnum.ABNORMAL.getValue());
        childCard.setType(CardTypeEnum.ABNORMAL.getValue());
        abnormalCardList.add(card);
        if (abnormalTaskList.size() > 1) {
            //有多个逾期/异常任务--显示共xx个任务异常/逾期
            childCard.setName(String.format(LocaleUtil.getMobileTextByKey(locale, "多个任务逾期异常"), abnormalCardList.size()));
        } else if (abnormalTaskList.size() == 1) {
            //有一个逾期/异常任务--显示该任务名称
            DigiwinAtmcTask task = abnormalTaskList.get(0);
            childCard.setName(task.getName());
            if (task.getOverdue()) {
                //是逾期任务，显示逾期天数
                List<Label> labelList = new ArrayList<>();
                labelList.add(
                        new Label(String.format(LocaleUtil.getMobileTextByKey(locale, "逾期天数"), task.getOverdueDays()), LabelTypeEnum.OVERDUE.getType()));
                childCard.setLabelList(labelList);
            }
        } else {
            //没有逾期任务，但是项目逾期了
            //是逾期任务，显示逾期天数
            List<Label> labelList = new ArrayList<>();
            labelList.add(
                    new Label(String.format(LocaleUtil.getMobileTextByKey(locale, "逾期天数"), DateTimeUtil.getDateTimePeriod(project.getEndTime(), DateTimeUtil.getTodayTimeUseDefaultPattern())), LabelTypeEnum.OVERDUE.getType()));
            childCard.setLabelList(labelList);
        }
    }

    public void handleCurrentlyNotSupportedProject(DigiwinAtmcProject project,
                                                   List<Card> normalCardList,
                                                   List<Card> abnormalCardList, String iamUserToken,
                                                   String locale, String tenantId, boolean isTeamProject) {
        Card card = new Card();
        card.setIsTeamProject(isTeamProject);
        card.setId(project.getId().toString());
        card.setName(project.getName());
        card.setStartTime(project.getStartTime());
        card.setEndTime(project.getEndTime());
        card.setCreateTime(project.getCreateTime());
        card.setImportance(project.getImportance());
        List<CardListFilter> filterContentList = new ArrayList<>();
        filterContentList.add(project.getProjectType(locale));
        filterContentList.add(project.getProjectStatus(locale));
        card.setFilterContentList(filterContentList);
        Map<String, Object> rawData = new HashMap<>(1);
        if (project.getPinning() != null && project.getPinning()) {
            //被分享的项目卡
            String text = String.format(LocaleUtil.getMobileTextByKey(locale, "来自%s的分享"), project.getPersonInChargeName());
            card.setBottomTipText(text);
            rawData.put("isShare", true);
        } else {
            rawData.put("isShare", false);
        }
        if (project.getReadCount() != null && project.getReadCount() == 0) {
            //未读，加标签NEW
            Label label = Label.create("NEW", LabelTypeEnum.NEW.getType());
            List<Label> labelList = new ArrayList<>();
            labelList.add(label);
            card.setLabelList(labelList);
        }
        //action、childCard、type

        // action设置
        Action action = new Action();
        action.setDataId(String.valueOf(project.getId()));
        action.setType(ActionTypeEnum.OPEN_NEW_PAGE.getValue());
        action.setRequestUrl("/mobile/v1/pattern/jump/info?type=search");
        DigiwinAtmcSearchResult result = new DigiwinAtmcSearchResult();
        result.setEndTime(project.getEndTime());
        result.setId(project.getId());
        result.setName(project.getName());
        result.setStartTime(project.getStartTime());
        result.setType(PATTERN_TYPE_PROJECT);
        result.setCode(String.valueOf(project.getId()));
        result.setEngineType("TaskEngine");
        action.setRawData(result);
        card.setAction(action);

        /**
         * TODO 2022年12月迭代，设计变更详情页面前端还有内容未更改，故还是“暂时不支持查看”。
         * 后续更改完成后，再恢复上方注释，同时删除下方setAction的代码
         */
//        card.setAction(new Action(project.getId() + "",
//                "",
//                "", ButtonActionTypeEnum.SHOW_TOAST.getValue(),
//                LocaleUtil.getMobileTextByKey(locale,
//                        "暂不支持查看")));

        ChildCard childCard = new ChildCard();
        card.setChildCard(childCard);
        List<DigiwinAtmcTask> taskList = project.getTasks();
        //逾期/异常任务
        List<DigiwinAtmcTask> abnormalTaskList = new ArrayList<>();
        if (CollectionUtils.isEmpty(taskList)) {
            //无任务进行中
            card.setType(CardTypeEnum.GENERAL.getValue());
            childCard.setName(LocaleUtil.getMobileTextByKey(locale, "暂无任务进行中"));
            childCard.setType(CardTypeEnum.GENERAL.getValue());
            normalCardList.add(card);
            return;
        } else {
            abnormalTaskList = taskList.stream().filter(
                    task -> (Optional.ofNullable(task.getException()).orElse(false))
                            || (Optional.ofNullable(task.getOverdue()).orElse(false))
            ).collect(Collectors.toList());
        }
        //项目是否正常
        boolean normal = true;
        //无逾期/异常任务
        if (abnormalTaskList.isEmpty()) {
            card.setType(CardTypeEnum.NORMAL.getValue());
            childCard.setType(CardTypeEnum.NORMAL.getValue());
            normalCardList.add(card);
            if (taskList.size() == 1) {
                //无逾期/异常任务,且只有一个任务进行中--显示任务名称
                childCard.setName(taskList.get(0).getName());
            } else {
                //无逾期/异常任务,且有多个任务进行中--显示-共xx个任务正在进行中
                childCard.setName(String.format(LocaleUtil.getMobileTextByKey(locale, "多个任务正在进行中"), taskList.size()));
            }
            return;
        }
        //有逾期/异常任务
        card.setType(CardTypeEnum.ABNORMAL.getValue());
        childCard.setType(CardTypeEnum.ABNORMAL.getValue());
        abnormalCardList.add(card);
        if (abnormalTaskList.size() > 1) {
            //有多个逾期/异常任务--显示共xx个任务异常/逾期
            childCard.setName(String.format(LocaleUtil.getMobileTextByKey(locale, "多个任务逾期异常"), abnormalCardList.size()));
        } else if (abnormalTaskList.size() == 1) {
            //有一个逾期/异常任务--显示该任务名称
            DigiwinAtmcTask task = abnormalTaskList.get(0);
            childCard.setName(task.getName());
            if (task.getOverdue()) {
                //是逾期任务，显示逾期天数
                List<Label> labelList = new ArrayList<>();
                labelList.add(
                        new Label(String.format(LocaleUtil.getMobileTextByKey(locale, "逾期天数"), task.getOverdueDays()), LabelTypeEnum.OVERDUE.getType()));
                childCard.setLabelList(labelList);
            }
        } else {
            //没有逾期任务，但是项目逾期了
            //是逾期任务，显示逾期天数
            List<Label> labelList = new ArrayList<>();
            labelList.add(
                    new Label(String.format(LocaleUtil.getMobileTextByKey(locale, "逾期天数"), DateTimeUtil.getDateTimePeriod(project.getEndTime(), DateTimeUtil.getTodayTimeUseDefaultPattern())), LabelTypeEnum.OVERDUE.getType()));
            childCard.setLabelList(labelList);
        }
    }

    public Card handleCurrentlyNotSupportedProject(DigiwinAtmcProject project, String iamUserToken, String locale, String tenantId, boolean isTeamProject) {
        Card card = new Card();
        card.convert(project);
        card.setIsTeamProject(isTeamProject);
        if (project.getPinning() != null && project.getPinning()) {
            //被分享的项目卡
            String text = String.format(LocaleUtil.getMobileTextByKey(locale, "来自%s的分享"), project.getPersonInChargeName());
            card.setBottomTipText(text);
        }
        if (project.getReadCount() != null && project.getReadCount() == 0) {
            //未读，加标签NEW
            Label label = Label.create("NEW", LabelTypeEnum.NEW.getType());
            List<Label> labelList = new ArrayList<>();
            labelList.add(label);
            card.setLabelList(labelList);
        }
        Action action = new Action();
        action.setDataId(String.valueOf(project.getId()));
        action.setType(ActionTypeEnum.OPEN_NEW_PAGE.getValue());
        action.setRequestUrl("/mobile/v1/pattern/jump/info?type=search");
        DigiwinAtmcSearchResult result = new DigiwinAtmcSearchResult();
        result.setEndTime(project.getEndTime());
        result.setId(project.getId());
        result.setName(project.getName());
        result.setStartTime(project.getStartTime());
        result.setType(PATTERN_TYPE_PROJECT);
        result.setCode(String.valueOf(project.getId()));
        result.setEngineType("TaskEngine");
        action.setRawData(result);
        card.setAction(action);

        ChildCard childCard = new ChildCard();
        card.setChildCard(childCard);
        List<DigiwinAtmcTask> taskList = project.getTasks();
        //逾期/异常任务
        List<DigiwinAtmcTask> abnormalTaskList = new ArrayList<>();
        if (CollectionUtils.isEmpty(taskList)) {
            //无任务进行中
            card.setType(CardTypeEnum.GENERAL.getValue());
            childCard.setName(LocaleUtil.getMobileTextByKey(locale, "暂无任务进行中"));
            childCard.setType(CardTypeEnum.GENERAL.getValue());
            return card;
        } else {
            abnormalTaskList = taskList.stream().filter(
                    task -> (Optional.ofNullable(task.getException()).orElse(false))
                            || (Optional.ofNullable(task.getOverdue()).orElse(false))
            ).collect(Collectors.toList());
        }
        //项目是否正常
        boolean normal = true;
        //无逾期/异常任务
        if (abnormalTaskList.isEmpty()) {
            card.setType(CardTypeEnum.NORMAL.getValue());
            childCard.setType(CardTypeEnum.NORMAL.getValue());
            if (taskList.size() == 1) {
                //无逾期/异常任务,且只有一个任务进行中--显示任务名称
                childCard.setName(taskList.get(0).getName());
            } else {
                //无逾期/异常任务,且有多个任务进行中--显示-共xx个任务正在进行中
                childCard.setName(String.format(LocaleUtil.getMobileTextByKey(locale, "多个任务正在进行中"), taskList.size()));
            }
            return card;
        }
        //有逾期/异常任务
        card.setType(CardTypeEnum.ABNORMAL.getValue());
        childCard.setType(CardTypeEnum.ABNORMAL.getValue());
        if (abnormalTaskList.size() > 1) {
            //有多个逾期/异常任务--显示共xx个任务异常/逾期
            childCard.setName(String.format(LocaleUtil.getMobileTextByKey(locale, "多个任务逾期异常"), abnormalTaskList.size()));
        } else if (abnormalTaskList.size() == 1) {
            //有一个逾期/异常任务--显示该任务名称
            DigiwinAtmcTask task = abnormalTaskList.get(0);
            childCard.setName(task.getName());
            if (task.getOverdue()) {
                //是逾期任务，显示逾期天数
                List<Label> labelList = new ArrayList<>();
                labelList.add(
                        new Label(String.format(LocaleUtil.getMobileTextByKey(locale, "逾期天数"), task.getOverdueDays()), LabelTypeEnum.OVERDUE.getType()));
                childCard.setLabelList(labelList);
            }
        } else {
            //没有逾期任务，但是项目逾期了
            //是逾期任务，显示逾期天数
            List<Label> labelList = new ArrayList<>();
            labelList.add(
                    new Label(String.format(LocaleUtil.getMobileTextByKey(locale, "逾期天数"), DateTimeUtil.getDateTimePeriod(project.getEndTime(), DateTimeUtil.getTodayTimeUseDefaultPattern())), LabelTypeEnum.OVERDUE.getType()));
            childCard.setLabelList(labelList);
        }
        return card;
    }

    private List<ProjectAndDoingTask> getFilteredProjectAndDoingTasks(List<DigiwinAtmcProject>
                                                                              allProjectList, String tenantId) {
        /**
         * 筛选出支持的项目卡
         */
        List<DigiwinAtmcProject> supportedProjectList = allProjectList.stream()
                .filter(project -> projectFilterService.isSupportedProjectByDigiwinAtmcProject(project, tenantId))
                .collect(Collectors.toList());
        /**
         * 转换成新对象
         */
        List<ProjectAndDoingTask> projectAndDoingTaskList = supportedProjectList.stream()
                .map(project -> {
                    if (CollectionUtils.isEmpty(project.getTasks())) {
                        return new ProjectAndDoingTask(project, new DigiwinAtmcTask());
                    } else {
                        return new ProjectAndDoingTask(project, project.getTasks().get(0));
                    }
                })
                .sorted((o1, o2) -> {
                    LocalDateTime o1STime = DateTimeUtil.parseUseDefaultPattern(o1.getProject().getStartTime());
                    LocalDateTime o2STime = DateTimeUtil.parseUseDefaultPattern(o2.getProject().getStartTime());
                    int sTimeCompareValue = o1STime.compareTo(o2STime);

                    LocalDateTime o1ETime = DateTimeUtil.parseUseDefaultPattern(o1.getProject().getEndTime());
                    LocalDateTime o2ETime = DateTimeUtil.parseUseDefaultPattern(o2.getProject().getEndTime());
                    return sTimeCompareValue == 0 ? o1ETime.compareTo(o2ETime) : sTimeCompareValue;
                })
                .collect(Collectors.toList());
        return projectAndDoingTaskList;
    }

    public List<ProjectAndDoingTask> convertProjectAndDoingTasks(List<DigiwinAtmcProject> allProjectList) {
        /**
         * 转换成新对象
         */
        List<ProjectAndDoingTask> projectAndDoingTaskList = allProjectList.stream()
                .map(project -> {
                    if (CollectionUtils.isEmpty(project.getTasks())) {
                        return new ProjectAndDoingTask(project, new DigiwinAtmcTask());
                    } else {
                        return new ProjectAndDoingTask(project, project.getTasks().get(0));
                    }
                })
                .sorted((o1, o2) -> {
                    LocalDateTime o1STime = DateTimeUtil.parseUseDefaultPattern(o1.getProject().getStartTime());
                    LocalDateTime o2STime = DateTimeUtil.parseUseDefaultPattern(o2.getProject().getStartTime());
                    int sTimeCompareValue = o1STime.compareTo(o2STime);

                    LocalDateTime o1ETime = DateTimeUtil.parseUseDefaultPattern(o1.getProject().getEndTime());
                    LocalDateTime o2ETime = DateTimeUtil.parseUseDefaultPattern(o2.getProject().getEndTime());
                    return sTimeCompareValue == 0 ? o1ETime.compareTo(o2ETime) : sTimeCompareValue;
                })
                .collect(Collectors.toList());
        return projectAndDoingTaskList;
    }

    @Override
    public List<LaunchableProject> getLaunchableProjectList(String iamUserToken, String tenantId, String locale) {
        // 基础查询。获取租户下当前用户所有可发起的项目
        List<KnowledgeMapsProject> knowledgeMapsProjects = this.digiwinKnowledgemapsProxyService.getLaunchableProjects(iamUserToken, tenantId, locale);

        List<LaunchableProject> projectList = new ArrayList<>();
        if (!CollectionUtils.isEmpty(knowledgeMapsProjects)) {
            String tenantVersion = activityFilterService.tenantVersion(iamUserToken, tenantId, locale);
            // 获取type为发起项目的数据
            List<MobileDesignerConfig> mobileDesignerConfigList = activityFilterService.findByTypeOfMobileDesignerConfig(MobileDesignerConfigTypeEnum.START_PROJECT.getValue(), tenantId, tenantVersion);
            for (KnowledgeMapsProject knowledgeMapsProject : knowledgeMapsProjects) {
                LaunchableProject project = new LaunchableProject();
                project.setName(knowledgeMapsProject.getName());
                project.setCode(knowledgeMapsProject.getCode());

                // 原先存在对应关系先保持不动
                if (this.codeSuffixMap.containsKey(knowledgeMapsProject.getCode())) {
                    project.setCreateSuffix(this.codeSuffixMap.get(knowledgeMapsProject.getCode()));
                    projectList.add(project);
                    continue;
                }

                if (CollectionUtils.isEmpty(knowledgeMapsProject.getPresetActivities())) {
                    continue;
                }

                Optional<Map<String, Object>> presetActivityItem = knowledgeMapsProject.getPresetActivities().stream().filter(e -> knowledgeMapsProject.getCode().equals(e.get("project"))).findFirst();
                if (presetActivityItem.isPresent()) {

                    // presetActivityItem 的project 对应项目code， presetActivityItem 的 code 对应 获取创建项目字段时需要的url后缀
                    String tmActivityId = (String) MapUtil.getOrDefault(presetActivityItem.get(), "code", "");
                    project.setCreateSuffix(tmActivityId);

                    for (MobileDesignerConfig designerConfig : mobileDesignerConfigList) {
                        boolean match = TaskFilter.match(tmActivityId, designerConfig.getTmActivityId(), designerConfig.getTmActivityIdMatch());
                        if (match) {
                            project.setIsDesigner(true);

                            // 设置项目介绍
                            Object projectDesc = Optional.ofNullable(knowledgeMapsProject.getLang()).map(e -> (Map<String, Object>) e.get("description")).orElse(Collections.emptyMap()).get(locale);
                            if (ObjectUtils.isNotEmpty(projectDesc)) {
                                project.setDesc((String) projectDesc);
                            }
                            projectList.add(project);
                            break;
                        }
                    }
                }
            }
        }
        List<String> tmActivityIdList = projectList.stream().map(LaunchableProject::getCode).collect(Collectors.toList());

        // 鉴权。获取当前用户有权限发起的项目
        List<DigiwinAudcActivityAccessible> projectAccessibleList = this.audcProxyService
                .getAccessibleList(locale, iamUserToken, tenantId, tmActivityIdList, 2, "start-project");

        // 无权限可发起的项目时，直接返回空列表
        if (CollectionUtils.isEmpty(projectAccessibleList)) {
            return Collections.emptyList();
        }

        List<LaunchableProject> accessibleLaunchableProjectListReturn = new ArrayList<>(projectAccessibleList.size());
        // 交叉匹配：返回所有可发起项目中有权限发起的项目
        for (DigiwinAudcActivityAccessible accessible : projectAccessibleList) {
            for (LaunchableProject launchableProject : projectList) {
                if (accessible.getTmActivityId().equals(launchableProject.getCode())
                        && accessible.getAccess().equalsIgnoreCase("allow")) {
                    accessibleLaunchableProjectListReturn.add(launchableProject);
                    break;
                }
            }
        }

        /**
         * FIXME 过滤出移动端支持发起的项目
         * 2021.11 实现 "项目备案"
         * 2022.03 实现 "问题快反"
         * 2022.08 实现 "除外回报"
         * 2023.02 实现 "发起排程修改" 2023.03 暂时隐藏入口
         * 2023.03 实现 "发起出厂物流"
         */
        accessibleLaunchableProjectListReturn = accessibleLaunchableProjectListReturn.stream()
                .filter(e -> e.isSupportedLaunchableProject() || BooleanUtils.isTrue(e.getIsDesigner()))
                .collect(Collectors.toList());

        return accessibleLaunchableProjectListReturn;
    }

    @Override
    public List<AppFunctionTouchable> getLaunchableProjectAppList(String iamUserToken, String tenantId, String locale) {
        List<LaunchableProject> accessibleLaunchableProjectList = getLaunchableProjectList(iamUserToken, tenantId, locale);
        List<AppFunctionTouchable> appList = new ArrayList<>();
        if (CollectionUtils.isEmpty(accessibleLaunchableProjectList)) {
            return appList;
        }
        appList = accessibleLaunchableProjectList.stream()
                .map(project -> {
                    AppFunctionTouchable appFunctionTouchable = new AppFunctionTouchable();
                    appFunctionTouchable.setId(project.getCode());
                    appFunctionTouchable.setName(project.getName());

                    //处理智慧检料图标
                    if (ProjectConstant.INTELLIGENT_MATERIAL_INSPECTION_PROJECT_CODE.equals(project.getCode())) {
                        appFunctionTouchable.setImage("IMAGE_" + ProjectConstant.LAUNCH_EXCLUDED_PROJECT_TM_ACTIVITY_ID);
                    } else {
                        appFunctionTouchable.setImage("IMAGE_" + project.getCreateSuffix());
                    }

                    Action action = new Action();
                    if (BooleanUtils.isTrue(project.getIsDesigner())) {
                        action.setType(ActionTypeEnum.OPEN_NEW_PAGE.getValue());
                        action.setJumpPageId(PageSettingIdPresetEnum.MOBILE_ATHENA_PROJECT_CREATE_V2_PAGE.toString());
                        action.setJumpPageTitle(project.getName());
                        Map<String, Object> rawData = new HashMap<>(2);
                        rawData.put("project_introduce", project.getDesc());
                        rawData.put("project_name", project.getName());
                        rawData.put("dataId", project.getCreateSuffix());
                        action.setRawData(rawData);
                    } else {
                        action.setType(ActionTypeEnum.OPEN_NEW_PAGE.getValue());
                        action.setJumpPageId(PageSettingIdPresetEnum.MOBILE_PROJECT_CREATE.toString());
                        action.setJumpPageTitle(project.getName());
                        action.setDataId(project.getCreateSuffix());
                    }
                    appFunctionTouchable.setAction(action);
                    AppEntranceAccessRecord accessRecord = new AppEntranceAccessRecord();
                    accessRecord.setSendRecord(true);
                    accessRecord.setRecordUrl("/mobile/v1/appEntranceRecord/add");
                    appFunctionTouchable.setAccessRecord(accessRecord);
                    return appFunctionTouchable;
                }).collect(Collectors.toList());
        return appList;
    }

    /**
     * 处理标准项目（鼎华的应用使用过的项目）
     *
     * @param project
     * @param normalCardList
     * @param abnormalCardList
     * @param iamUserToken
     * @param locale
     * @param isTeamProject
     */
    public void handleStandardProject(DigiwinAtmcProject project,
                                      List<Card> normalCardList,
                                      List<Card> abnormalCardList,
                                      String iamUserToken,
                                      String locale,
                                      String tenantId, boolean isTeamProject) {
        Card card = new Card();
        card.setId(project.getId().toString());
        card.setIsTeamProject(isTeamProject);
        card.setName(project.getName());
        card.setStartTime(project.getStartTime());
        card.setEndTime(project.getEndTime());
        card.setCreateTime(project.getCreateTime());
        card.setImportance(project.getImportance());
        List<CardListFilter> filterContentList = new ArrayList<>();
        filterContentList.add(project.getProjectType(locale));
        filterContentList.add(project.getProjectStatus(locale));
        card.setFilterContentList(filterContentList);
        Map<String, Object> rawData = new HashMap<>(1);
        if (project.getPinning() != null && project.getPinning()) {
            //被分享的项目卡
            String text = String.format(LocaleUtil.getMobileTextByKey(locale, "来自%s的分享"), project.getPersonInChargeName());
            card.setBottomTipText(text);
            rawData.put("isShare", true);
        } else {
            rawData.put("isShare", false);
        }
        if (project.getReadCount() != null && project.getReadCount() == 0) {
            //未读，加标签NEW
            Label label = Label.create("NEW", LabelTypeEnum.NEW.getType());
            List<Label> labelList = new ArrayList<>();
            labelList.add(label);
            card.setLabelList(labelList);
        }
        //action、childCard、type

        // action设置
/*        if (projectFilterService.isStandardProject(project, tenantId)) {
            Action action = new Action(project.getId() + "",
                    PageSettingIdPresetEnum.MOBILE_ATHENA_STANDARD_PROJECT_DETAIL.toString(),
                    "", ActionTypeEnum.OPEN_NEW_PAGE.getValue(),
                    "");
            action.setRawData(rawData);
            card.setAction(action);
        } else if (projectFilterService.isECNProject(project, tenantId)) {
//            card.setAction(new Action(project.getId() + "",
//                    PageSettingIdPresetEnum.MOBILE_ATHENA_ECN_PROJECT_DETAIL.toString(),
//                    "", ButtonActionTypeEnum.OPEN_NEW_PAGE.getValue(),
//                    ""));
            //设计变更详情页不可查看
            card.setAction(new Action(project.getId() + "",
                    ActionTypeEnum.SHOW_TOAST.getValue(),
                    LocaleUtil.getMobileTextByKey(locale, "暂不支持查看")));
        }*/
        Action action = new Action(project.getId() + "",
                PageSettingIdPresetEnum.MOBILE_ATHENA_STANDARD_PROJECT_DETAIL.toString(),
                "", ActionTypeEnum.OPEN_NEW_PAGE.getValue(),
                "");
        action.setRawData(rawData);
        card.setAction(action);

        /**
         * TODO 2022年12月迭代，设计变更详情页面前端还有内容未更改，故还是“暂时不支持查看”。
         * 后续更改完成后，再恢复上方注释，同时删除下方setAction的代码
         */
//        card.setAction(new Action(project.getId() + "",
//                "",
//                "", ButtonActionTypeEnum.SHOW_TOAST.getValue(),
//                LocaleUtil.getMobileTextByKey(locale,
//                        "暂不支持查看")));

        ChildCard childCard = new ChildCard();
        card.setChildCard(childCard);
        List<DigiwinAtmcTask> taskList = project.getTasks();
        //逾期/异常任务
        List<DigiwinAtmcTask> abnormalTaskList = new ArrayList<>();
        if (CollectionUtils.isEmpty(taskList)) {
            //无任务进行中
            card.setType(CardTypeEnum.GENERAL.getValue());
            childCard.setName(LocaleUtil.getMobileTextByKey(locale, "暂无任务进行中"));
            childCard.setType(CardTypeEnum.GENERAL.getValue());
            normalCardList.add(card);
            return;
        } else {
            abnormalTaskList = taskList.stream().filter(
                    task -> (Optional.ofNullable(task.getException()).orElse(false))
                            || (Optional.ofNullable(task.getOverdue()).orElse(false))
            ).collect(Collectors.toList());
        }
        //项目是否正常
        boolean normal = true;
        //无逾期/异常任务
        if (abnormalTaskList.isEmpty()) {
            card.setType(CardTypeEnum.NORMAL.getValue());
            childCard.setType(CardTypeEnum.NORMAL.getValue());
            normalCardList.add(card);
            if (taskList.size() == 1) {
                //无逾期/异常任务,且只有一个任务进行中--显示任务名称
                childCard.setName(taskList.get(0).getName());
            } else {
                //无逾期/异常任务,且有多个任务进行中--显示-共xx个任务正在进行中
                childCard.setName(String.format(LocaleUtil.getMobileTextByKey(locale, "多个任务正在进行中"), taskList.size()));
            }
            return;
        }
        //有逾期/异常任务
        card.setType(CardTypeEnum.ABNORMAL.getValue());
        childCard.setType(CardTypeEnum.ABNORMAL.getValue());
        abnormalCardList.add(card);
        if (abnormalTaskList.size() > 1) {
            //有多个逾期/异常任务--显示共xx个任务异常/逾期
            childCard.setName(String.format(LocaleUtil.getMobileTextByKey(locale, "多个任务逾期异常"), abnormalCardList.size()));
        } else if (abnormalTaskList.size() == 1) {
            //有一个逾期/异常任务--显示该任务名称
            DigiwinAtmcTask task = abnormalTaskList.get(0);
            childCard.setName(task.getName());
            if (task.getOverdue()) {
                //是逾期任务，显示逾期天数
                List<Label> labelList = new ArrayList<>();
                labelList.add(
                        new Label(String.format(LocaleUtil.getMobileTextByKey(locale, "逾期天数"), task.getOverdueDays()), LabelTypeEnum.OVERDUE.getType()));
                childCard.setLabelList(labelList);
            }
        } else {
            //没有逾期任务，但是项目逾期了
            //是逾期任务，显示逾期天数
            List<Label> labelList = new ArrayList<>();
            labelList.add(
                    new Label(String.format(LocaleUtil.getMobileTextByKey(locale, "逾期天数"), DateTimeUtil.getDateTimePeriod(project.getEndTime(), DateTimeUtil.getTodayTimeUseDefaultPattern())), LabelTypeEnum.OVERDUE.getType()));
            childCard.setLabelList(labelList);
        }
    }

    public Card handleStandardProject(DigiwinAtmcProject project, String iamUserToken, String locale, String tenantId, boolean isTeamProject) {
        Card card = new Card();
        card.convert(project);
        card.setIsTeamProject(isTeamProject);
        Map<String, Object> rawData = new HashMap<>(1);
        if (project.getPinning() != null && project.getPinning()) {
            //被分享的项目卡
            String text = String.format(LocaleUtil.getMobileTextByKey(locale, "来自%s的分享"), project.getPersonInChargeName());
            card.setBottomTipText(text);
            rawData.put("isShare", true);
        } else {
            rawData.put("isShare", false);
        }
        if (project.getReadCount() != null && project.getReadCount() == 0) {
            //未读，加标签NEW
            Label label = Label.create("NEW", LabelTypeEnum.NEW.getType());
            List<Label> labelList = new ArrayList<>();
            labelList.add(label);
            card.setLabelList(labelList);
        }
        Action action = new Action(project.getId() + "",
                PageSettingIdPresetEnum.MOBILE_ATHENA_STANDARD_PROJECT_DETAIL.toString(),
                "", ActionTypeEnum.OPEN_NEW_PAGE.getValue(),
                "");
        action.setRawData(rawData);
        card.setAction(action);
        ChildCard childCard = new ChildCard();
        card.setChildCard(childCard);
        List<DigiwinAtmcTask> taskList = project.getTasks();
        //逾期/异常任务
        List<DigiwinAtmcTask> abnormalTaskList = new ArrayList<>();
        if (CollectionUtils.isEmpty(taskList)) {
            //无任务进行中
            card.setType(CardTypeEnum.GENERAL.getValue());
            childCard.setName(LocaleUtil.getMobileTextByKey(locale, "暂无任务进行中"));
            childCard.setType(CardTypeEnum.GENERAL.getValue());
            return card;
        } else {
            abnormalTaskList = taskList.stream().filter(
                    task -> (Optional.ofNullable(task.getException()).orElse(false))
                            || (Optional.ofNullable(task.getOverdue()).orElse(false))
            ).collect(Collectors.toList());
        }
        //项目是否正常
        boolean normal = true;
        //无逾期/异常任务
        if (abnormalTaskList.isEmpty()) {
            card.setType(CardTypeEnum.NORMAL.getValue());
            childCard.setType(CardTypeEnum.NORMAL.getValue());
            if (taskList.size() == 1) {
                //无逾期/异常任务,且只有一个任务进行中--显示任务名称
                childCard.setName(taskList.get(0).getName());
            } else {
                //无逾期/异常任务,且有多个任务进行中--显示-共xx个任务正在进行中
                childCard.setName(String.format(LocaleUtil.getMobileTextByKey(locale, "多个任务正在进行中"), taskList.size()));
            }
            return card;
        }
        //有逾期/异常任务
        card.setType(CardTypeEnum.ABNORMAL.getValue());
        childCard.setType(CardTypeEnum.ABNORMAL.getValue());
        if (abnormalTaskList.size() > 1) {
            //有多个逾期/异常任务--显示共xx个任务异常/逾期
            childCard.setName(String.format(LocaleUtil.getMobileTextByKey(locale, "多个任务逾期异常"), abnormalTaskList.size()));
        } else if (abnormalTaskList.size() == 1) {
            //有一个逾期/异常任务--显示该任务名称
            DigiwinAtmcTask task = abnormalTaskList.get(0);
            childCard.setName(task.getName());
            if (task.getOverdue()) {
                //是逾期任务，显示逾期天数
                List<Label> labelList = new ArrayList<>();
                labelList.add(
                        new Label(String.format(LocaleUtil.getMobileTextByKey(locale, "逾期天数"), task.getOverdueDays()), LabelTypeEnum.OVERDUE.getType()));
                childCard.setLabelList(labelList);
            }
        } else {
            //没有逾期任务，但是项目逾期了
            //是逾期任务，显示逾期天数
            List<Label> labelList = new ArrayList<>();
            labelList.add(
                    new Label(String.format(LocaleUtil.getMobileTextByKey(locale, "逾期天数"), DateTimeUtil.getDateTimePeriod(project.getEndTime(), DateTimeUtil.getTodayTimeUseDefaultPattern())), LabelTypeEnum.OVERDUE.getType()));
            childCard.setLabelList(labelList);
        }
        return card;
    }

    /**
     * 处理体系云管家
     *
     * @param project
     * @param doingTask
     * @param normalCardList
     * @param iamUserToken
     * @param locale
     * @param isTeamProject
     */
    public void handleSystemCloudKeeperProject(DigiwinAtmcProject project,
                                               DigiwinAtmcTask doingTask,
                                               List<Card> normalCardList,
                                               List<Card> abnormalCardList,
                                               String iamUserToken,
                                               String tenantId,
                                               String locale, boolean isTeamProject) {

        //体系云管家数据处理
        Card card = new Card();
        card.setIsTeamProject(isTeamProject);
        Boolean isException = true;
        if (!project.getException() && !project.getOverdue()) {
            normalCardList.add(card);
            isException = false;
        } else {
            abnormalCardList.add(card);
        }
        card.setId(project.getId().toString());
        card.setName(project.getName());
        card.setStartTime(project.getStartTime());
        card.setEndTime(project.getEndTime());
        card.setCreateTime(project.getCreateTime());
        card.setImportance(project.getImportance());
        List<CardListFilter> filterContentList = new ArrayList<>();
        filterContentList.add(project.getProjectType(locale));
        filterContentList.add(project.getProjectStatus(locale));
        card.setFilterContentList(filterContentList);
        Boolean isShare = false;
        if (Boolean.TRUE.equals(project.getPinning())) {
            //被分享的项目卡
            String text = String.format(LocaleUtil.getMobileTextByKey(locale, "来自%s的分享"), project.getPersonInChargeName());
            card.setBottomTipText(text);
            isShare = true;
        }
        if (project.getReadCount() != null && project.getReadCount() == 0) {
            //未读，加标签NEW
            Label label = Label.create("NEW", LabelTypeEnum.NEW.getType());
            List<Label> labelList = new ArrayList<>();
            labelList.add(label);
            card.setLabelList(labelList);
        }
        //进度追踪，显示完成率、逾期率
        card.setType(CardTypeEnum.NORMAL.getValue());
        CardProgress cardProgress = new CardProgress();
        String urlTemplate = AppContext.getBaseUrl()
                + "/mobile/v1/project/progress/list?"
                + "iamUserToken=%s&tenantId=%s&locale=%s&projectId=%s&isShare=%s&type=KSC&pccProjectTypeIsNew=%s&isException=%s";
        String url = String.format(urlTemplate,
                iamUserToken,
                tenantId,
                locale,
                project.getId(),
                isShare,
                true,
                isException);
        cardProgress.setUrl(url);
        card.setProgress(cardProgress);
        Map<String, Object> rawData = new HashMap<>();
        rawData.put("project_type", project.getTmTaskId());
    }

    public Card handleSystemCloudKeeperProject(DigiwinAtmcProject project, String iamUserToken, String tenantId, String locale, boolean isTeamProject) {

        //体系云管家数据处理
        Card card = new Card();
        card.convert(project);
        card.setIsTeamProject(isTeamProject);
        boolean isException = true;
        if (!project.getException() && !project.getOverdue()) {
            isException = false;
        }
        boolean isShare = false;
        if (Boolean.TRUE.equals(project.getPinning())) {
            //被分享的项目卡
            String text = String.format(LocaleUtil.getMobileTextByKey(locale, "来自%s的分享"), project.getPersonInChargeName());
            card.setBottomTipText(text);
            isShare = true;
        }
        if (project.getReadCount() != null && project.getReadCount() == 0) {
            //未读，加标签NEW
            Label label = Label.create("NEW", LabelTypeEnum.NEW.getType());
            List<Label> labelList = new ArrayList<>();
            labelList.add(label);
            card.setLabelList(labelList);
        }
        //进度追踪，显示完成率、逾期率
        card.setType(CardTypeEnum.NORMAL.getValue());
        CardProgress cardProgress = new CardProgress();
        String urlTemplate = AppContext.getBaseUrl()
                + "/mobile/v1/project/progress/list?"
                + "iamUserToken=%s&tenantId=%s&locale=%s&projectId=%s&isShare=%s&type=KSC&pccProjectTypeIsNew=%s&isException=%s";
        String url = String.format(urlTemplate, iamUserToken, tenantId, locale, project.getId(), isShare, true, isException);
        cardProgress.setUrl(url);
        card.setProgress(cardProgress);
        return card;
    }

    /**
     * 处理项目中控台类型项目（包含旧：使用流程引擎，新：使用任务引擎）
     *
     * @param project
     * @param doingTask
     * @param normalCardList
     * @param iamUserToken
     * @param locale
     * @param isTeamProject
     */
    public void handleCenterConsoleProject(DigiwinAtmcProject project,
                                           DigiwinAtmcTask doingTask,
                                           List<Card> normalCardList,
                                           List<Card> abnormalCardList,
                                           String iamUserToken,
                                           String tenantId,
                                           String locale, boolean isTeamProject) {
        boolean pccProjectTypeIsNew = ProjectStringUtil.PCC_NEW_PROJECT_TYPE.equalsIgnoreCase(project.getTmTaskId());

        //项目中控台数据处理
        Card card = new Card();
        card.setIsTeamProject(isTeamProject);
        if (!project.getException() && !project.getOverdue()) {
            normalCardList.add(card);
        } else {
            abnormalCardList.add(card);
        }
        card.setId(project.getId().toString());
        card.setName(project.getName());
        card.setStartTime(project.getStartTime());
        card.setEndTime(project.getEndTime());
        card.setCreateTime(project.getCreateTime());
        card.setImportance(project.getImportance());
        List<CardListFilter> filterContentList = new ArrayList<>();
        filterContentList.add(project.getProjectType(locale));
        filterContentList.add(project.getProjectStatus(locale));
        card.setFilterContentList(filterContentList);
        Boolean isShare = false;
        if (Boolean.TRUE.equals(project.getPinning())) {
            //被分享的项目卡
            String text = String.format(LocaleUtil.getMobileTextByKey(locale, "来自%s的分享"), project.getPersonInChargeName());
            card.setBottomTipText(text);
            isShare = true;
        }
        if (project.getReadCount() != null && project.getReadCount() == 0) {
            //未读，加标签NEW
            Label label = Label.create("NEW", LabelTypeEnum.NEW.getType());
            List<Label> labelList = new ArrayList<>();
            labelList.add(label);
            card.setLabelList(labelList);
        }
        //进度追踪，显示完成率、逾期率
        card.setType(CardTypeEnum.NORMAL.getValue());
        CardProgress cardProgress = new CardProgress();
        String urlTemplate = AppContext.getBaseUrl()
                + "/mobile/v1/project/progress/list?"
                + "iamUserToken=%s&tenantId=%s&locale=%s&projectId=%s&isShare=%s&pccProjectTypeIsNew=%s";
        String url = String.format(urlTemplate,
                iamUserToken,
                tenantId,
                locale,
                project.getId(),
                isShare,
                pccProjectTypeIsNew);
        cardProgress.setUrl(url);
        card.setProgress(cardProgress);
        Map<String, Object> rawData = new HashMap<>();
        rawData.put("project_type", project.getTmTaskId());
    }

    public Card handleCenterConsoleProject(DigiwinAtmcProject project, String iamUserToken, String tenantId, String locale, boolean isTeamProject) {
        boolean pccProjectTypeIsNew = ProjectStringUtil.PCC_NEW_PROJECT_TYPE.equalsIgnoreCase(project.getTmTaskId());
        //项目中控台数据处理
        Card card = new Card();
        card.convert(project);
        card.setIsTeamProject(isTeamProject);
        boolean isShare = false;
        if (Boolean.TRUE.equals(project.getPinning())) {
            //被分享的项目卡
            String text = String.format(LocaleUtil.getMobileTextByKey(locale, "来自%s的分享"), project.getPersonInChargeName());
            card.setBottomTipText(text);
            isShare = true;
        }
        if (project.getReadCount() != null && project.getReadCount() == 0) {
            //未读，加标签NEW
            Label label = Label.create("NEW", LabelTypeEnum.NEW.getType());
            List<Label> labelList = new ArrayList<>();
            labelList.add(label);
            card.setLabelList(labelList);
        }
        //进度追踪，显示完成率、逾期率
        card.setType(CardTypeEnum.NORMAL.getValue());
        CardProgress cardProgress = new CardProgress();
        String urlTemplate = AppContext.getBaseUrl()
                + "/mobile/v1/project/progress/list?"
                + "iamUserToken=%s&tenantId=%s&locale=%s&projectId=%s&isShare=%s&pccProjectTypeIsNew=%s";
        String url = String.format(urlTemplate, iamUserToken, tenantId, locale, project.getId(), isShare, pccProjectTypeIsNew);
        cardProgress.setUrl(url);
        card.setProgress(cardProgress);
        return card;
    }

    /*    @Cacheable(
                cacheNames = "pccProjectProgressRate",
                keyGenerator = "cacheKeyGenerator",
                cacheManager = "projectCacheManager",
                sync = true
        )*/
    @Override
    public CardProgress getPccProjectProgress(String locale, String iamUserToken, String tenantId, String projectId, Boolean isShare, Boolean pccProjectTypeIsNew) {
        String tmTaskId = "";
        String tmActivityId = "";

        if (!pccProjectTypeIsNew) {
            //旧项目中控台项目详情的可视化看板数据（基于流程引擎）
            DigiwinAtmcPanel atmcPanel = this.digiwinAtmcProxyService.getTaskPanel(locale, iamUserToken, tenantId, projectId);
            if (null != atmcPanel.getTask()) {
                tmTaskId = atmcPanel.getTask().getTmTaskId();
            }
            if (!CollectionUtils.isEmpty(atmcPanel.getActivities())) {
                Optional<DigiwinAtmcActivity> optActivity =
                        atmcPanel.getActivities().stream().filter(atmcActivity ->
                                ProjectStringUtil.PCC_OLD_PROJECT_TYPE_PROGRESS_TRACK_ACTIVITY.equalsIgnoreCase(atmcActivity.getId())
                                        && (DigiwinAtmcActivityStateEnum.DOING.getState().equals(atmcActivity.getState())
                                        || DigiwinAtmcActivityStateEnum.DONE.getState().equals(atmcActivity.getState()))
                        ).findFirst();
                if (optActivity.isPresent()) {
                    tmActivityId = optActivity.get().getId();
                }
            }
        } else {
            //新项目中控台项目详情的可视化看板数据（基于任务引擎）
            DigiwinAtmcEnginePanelInfo atmcEnginePanelInfo = this.digiwinAtmcProxyService.getEnginePanelInfoByTaskId(
                    projectId, "", iamUserToken, tenantId, locale);

            tmTaskId = atmcEnginePanelInfo.getTask().getTmTaskId();
            List<DigiwinAtmcEngineActivity> activityList = atmcEnginePanelInfo.getRoute().getSelectRoute().getActivities();
            List<DigiwinAtmcEngineTaskCode> taskCodes = new ArrayList<>();
            for (DigiwinAtmcEngineActivity activity : activityList) {
                if (activity != null && activity.getTaskCodes() != null) {
                    taskCodes.addAll(activity.getTaskCodes());
                }
            }
            Optional<DigiwinAtmcEngineTaskCode> taskCodeOptional = taskCodes.stream()
                    .filter(taskCode ->
                            ProjectStringUtil.PCC_NEW_PROJECT_TYPE_PROGRESS_TRACK_ACTIVITY.equalsIgnoreCase(taskCode.getId())
                                    && (DigiwinAtmcActivityStateEnum.DOING.getState().equals(taskCode.getState())
                                    || DigiwinAtmcActivityStateEnum.DONE.getState().equals(taskCode.getState()))
                    ).findFirst();
            if (taskCodeOptional.isPresent()) {
                tmActivityId = taskCodeOptional.get().getId();
            }
        }
        // 非空字符串，表示找到了进行中的活动
        if (StringUtils.hasLength(tmActivityId)) {
            // 如果是PCC的新的活动名称，转成旧的，统一处理进度数据
            if (ProjectStringUtil.PCC_NEW_PROJECT_TYPE_PROGRESS_TRACK_ACTIVITY.equalsIgnoreCase(tmActivityId)) {
                tmActivityId = ProjectStringUtil.PCC_OLD_PROJECT_TYPE_PROGRESS_TRACK_ACTIVITY;
            }
        }
        return this.transCardDetailToPccProjectProgress(locale, iamUserToken, tenantId, projectId, isShare, tmActivityId, tmTaskId);
    }

    @Override
    public CardProgress getKscProjectProgress(String locale, String iamUserToken, String tenantId,
                                              String projectId, Boolean isShare, Boolean isException) {
        String tmActivityId = "";
        //新项目中控台项目详情的可视化看板数据（基于任务引擎）
        DigiwinAtmcEnginePanelInfo atmcEnginePanelInfo = this.digiwinAtmcProxyService.getEnginePanelInfoByTaskId(
                projectId, "", iamUserToken, tenantId, locale);

        List<DigiwinAtmcEngineActivity> activityList = atmcEnginePanelInfo.getRoute().getSelectRoute().getActivities();
        List<DigiwinAtmcEngineTaskCode> taskCodes = new ArrayList<>();
        for (DigiwinAtmcEngineActivity activity : activityList) {
            if (activity != null && activity.getTaskCodes() != null) {
                taskCodes.addAll(activity.getTaskCodes());
            }
        }
//        // 获取用户配置移动设计器DSL的当责者任务
//        String tenantVersion = activityFilterService.tenantVersion(iamUserToken, tenantId, locale);
//        Set<String> tmActivityIds = activityFilterService.findByTypeOfMobileDesignerConfig(MobileDesignerConfigTypeEnum.PROJECT.getValue(), tenantId, tenantVersion)
//                .stream().map(MobileDesignerConfig::getTmActivityId).collect(Collectors.toSet());
        Optional<DigiwinAtmcEngineTaskCode> taskCodeOptional = taskCodes.stream()
                .filter(taskCode ->
                                (ProjectStringUtil.KSC_PROJECT_TYPE_PROGRESS_TRACK_ACTIVITY.equalsIgnoreCase(taskCode.getId())
//                                || tmActivityIds.contains(taskCode.getId())
                                ) && (DigiwinAtmcActivityStateEnum.DOING.getState().equals(taskCode.getState())
                                        || DigiwinAtmcActivityStateEnum.DONE.getState().equals(taskCode.getState()))
                ).findFirst();
//        String tmPattern = "";
//        String tmCategory = "";
        if (taskCodeOptional.isPresent()) {
            DigiwinAtmcEngineTaskCode taskCode = taskCodeOptional.get();
            tmActivityId = taskCode.getId();
//            tmPattern = taskCode.getTmPattern();
//            tmCategory = taskCode.getTmCategory();
        }
//        if (tmActivityIds.contains(tmActivityId)) {
//            String tmTaskId = atmcEnginePanelInfo.getTask().getTmTaskId();
//            // 走设计器加载流程
//            return this.transCardDetailToKscProjectProgressV2(tmPattern, tmCategory, tmTaskId, tmActivityId, projectId, locale, isShare);
//        }
        return transCardDetailToKscProjectProgress(locale, iamUserToken, tenantId, projectId, isShare, tmActivityId, isException);
    }

//    private CardProgress transCardDetailToKscProjectProgressV2(String tmPattern, String tmCategory, String tmTaskId, String tmActivityId, String projectId, String locale, Boolean isShare) {
//        CardProgress resultCardProgress = new CardProgress();
//
//        ChildCard childCard = new ChildCard();
//        childCard.setName(this.localeService.getLanguageValue(locale, "暂无数据"));
//        childCard.setType(CardTypeEnum.GENERAL.getValue());
//        resultCardProgress.setChildCard(childCard);
//
//        Map<String, Object> rawData = new HashMap<>();
//        rawData.put("tmPattern", tmPattern);
//        rawData.put("tmCategory", tmCategory);
//        rawData.put("tmTaskId", tmTaskId);
//        rawData.put("tmActivityId", tmActivityId);
//        rawData.put("isShare", isShare);
//        resultCardProgress.setAction(new Action(projectId,
//                PageSettingIdPresetEnum.MOBILE_ATHENA_PROJECT_MILESTONE_PAGE.name(),
//                "", ActionTypeEnum.OPEN_NEW_PAGE.getValue(), rawData));
//        return resultCardProgress;
//    }

    @Override
    public CardProgress transCardDetailToKscProjectProgress(String locale, String iamUserToken,
                                                            String tenantId, String projectId,
                                                            Boolean isShare, String tmActivityId,
                                                            Boolean isException) {
        CardProgress resultCardProgress = new CardProgress();
//        // 空字符串，表示没找到进行中的活动，不支持点击打开详情
//        if (!StringUtils.hasLength(tmActivityId)) {
//            return buildNotStarted(resultCardProgress, locale, projectId.trim());
//        }
//        // 不是移动支持打开的进度追踪的活动，不支持点击打开详情
//        if (!ProjectStringUtil.KSC_PROJECT_TYPE_PROGRESS_TRACK_ACTIVITY.equalsIgnoreCase(tmActivityId)) {
//
//            return buildToWeb(resultCardProgress, locale, projectId.trim());
//        }

        DigiwinAtmcCardDetail taskCardDetail = this.digiwinAtmcProxyService
                .getDigiwinAtmcProjectCardDetail(locale, iamUserToken, tenantId, projectId);

        DigiwinAtmcSummaryLayout summaryLayout = taskCardDetail.getSummaryLayout();
        // 防护处理
        if (null == summaryLayout || CollectionUtils.isEmpty(summaryLayout.getLayout())) {
            ChildCard childCard = new ChildCard();
            childCard.setName(this.localeService.getLanguageValue(locale, "暂无数据"));
            childCard.setType(CardTypeEnum.GENERAL.getValue());
            resultCardProgress.setChildCard(childCard);

            Map<String, Object> rawData = new HashMap<>();
            rawData.put("project_type", taskCardDetail.getTmTaskId());
            rawData.put("isShare", isShare);
            resultCardProgress.setAction(new Action(
                    taskCardDetail.getProjectId().trim(),
                    PageSettingIdPresetEnum.MOBILE_ATHENA_SYSTEM_CLOUD_HOUSE_KEEPER_PROJECT_DETAIL.toString(),
                    "", ActionTypeEnum.OPEN_NEW_PAGE.getValue(), rawData));

            return resultCardProgress;
        }

        List<UiBotLayout> layoutList = summaryLayout.getLayout();
        UiBotPageData pageData = summaryLayout.getPageData();

        Map<String, Object> bizData = new HashMap<>();
        for (UiBotLayout layout : layoutList) {
            if (layout.isTypeOfTable()) {
                List<Map<String, Object>> bizDataList = (List<Map<String, Object>>) pageData.get(layout.getSchema());
                if (!CollectionUtils.isEmpty(bizDataList)) {
                    bizData = bizDataList.get(0);
                }
            } else if (layout.isTypeOfFormList()) {
                Object o = pageData.get(layout.getSchema());
                if (o instanceof Map) {
                    Map<String, Object> map = (Map<String, Object>) o;
                    if (!CollectionUtils.isEmpty(map)) {
                        bizData = map;
                    }
                }
            }
        }
        String projectStatus = String.valueOf(bizData.get("project_status"));
        switch (projectStatus) {
            case "20":
                //项目签核中
                buildApprovals(resultCardProgress, locale, taskCardDetail);
                break;
            case "10": // 未开始
            case "30": // 进行中
            case "50": // 暂停
                //显示逾期率、完成率
                String notice = (String) bizData.get("project_progress_message");
                ChildCard childCard = resultCardProgress.getChildCard();
                if (childCard == null) {
                    childCard = new ChildCard();
                }
                childCard.setName(notice);
                if (isException) {
                    childCard.setType(CardTypeEnum.ABNORMAL.getValue());
                } else {
                    childCard.setType(CardTypeEnum.NORMAL.getValue());
                }
                resultCardProgress.setChildCard(childCard);
                Map<String, Object> rawData = new HashMap<>(2);
                rawData.put("project_type", taskCardDetail.getTmTaskId());
                rawData.put("isShare", isShare);
                rawData.put("project_status", projectStatus);
                resultCardProgress.setAction(new Action(
                        taskCardDetail.getProjectId() + "",
                        PageSettingIdPresetEnum.MOBILE_ATHENA_SYSTEM_CLOUD_HOUSE_KEEPER_PROJECT_DETAIL.toString(),
                        "", ActionTypeEnum.OPEN_NEW_PAGE.getValue(), rawData));
                break;
            case "60":
                // 指定结案。理论上不应该能看到该数据，但当数据异常时会出现，故还是做兼容来提示用户
                buildCloseCase(resultCardProgress, locale, taskCardDetail);
                break;
            default:
                // 项目未启动
                buildNotStarted(resultCardProgress, locale, taskCardDetail.getProjectId().trim());
                break;
        }
        return resultCardProgress;
    }

    private CardProgress buildToWeb(CardProgress resultCardProgress, String locale,
                                    String id) {
        ChildCard notSupportedActivityChildCard = new ChildCard();
        notSupportedActivityChildCard.setName(this.localeService.getLanguageValue(locale, "专案当前里程碑不支持展示"));
        notSupportedActivityChildCard.setType(CardTypeEnum.GENERAL.getValue());
        resultCardProgress.setChildCard(notSupportedActivityChildCard);
        resultCardProgress.setAction(new Action(id,
                ActionTypeEnum.SHOW_TOAST.getValue(),
                this.localeService.getLanguageValue(locale, "专案当前里程碑不支持展示，建议去PC端查看")));

        return resultCardProgress;
    }

    /**
     * 返回专案签核中
     *
     * @param resultCardProgress
     * @param locale
     * @param taskCardDetail
     * @return
     */
    private CardProgress buildApprovals(CardProgress resultCardProgress, String locale,
                                        DigiwinAtmcCardDetail taskCardDetail) {
        ChildCard childCard = new ChildCard();
        childCard.setName(this.localeService.getLanguageValue(locale, "专案签核中"));
        childCard.setType(CardTypeEnum.GENERAL.getValue());
        resultCardProgress.setChildCard(childCard);
        resultCardProgress.setAction(new Action(taskCardDetail.getProjectId().trim(),
                ActionTypeEnum.SHOW_TOAST.getValue(),
                this.localeService.getLanguageValue(locale, "此专案正在签核中，建议去PC端维护")));
        return resultCardProgress;
    }

    /**
     * 已指定结案
     *
     * @param resultCardProgress
     * @param locale
     * @param taskCardDetail
     * @return
     */
    private CardProgress buildCloseCase(CardProgress resultCardProgress, String locale,
                                        DigiwinAtmcCardDetail taskCardDetail) {
        ChildCard endChildCard = new ChildCard();
        endChildCard.setName(this.localeService.getLanguageValue(locale, "专案已指定结案"));
        endChildCard.setType(CardTypeEnum.GENERAL.getValue());
        resultCardProgress.setChildCard(endChildCard);
        resultCardProgress.setAction(new Action(taskCardDetail.getProjectId().trim(),
                ActionTypeEnum.SHOW_TOAST.getValue(),
                this.localeService.getLanguageValue(locale, "此专案已指定结案，属于数据异常，请检查后台数据")));
        return resultCardProgress;
    }

    /**
     * 未启动
     *
     * @param resultCardProgress
     * @param locale
     * @param id
     * @return
     */
    private CardProgress buildNotStarted(CardProgress resultCardProgress, String locale,
                                         String id) {
        ChildCard notStartChildCard = new ChildCard();
        notStartChildCard.setName(this.localeService.getLanguageValue(locale, "专案未启动"));
        notStartChildCard.setType(CardTypeEnum.GENERAL.getValue());
        resultCardProgress.setChildCard(notStartChildCard);
        resultCardProgress.setAction(new Action(id,
                ActionTypeEnum.SHOW_TOAST.getValue(),
                this.localeService.getLanguageValue(locale, "此专案尚未启动，建议去PC端维护")));
        return resultCardProgress;
    }

    /**
     * 未启动
     *
     * @param resultCardProgress
     * @param locale
     * @param id
     * @return
     */
    private CardProgress buildPccNotStarted(CardProgress resultCardProgress, String locale,
                                            String id, Map<String, Object> rawData) {
        ChildCard notStartChildCard = new ChildCard();
        notStartChildCard.setName(this.localeService.getLanguageValue(locale, "专案未启动"));
        notStartChildCard.setType(CardTypeEnum.GENERAL.getValue());
        resultCardProgress.setChildCard(notStartChildCard);
        resultCardProgress.setAction(new Action(
                id,
                PageSettingIdPresetEnum.MOBILE_ATHENA_PROJECT_PCC_EMPTY_PAGE.toString(),
                ActionTypeEnum.OPEN_NEW_PAGE.getValue(), rawData));
        return resultCardProgress;
    }

    @Override
    public CardProgress transCardDetailToPccProjectProgress(String locale, String iamUserToken, String tenantId,
                                                            String projectId, Boolean isShare, String tmActivityId, String tmTaskId) {

        DigiwinAtmcCardDetail taskCardDetail = this.digiwinAtmcProxyService
                .getDigiwinAtmcProjectCardDetail(locale, iamUserToken, tenantId, projectId);

        CardProgress resultCardProgress = new CardProgress();
        // 空字符串，表示没找到进行中的活动，不支持点击打开详情
        if (!StringUtils.hasLength(tmActivityId)) {
            Map<String, Object> rawData = new HashMap<>();
            rawData.put("project_type", taskCardDetail.getTmTaskId());
            rawData.put("isShare", isShare);
            rawData.put("tmTaskId", tmTaskId);
            return buildPccNotStarted(resultCardProgress, locale, projectId.trim(), rawData);
        }
        // 不是移动支持打开的进度追踪的活动，不支持点击打开详情
        if (!ProjectStringUtil.PCC_OLD_PROJECT_TYPE_PROGRESS_TRACK_ACTIVITY.equalsIgnoreCase(tmActivityId)) {
            return buildToWeb(resultCardProgress, locale, projectId.trim());
        }

        DigiwinAtmcSummaryLayout summaryLayout = taskCardDetail.getSummaryLayout();
        // 防护处理
        if (null == summaryLayout || CollectionUtils.isEmpty(summaryLayout.getLayout())) {
            ChildCard childCard = new ChildCard();
            childCard.setName(this.localeService.getLanguageValue(locale, "暂无数据"));
            childCard.setType(CardTypeEnum.GENERAL.getValue());
            resultCardProgress.setChildCard(childCard);

            Map<String, Object> rawData = new HashMap<>();
            rawData.put("project_type", taskCardDetail.getTmTaskId());
            rawData.put("isShare", isShare);
            resultCardProgress.setAction(new Action(
                    taskCardDetail.getProjectId().trim(),
                    PageSettingIdPresetEnum.MOBILE_ATHENA_PROJECT_CC_PROJECT_DETAIL.toString(),
                    "", ActionTypeEnum.OPEN_NEW_PAGE.getValue(), rawData));

            return resultCardProgress;
        }

        List<UiBotLayout> layoutList = summaryLayout.getLayout();
        UiBotPageData pageData = summaryLayout.getPageData();

        Map<String, Object> bizData = new HashMap<>();
        for (UiBotLayout layout : layoutList) {
            if (layout.isTypeOfTable()) {
                List<Map<String, Object>> bizDataList = (List<Map<String, Object>>) pageData.get(layout.getSchema());
                if (!CollectionUtils.isEmpty(bizDataList)) {
                    bizData = bizDataList.get(0);
                }
            } else if (layout.isTypeOfFormList()) {
                Object o = pageData.get(layout.getSchema());
                if (o instanceof Map) {
                    Map<String, Object> map = (Map<String, Object>) o;
                    if (!CollectionUtils.isEmpty(map)) {
                        bizData = map;
                    }
                }
            }
        }
        String projectStatus = String.valueOf(bizData.get("project_status"));
        switch (projectStatus) {
            case "20":
                //项目签核中
                buildApprovals(resultCardProgress, locale, taskCardDetail);
                break;
            case "30": // 进行中
            case "50": // 暂停
                //显示逾期率、完成率
                List<ProgressRate> progressRateList = new ArrayList<>();
                double projectCompleteRate = Double.valueOf(bizData.getOrDefault("project_complete_rate", 0.0).toString());
                //完成率
                ProgressRate completeProgressRate = new ProgressRate();
                completeProgressRate.setRate(MathUtil.getDecimalPoint(projectCompleteRate, 2));
                completeProgressRate.setType(ProgressRateTypeEnum.NORMAL.getType());
                completeProgressRate.setName(this.localeService.getLanguageValue(locale, "完成率"));
                progressRateList.add(completeProgressRate);
                //逾期率
                double projectOverdueRate = Double.valueOf(bizData.getOrDefault("project_overdue_rate", 0.0).toString());
                if (projectOverdueRate != 0) {
                    ProgressRate overdueProgressRate = new ProgressRate();
                    overdueProgressRate.setRate(MathUtil.getDecimalPoint(projectOverdueRate, 2));
                    overdueProgressRate.setType(ProgressRateTypeEnum.EXCEPTION.getType());
                    overdueProgressRate.setName(this.localeService.getLanguageValue(locale, "逾期率"));
                    progressRateList.add(overdueProgressRate);
                }
                resultCardProgress.setProgressRateList(progressRateList);

                Map<String, Object> rawData = new HashMap<>();
                rawData.put("project_type", taskCardDetail.getTmTaskId());
                rawData.put("isShare", isShare);
                resultCardProgress.setAction(new Action(
                        taskCardDetail.getProjectId() + "",
                        PageSettingIdPresetEnum.MOBILE_ATHENA_PROJECT_CC_PROJECT_DETAIL.toString(),
                        "", ActionTypeEnum.OPEN_NEW_PAGE.getValue(), rawData));
                break;
            case "60":
                // 指定结案。理论上不应该能看到该数据，但当数据异常时会出现，故还是做兼容来提示用户
                buildCloseCase(resultCardProgress, locale, taskCardDetail);
                break;
            default:
                // 项目未启动
                buildNotStarted(resultCardProgress, locale, taskCardDetail.getProjectId().trim());
                break;
        }
        return resultCardProgress;
    }


    /**
     * 项目名称(简体：首字首字母升序，繁体：首字笔画升序)->查阅状态(未读优先)->项目时间(按结束时间升序)
     */
    @Override
    public List<Card> sortCardList(List<Card> cardList, String locale) {
        // 简体中文下和pc保持一致，pc为数字-》中文-》英文
        // SIMPLIFIED_CHINESE compare 排序后为数字-》英文-》中文， 拆分开英文和非英文，各种排序后再合并
        if ("zh_CN".equalsIgnoreCase(locale)) {
            List<Card> startWithEnList = new LinkedList<>();
            List<Card> startWithNotEnList = new LinkedList<>();
            for (Card card : cardList) {
                if (org.apache.commons.lang3.StringUtils.isNotEmpty(card.getName())
                        && String.valueOf(card.getName().charAt(0)).matches("^[a-zA-Z]")) {
                    startWithEnList.add(card);
                } else {
                    startWithNotEnList.add(card);
                }
            }
            startWithNotEnList = sortCardListByType(startWithNotEnList, locale);
            startWithEnList = sortCardListByType(startWithEnList, locale);
            startWithNotEnList.addAll(startWithEnList);
            return startWithNotEnList;
        } else {
            return sortCardListByType(cardList, locale);
        }
    }

    public List<Card> sortCardListByType(List<Card> cardList, String locale) {
        return cardList.stream().sorted((o1, o2) -> {
                    Collator collator;
                    if ("zh_CN".equalsIgnoreCase(locale)) {
                        collator = Collator.getInstance(Locale.SIMPLIFIED_CHINESE);
                    } else {
                        collator = Collator.getInstance(Locale.TRADITIONAL_CHINESE);
                    }

                    if (!org.apache.commons.lang3.StringUtils.equals(o1.getName(), o2.getName())) {
                        return collator.compare(o1.getName(), o2.getName());
                    }

                    boolean isNewO1 = !CollectionUtils.isEmpty(o1.getLabelList()) ?
                            o1.getLabelList().stream().anyMatch(e -> "NEW".equalsIgnoreCase(e.getType())) : false;
                    boolean isNewO2 = !CollectionUtils.isEmpty(o2.getLabelList()) ?
                            o2.getLabelList().stream().anyMatch(e -> "NEW".equalsIgnoreCase(e.getType())) : false;
                    if (isNewO1 ^ isNewO2) {
                        return isNewO1 ? -1 : 1;
                    }

                    LocalDateTime o1EndTime = DateTimeUtil.parseUseDefaultPattern(o1.getEndTime());
                    LocalDateTime o2EndTime = DateTimeUtil.parseUseDefaultPattern(o2.getEndTime());
                    if (!o1EndTime.isEqual(o2EndTime)) {
                        return o1EndTime.isBefore(o2EndTime) ? -1 : 1;
                    }
                    return 0;
                }
        ).collect(Collectors.toList());
    }


    /**
     * 是否是指定的体系云租户
     */
    public boolean validIsMatchTenantId(String tenantId) {
        return NEED_SORTCARD_BY_TENANTID_LIST.contains(tenantId);
    }

    /**
     * 按创建时间降序
     */
    public List<Card> sortCardListByCreateTime(List<Card> cardList) {
        return cardList.stream().sorted((o1, o2) -> {
                    LocalDateTime o1CreateTime = DateTimeUtil.parseUseDefaultPattern(o1.getCreateTime());
                    LocalDateTime o2CreateTime = DateTimeUtil.parseUseDefaultPattern(o2.getCreateTime());
                    return o2CreateTime.compareTo(o1CreateTime);
                }
        ).collect(Collectors.toList());
    }


    @Data
    public class ProjectAndDoingTask {
        private DigiwinAtmcProject project;
        /**
         * 项目正在进行的任务
         */
        private DigiwinAtmcTask task;

        /**
         * 旧项目中控台正在进行的任务
         */
        @Deprecated
        private DigiwinAtmcActivity activity;
        /**
         * 新项目中控台正在进行的任务
         */
        @Deprecated
        private DigiwinAtmcEngineTaskCode taskCode;

        private ProjectAndDoingTask() {
        }

        public ProjectAndDoingTask(DigiwinAtmcProject project, DigiwinAtmcTask task) {
            this.project = project;
            this.task = task;
        }

        @Deprecated
        public ProjectAndDoingTask(DigiwinAtmcProject project, DigiwinAtmcActivity activity) {
            this.project = project;
            this.activity = activity;
        }

        @Deprecated
        public ProjectAndDoingTask(DigiwinAtmcProject project, DigiwinAtmcEngineTaskCode taskCode) {
            this.project = project;
            this.taskCode = taskCode;
        }
    }
}