package com.digiwin.mobile.mobileuibot.project.strategy.custom;

import com.alibaba.fastjson.JSONObject;
import com.digiwin.mobile.mobileuibot.api.ApiRequest;
import com.digiwin.mobile.mobileuibot.api.ApiRequestSubmit;
import com.digiwin.mobile.mobileuibot.common.context.AppContext;
import com.digiwin.mobile.mobileuibot.common.datetime.DateTimeUtil;
import com.digiwin.mobile.mobileuibot.common.exception.ServiceException;
import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.common.map.MapUtil;
import com.digiwin.mobile.mobileuibot.core.component.action.Action;
import com.digiwin.mobile.mobileuibot.core.component.action.ActionSubmitParam;
import com.digiwin.mobile.mobileuibot.core.component.action.ActionSubmitResult;
import com.digiwin.mobile.mobileuibot.core.component.action.ActionTypeEnum;
import com.digiwin.mobile.mobileuibot.core.component.basic.MessageBar;
import com.digiwin.mobile.mobileuibot.core.component.button.BottomButtonDigiwinAthena;
import com.digiwin.mobile.mobileuibot.core.component.group.CustomGroup;
import com.digiwin.mobile.mobileuibot.core.component.group.CustomGroupContent;
import com.digiwin.mobile.mobileuibot.core.component.input.calendar.InputCalendar;
import com.digiwin.mobile.mobileuibot.core.component.input.calendar.InputCalendarDate;
import com.digiwin.mobile.mobileuibot.core.component.input.windowselect.InputWindowSelectStyleEnum;
import com.digiwin.mobile.mobileuibot.core.component.input.windowselect.single.InputWindowSingleSelect;
import com.digiwin.mobile.mobileuibot.core.pagesetting.PageSetting;
import com.digiwin.mobile.mobileuibot.core.pagesetting.PageSettingIdPresetEnum;
import com.digiwin.mobile.mobileuibot.core.strategy.modelbuild.UiBotModelBuildStrategy;
import com.digiwin.mobile.mobileuibot.locale.service.LocaleService;
import com.digiwin.mobile.mobileuibot.proxy.DigiwinAthenaApiResponse;
import com.digiwin.mobile.mobileuibot.proxy.atdm.service.DigiwinAtdmProxyService;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.UiBotModel;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.layout.UiBotLayout;
import com.fasterxml.jackson.core.type.TypeReference;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.StopWatch;

import java.io.IOException;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * <p>功能描述：体系云管家的项目详情选择模板界面构造</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: ProjectDetailBuildSystemCloudHouseKeeperChooseTemplateStrategy
 * @Author: xujinc
 * @Date: 2024/10/21
 */
@Component("projectDetailBuildSystemCloudHouseKeeperChooseTemplateStrategy")
public class ProjectDetailBuildSystemCloudHouseKeeperChooseTemplateStrategy implements UiBotModelBuildStrategy {
    private static final Logger logger =
            LoggerFactory.getLogger(ProjectDetailBuildSystemCloudHouseKeeperChooseTemplateStrategy.class);

    @Autowired
    private DigiwinAtdmProxyService digiwinAtdmProxyService;

    @Autowired
    private LocaleService localeService;

    @Override
    public UiBotModel buildModel(PageSetting pageSetting, ApiRequest apiRequest)
            throws IOException {
        StopWatch stopWatch = new StopWatch("SystemCloudHouseKeeper Choose Template Page");
        UiBotModel uiBotModel = null;
        Boolean useMockData = AppContext.getUseMockData();
        if (!useMockData) {
            uiBotModel = this.doActually(pageSetting, apiRequest, stopWatch);
        } else {
            uiBotModel = this.doMockData(pageSetting, apiRequest);
        }
        return uiBotModel;
    }

    private UiBotModel doActually(PageSetting pageSetting, ApiRequest apiRequest,
                                  StopWatch stopWatch) {
        UiBotModel uiBotModel = pageSetting.getPageModel();
        String locale = apiRequest.getLocale();
        Map<String, Object> rawData = apiRequest.getRawData().getInnerMap();

        Map<String, Object> projectInfoMap = (Map<String, Object>) MapUtil.getOrDefault(rawData, "project_info", Collections.emptyMap());
        List<Map<String, Object>> projectInfoList = (List<Map<String, Object>>) MapUtil.getOrDefault(projectInfoMap, "projectInfoList", Collections.emptyList());
        Map<String, Object> projectInfo = CollectionUtils.isNotEmpty(projectInfoList) ? projectInfoList.get(0) : Collections.emptyMap();

        // 拼成最终数据
        stopWatch.start("Build SystemCloudHouseKeeper Choose Template Page Model...");
        List<UiBotLayout> uiBotLayouts = uiBotModel.getLayout();
        for (UiBotLayout layout : uiBotLayouts) {
            String schema = layout.getSchema();
            switch (layout.getType()) {
                case MessageBar.COMPONENT_TYPE:
                    MessageBar messageBar = JsonUtil.objectToJavaObject(uiBotModel.getPageData().get(schema), MessageBar.class);
                    messageBar.setContent(localeService.getLanguageValue(locale, messageBar.getContent()));
                    uiBotModel.getPageData().put(schema, messageBar);
                    break;
                case CustomGroup.COMPONENT_TYPE:
                    CustomGroup customGroup = JsonUtil.objectToJavaObject(uiBotModel.getPageData().get(schema), CustomGroup.class);
                    handleCustomGroup(customGroup, projectInfo, locale);
                    uiBotModel.getPageData().put(schema, customGroup);
                    break;
                case BottomButtonDigiwinAthena.COMPONENT_TYPE:
                    List<BottomButtonDigiwinAthena> buttonList = JsonUtil.objectToJavaObject(uiBotModel.getPageData().get(schema), new TypeReference<List<BottomButtonDigiwinAthena>>() {
                    });
                    buttonList.forEach(bottomButton -> {
                        bottomButton.setName(localeService.getLanguageValue(locale, bottomButton.getName()));
                        if (bottomButton.getIsCheckButton()) {
                            Action action = bottomButton.getAction();
                            action.setRefreshPageId(apiRequest.getRefreshPageId());
                            rawData.put("actionId", "bm.pisc.project.template.copy.process");
                            rawData.put("project_no", projectInfo.get("projectNo"));
                            action.setRawData(rawData);
                        }
                    });
                    uiBotModel.getPageData().put(schema, buttonList);
                    break;
                default:
                    break;
            }
        }

        stopWatch.stop();
        return uiBotModel;
    }

    private void handleCustomGroup(CustomGroup customGroup, Map<String, Object> projectInfo, String locale) {
        CustomGroupContent customGroupContent = customGroup.getContent();
        List<UiBotLayout> customGroupUiBotLayouts = customGroupContent.getLayout();
        for (UiBotLayout customGroupUiBotLayout : customGroupUiBotLayouts) {
            String customGroupSchema = customGroupUiBotLayout.getSchema();
            switch (customGroupUiBotLayout.getType()) {
                case InputWindowSingleSelect.COMPONENT_TYPE:
                    InputWindowSingleSelect inputWindowSingleSelect = buildInputWindowSingleSelect(customGroupUiBotLayout, projectInfo,
                            customGroupSchema, locale);
                    customGroupContent.getPageData().put(customGroupSchema, inputWindowSingleSelect);
                    break;
                case InputCalendar.COMPONENT_TYPE:
                    InputCalendar inputCalendar = buildInputCalendar(customGroupUiBotLayout, projectInfo, locale);
                    customGroupContent.getPageData().put(customGroupSchema, inputCalendar);
                    break;
            }
        }
    }

    /**
     * 构建单选组件
     *
     * @param uiBotLayout
     * @param projectInfo
     * @param schema
     * @param locale
     * @return
     */
    private InputWindowSingleSelect buildInputWindowSingleSelect(UiBotLayout uiBotLayout, Map<String, Object> projectInfo, String schema, String locale) {
        InputWindowSingleSelect inputWindowSingleSelect = new InputWindowSingleSelect();
        inputWindowSingleSelect.setSchema(uiBotLayout.getSchema());
        inputWindowSingleSelect.setTitle(localeService.getLanguageValue(locale, uiBotLayout.getLabel()));
        inputWindowSingleSelect.setUrl(AppContext.getBaseUrl() + "/mobile/v1/proxy/window/data/list?componentSchema=" + schema);
        inputWindowSingleSelect.setRawData(buildInputWindowMultiSelectRawData(projectInfo));
        inputWindowSingleSelect.setType(InputWindowSingleSelect.COMPONENT_TYPE);
        inputWindowSingleSelect.setStyle(InputWindowSelectStyleEnum.BG_STYLE_WHITE.getValue());
        inputWindowSingleSelect.setPlaceholder(localeService.getLanguageValue(locale, "请选择"));
        inputWindowSingleSelect.setRequired(true);
        return inputWindowSingleSelect;
    }

    /**
     * 构建日期组件
     *
     * @param uiBotLayout
     * @param projectInfo
     * @param locale
     * @return
     */
    private InputCalendar buildInputCalendar(UiBotLayout uiBotLayout, Map<String, Object> projectInfo, String locale) {
        uiBotLayout.setHeaderName(localeService.getLanguageValue(locale, uiBotLayout.getLabel()));
        InputCalendar inputCalendar = InputCalendar.create(uiBotLayout, null, locale);
        String planStartDate = MapUtils.getString(projectInfo, "planStartDate");
        if (StringUtils.isNotEmpty(planStartDate)) {
            LocalDate localDate = DateTimeUtil.parseLocalDateUseDefaultPattern(planStartDate);
            InputCalendarDate inputCalendarDate = InputCalendarDate.create(localDate.getYear(), localDate.getMonthValue(), localDate.getDayOfMonth());
            inputCalendar.setMinDateSelectable(inputCalendarDate);
            inputCalendar.setDateValueList(Collections.singletonList(inputCalendarDate));
        }
        inputCalendar.setRequired(true);
        inputCalendar.setHidden(true);
        return inputCalendar;
    }

    private Map<String, Object> buildInputWindowMultiSelectRawData(Map<String, Object> projectInfo) {
        List<Map<String, Object>> searchInfoList = new ArrayList<>(3);
        Map<String, Object> searchItemOrConditionMap = new HashMap<>(6);
        searchItemOrConditionMap.put("bracket", "(");
        searchItemOrConditionMap.put("logic", "OR");
        searchItemOrConditionMap.put("order", 1);
        searchItemOrConditionMap.put("search_field", "project_type_no");
        searchItemOrConditionMap.put("search_operator", "equal");
        searchItemOrConditionMap.put("search_value", Collections.singletonList(MapUtils.getString(projectInfo, "projectTypeNo")));
        searchInfoList.add(searchItemOrConditionMap);

        Map<String, Object> searchItemAndConditionMap = new HashMap<>(6);
        searchItemAndConditionMap.put("bracket", ")");
        searchItemAndConditionMap.put("logic", "AND");
        searchItemAndConditionMap.put("order", 2);
        searchItemAndConditionMap.put("search_field", "project_type_no");
        searchItemAndConditionMap.put("search_operator", "equal");
        searchItemAndConditionMap.put("search_value", Collections.singletonList(""));
        searchInfoList.add(searchItemAndConditionMap);

        Map<String, Object> searchItemNullConditionMap = new HashMap<>(6);
        searchItemNullConditionMap.put("order", 3);
        searchItemNullConditionMap.put("search_field", "is_default");
        searchItemNullConditionMap.put("search_operator", "equal");
        searchItemNullConditionMap.put("search_value", Collections.singletonList(false));
        searchInfoList.add(searchItemNullConditionMap);

        Map<String, Object> parasMap = new HashMap<>(2);
        parasMap.put("query_condition", "M1");
        parasMap.put("search_info", searchInfoList);

        Map<String, Object> requestMap = new HashMap<>(2);
        requestMap.put("actionId", "bm.pisc.project.template.get");
        requestMap.put("parameter", parasMap);

        Map<String, Object> rawDataMap = new HashMap<>(2);
        rawDataMap.put("requestParam", requestMap);
        rawDataMap.put("type", "isSCHEditPlanTemplateInfo");
        return rawDataMap;
    }

    /**
     * 提交
     *
     * @param apiRequestSubmit
     * @return
     */
    public ActionSubmitResult submit(ApiRequestSubmit apiRequestSubmit) {
        String locale = apiRequestSubmit.getLocale();
        Map<String, Object> rawData = apiRequestSubmit.getRawData().getInnerMap();

        Map<String, Object> projectInfoMap = new HashMap<>(3);
        projectInfoMap.put("project_no", rawData.get("project_no"));
        for (ActionSubmitParam actionSubmitParam : apiRequestSubmit.getSubmitParams()) {
            switch (actionSubmitParam.getType()) {
                case InputWindowSingleSelect.COMPONENT_TYPE:
                    InputWindowSingleSelect inputWindowSingleSelect = JsonUtil.objectToJavaObject(actionSubmitParam.getParams(), InputWindowSingleSelect.class);
                    String projectTemplateNo = Optional.ofNullable(inputWindowSingleSelect.getDetail()).map(e -> MapUtils.getString(e, "project_template_no")).orElse("");
                    projectInfoMap.put("project_template_no", projectTemplateNo);
                    break;
                case InputCalendar.COMPONENT_TYPE:
                    InputCalendar inputCalendar = (InputCalendar) actionSubmitParam.getParams();
                    if (StringUtils.isNotEmpty(inputCalendar.getInputText())) {
                        projectInfoMap.put("plan_rearrange_base_date", inputCalendar.getInputText().replaceAll("\\.", "-"));
                    }
                    break;
            }
        }

        Map<String, Object> parameterMap = new HashMap<>(2);
        parameterMap.put("is_clear_person_info", true);
        parameterMap.put("project_info", Collections.singletonList(projectInfoMap));

        Map<String, Object> requestMap = new HashMap<>(2);
        requestMap.put("actionId", rawData.get("actionId"));
        requestMap.put("parameter", parameterMap);
        DigiwinAthenaApiResponse<JSONObject> resp = digiwinAtdmProxyService
                .handleByActionId(locale, apiRequestSubmit.getIamUserToken(), apiRequestSubmit.getTenantId(), "", requestMap);

        if (resp.getStatus() != 200 || StringUtils.isNotEmpty(resp.getErrorMessage())) {
            throw new ServiceException(resp.getErrorMessage());
        } else {
            List<Map<String, Object>> projectInfoList = Optional.ofNullable(resp.getResponse())
                    .map(e -> (List<Map<String, Object>>) MapUtil.getOrDefault(e, "project_info", Collections.emptyList()))
                    .orElse(Collections.emptyList());
            String errorMsg = projectInfoList.stream()
                    .filter(e -> StringUtils.isNotEmpty(MapUtils.getString(e, "error_msg")))
                    .findFirst().map(e -> MapUtils.getString(e, "error_msg"))
                    .orElse("");
            if (StringUtils.isNotEmpty(errorMsg)) {
                throw new ServiceException(errorMsg);
            }

            Action action = new Action<>();
            action.setType(ActionTypeEnum.BACK_ONE_WITH_REFRESH.getValue());
            action.setRefreshPageId(PageSettingIdPresetEnum.MOBILE_ATHENA_SYSTEM_CLOUD_HOUSE_KEEPER_PROJECT_DETAIL.name());
            action.setSuccessMsg(localeService.getLanguageValue(locale, "提交成功"));
            return ActionSubmitResult.create(true, action);
        }
    }

    private UiBotModel doMockData(PageSetting pageSetting, ApiRequest apiRequest) {
        return null;
    }
}
