package com.digiwin.mobile.mobileuibot.project.strategy.custom;

import com.alibaba.fastjson.JSONObject;
import com.digiwin.mobile.mobileuibot.api.ApiRequest;
import com.digiwin.mobile.mobileuibot.api.ApiRequestSubmit;
import com.digiwin.mobile.mobileuibot.common.exception.ServiceException;
import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.common.localization.LocaleUtil;
import com.digiwin.mobile.mobileuibot.common.map.MapUtil;
import com.digiwin.mobile.mobileuibot.common.math.MathUtil;
import com.digiwin.mobile.mobileuibot.core.component.action.Action;
import com.digiwin.mobile.mobileuibot.core.component.action.ActionSubmitResult;
import com.digiwin.mobile.mobileuibot.core.component.action.ActionTypeEnum;
import com.digiwin.mobile.mobileuibot.core.component.action.ConfirmPop;
import com.digiwin.mobile.mobileuibot.core.component.basic.AutoFill;
import com.digiwin.mobile.mobileuibot.core.component.basic.AutoFillData;
import com.digiwin.mobile.mobileuibot.core.component.basic.Empty;
import com.digiwin.mobile.mobileuibot.core.component.button.BottomButtonDigiwinAthena;
import com.digiwin.mobile.mobileuibot.core.component.button.ButtonGroupAlignEnum;
import com.digiwin.mobile.mobileuibot.core.component.divider.Divider;
import com.digiwin.mobile.mobileuibot.core.component.divider.DividerTypeEnum;
import com.digiwin.mobile.mobileuibot.core.component.navbar.pagedesc.PageDescBuilderFactory;
import com.digiwin.mobile.mobileuibot.core.component.progress.ProgressRate;
import com.digiwin.mobile.mobileuibot.core.component.progress.ProgressRateTypeEnum;
import com.digiwin.mobile.mobileuibot.core.component.tab.customtab.CustomTabs;
import com.digiwin.mobile.mobileuibot.core.component.tab.customtab.CustomTabsBuilder;
import com.digiwin.mobile.mobileuibot.core.component.title.titlebody.TitleBody;
import com.digiwin.mobile.mobileuibot.core.component.title.titledesc.TitleDesc;
import com.digiwin.mobile.mobileuibot.core.component.title.titledesc.TitleDescProjectCCBuilder;
import com.digiwin.mobile.mobileuibot.core.pagesetting.PageSetting;
import com.digiwin.mobile.mobileuibot.core.pagesetting.PageSettingIdPresetEnum;
import com.digiwin.mobile.mobileuibot.locale.service.LocaleService;
import com.digiwin.mobile.mobileuibot.project.common.ProjectTypeUtil;
import com.digiwin.mobile.mobileuibot.project.strategy.ProjectDetailBuildStrategy;
import com.digiwin.mobile.mobileuibot.proxy.DigiwinAthenaApiResponse;
import com.digiwin.mobile.mobileuibot.proxy.atdm.model.pcc.PccApiDataWrapper;
import com.digiwin.mobile.mobileuibot.proxy.atdm.model.pcc.PccProjectInfo;
import com.digiwin.mobile.mobileuibot.proxy.atdm.model.pcc.PccProjectTaskInfo;
import com.digiwin.mobile.mobileuibot.proxy.atdm.service.DigiwinAtdmProxyService;
import com.digiwin.mobile.mobileuibot.proxy.atmc.model.DigiwinAtmcPanel;
import com.digiwin.mobile.mobileuibot.proxy.knowledgemaps.service.DigiwinKnowledgeMapsProxyService;
import com.digiwin.mobile.mobileuibot.proxy.taskengine.service.DigiwinTaskEngineProxyService;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.UiBotModel;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.layout.UiBotLayout;
import com.fasterxml.jackson.core.type.TypeReference;
import lombok.SneakyThrows;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StopWatch;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.stream.Collectors;

/**
 * <p>功能描述：体系云管家的项目详情界面构造</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: ProjectDetailBuildSystemCloudHouseKeeperEditPlanStrategy
 * @Author: xujinc
 * @Date: 2024/10/21
 */
@Component("projectDetailBuildSystemCloudHouseKeeperEditPlanStrategy")
public class ProjectDetailBuildSystemCloudHouseKeeperEditPlanStrategy implements ProjectDetailBuildStrategy {

    private static final Logger logger = LoggerFactory.getLogger(ProjectDetailBuildSystemCloudHouseKeeperEditPlanStrategy.class);

    @Resource(name = "defaultThreadPool")
    private ThreadPoolTaskExecutor threadPoolTaskExecutor;

    @Autowired
    private DigiwinAtdmProxyService digiwinAtdmProxyService;

    @Autowired
    private DigiwinTaskEngineProxyService digiwinTaskEngineProxyService;

    @Autowired
    private DigiwinKnowledgeMapsProxyService digiwinKnowledgeMapsProxyService;

    @Autowired
    private TitleDescProjectCCBuilder titleDescProjectCCBuilder;

    @Autowired
    private CustomTabsBuilder customTabsBuilder;

    @Autowired
    private PageDescBuilderFactory pageDescBuilderFactory;

    @Autowired
    private LocaleService localeService;

    @Override
    public String getProjectDetailType() {
        return ProjectTypeUtil.KSC_PROJECT_EDIT_PLAN_TYPE;
    }

    @Override
    @SneakyThrows
    public Map<String, Object> build(ApiRequest apiRequest, String iamUserToken, PageSetting pageSetting,
                                     UiBotModel pcUiBotModel, DigiwinAtmcPanel digiwinAtmcPanel, Object... args) {
        String locale = apiRequest.getLocale();
        String tenantId = apiRequest.getTenantId();

        /**
         * 1.获取项目详情
         * 根据ESP的actionId获取业务数据(actionId=project.info.get)
         * POST API：https://atdm-test.apps.digiwincloud.com.cn/api/atdm/v1/data/query/by/actionId
         *
         * 2.获取项目下属任务详情
         * 根据ESP的actionId获取业务数据(actionId=task.info.get)
         * POST API：https://atdm-test.apps.digiwincloud.com.cn/api/atdm/v1/data/query/by/actionId
         */
        StopWatch stopWatch = new StopWatch("Build Project-Detail Page Data");
        stopWatch.start("Get ProjectCC Detail Data From Athena ATDM");
        //使用Future方式执行多任务
        List<Future<?>> futureList = new ArrayList<>();
        Future<PccApiDataWrapper<PccProjectInfo>> futureAtdmProjectInfo =
                this.threadPoolTaskExecutor.submit(this.buildGetProjectInfoCallable(locale, iamUserToken, pcUiBotModel, this.digiwinAtdmProxyService, tenantId));
        futureList.add(futureAtdmProjectInfo);

        Future<PccApiDataWrapper<PccProjectTaskInfo>> futureAtdmProjectTaskInfo =
                this.threadPoolTaskExecutor.submit(this.buildGetProjectTaskInfoCallable(locale, iamUserToken, pcUiBotModel, this.digiwinAtdmProxyService, tenantId));
        futureList.add(futureAtdmProjectTaskInfo);

        PccApiDataWrapper<PccProjectInfo> projectInfo = null;
        PccApiDataWrapper<PccProjectTaskInfo> projectTaskInfo = null;
        //查询任务执行的结果
        try {
            for (Future<?> future : futureList) {
                // CPU高速轮询：每个future都并发轮循，判断完成状态然后获取结果，
                // 这一行，是本实现方案的精髓所在。即有10个future在高速轮询，完成一个future的获取结果，就关闭一个轮询
                while (true) {
                    // 获取future成功完成状态，如果想要限制每个任务的超时时间，取消本行的状态判断+future.get(1000*1, TimeUnit.MILLISECONDS)+catch超时异常使用即可。
                    if (future.isDone() && !future.isCancelled()) {
                        //获取结果
                        Object obj = future.get();
                        if (obj instanceof PccApiDataWrapper) {
                            PccApiDataWrapper container = (PccApiDataWrapper) obj;
                            if (null != container.getProjectInfoList() && !container.getProjectInfoList().isEmpty()) {
                                Object firstEle = container.getProjectInfoList().get(0);
                                if (firstEle instanceof PccProjectInfo) {
                                    projectInfo = container;
                                } else if (firstEle instanceof PccProjectTaskInfo) {
                                    projectTaskInfo = container;
                                } else {
                                }
                            }
                        }
                        break;//当前future获取结果完毕，跳出while
                    } else {
                        //每次轮询休息1毫秒（CPU纳秒级），避免CPU高速轮循耗空CPU---》新手别忘记这个
                        Thread.sleep(1);
                    }
                }
            }
        } catch (InterruptedException | ExecutionException e) {
            logger.error(e.getLocalizedMessage());
            e.printStackTrace();
            String message = e.getMessage();
            if (null != e.getCause()) {
                message = e.getCause().getMessage();
            }
            throw new ServiceException(message);
        } finally {

        }
        stopWatch.stop();

        if (null == projectInfo || CollectionUtils.isEmpty(projectInfo.getProjectInfoList())) {
            return Collections.emptyMap();
        }

        stopWatch.start("Build SystemCloudHouseKeeper Edit Plan Page Page");
        Map<String, Object> result = new HashMap<>();
        PccProjectInfo pccProjectInfo = projectInfo.getProjectInfoList().get(0);

        Map<String, Object> rawData = apiRequest.getRawData().getInnerMap();
        Map<String, Object> projectDetailDataState = (Map<String, Object>) MapUtil.getOrDefault(rawData,
                "projectDetailDataState", Collections.emptyMap());

        UiBotModel pageModel = pageSetting.getPageModel();
        Iterator<UiBotLayout> layoutIterator = pageModel.getLayout().iterator();
        while (layoutIterator.hasNext()) {
            UiBotLayout layout = layoutIterator.next();
            String schema = layout.getSchema();
            switch (layout.getType()) {
                case TitleDesc.COMPONENT_TYPE:
                    result.put(schema, this.titleDescProjectCCBuilder.build(apiRequest,
                            pageSetting, pcUiBotModel, TitleDesc.class, pccProjectInfo, digiwinAtmcPanel));
                    break;
                case TitleBody.COMPONENT_TYPE:
                    result.put(schema, TitleBody.create(apiRequest, pageSetting, pcUiBotModel,
                            pccProjectInfo, digiwinAtmcPanel, pageDescBuilderFactory));
                    break;
                case ProgressRate.COMPONENT_TYPE:
                    // FIXME 定制逻辑的代码，使用数据库中的schema来区分从哪个字段获取数据，后续再找机会调整。
                    if ("progressRateFinished".equalsIgnoreCase(schema)) {
                        double rate = MathUtil.getDecimalPoint(
                                Optional.ofNullable(pccProjectInfo.getProjectCompleteRate()).orElse(0.0),
                                2);
                        result.put(schema, new ProgressRate(
                                LocaleUtil.getMobileTextByKey(locale, "完成率") + String.format("%.0f", rate * 100.0) + "%",
                                ProgressRateTypeEnum.NORMAL.getType(),
                                rate
                        ));
                    } else if ("progressRateOverdue".equalsIgnoreCase(schema)) {
                        double rate = MathUtil.getDecimalPoint(
                                Optional.ofNullable(pccProjectInfo.getProjectOverdueRate()).orElse(0.0),
                                2);
                        // 逾期率为0时不展示
                        if (rate != 0.0) {
                            result.put(schema, new ProgressRate(
                                    LocaleUtil.getMobileTextByKey(locale, "逾期率") + String.format("%.0f", rate * 100.0) + "%",
                                    ProgressRateTypeEnum.EXCEPTION.getType(),
                                    rate
                            ));
                        } else {
                            layoutIterator.remove();
                        }
                    } else {
                        // FIXME 未定义的schema，展示未定义的内容，进度为0
                        result.put(schema, new ProgressRate(
                                LocaleUtil.getMobileTextByKey(locale, "未定义"),
                                ProgressRateTypeEnum.NORMAL.getType(),
                                0.0
                        ));
                    }
                    break;
                case Divider.COMPONENT_TYPE:
                    Divider divider = Divider.create(DividerTypeEnum.BLOCK_DIVIDER);
                    String newSchema = schema + "_" + divider.getCmptUuid();
                    layout.setSchema(newSchema);
                    result.put(newSchema, divider);
                    break;
                case AutoFill.COMPONENT_TYPE:

                    List<AutoFillData> autoFillDataList = new ArrayList<>();

                    //存在数据，则展⽰过程备忘录按钮
                    List<Map<String, Object>> bmdPiscProMemoList = getBmdPiscProMemo(apiRequest, pccProjectInfo.getProjectNo());
                    if (!CollectionUtils.isEmpty(bmdPiscProMemoList)) {
                        AutoFillData processRemarkAutoFillData = new AutoFillData();
                        processRemarkAutoFillData.setName(localeService.getLanguageValue(locale, "过程备忘录"));
                        processRemarkAutoFillData.setNameColor("5037DB");
                        processRemarkAutoFillData.setNameSize("28");
                        Action basicDetailAction = new Action();
                        basicDetailAction.setJumpPageId(PageSettingIdPresetEnum.MOBILE_ATHENA_SYSTEM_CLOUD_HOUSE_KEEPER_PROCESS_REMARK.toString());
                        basicDetailAction.setJumpPageTitle(processRemarkAutoFillData.getName());
                        basicDetailAction.setType(ActionTypeEnum.OPEN_NEW_PAGE_FROM_BOTTOM.getValue());
                        Map<String, Object> basicDetailRawData = new HashMap<>();
                        basicDetailRawData.put("project_info", pccProjectInfo);
                        basicDetailRawData.put("isEdit", true);
                        basicDetailAction.setRawData(basicDetailRawData);
                        processRemarkAutoFillData.setAction(basicDetailAction);
                        autoFillDataList.add(processRemarkAutoFillData);
                    }

                    AutoFillData chooseTemplateAutoFillData = new AutoFillData();
                    chooseTemplateAutoFillData.setName(localeService.getLanguageValue(locale, "选择模板"));
                    chooseTemplateAutoFillData.setNameColor("5037DB");
                    chooseTemplateAutoFillData.setNameSize("28");
                    Action basicDetailAction = new Action();
                    basicDetailAction.setJumpPageId(PageSettingIdPresetEnum.MOBILE_ATHENA_SYSTEM_CLOUD_HOUSE_KEEPER_PROJECT_EDIT_PLAN_CHOOSE_TEMPLATE.toString());
                    basicDetailAction.setJumpPageTitle(chooseTemplateAutoFillData.getName());
                    basicDetailAction.setType(ActionTypeEnum.OPEN_NEW_PAGE_FROM_BOTTOM.getValue());
                    Map<String, Object> basicDetailRawData = new HashMap<>();
                    basicDetailRawData.put("project_info", projectInfo);
                    basicDetailAction.setRawData(basicDetailRawData);
                    chooseTemplateAutoFillData.setAction(basicDetailAction);
                    autoFillDataList.add(chooseTemplateAutoFillData);

                    AutoFill autoFill = new AutoFill();
                    autoFill.setData(autoFillDataList);
                    autoFill.setAlign(ButtonGroupAlignEnum.CENTER.getValue());
                    result.put(schema, autoFill);
                    break;
                case CustomTabs.COMPONENT_TYPE:
                    if (Objects.nonNull(projectTaskInfo)) {
                        List<PccProjectTaskInfo> pccProjectTaskInfo = projectTaskInfo.getProjectInfoList();
                        // 最后一个参数传入layout，方便后续计算
                        CustomTabs customTabs = this.customTabsBuilder.build(apiRequest, pageSetting, pcUiBotModel, CustomTabs.class,
                                layout, pccProjectTaskInfo,
                                pccProjectInfo.getProjectNo(), apiRequest.getTenantId());
                        result.put(schema, customTabs);
                    } else {
                        Empty empty = Empty.create(LocaleUtil.getMobileTextByKey(locale, "请选择模板，添加任务"));
                        layout.setType(Empty.COMPONENT_TYPE);
                        result.put(schema, empty);
                    }
                    break;
                case BottomButtonDigiwinAthena.COMPONENT_TYPE:
                    List<BottomButtonDigiwinAthena> buttonList = JsonUtil.objectToJavaObject(pageModel.getPageData().get(schema), new TypeReference<List<BottomButtonDigiwinAthena>>() {
                    });
                    PccApiDataWrapper<PccProjectTaskInfo> finalProjectTaskInfo = projectTaskInfo;
                    buttonList.forEach(bottomButton -> {
                        bottomButton.setName(localeService.getLanguageValue(locale, bottomButton.getName()));
                        if (bottomButton.getIsCheckButton()) {
                            Action action = bottomButton.getAction();
                            action.setRefreshPageId(apiRequest.getRefreshPageId());
                            rawData.put("projectDetailDataState", projectDetailDataState);
                            rawData.put("project_info", pccProjectInfo);
                            rawData.put("pageId", PageSettingIdPresetEnum.MOBILE_ATHENA_SYSTEM_CLOUD_HOUSE_KEEPER_PROJECT_EDIT_PLAN_DETAIL.toString());
                            action.setRawData(rawData);
                            ConfirmPop confirmPop = ConfirmPop.create(localeService.getLanguageValue(locale, "是否启动项目?"),
                                    localeService.getLanguageValue(locale, "请核对执行人/负责人/任务名称等信息"), true);
                            action.setConfirmPop(confirmPop);
                        }
                        if (Objects.isNull(finalProjectTaskInfo) || CollectionUtils.isEmpty(finalProjectTaskInfo.getProjectInfoList())) {
                            bottomButton.setEnable(false);
                        }
                    });
                    result.put(schema, buttonList);
                default:
                    break;

            }
        }
        stopWatch.stop();
        //先这样把项目信息传回
        result.put("project_info", pccProjectInfo);

        logger.debug(stopWatch.prettyPrint());

        return result;
    }

    /**
     * 返回一个获取项目详情的可执行任务
     *
     * @param locale
     * @param iamUserToken
     * @param pcUiBotModel
     * @param digiwinAtdmProxyService
     * @return
     */
    private Callable<PccApiDataWrapper<PccProjectInfo>>
    buildGetProjectInfoCallable(String locale, String iamUserToken,
                                UiBotModel pcUiBotModel, DigiwinAtdmProxyService digiwinAtdmProxyService, String tenantId) {
        return () -> {
            // 获取项目详情
            Map<String, Object> bodyPayloadMap = new HashMap<>(4);
            String actionId = "project.info.get";
            bodyPayloadMap.put("actionId", actionId);
            bodyPayloadMap.put("businessUnit", pcUiBotModel.getExecuteContext().get("businessUnit"));

            Map<String, Object> taskWithBacklogData =
                    (Map<String, Object>) pcUiBotModel.getExecuteContext().get("taskWithBacklogData");
            Map<String, Object> bpmData = (Map<String, Object>) taskWithBacklogData.get("bpmData");
            List<Map<String, Object>> projectDataList = (List<Map<String, Object>>) bpmData.get("project_data");

            Map<String, Object> parameter = new HashMap<>(1);
            parameter.put("project_info", projectDataList);

            bodyPayloadMap.put("parameter", parameter);

            return digiwinAtdmProxyService.dataQueryByActionProjectInfo(locale, iamUserToken,
                    JsonUtil.javaObjectToJsonString(bodyPayloadMap), tenantId);
        };
    }

    /**
     * 返回一个获取项目下属任务详情的可执行任务
     *
     * @param locale
     * @param iamUserToken
     * @param pcUiBotModel
     * @param digiwinAtdmProxyService
     * @return
     */
    private Callable<PccApiDataWrapper<PccProjectTaskInfo>>
    buildGetProjectTaskInfoCallable(String locale, String iamUserToken,
                                    UiBotModel pcUiBotModel, DigiwinAtdmProxyService digiwinAtdmProxyService, String tenantId) {
        return () -> {
            // 获取项目详情
            Map<String, Object> bodyPayloadMap = new HashMap<>(4);
            String actionId = "task.info.get";
            bodyPayloadMap.put("actionId", actionId);
            bodyPayloadMap.put("businessUnit", pcUiBotModel.getExecuteContext().get("businessUnit"));

            Map<String, Object> taskWithBacklogData =
                    (Map<String, Object>) pcUiBotModel.getExecuteContext().get("taskWithBacklogData");
            Map<String, Object> bpmData = (Map<String, Object>) taskWithBacklogData.get("bpmData");
            List<Map<String, Object>> projectDataList = (List<Map<String, Object>>) bpmData.get("project_data");
            if (null != projectDataList && !projectDataList.isEmpty()) {
                projectDataList.get(0).put("control_mode", "1");

                //1.项目;2.项目模版
                //未启动项目查询传值1
                projectDataList.get(0).put("task_property", "1");
            }
            Map<String, Object> parameter = new HashMap<>(1);
            parameter.put("project_info", projectDataList);

            bodyPayloadMap.put("parameter", parameter);

            return digiwinAtdmProxyService.dataQueryByActionProjectTaskInfo(locale, iamUserToken,
                    JsonUtil.javaObjectToJsonString(bodyPayloadMap), tenantId);
        };
    }

    private List<Map<String, Object>> getBmdPiscProMemo(ApiRequest apiRequest, String projectNo) {
        Map<String, Object> projectProcessMemorandumInfoMap = new HashMap<>(2);
        projectProcessMemorandumInfoMap.put("project_no", projectNo);
        projectProcessMemorandumInfoMap.put("memorandum_property", 1);

        Map<String, Object> parameterMap = new HashMap<>(2);
        parameterMap.put("project_process_memorandum_info", Collections.singletonList(projectProcessMemorandumInfoMap));

        Map<String, Object> requestMap = new HashMap<>(2);
        requestMap.put("actionId", "bmd.pisc.pro.memo.get");
        requestMap.put("parameter", parameterMap);
        DigiwinAthenaApiResponse<JSONObject> resp = digiwinAtdmProxyService
                .handleByActionId(apiRequest.getLocale(), apiRequest.getIamUserToken(), apiRequest.getTenantId(), "", requestMap);

        if (resp.getStatus() == 200 && StringUtils.isEmpty(resp.getErrorMessage())) {
            return Optional.ofNullable(resp.getResponse())
                    .map(e -> (List<Map<String, Object>>) e.get("project_process_memorandum_info"))
                    .orElse(Collections.emptyList());
        } else {
            return Collections.emptyList();
        }
    }

    private List<Map<String, Object>> getBmPiscAttachmentJoinRelation(ApiRequest apiRequest, String projectNo) {
        Map<String, Object> taskInfoMap = new HashMap<>(3);
        taskInfoMap.put("project_no", projectNo);
        taskInfoMap.put("task_property", "1");

        Map<String, Object> parameterMap = new HashMap<>(2);
        parameterMap.put("query_condition", "ALL");
        parameterMap.put("task_info", Collections.singletonList(taskInfoMap));

        Map<String, Object> requestMap = new HashMap<>(2);
        requestMap.put("actionId", "bm.pisc.attachment.join.relation.get");
        requestMap.put("parameter", parameterMap);
        DigiwinAthenaApiResponse<JSONObject> resp = digiwinAtdmProxyService
                .handleByActionId(apiRequest.getLocale(), apiRequest.getIamUserToken(), apiRequest.getTenantId(), "", requestMap);

        if (resp.getStatus() == 200 && StringUtils.isEmpty(resp.getErrorMessage())) {
            return Optional.ofNullable(resp.getResponse())
                    .map(e -> (List<Map<String, Object>>) e.get("task_info"))
                    .orElse(Collections.emptyList());
        }
        return Collections.emptyList();
    }

    /**
     * 提交-启动项目
     *
     * @param apiRequestSubmit
     * @return
     */
    public ActionSubmitResult submit(ApiRequestSubmit apiRequestSubmit) throws ServiceException {
        String locale = apiRequestSubmit.getLocale();

        //校验过程备忘录过程方案必填情况下，过程方案是否有值
        this.validProcessRemarkRequired(apiRequestSubmit);

        //协同排定计划检查
        this.validateAssistSchedule(apiRequestSubmit);

        //项⽬启动重复操作检查
        this.validateProjectInfo(apiRequestSubmit);

        // 判断是否需要签核
        boolean isNeedApprove = this.getProjectIsApproveByProjectType(apiRequestSubmit);

        this.updateProjectStatus(apiRequestSubmit, isNeedApprove);

        if (isNeedApprove) {
            this.executionEngine(apiRequestSubmit);
        } else {
            this.submitData(apiRequestSubmit);
            this.dispatch(apiRequestSubmit);
        }

        Action action = new Action<>();
        action.setType(ActionTypeEnum.BACK_ONE_WITH_REFRESH.getValue());
        action.setRefreshPageId(PageSettingIdPresetEnum.MOBILE_ATHENA_SYSTEM_CLOUD_HOUSE_KEEPER_PROJECT_DETAIL.name());
        action.setSuccessMsg(localeService.getLanguageValue(locale, "提交成功"));
        return ActionSubmitResult.create(true, action);
    }

    private void validProcessRemarkRequired(ApiRequestSubmit apiRequestSubmit) {
        ApiRequest apiRequest = new ApiRequest();
        apiRequest.setLocale(apiRequestSubmit.getLocale());
        apiRequest.setTenantId(apiRequestSubmit.getTenantId());
        apiRequest.setIamUserToken(apiRequestSubmit.getIamUserToken());

        Map<String, Object> rawData = apiRequestSubmit.getRawData().getInnerMap();
        Map<String, Object> projectInfo = (Map<String, Object>) rawData.get("project_info");
        String projectNo = MapUtils.getString(projectInfo, "projectNo");

        //过程备忘录数据
        List<Map<String, Object>> bmdPiscProMemoList = getBmdPiscProMemo(apiRequest, projectNo);

        //任务数据
        List<Map<String, Object>> bmPiscAttachmentJoinRelationList = getBmPiscAttachmentJoinRelation(apiRequest, projectNo);
        Map<String, Map<String, Object>> bmPiscAttachmentJoinRelationListMapByTaskNo = bmPiscAttachmentJoinRelationList.stream()
                .collect(Collectors.toMap(item -> MapUtils.getString(item, "task_no"), item -> item, (v1, v2) -> v1));

        List<String> validFailedTaskNameList = new ArrayList<>();
        for (Map<String, Object> bmdPiscProMemo : bmdPiscProMemoList) {
            //归属任务编号
            String belongTaskNo = MapUtils.getString(bmdPiscProMemo, "belong_task_no");

            //过程方案
            String processPlan = MapUtils.getString(bmdPiscProMemo, "process_plan");

            Map<String, Object> bmPiscAttachmentJoinRelationMapByTaskNo = bmPiscAttachmentJoinRelationListMapByTaskNo.get(belongTaskNo);

            //need_issue_task_card=false,则过程方案必填
            boolean isRequired = CollectionUtils.isEmpty(bmPiscAttachmentJoinRelationMapByTaskNo) ?
                    false : !MapUtils.getBooleanValue(bmPiscAttachmentJoinRelationMapByTaskNo, "need_issue_task_card", true);

            //过程方案必填情况下，过程方案未填，则校验不通过
            if (isRequired && StringUtils.isEmpty(processPlan)) {
                validFailedTaskNameList.add(MapUtils.getString(bmPiscAttachmentJoinRelationMapByTaskNo, "task_name"));
            }
        }

        if (!CollectionUtils.isEmpty(validFailedTaskNameList)) {
            String errorMessage = StringUtils.join(validFailedTaskNameList, "，") +
                    localeService.getLanguageValue(apiRequestSubmit.getLocale(), "任务不需执行，过程备忘录必填");
            throw new ServiceException(errorMessage);
        }
    }

    private void validateAssistSchedule(ApiRequestSubmit apiRequestSubmit) {
        String locale = apiRequestSubmit.getLocale();
        Map<String, Object> rawData = apiRequestSubmit.getRawData().getInnerMap();
        Map<String, Object> projectInfo = (Map<String, Object>) rawData.get("project_info");

        Map<String, Object> assistScheduleInfoMap = new HashMap<>(2);
        assistScheduleInfoMap.put("project_no", projectInfo.get("projectNo"));
        assistScheduleInfoMap.put("schedule_status", "1");

        Map<String, Object> parameterMap = new HashMap<>(2);
        parameterMap.put("assist_schedule_info", Collections.singletonList(assistScheduleInfoMap));

        Map<String, Object> requestMap = new HashMap<>(2);
        requestMap.put("actionId", "bm.pisc.assist.schedule.get");
        requestMap.put("parameter", parameterMap);
        DigiwinAthenaApiResponse<JSONObject> resp = digiwinAtdmProxyService
                .handleByActionId(locale, apiRequestSubmit.getIamUserToken(), apiRequestSubmit.getTenantId(), "", requestMap);

        if (resp.getStatus() == 200 && StringUtils.isEmpty(resp.getErrorMessage())) {
            List<Map<String, Object>> assistScheduleInfoResp = Optional.ofNullable(resp.getResponse())
                    .map(e -> (List<Map<String, Object>>) e.get("assist_schedule_info"))
                    .orElse(Collections.emptyList());
            if (!CollectionUtils.isEmpty(assistScheduleInfoResp)) {
                throw new ServiceException(localeService.getLanguageValue(locale, "当前有协同排定计划任务未完成，请催办！"));
            }
        } else {
            throw new ServiceException(resp.getErrorMessage());
        }
    }

    private void validateProjectInfo(ApiRequestSubmit apiRequestSubmit) {
        String locale = apiRequestSubmit.getLocale();
        Map<String, Object> rawData = apiRequestSubmit.getRawData().getInnerMap();
        Map<String, Object> projectInfo = (Map<String, Object>) rawData.get("project_info");

        Map<String, Object> projectInfoMap = new HashMap<>(1);
        projectInfoMap.put("project_no", projectInfo.get("projectNo"));

        Map<String, Object> parameterMap = new HashMap<>(1);
        parameterMap.put("project_info", Collections.singletonList(projectInfoMap));

        Map<String, Object> requestMap = new HashMap<>(2);
        requestMap.put("actionId", "bm.pisc.project.get");
        requestMap.put("parameter", parameterMap);
        DigiwinAthenaApiResponse<JSONObject> resp = digiwinAtdmProxyService
                .handleByActionId(locale, apiRequestSubmit.getIamUserToken(), apiRequestSubmit.getTenantId(), "", requestMap);

        if (resp.getStatus() == 200 && StringUtils.isEmpty(resp.getErrorMessage())) {
            Map<String, Object> projectInfoResp = Optional.ofNullable(resp.getResponse())
                    .map(e -> (List<Map<String, Object>>) e.get("project_info"))
                    .filter(e -> !CollectionUtils.isEmpty(e))
                    .map(e -> e.get(0))
                    .orElse(Collections.emptyMap());

            String projectStatus = MapUtils.getString(projectInfoResp, "project_status");
            String approveAction = MapUtils.getString(projectInfoResp, "approve_action");

            //签核动作=30且项⽬状态=10（未开始），弹窗提⽰：当前项⽬正在启动签核中，请等待签核完成
            if (Objects.equals(projectStatus, "10") && Objects.equals(approveAction, "30")) {
                throw new ServiceException(localeService.getLanguageValue(locale, "当前项⽬正在启动签核中，请等待签核完成！"));
            }

            //项⽬状态=30.进⾏中，提⽰：当前项⽬已经签核完成，请刷新⻚⾯
            if (Objects.equals(projectStatus, "30")) {
                throw new ServiceException(localeService.getLanguageValue(locale, "当前项⽬已经签核完成，请刷新⻚⾯！"));
            }
        } else {
            throw new ServiceException(resp.getErrorMessage());
        }
    }

    private boolean getProjectIsApproveByProjectType(ApiRequestSubmit apiRequestSubmit) {
        String locale = apiRequestSubmit.getLocale();
        Map<String, Object> rawData = apiRequestSubmit.getRawData().getInnerMap();
        Map<String, Object> projectInfo = (Map<String, Object>) rawData.get("project_info");

        Map<String, Object> projectInfoMap = new HashMap<>(1);
        projectInfoMap.put("project_type_no", projectInfo.get("projectTypeNo"));

        Map<String, Object> parameterMap = new HashMap<>(1);
        parameterMap.put("project_type_info", Collections.singletonList(projectInfoMap));

        Map<String, Object> requestMap = new HashMap<>(2);
        requestMap.put("actionId", "bm.pisc.project.type.get");
        requestMap.put("parameter", parameterMap);
        DigiwinAthenaApiResponse<JSONObject> resp = digiwinAtdmProxyService
                .handleByActionId(locale, apiRequestSubmit.getIamUserToken(), apiRequestSubmit.getTenantId(), "", requestMap);

        if (resp.getStatus() == 200 && StringUtils.isEmpty(resp.getErrorMessage())) {
            Map<String, Object> projectTypeInfoResp = Optional.ofNullable(resp.getResponse())
                    .map(e -> (List<Map<String, Object>>) e.get("project_type_info"))
                    .filter(e -> !CollectionUtils.isEmpty(e))
                    .map(e -> e.get(0))
                    .orElse(Collections.emptyMap());
            return MapUtils.getBooleanValue(projectTypeInfoResp, "project_is_approve");

        } else {
            throw new ServiceException(resp.getErrorMessage());
        }
    }

    private void updateProjectStatus(ApiRequestSubmit apiRequestSubmit, boolean isApprove) {
        String locale = apiRequestSubmit.getLocale();
        Map<String, Object> rawData = apiRequestSubmit.getRawData().getInnerMap();
        Map<String, Object> projectInfo = (Map<String, Object>) rawData.get("project_info");

        Map<String, Object> projectInfoMap = new HashMap<>(1);
        projectInfoMap.put("project_no", projectInfo.get("projectNo"));
        projectInfoMap.put("is_approve", isApprove);
        //签核状态， 项⽬启动签核否 == true 则为V（无需签核），否则为N（签核中）
        projectInfoMap.put("approve_status", isApprove ? "N" : "V");
        projectInfoMap.put("project_status", "30");

        Map<String, Object> parameterMap = new HashMap<>(1);
        parameterMap.put("project_info", Collections.singletonList(projectInfoMap));
        parameterMap.put("sync_steady_state", "N");

        Map<String, Object> requestMap = new HashMap<>(2);
        requestMap.put("actionId", "bm.pisc.project.status.update");
        requestMap.put("parameter", parameterMap);
        DigiwinAthenaApiResponse<JSONObject> resp = digiwinAtdmProxyService
                .handleByActionId(locale, apiRequestSubmit.getIamUserToken(), apiRequestSubmit.getTenantId(), "", requestMap);

        if (resp.getStatus() != 200 && StringUtils.isNotEmpty(resp.getErrorMessage())) {
            throw new ServiceException(resp.getErrorMessage());
        }
    }

    private void submitData(ApiRequestSubmit apiRequestSubmit) {
        String locale = apiRequestSubmit.getLocale();
        Map<String, Object> rawData = apiRequestSubmit.getRawData().getInnerMap();
        Map<String, Object> projectInfo = (Map<String, Object>) rawData.get("project_info");
        Map<String, Object> projectDetailDataState = (Map<String, Object>) rawData.get("projectDetailDataState");

        Map<String, Object> workItemInfo = Optional.ofNullable(projectDetailDataState)
                .map(e -> (Map<String, Object>) MapUtil.getOrDefault(e, "executeContext", Collections.emptyMap()))
                .map(e -> (Map<String, Object>) MapUtil.getOrDefault(e, "taskWithBacklogData", Collections.emptyMap()))
                .map(e -> (List<Map<String, Object>>) MapUtil.getOrDefault(e, "backlog", Collections.emptyList()))
                .map(e -> e.get(0))
                .map(e -> (List<Map<String, Object>>) MapUtil.getOrDefault(e, "workitemList", Collections.emptyList()))
                .map(e -> e.get(0)).orElse(Collections.emptyMap());

        Map<String, Object> dispatchDataMap = new HashMap<>(1);
        dispatchDataMap.put("project_no", projectInfo.get("projectNo"));

        Map<String, Object> processVariablesMap = new HashMap<>(1);
        processVariablesMap.put("project_no", projectInfo.get("projectNo"));

        Map<String, Object> parameterMap = new HashMap<>(1);
        parameterMap.put("dispatchData", Collections.singletonList(dispatchDataMap));
        parameterMap.put("processVariables", processVariablesMap);
        parameterMap.put("performerId", apiRequestSubmit.getUserId());
        parameterMap.put("workitemId", workItemInfo.get("workitemId"));
        parameterMap.put("locale", locale);

        Map<String, Object> resp = digiwinTaskEngineProxyService.submitData(apiRequestSubmit.getTenantId(), apiRequestSubmit.getIamUserToken(), parameterMap);
        if (!MapUtils.getBooleanValue(resp, "success")) {
            throw new ServiceException(MapUtils.getString(resp, "message"));
        }
    }

    private void dispatch(ApiRequestSubmit apiRequestSubmit) {
        String locale = apiRequestSubmit.getLocale();
        Map<String, Object> rawData = apiRequestSubmit.getRawData().getInnerMap();
        Map<String, Object> projectDetailDataState = (Map<String, Object>) rawData.get("projectDetailDataState");

        Map<String, Object> workItemInfo = Optional.ofNullable(projectDetailDataState)
                .map(e -> (Map<String, Object>) MapUtil.getOrDefault(e, "executeContext", Collections.emptyMap()))
                .map(e -> (Map<String, Object>) MapUtil.getOrDefault(e, "taskWithBacklogData", Collections.emptyMap()))
                .map(e -> (List<Map<String, Object>>) MapUtil.getOrDefault(e, "backlog", Collections.emptyList()))
                .map(e -> e.get(0))
                .map(e -> (List<Map<String, Object>>) MapUtil.getOrDefault(e, "workitemList", Collections.emptyList()))
                .map(e -> e.get(0)).orElse(Collections.emptyMap());

        Map<String, Object> parameterMap = new HashMap<>(1);
        parameterMap.put("comment", "");
        parameterMap.put("performerId", workItemInfo.get("performerId"));
        parameterMap.put("workitemId", workItemInfo.get("workitemId"));
        parameterMap.put("locale", locale);

        Map<String, Object> resp = digiwinTaskEngineProxyService.manualDispatch(apiRequestSubmit.getTenantId(), apiRequestSubmit.getIamUserToken(), parameterMap);
        if (!MapUtils.getBooleanValue(resp, "success")) {
            throw new ServiceException(MapUtils.getString(resp, "message"));
        }
    }

    private void executionEngine(ApiRequestSubmit apiRequestSubmit) {
        Map<String, Object> rawData = apiRequestSubmit.getRawData().getInnerMap();
        Map<String, Object> projectInfo = (Map<String, Object>) rawData.get("project_info");

        Map<String, Object> projectInfoMap = new HashMap<>(5);
        projectInfoMap.put("project_no", projectInfo.get("projectNo"));
        projectInfoMap.put("project_leader_code", projectInfo.get("projectLeaderCode"));
        projectInfoMap.put("project_leader_id", apiRequestSubmit.getUserId());
        projectInfoMap.put("project_status", "30");
        projectInfoMap.put("action", "start");

        Map<String, Object> parameterMap = new HashMap<>(1);
        parameterMap.put("project_info", Collections.singletonList(projectInfoMap));
        Map<String, Object> resp = digiwinKnowledgeMapsProxyService.executionEngine(apiRequestSubmit.getIamUserToken(),
                apiRequestSubmit.getTenantId(), "startServiceComposer_sch_startProjectCard", Collections.singletonList(parameterMap));
        Map<String, Object> output = Optional.ofNullable(resp)
                .map(e -> (Map<String, Object>) MapUtil.getOrDefault(e, "output", Collections.emptyMap()))
                .orElse(Collections.emptyMap());
        if (!MapUtils.getBooleanValue(output, "success")) {
            throw new ServiceException(MapUtils.getString(output, "errorMessage"));
        }
    }
}
