package com.digiwin.mobile.mobileuibot.project.strategy.custom;

import com.alibaba.fastjson.JSONObject;
import com.digiwin.mobile.mobileuibot.api.ApiRequest;
import com.digiwin.mobile.mobileuibot.api.ApiRequestSubmit;
import com.digiwin.mobile.mobileuibot.common.context.AppContext;
import com.digiwin.mobile.mobileuibot.common.context.AppRequestContext;
import com.digiwin.mobile.mobileuibot.common.datetime.DateTimeUtil;
import com.digiwin.mobile.mobileuibot.common.exception.ServiceException;
import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.common.map.MapUtil;
import com.digiwin.mobile.mobileuibot.core.component.MobileRenderMetaData;
import com.digiwin.mobile.mobileuibot.core.component.action.Action;
import com.digiwin.mobile.mobileuibot.core.component.action.ActionSubmitParam;
import com.digiwin.mobile.mobileuibot.core.component.action.ActionSubmitResult;
import com.digiwin.mobile.mobileuibot.core.component.action.ActionTypeEnum;
import com.digiwin.mobile.mobileuibot.core.component.button.BottomButtonDigiwinAthena;
import com.digiwin.mobile.mobileuibot.core.component.group.CustomGroup;
import com.digiwin.mobile.mobileuibot.core.component.group.CustomGroupContent;
import com.digiwin.mobile.mobileuibot.core.component.input.calendar.InputCalendar;
import com.digiwin.mobile.mobileuibot.core.component.input.calendar.InputCalendarDate;
import com.digiwin.mobile.mobileuibot.core.component.input.numeric.InputNumeric;
import com.digiwin.mobile.mobileuibot.core.component.input.numeric.InputNumericTypeEnum;
import com.digiwin.mobile.mobileuibot.core.component.input.singletext.InputSingleText;
import com.digiwin.mobile.mobileuibot.core.component.input.windowselect.InputWindowSelectStyleEnum;
import com.digiwin.mobile.mobileuibot.core.component.input.windowselect.multi.InputWindowMultiSelect;
import com.digiwin.mobile.mobileuibot.core.component.input.windowselect.single.InputWindowSingleSelect;
import com.digiwin.mobile.mobileuibot.core.hooks.MobileUiBotHooks;
import com.digiwin.mobile.mobileuibot.core.hooks.MobileUiBotHooksService;
import com.digiwin.mobile.mobileuibot.core.pagesetting.PageSetting;
import com.digiwin.mobile.mobileuibot.core.pagesetting.PageSettingIdPresetEnum;
import com.digiwin.mobile.mobileuibot.core.rule.JsRule;
import com.digiwin.mobile.mobileuibot.core.rule.JsRuleService;
import com.digiwin.mobile.mobileuibot.core.strategy.modelbuild.UiBotModelBuildStrategy;
import com.digiwin.mobile.mobileuibot.locale.service.LocaleService;
import com.digiwin.mobile.mobileuibot.proxy.DigiwinAthenaApiResponse;
import com.digiwin.mobile.mobileuibot.proxy.atdm.service.DigiwinAtdmProxyService;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.UiBotModel;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.UiBotPageData;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.layout.UiBotLayout;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.rule.UiBotRule;
import com.fasterxml.jackson.core.type.TypeReference;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.StopWatch;

import java.io.IOException;
import java.math.BigDecimal;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * <p>功能描述：体系云管家的项目详情编辑任务界面构造</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: ProjectDetailBuildSystemCloudHouseKeeperChooseTemplateStrategy
 * @Author: xujinc
 * @Date: 2024/10/21
 */
@Component("projectDetailBuildSystemCloudHouseKeeperEditTaskStrategy")
public class ProjectDetailBuildSystemCloudHouseKeeperEditTaskStrategy implements UiBotModelBuildStrategy {
    private static final Logger logger =
            LoggerFactory.getLogger(ProjectDetailBuildSystemCloudHouseKeeperEditTaskStrategy.class);

    @Autowired
    private DigiwinAtdmProxyService digiwinAtdmProxyService;

    @Autowired
    private MobileUiBotHooksService mobileUiBotHooksService;

    @Autowired
    private JsRuleService jsRuleService;

    @Autowired
    private LocaleService localeService;

    @Override
    public UiBotModel buildModel(PageSetting pageSetting, ApiRequest apiRequest)
            throws IOException {
        AppRequestContext.getContextEntity().setRenderVersion(MobileRenderMetaData.RENDER_VERSION_2);
        StopWatch stopWatch = new StopWatch("SystemCloudHouseKeeper Edit Task Page");
        UiBotModel uiBotModel = null;
        Boolean useMockData = AppContext.getUseMockData();
        if (!useMockData) {
            uiBotModel = this.doActually(pageSetting, apiRequest, stopWatch);
        } else {
            uiBotModel = this.doMockData(pageSetting, apiRequest);
        }

        // v2规则目前无法从km获取，先从mongodb获取
        List<JsRule> jsRuleList = jsRuleService.findByPageId(pageSetting.getPageId());
        if (!org.springframework.util.CollectionUtils.isEmpty(jsRuleList)) {
            List<UiBotRule> uiBotRuleList = new ArrayList<>(1);
            jsRuleList.forEach(jsRule -> {
                UiBotRule uiBotRule = new UiBotRule();
                BeanUtils.copyProperties(jsRule, uiBotRule);
                uiBotRuleList.add(uiBotRule);
            });
            if(uiBotModel != null){
                uiBotModel.setRules(uiBotRuleList);
            }
        }
        return uiBotModel;
    }

    private UiBotModel doActually(PageSetting pageSetting, ApiRequest apiRequest,
                                  StopWatch stopWatch) {
        UiBotModel uiBotModel = pageSetting.getPageModel();
        String locale = apiRequest.getLocale();
        Map<String, Object> rawData = apiRequest.getRawData().getInnerMap();
        Map<String, Object> currentProjectTaskInfoMap = (Map<String, Object>) MapUtil.getOrDefault(rawData, "currentProjectTaskInfo", Collections.emptyMap());
        Map<String, Object> taskInfoMap = getInitData(apiRequest, currentProjectTaskInfoMap);

        List<MobileUiBotHooks> mobileUiBotHooksList = mobileUiBotHooksService.findByPageId(pageSetting.getPageId());
//        Map<String, List<Map<String, Object>>> hooksBySchema = mobileUiBotHooksList.stream().collect(Collectors.groupingBy(MobileUiBotHooks::getSchema, Collectors.mapping(e -> e.getData().getInnerMap(), Collectors.toList())));
        Map<String, Map<String, Object>> hooksBySchema = mobileUiBotHooksList.stream().collect(Collectors.toMap(MobileUiBotHooks::getSchema, e -> e.getData().getInnerMap(), (a, b) -> a));

        // 拼成最终数据
        stopWatch.start("Build SystemCloudHouseKeeper Edit Task Page Model...");
        List<UiBotLayout> uiBotLayouts = uiBotModel.getLayout();
        for (UiBotLayout layout : uiBotLayouts) {
            String schema = layout.getSchema();
            switch (layout.getType()) {
                case CustomGroup.COMPONENT_TYPE:
                    CustomGroup customGroup = JsonUtil.objectToJavaObject(uiBotModel.getPageData().get(schema), CustomGroup.class);
                    handleCustomGroup(customGroup, taskInfoMap, hooksBySchema, locale);
                    uiBotModel.getPageData().put(schema, customGroup);
                    break;
                case BottomButtonDigiwinAthena.COMPONENT_TYPE:
                    List<BottomButtonDigiwinAthena> buttonList = JsonUtil.objectToJavaObject(uiBotModel.getPageData().get(schema), new TypeReference<List<BottomButtonDigiwinAthena>>() {
                    });
                    buttonList.forEach(bottomButton -> {
                        bottomButton.setName(localeService.getLanguageValue(locale, bottomButton.getName()));
                        if (bottomButton.getIsCheckButton()) {
                            Action action = bottomButton.getAction();
                            action.setRefreshPageId(apiRequest.getRefreshPageId());
                            rawData.put("actionId", "bm.pisc.attachment.join.relation.update");
                            rawData.put("taskInfoMap", taskInfoMap);
                            rawData.put("pageId", apiRequest.getPageId());
                            rawData.put("fromPageId", MapUtils.getString(rawData, "fromPageId"));
                            action.setRawData(rawData);
                        }
                    });
                    uiBotModel.getPageData().put(schema, buttonList);
                    break;
                default:
                    break;
            }
        }

        UiBotPageData rawUiBotPageData = new UiBotPageData();
        rawUiBotPageData.put("task_info", taskInfoMap);
        uiBotModel.setRawUiBotPageData(rawUiBotPageData);

        stopWatch.stop();
        return uiBotModel;
    }

    private Map<String, Object> getInitData(ApiRequest apiRequest, Map<String, Object> currentProjectTaskInfo) {
        Map<String, Object> taskInfoMap = new HashMap<>(3);
        taskInfoMap.put("project_no", currentProjectTaskInfo.get("project_no"));
        taskInfoMap.put("task_no", currentProjectTaskInfo.get("task_no"));
        taskInfoMap.put("task_property", "1");

        Map<String, Object> parameterMap = new HashMap<>(2);
        parameterMap.put("query_condition", "ALL");
        parameterMap.put("source_type", "4");
        parameterMap.put("task_info", Collections.singletonList(taskInfoMap));

        Map<String, Object> requestMap = new HashMap<>(2);
        requestMap.put("actionId", "bm.pisc.attachment.join.relation.get");
        requestMap.put("parameter", parameterMap);
        DigiwinAthenaApiResponse<JSONObject> resp = digiwinAtdmProxyService
                .handleByActionId(apiRequest.getLocale(), apiRequest.getIamUserToken(), apiRequest.getTenantId(), "", requestMap);

        if (resp.getStatus() == 200 && StringUtils.isEmpty(resp.getErrorMessage())) {
            return Optional.ofNullable(resp.getResponse())
                    .map(e -> (List<Map<String, Object>>) e.get("task_info"))
                    .filter(e -> CollectionUtils.isNotEmpty(e))
                    .map(e -> e.get(0))
                    .orElse(Collections.emptyMap());
        }
        return Collections.emptyMap();
    }

    private void handleCustomGroup(CustomGroup customGroup, Map<String, Object> taskInfoMap, Map<String, Map<String, Object>> hooksBySchema, String locale) {
        CustomGroupContent customGroupContent = customGroup.getContent();
        List<UiBotLayout> customGroupUiBotLayouts = customGroupContent.getLayout();
        for (UiBotLayout customGroupUiBotLayout : customGroupUiBotLayouts) {
            String customGroupSchema = customGroupUiBotLayout.getSchema();
            switch (customGroupUiBotLayout.getType()) {
                case InputSingleText.COMPONENT_TYPE:
                    InputSingleText inputSingleText = InputSingleText.create(
                            localeService.getLanguageValue(locale, customGroupUiBotLayout.getLabel()), MapUtils.getString(taskInfoMap, customGroupSchema));
                    inputSingleText.setRequired(true);
                    customGroupContent.getPageData().put(customGroupSchema, inputSingleText);
                    break;
                case InputNumeric.COMPONENT_TYPE:
                    // 工作量单位日，预计工时单位小时
                    String unit = localeService.getLanguageValue(locale, customGroupSchema.equals("workload_qty") ? "日" : "小时");
                    String value = MapUtils.getString(taskInfoMap, customGroupSchema);
                    value = StringUtils.isEmpty(value) ? "0" : value;
                    String defaultValue = new BigDecimal(value).stripTrailingZeros().toPlainString();
                    InputNumeric inputNumeric = InputNumeric.create(99999999, 0,
                            defaultValue, localeService.getLanguageValue(locale, customGroupUiBotLayout.getLabel()),
                            InputNumericTypeEnum.INTEGERWITHUNIT.getValue(), false, unit);
                    // 预计工时不可编辑
                    if (customGroupSchema.equals("plan_work_hours")) {
                        inputNumeric.setEnable(false);
                    }

                    // 工作量
                    if (customGroupSchema.equals("workload_qty")) {
                        inputNumeric.setHooks(hooksBySchema.get(customGroupSchema));
                    }
                    customGroupContent.getPageData().put(customGroupSchema, inputNumeric);
                    break;
                case InputWindowMultiSelect.COMPONENT_TYPE:
                    InputWindowMultiSelect inputWindowMultiSelect = buildInputWindowMultiSelect(customGroupUiBotLayout,
                            taskInfoMap, customGroupSchema, locale);
                    customGroupContent.getPageData().put(customGroupSchema, inputWindowMultiSelect);
                    break;
                case InputCalendar.COMPONENT_TYPE:
                    String dateValue = MapUtils.getString(taskInfoMap, customGroupSchema);
                    InputCalendar inputCalendar = buildInputCalendar(customGroupUiBotLayout, dateValue, locale);
                    inputCalendar.setHooks(hooksBySchema.get(customGroupSchema));
                    customGroupContent.getPageData().put(customGroupSchema, inputCalendar);
                    break;
            }
        }
    }

    /**
     * 构建多选组件
     *
     * @param uiBotLayout
     * @param schema
     * @param locale
     * @return
     */
    private InputWindowMultiSelect buildInputWindowMultiSelect(UiBotLayout uiBotLayout, Map<String, Object> taskInfoMap, String schema, String locale) {
        InputWindowMultiSelect inputWindowMultiSelect = new InputWindowMultiSelect();
        inputWindowMultiSelect.setSchema(uiBotLayout.getSchema());
        inputWindowMultiSelect.setTitle(localeService.getLanguageValue(locale, uiBotLayout.getLabel()));
        inputWindowMultiSelect.setUrl(AppContext.getBaseUrl() + "/mobile/v1/proxy/window/data/list?componentSchema=" + schema);
        inputWindowMultiSelect.setRawData(buildInputWindowMultiSelectRawData());
        inputWindowMultiSelect.setType(InputWindowSingleSelect.COMPONENT_TYPE);
        inputWindowMultiSelect.setStyle(InputWindowSelectStyleEnum.BG_STYLE_WHITE.getValue());
        inputWindowMultiSelect.setPlaceholder(localeService.getLanguageValue(locale, "请选择"));
        inputWindowMultiSelect.setUseHasNext(false);

        List<Map<String, Object>> taskMemberInfoList = (List<Map<String, Object>>) MapUtil.getOrDefault(taskInfoMap, "task_member_info", Collections.emptyList());
        inputWindowMultiSelect.setSelectIdList(taskMemberInfoList.stream().map(e -> MapUtils.getString(e, "executor_no")).collect(Collectors.toList()));
        inputWindowMultiSelect.setTextList(taskMemberInfoList.stream().map(e -> MapUtils.getString(e, "executor_name")).collect(Collectors.toList()));
        inputWindowMultiSelect.setRequired(true);
        return inputWindowMultiSelect;
    }

    /**
     * 构建日期组件
     *
     * @param uiBotLayout
     * @param dateValue
     * @param locale
     * @return
     */
    private InputCalendar buildInputCalendar(UiBotLayout uiBotLayout, String dateValue, String locale) {
        uiBotLayout.setHeaderName(localeService.getLanguageValue(locale, uiBotLayout.getLabel()));
        InputCalendar inputCalendar = InputCalendar.create(uiBotLayout, null, locale);
        if (StringUtils.isNotEmpty(dateValue)) {
            LocalDate localDate = DateTimeUtil.parseLocalDateUseDefaultPattern(dateValue);
            InputCalendarDate inputCalendarDate = InputCalendarDate.create(localDate.getYear(),
                    localDate.getMonthValue(), localDate.getDayOfMonth());
            inputCalendar.setMinDateSelectable(inputCalendarDate);
            inputCalendar.setDateValueList(Collections.singletonList(inputCalendarDate));
        }
        inputCalendar.setRequired(true);
        return inputCalendar;
    }

    private Map<String, Object> buildInputWindowMultiSelectRawData() {
        Map<String, Object> parasMap = new HashMap<>(2);
        parasMap.put("appid", "sch");
        parasMap.put("is_show_main_department", true);

        Map<String, Object> requestMap = new HashMap<>(2);
        requestMap.put("actionId", "bm.basc.app.auth.employee.get");
        requestMap.put("parameter", parasMap);

        Map<String, Object> rawDataMap = new HashMap<>(2);
        rawDataMap.put("requestParam", requestMap);
        rawDataMap.put("type", "isSCHEditTaskQueryEmployee");
        return rawDataMap;
    }

    /**
     * 提交
     *
     * @param apiRequestSubmit
     * @return
     */
    public ActionSubmitResult submit(ApiRequestSubmit apiRequestSubmit) {
        String locale = apiRequestSubmit.getLocale();
        Map<String, Object> rawData = apiRequestSubmit.getRawData().getInnerMap();
        Map<String, Object> taskInfoMap = (Map<String, Object>) rawData.get("taskInfoMap");

        for (ActionSubmitParam actionSubmitParam : apiRequestSubmit.getSubmitParams()) {
            switch (actionSubmitParam.getType()) {
                case InputSingleText.COMPONENT_TYPE:
                    // 任务名称
                    InputSingleText inputSingleText = JsonUtil.objectToJavaObject(actionSubmitParam.getParams(), InputSingleText.class);
                    taskInfoMap.put(actionSubmitParam.getSchema(), inputSingleText.getText());
                    break;
                case InputNumeric.COMPONENT_TYPE:
                    // 工作量，预计工时
                    InputNumeric inputNumeric = JsonUtil.objectToJavaObject(actionSubmitParam.getParams(), InputNumeric.class);
                    taskInfoMap.put(actionSubmitParam.getSchema(), inputNumeric.getText());
                    break;
                case InputWindowMultiSelect.COMPONENT_TYPE:
                    // 执行人
                    InputWindowMultiSelect inputWindowMultiSelect = JsonUtil.objectToJavaObject(actionSubmitParam.getParams(), InputWindowMultiSelect.class);
                    List<Object> selectDetailList = inputWindowMultiSelect.getSelectDetailList();
                    if (CollectionUtils.isNotEmpty(selectDetailList)) {
                        List<Map<String, Object>> taskMemberInfoList = selectDetailList.stream().map(e -> {
                            Map<String, Object> selectDetailItem = (Map<String, Object>) e;
                            Map<String, Object> taskMemberInfoItem = new HashMap<>(4);
                            taskMemberInfoItem.put("executor_no", selectDetailItem.get("employee_no"));
                            taskMemberInfoItem.put("executor_name", selectDetailItem.get("employee_name"));
                            taskMemberInfoItem.put("executor_department_no", selectDetailItem.get("department_no"));
                            taskMemberInfoItem.put("executor_department_name", selectDetailItem.get("department_name"));
                            return taskMemberInfoItem;
                        }).collect(Collectors.toList());
                        taskInfoMap.put(actionSubmitParam.getSchema(), taskMemberInfoList);
                    }
                    break;
                case InputCalendar.COMPONENT_TYPE:
                    // 计划开始日，计划结束日
                    InputCalendar inputCalendar = (InputCalendar) actionSubmitParam.getParams();
                    if (StringUtils.isNotEmpty(inputCalendar.getInputText())) {
                        taskInfoMap.put(actionSubmitParam.getSchema(), inputCalendar.getInputText().replaceAll("\\.", "-"));
                    }
                    break;
            }
        }
        Map<String, Object> parameterMap = new HashMap<>(2);
        parameterMap.put("task_info", Collections.singletonList(taskInfoMap));

        Map<String, Object> requestMap = new HashMap<>(2);
        requestMap.put("actionId", rawData.get("actionId"));
        requestMap.put("parameter", parameterMap);
        DigiwinAthenaApiResponse<JSONObject> resp = digiwinAtdmProxyService
                .handleByActionId(locale, apiRequestSubmit.getIamUserToken(), apiRequestSubmit.getTenantId(), "", requestMap);

        if (resp.getStatus() != 200 || StringUtils.isNotEmpty(resp.getErrorMessage())) {
            throw new ServiceException(resp.getErrorMessage());
        } else {
            List<Map<String, Object>> projectInfoList = Optional.ofNullable(resp.getResponse())
                    .map(e -> (List<Map<String, Object>>) MapUtil.getOrDefault(e, "task_info", Collections.emptyList()))
                    .orElse(Collections.emptyList());
            String errorMsg = projectInfoList.stream()
                    .filter(e -> StringUtils.isNotEmpty(MapUtils.getString(e, "error_msg")))
                    .findFirst().map(e -> MapUtils.getString(e, "error_msg"))
                    .orElse("");
            if (StringUtils.isNotEmpty(errorMsg)) {
                throw new ServiceException(errorMsg);
            }

            String fromPageId = MapUtils.getString(rawData, "fromPageId");
            Integer type = Objects.equals(PageSettingIdPresetEnum.MOBILE_ATHENA_SYSTEM_CLOUD_HOUSE_KEEPER_PROJECT_DETAIL.toString(), fromPageId)
                    ? ActionTypeEnum.BACK_ONE_WITH_REFRESH.getValue() : ActionTypeEnum.BACK_TWO_WITH_REFRESH.getValue();
            Action action = new Action<>();
            action.setType(type);
            action.setRefreshPageId(PageSettingIdPresetEnum.MOBILE_ATHENA_SYSTEM_CLOUD_HOUSE_KEEPER_PROJECT_DETAIL.name());
            action.setSuccessMsg(localeService.getLanguageValue(locale, "提交成功"));
            return ActionSubmitResult.create(true, action);
        }
    }

    private UiBotModel doMockData(PageSetting pageSetting, ApiRequest apiRequest) {
        return null;
    }
}
