package com.digiwin.mobile.mobileuibot.project.strategy.custom;

import com.alibaba.fastjson.JSONObject;
import com.digiwin.mobile.mobileuibot.api.ApiRequest;
import com.digiwin.mobile.mobileuibot.common.exception.ServiceException;
import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.common.localization.LocaleUtil;
import com.digiwin.mobile.mobileuibot.common.math.MathUtil;
import com.digiwin.mobile.mobileuibot.core.component.action.Action;
import com.digiwin.mobile.mobileuibot.core.component.action.ActionTypeEnum;
import com.digiwin.mobile.mobileuibot.core.component.basic.AutoFill;
import com.digiwin.mobile.mobileuibot.core.component.basic.AutoFillData;
import com.digiwin.mobile.mobileuibot.core.component.button.ButtonGroupAlignEnum;
import com.digiwin.mobile.mobileuibot.core.component.divider.Divider;
import com.digiwin.mobile.mobileuibot.core.component.divider.DividerTypeEnum;
import com.digiwin.mobile.mobileuibot.core.component.navbar.pagedesc.PageDescBuilderFactory;
import com.digiwin.mobile.mobileuibot.core.component.progress.ProgressRate;
import com.digiwin.mobile.mobileuibot.core.component.progress.ProgressRateTypeEnum;
import com.digiwin.mobile.mobileuibot.core.component.tab.customtab.CustomTabs;
import com.digiwin.mobile.mobileuibot.core.component.tab.customtab.CustomTabsBuilder;
import com.digiwin.mobile.mobileuibot.core.component.title.titlebody.TitleBody;
import com.digiwin.mobile.mobileuibot.core.component.title.titledesc.TitleDesc;
import com.digiwin.mobile.mobileuibot.core.component.title.titledesc.TitleDescProjectCCBuilder;
import com.digiwin.mobile.mobileuibot.core.pagesetting.PageSetting;
import com.digiwin.mobile.mobileuibot.core.pagesetting.PageSettingIdPresetEnum;
import com.digiwin.mobile.mobileuibot.locale.service.LocaleService;
import com.digiwin.mobile.mobileuibot.project.common.ProjectTypeUtil;
import com.digiwin.mobile.mobileuibot.project.strategy.ProjectDetailBuildStrategy;
import com.digiwin.mobile.mobileuibot.proxy.DigiwinAthenaApiResponse;
import com.digiwin.mobile.mobileuibot.proxy.atdm.model.pcc.PccApiDataWrapper;
import com.digiwin.mobile.mobileuibot.proxy.atdm.model.pcc.PccProjectInfo;
import com.digiwin.mobile.mobileuibot.proxy.atdm.model.pcc.PccProjectTaskInfo;
import com.digiwin.mobile.mobileuibot.proxy.atdm.service.DigiwinAtdmProxyService;
import com.digiwin.mobile.mobileuibot.proxy.atmc.model.DigiwinAtmcPanel;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.UiBotModel;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.layout.UiBotLayout;
import lombok.SneakyThrows;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StopWatch;

import javax.annotation.Resource;
import java.util.*;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;

/**
 * <p>功能描述：体系云管家的项目详情界面构造</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: ProjectDetailBuildSystemCloudHouseKeeperStrategy
 * @Author: Zaregoto
 * @Date: 20230609
 */
@Component("projectDetailBuildSystemCloudHouseKeeperStrategy")
public class ProjectDetailBuildSystemCloudHouseKeeperStrategy implements ProjectDetailBuildStrategy {

    private static final Logger logger = LoggerFactory.getLogger(ProjectDetailBuildSystemCloudHouseKeeperStrategy.class);

    @Resource(name = "defaultThreadPool")
    private ThreadPoolTaskExecutor threadPoolTaskExecutor;

    @Autowired
    private DigiwinAtdmProxyService digiwinAtdmProxyService;

    @Autowired
    private TitleDescProjectCCBuilder titleDescProjectCCBuilder;

    @Autowired
    private CustomTabsBuilder customTabsBuilder;

    @Autowired
    private PageDescBuilderFactory pageDescBuilderFactory;

    @Autowired
    private LocaleService localeService;

    @Override
    public String getProjectDetailType() {
        return ProjectTypeUtil.KSC_PROJECT_TYPE;
    }

    @Override
    @SneakyThrows
    public Map<String, Object> build(ApiRequest apiRequest, String iamUserToken, PageSetting pageSetting,
                                     UiBotModel pcUiBotModel, DigiwinAtmcPanel digiwinAtmcPanel, Object... args) {
        String locale = apiRequest.getLocale();
        String tenantId = apiRequest.getTenantId();

        /**
         * 1.获取项目详情
         * 根据ESP的actionId获取业务数据(actionId=project.info.get)
         * POST API：https://atdm-test.apps.digiwincloud.com.cn/api/atdm/v1/data/query/by/actionId
         *
         * 2.获取项目下属任务详情
         * 根据ESP的actionId获取业务数据(actionId=task.info.get)
         * POST API：https://atdm-test.apps.digiwincloud.com.cn/api/atdm/v1/data/query/by/actionId
         */
        StopWatch stopWatch = new StopWatch("Build Project-Detail Page Data");
        stopWatch.start("Get ProjectCC Detail Data From Athena ATDM");
        //使用Future方式执行多任务
        List<Future<?>> futureList = new ArrayList<>();
        Future<PccApiDataWrapper<PccProjectInfo>> futureAtdmProjectInfo =
                this.threadPoolTaskExecutor.submit(this.buildGetProjectInfoCallable(locale, iamUserToken, pcUiBotModel, this.digiwinAtdmProxyService, tenantId));
        futureList.add(futureAtdmProjectInfo);

        Future<PccApiDataWrapper<PccProjectTaskInfo>> futureAtdmProjectTaskInfo =
                this.threadPoolTaskExecutor.submit(this.buildGetProjectTaskInfoCallable(locale, iamUserToken, pcUiBotModel, this.digiwinAtdmProxyService, tenantId));
        futureList.add(futureAtdmProjectTaskInfo);

        PccApiDataWrapper<PccProjectInfo> projectInfo = null;
        PccApiDataWrapper<PccProjectTaskInfo> projectTaskInfo = null;
        //查询任务执行的结果
        try {
            for (Future<?> future : futureList) {
                // CPU高速轮询：每个future都并发轮循，判断完成状态然后获取结果，
                // 这一行，是本实现方案的精髓所在。即有10个future在高速轮询，完成一个future的获取结果，就关闭一个轮询
                while (true) {
                    // 获取future成功完成状态，如果想要限制每个任务的超时时间，取消本行的状态判断+future.get(1000*1, TimeUnit.MILLISECONDS)+catch超时异常使用即可。
                    if (future.isDone() && !future.isCancelled()) {
                        //获取结果
                        Object obj = future.get();
                        if (obj instanceof PccApiDataWrapper) {
                            PccApiDataWrapper container = (PccApiDataWrapper) obj;
                            if (null != container.getProjectInfoList() && !container.getProjectInfoList().isEmpty()) {
                                Object firstEle = container.getProjectInfoList().get(0);
                                if (firstEle instanceof PccProjectInfo) {
                                    projectInfo = container;
                                } else if (firstEle instanceof PccProjectTaskInfo) {
                                    projectTaskInfo = container;
                                } else {
                                }
                            }
                        }
                        break;//当前future获取结果完毕，跳出while
                    } else {
                        //每次轮询休息1毫秒（CPU纳秒级），避免CPU高速轮循耗空CPU---》新手别忘记这个
                        Thread.sleep(1);
                    }
                }
            }
        } catch (InterruptedException | ExecutionException e) {
            logger.error(e.getLocalizedMessage());
            e.printStackTrace();
            String message = e.getMessage();
            if (null != e.getCause()) {
                message = e.getCause().getMessage();
            }
            throw new ServiceException(message);
        } finally {

        }
        stopWatch.stop();

        if (null == projectInfo || CollectionUtils.isEmpty(projectInfo.getProjectInfoList())) {
            return Collections.emptyMap();
        }

        stopWatch.start("Add Data to ProjectCC PageModel");
        Map<String, Object> result = new HashMap<>();
        PccProjectInfo pccProjectInfo = projectInfo.getProjectInfoList().get(0);

        UiBotModel pageModel = pageSetting.getPageModel();
        Iterator<UiBotLayout> layoutIterator = pageModel.getLayout().iterator();
        while (layoutIterator.hasNext()) {
            UiBotLayout layout = layoutIterator.next();
            String schema = layout.getSchema();
            if (layout.getType().equalsIgnoreCase(TitleDesc.COMPONENT_TYPE)) {
                result.put(schema,
                        this.titleDescProjectCCBuilder.build(apiRequest, pageSetting, pcUiBotModel, TitleDesc.class,
                                pccProjectInfo, digiwinAtmcPanel));
            } else if (layout.getType().equalsIgnoreCase(TitleBody.COMPONENT_TYPE)) {
                result.put(schema, TitleBody.create(apiRequest, pageSetting, pcUiBotModel,
                        pccProjectInfo, digiwinAtmcPanel, pageDescBuilderFactory));
            } else if (layout.getType().equalsIgnoreCase(ProgressRate.COMPONENT_TYPE)) {
                // FIXME 定制逻辑的代码，使用数据库中的schema来区分从哪个字段获取数据，后续再找机会调整。
                if ("progressRateFinished".equalsIgnoreCase(schema)) {
                    double rate = MathUtil.getDecimalPoint(
                            Optional.ofNullable(pccProjectInfo.getProjectCompleteRate()).orElse(0.0),
                            2);
                    result.put(schema, new ProgressRate(
                            LocaleUtil.getMobileTextByKey(locale, "完成率") + String.format("%.0f", rate * 100.0) + "%",
                            ProgressRateTypeEnum.NORMAL.getType(),
                            rate
                    ));
                } else if ("progressRateOverdue".equalsIgnoreCase(schema)) {
                    double rate = MathUtil.getDecimalPoint(
                            Optional.ofNullable(pccProjectInfo.getProjectOverdueRate()).orElse(0.0),
                            2);
                    // 逾期率为0时不展示
                    if (rate != 0.0) {
                        result.put(schema, new ProgressRate(
                                LocaleUtil.getMobileTextByKey(locale, "逾期率") + String.format("%.0f", rate * 100.0) + "%",
                                ProgressRateTypeEnum.EXCEPTION.getType(),
                                rate
                        ));
                    } else {
                        layoutIterator.remove();
                    }
                } else {
                    // FIXME 未定义的schema，展示未定义的内容，进度为0
                    result.put(schema, new ProgressRate(
                            LocaleUtil.getMobileTextByKey(locale, "未定义"),
                            ProgressRateTypeEnum.NORMAL.getType(),
                            0.0
                    ));
                }
            } else if (layout.getType().equalsIgnoreCase(Divider.COMPONENT_TYPE)) {
                Divider divider = Divider.create(DividerTypeEnum.BLOCK_DIVIDER);
                String newSchema = schema + "_" + divider.getCmptUuid();
                layout.setSchema(newSchema);
                result.put(newSchema, divider);
            } else if (layout.getType().equalsIgnoreCase(AutoFill.COMPONENT_TYPE)) {
                List<AutoFillData> autoFillDataList = new ArrayList<>();
                //存在数据，则展⽰过程备忘录按钮
                List<Map<String, Object>> bmdPiscProMemoList = getBmdPiscProMemo(apiRequest, pccProjectInfo.getProjectNo());
                if (!CollectionUtils.isEmpty(bmdPiscProMemoList)) {
                    AutoFillData processRemarkAutoFillData = new AutoFillData();
                    processRemarkAutoFillData.setName(localeService.getLanguageValue(locale, "过程备忘录"));
                    processRemarkAutoFillData.setNameColor("5037DB");
                    processRemarkAutoFillData.setNameSize("28");
                    Action basicDetailAction = new Action();
                    basicDetailAction.setJumpPageId(PageSettingIdPresetEnum.MOBILE_ATHENA_SYSTEM_CLOUD_HOUSE_KEEPER_PROCESS_REMARK.toString());
                    basicDetailAction.setJumpPageTitle(processRemarkAutoFillData.getName());
                    basicDetailAction.setType(ActionTypeEnum.OPEN_NEW_PAGE_FROM_BOTTOM.getValue());
                    Map<String, Object> basicDetailRawData = new HashMap<>();
                    basicDetailRawData.put("project_info", pccProjectInfo);
                    basicDetailRawData.put("isEdit", false);
                    basicDetailAction.setRawData(basicDetailRawData);
                    processRemarkAutoFillData.setAction(basicDetailAction);
                    autoFillDataList.add(processRemarkAutoFillData);

                    AutoFill autoFill = new AutoFill();
                    autoFill.setData(autoFillDataList);
                    autoFill.setAlign(ButtonGroupAlignEnum.CENTER.getValue());
                    result.put(schema, autoFill);
                } else {
                    layoutIterator.remove();
                }
            } else if (layout.getType().equalsIgnoreCase(this.customTabsBuilder.getMobileComponentType())) {
                if (projectTaskInfo == null) {
                    continue;
                }
                List<PccProjectTaskInfo> pccProjectTaskInfo = projectTaskInfo.getProjectInfoList();
                // 最后一个参数传入layout，方便后续计算
                CustomTabs customTabs = this.customTabsBuilder.build(apiRequest, pageSetting, pcUiBotModel, CustomTabs.class,
                        layout, pccProjectTaskInfo,
                        pccProjectInfo.getProjectNo(), apiRequest.getTenantId());
                result.put(schema, customTabs);
            }
        }
        stopWatch.stop();
        //先这样把项目信息传回
        result.put("project_info", pccProjectInfo);

        logger.debug(stopWatch.prettyPrint());

        return result;
    }

    /**
     * 返回一个获取项目详情的可执行任务
     *
     * @param locale
     * @param iamUserToken
     * @param pcUiBotModel
     * @param digiwinAtdmProxyService
     * @return
     */
    private Callable<PccApiDataWrapper<PccProjectInfo>>
    buildGetProjectInfoCallable(String locale, String iamUserToken,
                                UiBotModel pcUiBotModel, DigiwinAtdmProxyService digiwinAtdmProxyService, String tenantId) {
        return () -> {
            // 获取项目详情
            Map<String, Object> bodyPayloadMap = new HashMap<>(4);
            String actionId = "project.info.get";
            bodyPayloadMap.put("actionId", actionId);
            bodyPayloadMap.put("businessUnit", pcUiBotModel.getExecuteContext().get("businessUnit"));

            Map<String, Object> taskWithBacklogData =
                    (Map<String, Object>) pcUiBotModel.getExecuteContext().get("taskWithBacklogData");
            Map<String, Object> bpmData = (Map<String, Object>) taskWithBacklogData.get("bpmData");
            List<Map<String, Object>> projectDataList = (List<Map<String, Object>>) bpmData.get("project_data");

            Map<String, Object> parameter = new HashMap<>(1);
            parameter.put("project_info", projectDataList);

            bodyPayloadMap.put("parameter", parameter);

            return digiwinAtdmProxyService.dataQueryByActionProjectInfo(locale, iamUserToken,
                    JsonUtil.javaObjectToJsonString(bodyPayloadMap), tenantId);
        };
    }

    /**
     * 返回一个获取项目下属任务详情的可执行任务
     *
     * @param locale
     * @param iamUserToken
     * @param pcUiBotModel
     * @param digiwinAtdmProxyService
     * @return
     */
    private Callable<PccApiDataWrapper<PccProjectTaskInfo>>
    buildGetProjectTaskInfoCallable(String locale, String iamUserToken,
                                    UiBotModel pcUiBotModel, DigiwinAtdmProxyService digiwinAtdmProxyService, String tenantId) {
        return () -> {
            // 获取项目详情
            Map<String, Object> bodyPayloadMap = new HashMap<>(4);
            String actionId = "task.info.get";
            bodyPayloadMap.put("actionId", actionId);
            bodyPayloadMap.put("businessUnit", pcUiBotModel.getExecuteContext().get("businessUnit"));

            Map<String, Object> taskWithBacklogData =
                    (Map<String, Object>) pcUiBotModel.getExecuteContext().get("taskWithBacklogData");
            Map<String, Object> bpmData = (Map<String, Object>) taskWithBacklogData.get("bpmData");
            List<Map<String, Object>> projectDataList = (List<Map<String, Object>>) bpmData.get("project_data");
            if (null != projectDataList && !projectDataList.isEmpty()) {
                projectDataList.get(0).put("control_mode", "1");
            }
            Map<String, Object> parameter = new HashMap<>(1);
            parameter.put("project_info", projectDataList);

            bodyPayloadMap.put("parameter", parameter);

            return digiwinAtdmProxyService.dataQueryByActionProjectTaskInfo(locale, iamUserToken,
                    JsonUtil.javaObjectToJsonString(bodyPayloadMap), tenantId);
        };
    }

    private List<Map<String, Object>> getBmdPiscProMemo(ApiRequest apiRequest, String projectNo) {
        Map<String, Object> projectProcessMemorandumInfoMap = new HashMap<>(2);
        projectProcessMemorandumInfoMap.put("project_no", projectNo);
        projectProcessMemorandumInfoMap.put("memorandum_property", 1);

        Map<String, Object> parameterMap = new HashMap<>(2);
        parameterMap.put("project_process_memorandum_info", Collections.singletonList(projectProcessMemorandumInfoMap));

        Map<String, Object> requestMap = new HashMap<>(2);
        requestMap.put("actionId", "bmd.pisc.pro.memo.get");
        requestMap.put("parameter", parameterMap);
        DigiwinAthenaApiResponse<JSONObject> resp = digiwinAtdmProxyService
                .handleByActionId(apiRequest.getLocale(), apiRequest.getIamUserToken(), apiRequest.getTenantId(), "", requestMap);

        if (resp.getStatus() == 200 && StringUtils.isEmpty(resp.getErrorMessage())) {
            return Optional.ofNullable(resp.getResponse())
                    .map(e -> (List<Map<String, Object>>) e.get("project_process_memorandum_info"))
                    .orElse(Collections.emptyList());
        } else {
            return Collections.emptyList();
        }
    }
}
