package com.digiwin.mobile.mobileuibot.proxy;

import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.*;
import com.fasterxml.jackson.databind.deser.ContextualDeserializer;
import com.fasterxml.jackson.databind.node.TextNode;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import java.io.IOException;

/**
 * <p>功能描述：雅典娜API响应对象的反序列化处理器</p>
 * <p>参考了这里面的做法来获取泛型类型，存储为JavaType：https://stackoverflow.com/questions/47348029/get-the-detected-generic-type-inside-jacksons-jsondeserializer</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: DigiwinAthenaApiResponseJsonDeserializer
 * @Author: Zaregoto
 * @Date: 2023/7/3 11:31
 */
@Slf4j
public class DigiwinAthenaApiResponseJsonDeserializer extends JsonDeserializer<DigiwinAthenaApiResponse> implements ContextualDeserializer {

    private JavaType javaType;

    public DigiwinAthenaApiResponseJsonDeserializer() {
    }

    public DigiwinAthenaApiResponseJsonDeserializer(JavaType type) {
        this.javaType = type;
    }

    @Override
    public JsonDeserializer<?> createContextual(DeserializationContext deserializationContext, BeanProperty beanProperty) throws JsonMappingException {
        JavaType type = deserializationContext.getContextualType() != null
                ? deserializationContext.getContextualType()
                : beanProperty.getMember().getType();
        return new DigiwinAthenaApiResponseJsonDeserializer(type);
    }

    @Override
    public DigiwinAthenaApiResponse deserialize(JsonParser jsonParser, DeserializationContext deserializationContext) throws IOException, JsonProcessingException {
        log.debug("DigiwinAthenaApiResponseJsonDeserializer activated!!!");
        DigiwinAthenaApiResponse digiwinAthenaApiResponse = new DigiwinAthenaApiResponse<>();

        JsonNode node = jsonParser.getCodec().readTree(jsonParser);

        /**
         * 每个属性单独处理与赋值
         */
        JsonNode durationNode = node.get("duration");
        if (null == durationNode) {
            digiwinAthenaApiResponse.setDuration(-1L);
        } else {
            digiwinAthenaApiResponse.setDuration(durationNode.asLong(-1L));
        }

        JsonNode statusNode = node.get("status");
        if (null == statusNode) {
            digiwinAthenaApiResponse.setStatus(-1);
        } else {
            digiwinAthenaApiResponse.setStatus(statusNode.asInt(-1));
        }

        JsonNode statusDescriptionNode = node.get("statusDescription");
        if (null == statusDescriptionNode) {
            digiwinAthenaApiResponse.setStatusDescription("");
        } else {
            digiwinAthenaApiResponse.setStatusDescription(statusDescriptionNode.asText(""));
        }

        JsonNode responseNode = node.get("response");
        if (null == responseNode || (responseNode instanceof TextNode && StringUtils.isEmpty(responseNode.asText("")))) {
            digiwinAthenaApiResponse.setResponse(null);
        } else {
            // 因为当前类DigiwinAthenaApiResponse只有一个泛型，所以这里直接取索引为0的即可
            JavaType referencedType = this.javaType.getBindings().getBoundType(0);
            digiwinAthenaApiResponse.setResponse(JsonUtil.objectToJavaObject(responseNode, referencedType));
        }

        JsonNode errorMessageNode = node.get("errorMessage");
        if (null == errorMessageNode) {
            digiwinAthenaApiResponse.setErrorMessage("");
        } else {
            digiwinAthenaApiResponse.setErrorMessage(errorMessageNode.asText(""));
        }

        JsonNode codeNode = node.get("code");
        if (null == codeNode) {
            digiwinAthenaApiResponse.setCode("");
        } else {
            digiwinAthenaApiResponse.setCode(codeNode.asText(""));
        }

        JsonNode debugInfoNode = node.get("debugInfo");
        if (null == debugInfoNode) {
            digiwinAthenaApiResponse.setDebugInfo(null);
        } else {
            digiwinAthenaApiResponse.setDebugInfo(JsonUtil.objectToJavaObject(debugInfoNode, Object.class));
        }

        JsonNode descriptionNode = node.get("description");
        if (null == descriptionNode) {
            digiwinAthenaApiResponse.setDescription("");
        } else {
            digiwinAthenaApiResponse.setDescription(descriptionNode.asText(""));
        }

        JsonNode errorCodeNode = node.get("errorCode");
        if (null == errorCodeNode) {
            digiwinAthenaApiResponse.setErrorCode("");
        } else {
            digiwinAthenaApiResponse.setErrorCode(errorCodeNode.asText(""));
        }

        JsonNode traceId = node.get("traceId");
        if (null == traceId) {
            digiwinAthenaApiResponse.setErrorCode("");
        } else {
            digiwinAthenaApiResponse.setErrorCode(traceId.asText(""));
        }

        return digiwinAthenaApiResponse;
    }
}