package com.digiwin.mobile.mobileuibot.proxy.dmc.service.impl;

import com.digiwin.dap.middleware.dmc.DMC;
import com.digiwin.dap.middleware.dmc.model.DirInfo;
import com.digiwin.dap.middleware.dmc.model.FileInfo;
import com.digiwin.dap.middleware.dmc.model.ShareInfo;
import com.digiwin.mobile.mobileuibot.common.context.AppContext;
import com.digiwin.mobile.mobileuibot.common.http.HttpService;
import com.digiwin.mobile.mobileuibot.common.http.model.AthenaBasicHttpRequestHead;
import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.proxy.dmc.config.DigiwinDmcConfig;
import com.digiwin.mobile.mobileuibot.proxy.dmc.model.SegmentedUploadFileDTO;
import com.digiwin.mobile.mobileuibot.proxy.dmc.service.DigiwinDmcProxyService;
import io.micrometer.core.instrument.util.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.Resource;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;

import javax.annotation.PostConstruct;
import java.io.File;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.*;

/**
 * <p>功能描述：</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: DigiwinDmcProxyServiceImpl
 * @Author: Zaregoto
 * @Date: 2021/6/9 10:56
 */
@Service("digiwinDmcProxyService")
public class DigiwinDmcProxyServiceImpl implements DigiwinDmcProxyService {

    @Autowired(required = false)
    private DMC dmcMobileClient;

    @Autowired(required = false)
    private DMC dmcAthenaClient;

    @Autowired(required = false)
    private DMC dmcFrcClient;

    @Autowired(required = false)
    private DMC dmcSrmLoadingClient;

    @Autowired(required = false)
    private DMC dmcKscClient;

    @Autowired(required = false)
    private DMC dmcTbdsClient;

    @Autowired(required = false)
    private DMC dmcCtpClient;

    @Autowired(required = false)
    private DMC dmcSchClient;

    @Autowired(required = false)
    private DMC agileDataClient;

    @Autowired
    HttpService httpService;

    private final Map<String, DMC> dmcClientMap = new HashMap<>();

    @PostConstruct
    private void init() {
        dmcClientMap.put(DigiwinDmcConfig.MOBILE_BUCKET, dmcMobileClient);
        dmcClientMap.put(DigiwinDmcConfig.ATHENA_BUCKET, dmcAthenaClient);
        dmcClientMap.put(DigiwinDmcConfig.FRC_BUCKET, dmcFrcClient);
        dmcClientMap.put(DigiwinDmcConfig.SRM_LOADING_BUCKET, dmcSrmLoadingClient);
        dmcClientMap.put(DigiwinDmcConfig.KSC_BUCKET, dmcKscClient);
        dmcClientMap.put(DigiwinDmcConfig.TBDS_BUCKET, dmcTbdsClient);
        dmcClientMap.put(DigiwinDmcConfig.CTP_BUCKET, dmcCtpClient);
        dmcClientMap.put(DigiwinDmcConfig.SCH_BUCKET, dmcSchClient);
        dmcClientMap.put(DigiwinDmcConfig.AGILE_DATA_BUCKET, agileDataClient);
    }

    private static final Logger logger = LoggerFactory.getLogger(DigiwinDmcProxyServiceImpl.class);

    @Override
    public FileInfo uploadFile(File file, String dirId) {
        try {
            return dmcAthenaClient.upload(file, dirId);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return new FileInfo();
    }

    @Override
    public FileInfo uploadFile(InputStream var1, FileInfo var2) {
        try {
            return dmcClientMap.getOrDefault(var2.getBucket(), dmcAthenaClient).upload(var1, var2);
        } catch (Exception e) {
            logger.error("uploadFile异常:{}", e.getMessage(), e);
            return null;
        }
    }

    @Override
    public FileInfo uploadFile(byte[] fileBytes, String fileName, String dirId) {
        try {
            FileInfo fileInfo = new FileInfo();
            fileInfo.setFileName(fileName);
            fileInfo.setDirectoryId(dirId);
            return dmcAthenaClient.upload(fileBytes, fileInfo);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return new FileInfo();
    }

    @Override
    public FileInfo uploadFile(byte[] fileBytes, String fileName, String contentType, String dirId, String bucket) {
        try {
            FileInfo fileInfo = new FileInfo();
            fileInfo.setFileName(fileName);
            fileInfo.setDirectoryId(dirId);
            if (StringUtils.isNotBlank(contentType)) {
                fileInfo.setContentType(contentType);
            }
            if (StringUtils.isBlank(dirId)) {
                fileInfo.setDirectoryId(DEFAULT_DIR);
            }
            return dmcClientMap.getOrDefault(bucket, dmcAthenaClient).upload(fileBytes, fileInfo);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return new FileInfo();
    }

    @Override
    public String getDmcUserToken() {
        return dmcAthenaClient.getToken();
    }

    @Override
    public String getDmcUserToken(String bucket) {
        return dmcClientMap.getOrDefault(bucket, dmcAthenaClient).getToken();
    }

    @Override
    public ShareInfo shareFile(String fileId) {
        try {
            return dmcAthenaClient.share(fileId);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return new ShareInfo();
    }

    @Override
    public ShareInfo shareFile(String fileId, String bucket) {
        try {
            return dmcClientMap.getOrDefault(bucket, dmcAthenaClient).share(fileId);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return new ShareInfo();
    }

    @Override
    public String getDirIdByName(String dirName) {
        try {
            DirInfo dirInfo =
                    dmcAthenaClient.getDirInfoByName(DEFAULT_DIR, dirName);
            return dirInfo.getId();
        } catch (Exception e) {
            logger.error("获取目录id出错了....目录目录名称：" + dirName);
            e.printStackTrace();
        }
        return "";
    }

    @Override
    public String getDirIdByName(String dirName, String bucket) {
        try {
            DirInfo dirInfo = dmcClientMap.getOrDefault(bucket, dmcAthenaClient)
                    .getDirInfoByName(DEFAULT_DIR, dirName);
            return dirInfo.getId();
        } catch (Exception e) {
            logger.error("获取目录id出错了....目录目录名称：" + dirName);
            e.printStackTrace();
        }
        return "";
    }

    @Override
    public boolean deleteFile(String fileId) {
        try {
            //dmc中删除附件
            dmcAthenaClient.delete(fileId);
        } catch (Exception e) {
            logger.error("DMC系统删除附件出错了....fileID：" + fileId);
            e.printStackTrace();
            return false;
        }
        return true;
    }

    @Override
    public boolean deleteDir(String dirId, String bucket) {
        try {
            //dmc中删除附件
            dmcClientMap.getOrDefault(bucket, dmcAthenaClient).deleteDir(dirId);
        } catch (Exception e) {
            logger.error("DMC系统删除指定目录出错了....dirId:{},bucket:{},error:{}", dirId, bucket, e.getMessage(), e);
            return false;
        }
        return true;
    }

    @Override
    public boolean deleteFile(String fileId, String bucket) {
        try {
            //dmc中删除附件
            dmcClientMap.getOrDefault(bucket, dmcAthenaClient).delete(fileId);
        } catch (Exception e) {
            logger.error("DMC系统删除附件出错了....fileID：" + fileId);
            e.printStackTrace();
            return false;
        }
        return true;
    }

    @Override
    public boolean deleteShare(String shareId, String bucket) {
        try {
            //dmc中删除附件
            dmcClientMap.getOrDefault(bucket, dmcAthenaClient).deleteShare(shareId);
        } catch (Exception e) {
            logger.error("DMC系统删除分享出错了....shareId：" + shareId);
            e.printStackTrace();
            return false;
        }
        return true;
    }

    @Override
    public DirInfo createDir(String name) {
        DirInfo dirInfo = new DirInfo();
        try {
            dirInfo.setParentId(DEFAULT_DIR);
            dirInfo.setName(name);
            dirInfo.setDisplayName(name);
            //dmc创建目录
            dirInfo = dmcAthenaClient.createDir(dirInfo);
        } catch (Exception e) {
            logger.error("DMC系统创建目录出错了....dirName：" + name);
            e.printStackTrace();
            return null;
        }
        return dirInfo;
    }

    @Override
    public DirInfo createDir(String name, String bucket) {
        DirInfo dirInfo = new DirInfo();
        try {
            dirInfo.setParentId(DEFAULT_DIR);
            dirInfo.setName(name);
            dirInfo.setDisplayName(name);
            //dmc创建目录
            dirInfo = dmcClientMap.getOrDefault(bucket, dmcAthenaClient).createDir(dirInfo);
        } catch (Exception e) {
            logger.error("DMC系统创建目录出错了....dirName：" + name);
            e.printStackTrace();
            return null;
        }
        return dirInfo;
    }

    @Override
    public DirInfo createDir(String dirId, String name, String bucket) {
        DirInfo dirInfo = new DirInfo();
        try {
            dirInfo.setParentId(DEFAULT_DIR);
            dirInfo.setId(dirId);
            dirInfo.setName(name);
            dirInfo.setDisplayName(name);
            //dmc创建目录
            dirInfo = dmcClientMap.getOrDefault(bucket, dmcAthenaClient).createDir(dirInfo);
        } catch (Exception e) {
            logger.error("DMC系统创建目录出错了....dirName：" + name);
            return null;
        }
        return dirInfo;
    }

    @Override
    public String segmentedUploadFile(SegmentedUploadFileDTO dto, String bucket, String token) {
        String url = String.format("%s/api/dmc/v1/buckets/%s/files/segment", AppContext.getApiUrlSetting().getDmcUrl(), bucket);
        try {
            ResponseEntity<Map> responseEntity =
                    httpService.doPost(url, new AthenaBasicHttpRequestHead(token),
                            JsonUtil.javaObjectToJsonString(dto), Map.class);
            if (responseEntity.getStatusCode().is2xxSuccessful()) {
                return String.valueOf(Objects.requireNonNull(responseEntity.getBody()).get("id"));
            }
        } catch (Exception e) {
            logger.error("segmentedUploadFile分段上传-创建空文件失败:{}", e.getMessage());
        }
        return null;
    }

    @Override
    public String segmentUploadCreateFile(Resource resource, String token, String bucket, String fileId, Integer from, Integer to, Integer total) {
        String url = String.format("%s/api/dmc/v1/buckets/%s/files/%s/%s/%s/%s", AppContext.getApiUrlSetting().getDmcUrl(), bucket, fileId, from, to, total);
        RestTemplate restTemplate = new RestTemplate();
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_OCTET_STREAM);
        headers.add("digi-middleware-auth-user", token);
        HttpEntity request = new HttpEntity<>(resource, headers);
        try {
            Map map = restTemplate.postForObject(url, request, Map.class);
            if (map == null) {
                return null;
            }
            return String.valueOf(map.get("id"));
        } catch (RestClientException e) {
            logger.error("segmentUploadCreateFile分段上传文件失败:{}", e.getMessage());
            return null;
        }
    }

    @Override
    public List<FileInfo> getFileInfos(String dirId, String bucket) {
        try {
            return dmcClientMap.getOrDefault(bucket, dmcAthenaClient).getFileInfos(dirId);
        } catch (Exception e) {
            logger.error("DMC获取文件目录下文件异常,目录ID:{},error:{}", dirId, e.getMessage(), e);
            return Collections.emptyList();
        }
    }

    @Override
    public List<FileInfo> getFileInfoByIds(String bucket, List<String> ids) {
        try {
            return dmcClientMap.getOrDefault(bucket, dmcAthenaClient).getFileInfos(ids);
        } catch (Exception e) {
            logger.error("DMC获取文件目录下文件异常，error:{}", e.getMessage(), e);
            return Collections.emptyList();
        }
    }

    public static void main(String[] args) {
        try {
            MessageDigest sha256MessageDigest = MessageDigest.getInstance("SHA-256");
            sha256MessageDigest.update("Athena".getBytes(StandardCharsets.UTF_8));
            byte[] tempByte = sha256MessageDigest.digest();
            sha256MessageDigest.update(tempByte);
            System.out.println(new String(Base64.getEncoder().encode(sha256MessageDigest.digest())));
            System.out.println(getSha256("Athena"));
        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
        }
    }

    public static String getSha256(String str) {
        MessageDigest messageDigest;
        try {
            messageDigest = MessageDigest.getInstance("SHA-256");
            messageDigest.update(str.getBytes(StandardCharsets.UTF_8));
            byte[] bytes = messageDigest.digest();
            messageDigest.update(bytes);
            return Base64.getEncoder().encodeToString(messageDigest.digest());
        } catch (Exception e) {
            logger.error("加密失败", e);
        }
        return "";
    }
}