package com.digiwin.mobile.mobileuibot.proxy.iam.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.digiwin.mobile.mobileuibot.common.calculate.UUIDUtil;
import com.digiwin.mobile.mobileuibot.common.context.AppContext;
import com.digiwin.mobile.mobileuibot.common.crypto.digest.SecureAESUtil;
import com.digiwin.mobile.mobileuibot.common.crypto.password.RandomPasswordUtil;
import com.digiwin.mobile.mobileuibot.common.exception.ServiceException;
import com.digiwin.mobile.mobileuibot.common.http.HttpService;
import com.digiwin.mobile.mobileuibot.common.http.model.AthenaBasicHttpRequestHead;
import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.common.localization.LocaleUtil;
import com.digiwin.mobile.mobileuibot.config.SysEnvConfig;
import com.digiwin.mobile.mobileuibot.locale.service.LocaleService;
import com.digiwin.mobile.mobileuibot.proxy.digiwhale.model.DigiwhaleLoginResponse;
import com.digiwin.mobile.mobileuibot.proxy.digiwhale.model.IamAuth;
import com.digiwin.mobile.mobileuibot.proxy.digiwhale.model.PasswordChange;
import com.digiwin.mobile.mobileuibot.proxy.iam.model.*;
import com.digiwin.mobile.mobileuibot.proxy.iam.service.DigiwinIamProxyService;
import com.digiwin.mobile.mobileuibot.sso.dto.ThirdPartyLoginFreeQueryDTO;
import com.fasterxml.jackson.core.type.TypeReference;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.NumberUtils;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import javax.servlet.http.HttpServletRequest;
import java.util.*;

/**
 * <p>功能描述：鼎捷IAM系统代理服务</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: DigiwinIamProxyService
 * @Author: Zaregoto
 * @Date: 2021/4/20 14:26
 */
@Service("digiwinIamProxyService")
public class DigiwinIamProxyServiceImpl implements DigiwinIamProxyService {

    private static final String LOGIN_TYPE_FIRM_ACCOUNT = "firmAccount";

    private static final Logger logger = LoggerFactory.getLogger(DigiwinIamProxyServiceImpl.class);

    private final HttpService httpService;

    @Autowired
    private LocaleService localeService;

    @Autowired
    private SysEnvConfig sysEnvConfig;

    public DigiwinIamProxyServiceImpl(HttpService httpService) {
        this.httpService = httpService;
    }

    //TODO IAM响应格式未统一，遇到异常时需要代码再处理异常的格式

    @Override
    public List<DigiwinIamTenant> listUserTenantsInAthena(String userToken) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/tenant?appId={appId}";

        Map<String, String> uriVariables = new HashMap<>(1);
        uriVariables.put("appId", DigiwinIamProxyService.ATHENA_APP_ID);

        // 使用ParameterizedTypeReference对返回带泛型参数的包装
        ParameterizedTypeReference<List<DigiwinIamTenant>> responseType =
                new ParameterizedTypeReference<List<DigiwinIamTenant>>() {
                };

        ResponseEntity<List<DigiwinIamTenant>> responseEntity =
                this.httpService.doPost(url, new AthenaBasicHttpRequestHead(userToken), uriVariables, responseType);

        HttpStatus statusCode = responseEntity.getStatusCode();

        return statusCode.is2xxSuccessful() ?
                responseEntity.getBody() : Collections.emptyList();
    }

    @Override
    public DigiwinIamTokenRefreshTenant tokenRefreshTenant(String tenantId, String tenantSid, String userToken) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/identity/token/refresh/tenant";

        Map<String, Object> bodyPayload = new HashMap<>(1);
        if (StringUtils.hasLength(tenantSid)) {
            bodyPayload.put("tenantSid", NumberUtils.parseNumber(tenantSid, Long.class));
        }
        if (StringUtils.hasLength(tenantId)) {
            bodyPayload.put("tenantId", tenantId);
        }

        ResponseEntity<DigiwinIamTokenRefreshTenant> responseEntity =
                this.httpService.doPost(url, new AthenaBasicHttpRequestHead(userToken), JsonUtil.javaObjectToJsonString(bodyPayload),
                        DigiwinIamTokenRefreshTenant.class);

        HttpStatus statusCode = responseEntity.getStatusCode();

        return responseEntity.getBody();
    }

    @Override
    public DigiwinIamAnalyzedToken tokenAnalyze(String userToken) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/identity/token/analyze";

        ResponseEntity<Map> responseEntity =
                this.httpService.doPost(url, new AthenaBasicHttpRequestHead(userToken),
                        Map.class);
        HttpStatus statusCode = responseEntity.getStatusCode();
        if (statusCode.is2xxSuccessful()) {
            return JsonUtil.objectToJavaObject(responseEntity.getBody(), DigiwinIamAnalyzedToken.class);
        } else {
            return DigiwinIamAnalyzedToken.createEmpty();
        }
    }

    @Override
    public int tokenAnalyzeByCheck(String userToken) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/identity/token/analyze";

        ResponseEntity<DigiwinIamAnalyzedToken> responseEntity =
                this.httpService.doPost(url, new AthenaBasicHttpRequestHead(userToken),
                        DigiwinIamAnalyzedToken.class);

        HttpStatus statusCode = responseEntity.getStatusCode();


        return statusCode.value();
    }

    @Override
    public Integer logout(String userToken) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/identity/logout";

        ResponseEntity<Object> responseEntity =
                this.httpService.doPost(url, new AthenaBasicHttpRequestHead(userToken), Collections.emptyMap(), Object.class);

        HttpStatus statusCode = responseEntity.getStatusCode();

        return statusCode.value();
    }

    @Override
    public List<DigiwinIamUserInfo> userList(List<String> userIds, String userToken, String locale) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/user/list";
        ResponseEntity<DigiwinIamUserInfoResponse> responseEntity =
                this.httpService.doPost(url, new AthenaBasicHttpRequestHead(userToken), JsonUtil.javaObjectToJsonString(userIds), DigiwinIamUserInfoResponse.class);
        return responseEntity.getStatusCode().is2xxSuccessful() ? responseEntity.getBody().getData() : null;
    }

    @Override
    public DigiwinIamUserInfo userInfoById(String userId, String userToken, String locale) {
        if (StringUtils.isEmpty(userId)) {
            return null;
        }
        List<String> userIds = new ArrayList<>(1);
        userIds.add(userId);
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/user/list";
        ResponseEntity<DigiwinIamUserInfoResponse> responseEntity =
                this.httpService.doPost(url, new AthenaBasicHttpRequestHead(userToken), JsonUtil.javaObjectToJsonString(userIds), DigiwinIamUserInfoResponse.class);
        if (responseEntity.getStatusCode().is2xxSuccessful() && responseEntity.getBody().getData().size() != 0) {
            return responseEntity.getBody().getData().get(0);
        } else {
            return null;
        }
    }

    @Override
    public Boolean mobileIsRegister(String telephone, String locale) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/user/mobilephone/without/inner/exist";
        Map<String, String> params = new HashMap<>(1);
        params.put("telephone", telephone);
        ResponseEntity<Map> responseEntity =
                this.httpService.doPost(url, new AthenaBasicHttpRequestHead(), JsonUtil.javaObjectToJsonString(params), Map.class);
        if (responseEntity.getStatusCode().is2xxSuccessful()) {
            return (Boolean) responseEntity.getBody().get("isRegister");
        } else {
            return false;
        }
    }

    @Override
    public ResponseEntity<DigiwhaleLoginResponse> userRegisterByMobileOrEmail(String telephone, String verificationCode, String locale, String clientAgent) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX + "/user/register";
        final int PASSWORD_MAX_LENGTH = 16;

        String phoneNumSuffix;
        String cellphonePrefix;
        String userId;
        String password;
        String passwordHash;
        String telephoneHash;
        String emailHash = ""; // 假设email为空

        if (Objects.equals(locale.trim(), Locale.SIMPLIFIED_CHINESE.toString())) {
            phoneNumSuffix = telephone.substring(7, 11);
            cellphonePrefix = "+86";
            userId = UUIDUtil.getUuid().substring(0, 2) + phoneNumSuffix;
            password = RandomPasswordUtil.getRandomPwd(PASSWORD_MAX_LENGTH);
        } else if (Objects.equals(locale.trim(), Locale.TRADITIONAL_CHINESE.toString())) {
            phoneNumSuffix = telephone.substring(6, 10);
            cellphonePrefix = "+886";
            userId = "user" + telephone;
            password = "Digiwin666";
        } else {
            phoneNumSuffix = telephone;
            cellphonePrefix = "";
            userId = "user" + telephone;
            password = "Digiwin666";
        }

        passwordHash = SecureAESUtil.aesEncryptInIamRegister(password);
        telephoneHash = SecureAESUtil.aesEncryptInIamRegister(telephone);
        emailHash = SecureAESUtil.aesEncryptInIamRegister(emailHash);

        Map<String, Object> params = new HashMap<>(8);
        params.put("Id", userId);
        params.put("name", phoneNumSuffix + LocaleUtil.getMobileTextByKey(locale, "用户"));
        params.put("emailHash", emailHash);
        params.put("telephoneHash", telephoneHash);
        params.put("cellphonePrefix", cellphonePrefix);
        params.put("verificationCode", verificationCode);
        params.put("passwordHash", passwordHash);
        params.put("agreeAgreement", true);
        params.put("verificationCodeType", 1);
        params.put("inviteSid", "");

        AthenaBasicHttpRequestHead athenaBasicHttpRequestHead = new AthenaBasicHttpRequestHead();
        athenaBasicHttpRequestHead.setClientAgent(clientAgent);
        ResponseEntity<DigiwhaleLoginResponse> responseEntity = this.httpService.doPost(url, athenaBasicHttpRequestHead, JsonUtil.javaObjectToJsonString(params), DigiwhaleLoginResponse.class);

        if (!responseEntity.getStatusCode().is2xxSuccessful()) {
            String errorMessage = this.localeService.getLanguageValue(locale, "注册失败:") +
                    (responseEntity.getBody() != null ? this.localeService.getLanguageValue(locale, responseEntity.getBody().getErrorMessage()) : this.localeService.getLanguageValue(locale, "IAM服务异常，请稍后重试"));
            return buildError(responseEntity, errorMessage);
        } else {
            buildResponse(responseEntity.getBody());
        }
        return responseEntity;
    }


    @Override
    public ResponseEntity<DigiwhaleLoginResponse> userLoginByVerificationCode(String username, String password, String locale, String iamDeviceId, String clientAgent) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/identity/login";
        AthenaBasicHttpRequestHead header = new AthenaBasicHttpRequestHead();
        header.setLocale(locale);
        header.setClientAgent(clientAgent);
        header.setIamDeviceId(iamDeviceId);
        Map<String, Object> params = new HashMap<>(3);
        params.put("userId", username);
        params.put("verificationCode", password);
        params.put("identityType", "verificationCode");
        ResponseEntity<DigiwhaleLoginResponse> responseEntity =
                this.httpService.doPost(url, header, JsonUtil.javaObjectToJsonString(params), DigiwhaleLoginResponse.class);
        if (!responseEntity.getStatusCode().is2xxSuccessful()) {
            return buildError(responseEntity, responseEntity.getBody().getMessage());
        } else {
            buildResponse(responseEntity.getBody());

        }
        return responseEntity;
    }

    @Override
    public ResponseEntity<DigiwhaleLoginResponse> userLoginByEncodePassword(String publicKey, String username, String password, String locale, String tenantId, String clientAgent, HttpServletRequest request, String iamDeviceId) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/identity/login";
        return handleLoginAndpublicKey(publicKey, username, password, locale, tenantId, clientAgent, url, request, iamDeviceId);
    }

    @Override
    public ResponseEntity<DigiwhaleLoginResponse> firmAccountLogin(String publicKey, String username, String password, String locale, String tenantId, String clientAgent, HttpServletRequest request, String iamDeviceId) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/identity/internal/login";
        return handleLoginAndpublicKey(publicKey, username, password, locale, tenantId, clientAgent, url, request, iamDeviceId);
    }

    @NotNull
    private ResponseEntity handleLoginAndpublicKey(String publicKey, String username, String password, String locale, String tenantId, String clientEncryptPublicKey, String url, HttpServletRequest request, String iamDeviceId) {
        AthenaBasicHttpRequestHead header = new AthenaBasicHttpRequestHead();
        header.setClientAgent(clientEncryptPublicKey);
        header.setLocale(locale);
        header.setIamDeviceId(iamDeviceId);
        Map<String, Object> params = new HashMap<>(4);
        params.put("identityType", "token");
        if (!ObjectUtils.isEmpty(request)) {
            //企业用户二次验证登录需要userToken
            if (StringUtils.hasText(request.getParameter("userToken"))) {
                header.setUserToken(request.getParameter("userToken"));
            }
            //账密输入错误五次以上需要传入二维码相关的信息
            if (StringUtils.hasText(request.getParameter("code")) && StringUtils.hasText(request.getParameter("value"))) {
                params.put("code", Long.valueOf(request.getParameter("code")));
                params.put("value", String.valueOf(request.getParameter("value")));
            }
            //开启二次验证登录需要携带这些参数
            if (StringUtils.hasText(request.getParameter("identityType")) && StringUtils.hasText(request.getParameter("verificationCode"))) {
                params.put("identityType", String.valueOf(request.getParameter("identityType")));
                params.put("verificationCode", String.valueOf(request.getParameter("verificationCode")));
                //企业登录 租户$userId
                if (LOGIN_TYPE_FIRM_ACCOUNT.equals(request.getParameter("loginType"))) {
                    username = tenantId + "$" + username;
                }
            }
        }
        params.put("userId", username);
        params.put("passwordHash", password);
        params.put("tenantId", tenantId);
        params.put("clientEncryptPublicKey", publicKey);
        ResponseEntity<DigiwhaleLoginResponse> responseEntity =
                this.httpService.doPost(url, header, JsonUtil.javaObjectToJsonString(params), DigiwhaleLoginResponse.class);
        if (!responseEntity.getStatusCode().is2xxSuccessful()) {
            return buildError(responseEntity, Objects.requireNonNull(responseEntity.getBody()).getMessage());
        } else {
            buildResponse(responseEntity.getBody());
        }
        return responseEntity;
    }


    @Override
    public ResponseEntity<DigiwhaleLoginResponse> userLoginByAD(String username, String clientEncryptPublicKey, String password, String locale, String tenantId, String clientAgent, HttpServletRequest request, String iamDeviceId) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/identity/login";
        AthenaBasicHttpRequestHead header = new AthenaBasicHttpRequestHead();
        header.setClientAgent(clientAgent);
        header.setLocale(locale);
        header.setIamDeviceId(iamDeviceId);
        Map<String, Object> params = new HashMap<>();
        params.put("userId", username);
        params.put("passwordHash", password);
        params.put("identityType", "service");
        params.put("tenantId", tenantId);
        params.put("clientEncryptPublicKey", clientEncryptPublicKey);
        params.put("serviceName", "digiwin");
        if (!ObjectUtils.isEmpty(request)) {
            //账密输入错误五次以上需要传入二维码相关的信息
            if (StringUtils.hasText(request.getParameter("code")) && StringUtils.hasText(request.getParameter("value"))) {
                params.put("code", Long.valueOf(request.getParameter("code")));
                params.put("value", String.valueOf(request.getParameter("value")));
            }
            //开启二次验证登录需要携带这些参数
            if (StringUtils.hasText(request.getParameter("identityType")) && StringUtils.hasText(request.getParameter("verificationCode"))) {
                params.put("identityType", String.valueOf(request.getParameter("identityType")));
                params.put("verificationCode", String.valueOf(request.getParameter("verificationCode")));
            }
        }

        ResponseEntity<DigiwhaleLoginResponse> responseEntity =
                this.httpService.doPost(url, header, JsonUtil.javaObjectToJsonString(params), DigiwhaleLoginResponse.class);
        if (!responseEntity.getStatusCode().is2xxSuccessful()) {
            return buildError(responseEntity, responseEntity.getBody().getMessage());
        } else {
            buildResponse(responseEntity.getBody());

        }
        return responseEntity;
    }

    @Override
    public ResponseEntity<DigiwhaleLoginResponse>
    userLoginGetProxyUsers(String username, String clientEncryptPublicKey, String password, String locale,
                           String agentAuthorizationCode, String clientAgent) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/identity/login/proxy/users";
        AthenaBasicHttpRequestHead header = new AthenaBasicHttpRequestHead();
        header.setClientAgent(clientAgent);
        header.setLocale(locale);
        Map<String, Object> params = new HashMap<>(5);
        params.put("agentAuthorizationCode", agentAuthorizationCode);
        params.put("clientEncryptPublicKey", clientEncryptPublicKey);
        params.put("passwordHash", password);
        params.put("identityType", "agent");
        params.put("userId", username);

        ResponseEntity<DigiwhaleLoginResponse> responseEntity =
                this.httpService.doPost(url, header, JsonUtil.javaObjectToJsonString(params), DigiwhaleLoginResponse.class);
        if (!responseEntity.getStatusCode().is2xxSuccessful()) {
            return buildError(responseEntity, responseEntity.getBody().getMessage());
        }
        buildResponse(responseEntity.getBody());
        return responseEntity;
    }

    @Override
    public ResponseEntity<DigiwhaleLoginResponse>
    userLoginByProxy(String username, String agentToken, String locale, String clientAgent, String iamDeviceId) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/identity/login/proxy";
        AthenaBasicHttpRequestHead header = new AthenaBasicHttpRequestHead();
        header.setClientAgent(clientAgent);
        header.setIamDeviceId(iamDeviceId);
        header.setLocale(locale);
        Map<String, Object> params = new HashMap<>(2);
        params.put("agentToken", agentToken);
        params.put("userId", username);

        ResponseEntity<DigiwhaleLoginResponse> responseEntity =
                this.httpService.doPost(url, header, JsonUtil.javaObjectToJsonString(params), DigiwhaleLoginResponse.class);
        if (!responseEntity.getStatusCode().is2xxSuccessful()) {
            return buildError(responseEntity, responseEntity.getBody().getMessage());
        }
        buildResponse(responseEntity.getBody());
        return responseEntity;
    }

    @Override
    public ResponseEntity<DigiwhaleLoginResponse> userLoginByCard(String username, String clientEncryptPublicKey, String locale, String tenantId, String clientAgent, String loginType, String iamDeviceId) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/identity/login";
        AthenaBasicHttpRequestHead header = new AthenaBasicHttpRequestHead();
        header.setClientAgent(clientAgent);
        header.setLocale(locale);
        header.setIamDeviceId(iamDeviceId);
        Map<String, Object> params = new HashMap<>(4);
        params.put("secretKey", username);
        params.put("identityType", "cardKey");
        params.put("tenantId", tenantId);
        params.put("clientEncryptPublicKey", clientEncryptPublicKey);
        params.put("serviceName", loginType);
        ResponseEntity<DigiwhaleLoginResponse> responseEntity =
                this.httpService.doPost(url, header, JsonUtil.javaObjectToJsonString(params), DigiwhaleLoginResponse.class);
        if (!responseEntity.getStatusCode().is2xxSuccessful()) {
            return buildError(responseEntity, responseEntity.getBody().getMessage());
        } else {
            buildResponse(responseEntity.getBody());

        }
        return responseEntity;
    }

    @Override
    public String doGetPublicKey() {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/identity/publickey";

        ParameterizedTypeReference<Map<String, Object>> responseType =
                new ParameterizedTypeReference<Map<String, Object>>() {
                };

        ResponseEntity<Map<String, Object>> responseEntity = this.httpService.doGet(url, new AthenaBasicHttpRequestHead(), new HashMap<>(), responseType);
        if (responseEntity.getStatusCode().is2xxSuccessful()) {
            return (String) responseEntity.getBody().get("publicKey");
        } else {
            return "";
        }
    }

    @Override
    public String doGetAesKey(String clientEncryptPublicKey) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/identity/aeskey";
        Map<String, String> params = new HashMap<>(1);
        params.put("clientEncryptPublicKey", clientEncryptPublicKey);
        ResponseEntity<Map> responseEntity =
                this.httpService.doPost(url, new AthenaBasicHttpRequestHead(), JsonUtil.javaObjectToJsonString(params), Map.class);
        if (responseEntity.getStatusCode().is2xxSuccessful()) {
            return (String) responseEntity.getBody().get("encryptAesKey");
        } else {
            return "";
        }
    }

    private DigiwhaleLoginResponse buildResponse(DigiwhaleLoginResponse digiwhaleLoginResponse) {
        IamAuth iamAuth = new IamAuth();
        BeanUtils.copyProperties(digiwhaleLoginResponse, iamAuth);
        digiwhaleLoginResponse.setIamAuth(iamAuth);
        digiwhaleLoginResponse.setAccessToken(digiwhaleLoginResponse.getToken());
        return digiwhaleLoginResponse;
    }

    /**
     * 构造验证码错误的方法
     *
     * @param responseEntity
     * @param errorMessage
     * @return
     */
    private ResponseEntity buildError(ResponseEntity<DigiwhaleLoginResponse> responseEntity, String errorMessage) {
        responseEntity.getBody().setError(HttpStatus.INTERNAL_SERVER_ERROR.value() + "");
        responseEntity.getBody().setErrorDescription(errorMessage);
        return responseEntity;
    }

    @Override
    public String getAuthorizeCode(String userToken, String appId, String callbackUrl) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/oauth2/authorize";
        Map<String, String> params = new HashMap<>(2);
        params.put("appId", appId);
        params.put("callbackUrl", callbackUrl);
        ResponseEntity<Map> responseEntity =
                this.httpService.doPost(url, new AthenaBasicHttpRequestHead(userToken), JsonUtil.javaObjectToJsonString(params), Map.class);
        if (responseEntity.getStatusCode().is2xxSuccessful()) {
            Map<String, Object> map = (Map<String, Object>) responseEntity.getBody().get("data");
            return String.valueOf(map.get("code"));
        } else {
            return "";
        }
    }

    @Override
    public PasswordChange checkUserPasswordChange(String locale, String userToken) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/user/password/change/check";
        ResponseEntity<PasswordChange> responseEntity = this.httpService.doGet(url,
                new AthenaBasicHttpRequestHead(locale, userToken), Collections.emptyMap(), PasswordChange.class);
        if (responseEntity.getStatusCode().is2xxSuccessful()) {
            return responseEntity.getBody();
        } else {
            return new PasswordChange(false);
        }
    }

    @Override
    public Boolean updatePassword(String iamUserToken, Map<String, Object> parameters) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/user/update/password";
        ResponseEntity responseEntity =
                this.httpService.doPost(url, new AthenaBasicHttpRequestHead(iamUserToken), JsonUtil.javaObjectToJsonString(parameters), Map.class);
        System.out.println("1: " + JsonUtil.javaObjectToJsonString(responseEntity));
        if (responseEntity.getStatusCode().is2xxSuccessful()) {
            return true;
        } else {
            return false;
        }
    }

    @Override
    public Boolean renewalPassword(String iamUserToken) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/user/password/renewal";
        ResponseEntity responseEntity =
                this.httpService.doPost(url, new AthenaBasicHttpRequestHead(iamUserToken), Collections.emptyMap(), Map.class);
        if (responseEntity.getStatusCode().is2xxSuccessful()) {
            return true;
        } else {
            return false;
        }
    }

    @Override
    public Boolean isSuperAdmin(String userToken, String userId) {
        DigiwinIamUserRoleInfo digiwinIamUserRoleInfo = this.userRole(userToken, userId);
        if (null == digiwinIamUserRoleInfo) {
            return Boolean.FALSE;
        }
        List<DigiwinIamRole> roles = digiwinIamUserRoleInfo.getRoles();
        if (CollectionUtils.isEmpty(roles)) {
            return Boolean.FALSE;
        }
        return roles.stream().anyMatch(role -> "superadmin".equalsIgnoreCase(role.getId()));
    }

    @Override
    public DigiwinIamUserRoleInfo userRole(String userToken, String userId) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/user/role";

        Map<String, String> bodyPayLoad = new HashMap<>(1);
        bodyPayLoad.put("id", userId);

        // 使用ParameterizedTypeReference对返回带泛型参数的包装
        ParameterizedTypeReference<DigiwinIamUserRoleInfo> responseType =
                new ParameterizedTypeReference<DigiwinIamUserRoleInfo>() {
                };
        ResponseEntity<DigiwinIamUserRoleInfo> responseEntity =
                httpService.doPost(url, new AthenaBasicHttpRequestHead(userToken),
                        JsonUtil.javaObjectToJsonString(bodyPayLoad), responseType);

        if (responseEntity.getStatusCode().is2xxSuccessful()) {
            return responseEntity.getBody();
        }
        return null;
    }

    @Override
    public DigiwinIamUserAuthInfo getUserPermissionFunctional(String userToken, Map<String, Object> params) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/permission/user/functional";
        ResponseEntity<DigiwinIamUserAuthInfo> responseEntity =
                this.httpService.doPost(url, new AthenaBasicHttpRequestHead(userToken), JsonUtil.javaObjectToJsonString(params), DigiwinIamUserAuthInfo.class);
        return responseEntity.getStatusCode().is2xxSuccessful() ? responseEntity.getBody() : null;
    }

    @Override
    public JSONObject deleteBundle(String userId, String userToken, String unionId) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/mapping/delete";
        JSONObject param = new JSONObject();
        param.put("userId", userId);
        param.put("identityId", "Athena");
        param.put("verifyUserId", unionId);
        param.put("account", "line");

        ResponseEntity<JSONObject> responseEntity =
                this.httpService.doPost(url, new AthenaBasicHttpRequestHead(userToken), param.toJSONString(), JSONObject.class);
        return responseEntity.getStatusCode().is2xxSuccessful() ? responseEntity.getBody() : null;
    }

    /**
     * 绑定用户微信UUID/line userId信息
     *
     * @param
     * @param
     */
    @Override
    public JSONObject importThirdplatform(String comeFrom, String unionId, String iamId, String userToken, String tenantId) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + "/api/iam/integrate/import/thirdplatform";
        JSONObject param = new JSONObject();
        param.put("tenantId", tenantId);
        param.put("comeFrom", comeFrom);
        param.put("enterprise", false);
        param.put("action", 1);
        JSONObject users = new JSONObject().fluentPut("unionId", unionId).fluentPut("id", iamId);

        param.put("users", Collections.singletonList(users));
        JSONObject auth = new JSONObject().fluentPut("appId", "Athena");
        param.put("auth", auth);
        param.put("remark", "mock");

        ResponseEntity<JSONObject> responseEntity =
                this.httpService.doPost(url, new AthenaBasicHttpRequestHead(userToken), param.toJSONString(), JSONObject.class);
        return responseEntity.getStatusCode().is2xxSuccessful() ? responseEntity.getBody() : null;
    }

    @Override
    public DigiwhaleLoginResponse identityOauthTokenGrantAccess(ThirdPartyLoginFreeQueryDTO queryDTO) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/identity/oauth/token/grant/access";
        try {
            ResponseEntity<DigiwhaleLoginResponse> responseEntity = this.httpService.doPost(url, new AthenaBasicHttpRequestHead(), JsonUtil.javaObjectToJsonString(queryDTO), DigiwhaleLoginResponse.class);
            return responseEntity.getBody();
        } catch (Exception e) {
            logger.error("identityOauthTokenGrantAccess 异常:{}", e.getMessage());
        }
        return null;
    }

    @Override
    public String identityTokenTempAccess(String tempToken, String tenantId, String userId, String appId) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/identity/token/temp/access";
        try {
            Map<String, String> map = new HashMap<>(4);
            map.put("tempToken", tempToken);
            map.put("tenantId", tenantId);
            map.put("userId", userId);
            map.put("appId", appId);
            ResponseEntity<Object> responseEntity = this.httpService.doPost(url, new AthenaBasicHttpRequestHead(), JsonUtil.javaObjectToJsonString(map), Object.class);
            if (responseEntity.getStatusCode().is2xxSuccessful()) {
                JSONObject jsonObject = JsonUtil.objectToJavaObject(responseEntity.getBody(), new TypeReference<JSONObject>() {
                });
                return jsonObject.getJSONObject("data").getString("token");
            }
        } catch (Exception e) {
            logger.error("identityTokenTempAccess 异常:{}", e.getMessage());
        }
        return null;
    }

    @Override
    public DigiwhaleLoginResponse identityOauthTokenAccess(String code, String tenantId, String type, String appId) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/identity/oauth/token/access";
        try {
            Map<String, String> map = new HashMap<>(4);
            map.put("code", code);
            map.put("tenantId", tenantId);
            map.put("oauthType", type);
            map.put("appId", appId);
            ResponseEntity<DigiwhaleLoginResponse> responseEntity = this.httpService.doPost(url, new AthenaBasicHttpRequestHead(), JsonUtil.javaObjectToJsonString(map), DigiwhaleLoginResponse.class);
            if (responseEntity.getStatusCode().is2xxSuccessful()) {
                return responseEntity.getBody();
            }
        } catch (Exception e) {
            logger.error("identityTokenTempAccess 异常:{}", e.getMessage());
        }
        return null;
    }


    @Override
    public List<String> queryUserAppCodes(String iamUserToken) {
        try {
            String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                    + "/tenant/application";
            ResponseEntity<List> responseEntity = this.httpService.doGet(url, new AthenaBasicHttpRequestHead(iamUserToken), Collections.emptyMap(), List.class);
            List<Object> body = responseEntity.getStatusCode().is2xxSuccessful() ? responseEntity.getBody() : null;
            if (body == null) {
                return null;
            }
            List<String> appCodes = new ArrayList<String>();
            body.forEach(apps -> {
                if (apps != null) {
                    JSONObject apps_json = JsonUtil.objectToJavaObject(apps, JSONObject.class);
                    if (apps_json.get("id") != null && apps_json.get("expired") != null) {
                        if ("false".equals(apps_json.getString("expired"))) {
                            appCodes.add(apps_json.getString("id"));
                        }
                    }
                }
            });
            return appCodes;
        } catch (Exception ex) {
            return null;
        }
    }

    @Override
    public void userMetadataUserLanguage(String userToken, String userSid, String locale) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/usermetadata/user/language";
        Map<String, Object> map = new HashMap<>(5);
        map.put("catalogId", "basic");
        map.put("key", "dapAcceptLanguage");
        map.put("tenantSid", 0);
        map.put("userSid", userSid);
        map.put("value", locale);
        this.httpService.doPost(url, new AthenaBasicHttpRequestHead(userToken), JsonUtil.javaObjectToJsonString(map), Object.class);
    }

    @Override
    public ResponseEntity<Object> getUserMetadataUserLanguage(String userToken, String userSid, String userID) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/usermetadata/user/get";
        Map<String, Object> map = new HashMap<>(5);
        map.put("catalogId", "basic");
        map.put("key", "dapAcceptLanguage");
        map.put("tenantSid", 0);
        map.put("userSid", userSid);
        map.put("userID", userID);
        ResponseEntity<Object> ResponseEntity = this.httpService.doPost(url, new AthenaBasicHttpRequestHead(userToken), JsonUtil.javaObjectToJsonString(map), Object.class);
        return ResponseEntity;
    }

    @Override
    public Boolean associationRoleAddUsers(String iamUserToken, String roleId, List<String> userIds) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/association/role/addusers";
        try {
            Map<String, Object> map = new HashMap<>(2);
            map.put("roleId", roleId);
            map.put("userIds", userIds);

            // 使用ParameterizedTypeReference对返回带泛型参数的包装
            ParameterizedTypeReference<DigiwinIamRoleAddUserResponse> responseType =
                    new ParameterizedTypeReference<DigiwinIamRoleAddUserResponse>() {
                    };

            ResponseEntity<DigiwinIamRoleAddUserResponse> responseEntity = this.httpService.doPost(url,
                    new AthenaBasicHttpRequestHead(iamUserToken), JsonUtil.javaObjectToJsonString(map), responseType);
            if (responseEntity.getStatusCode().is2xxSuccessful()) {
                return true;
            }
        } catch (Exception e) {
            logger.error("associationRoleAddusers 异常:{}", e.getMessage());
        }
        return false;
    }

    @Override
    public DigiwinIamUserAddJoinTenant userAddJoinTenant(String iamUserToken, String tenantId, String tenantSid, String appId, String tenantVisible) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/user/add/join/tenant";

        Map<String, Object> bodyPayload = new HashMap<>(1);
        if (StringUtils.hasLength(tenantSid)) {
            bodyPayload.put("tenantSid", NumberUtils.parseNumber(tenantSid, Long.class));
        }
        if (StringUtils.hasLength(tenantId)) {
            bodyPayload.put("tenantId", tenantId);
        }
        if (StringUtils.hasLength(appId)) {
            bodyPayload.put("appId", appId);
        }
        if (StringUtils.hasLength(tenantVisible)) {
            bodyPayload.put("tenantVisible", tenantVisible);
        }

        ResponseEntity<DigiwinIamUserAddJoinTenant> responseEntity =
                this.httpService.doPost(url, new AthenaBasicHttpRequestHead(iamUserToken), JsonUtil.javaObjectToJsonString(bodyPayload),
                        DigiwinIamUserAddJoinTenant.class);

        return responseEntity.getBody();
    }

    @Override
    public Boolean checkUserInTenant(String user, String iamUserToken) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/tenant/check/userintenant";

        Map<String, Object> bodyPayload = new HashMap<>(1);
        bodyPayload.put("user", user);

        ResponseEntity<String> responseEntity =
                this.httpService.doPost(url, new AthenaBasicHttpRequestHead(iamUserToken), JsonUtil.javaObjectToJsonString(bodyPayload),
                        String.class);

        return responseEntity.getStatusCode().is2xxSuccessful() && Boolean.parseBoolean(responseEntity.getBody());
    }

    @Override
    public DigiwinIamAuthoredUser authCodeSubstitutionAccessToken(String authCode, String sourceAppId) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/identity/sso/accesstoken";

        Map<String, Object> bodyPayload = new HashMap<>(2);
        bodyPayload.put("authCode", authCode);
        bodyPayload.put("sourceAppId", sourceAppId);

        // 使用ParameterizedTypeReference对返回带泛型参数的包装
        ParameterizedTypeReference<DigiwinIamResult<DigiwinIamAuthoredUser>> responseType =
                new ParameterizedTypeReference<DigiwinIamResult<DigiwinIamAuthoredUser>>() {
                };

        ResponseEntity<DigiwinIamResult<DigiwinIamAuthoredUser>> responseEntity = this.httpService.doPost(url, new AthenaBasicHttpRequestHead(),
                JsonUtil.javaObjectToJsonString(bodyPayload), responseType);

        return responseEntity.getStatusCode().is2xxSuccessful() ?
                Objects.requireNonNull(responseEntity.getBody()).getData() : null;
    }

    @Override
    public String getAuthCode(String iamUserToken, String sourceAppId) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/identity/sso/auth/code";

        Map<String, Object> bodyPayload = new HashMap<>(1);
        bodyPayload.put("targetAppId", sourceAppId);

        // 使用ParameterizedTypeReference对返回带泛型参数的包装
        ParameterizedTypeReference<DigiwinIamResult> responseType =
                new ParameterizedTypeReference<DigiwinIamResult>() {
                };

        ResponseEntity<DigiwinIamResult> responseEntity = this.httpService.doPost(url, new AthenaBasicHttpRequestHead(iamUserToken),
                JsonUtil.javaObjectToJsonString(bodyPayload), responseType);
        if (responseEntity.getStatusCode().is2xxSuccessful()) {
            Map data = (Map) responseEntity.getBody().getData();
            return String.valueOf(data.get("authCode"));
        }

        return null;
    }

    @Override
    public JSONObject loginFailure(String userId) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX + "/user/login/failure";
        Map<String, String> body = new HashMap<>();
        body.put("userId", userId);

        ResponseEntity<JSONObject> responseEntity = this.httpService.doPost(url, new AthenaBasicHttpRequestHead().setProxyToken(sysEnvConfig.getAppToken()), JsonUtil.javaObjectToJsonString(body),
                JSONObject.class);

        if (responseEntity.getStatusCode().is2xxSuccessful()) {
            JSONObject data = responseEntity.getBody().getJSONObject("data");
            return data;
        }

        return new JSONObject();
    }

    @Override
    public JSONObject getImage(String userId) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX + "/captcha/image";
        Map<String, String> body = new HashMap<>();
        body.put("userId", userId);
        ResponseEntity<JSONObject> responseEntity = this.httpService.doPost(url, new AthenaBasicHttpRequestHead().setProxyToken(sysEnvConfig.getAppToken())
                , JsonUtil.javaObjectToJsonString(body), JSONObject.class);
        if (responseEntity.getStatusCode().is2xxSuccessful()) {
            JSONObject response = responseEntity.getBody();
            JSONObject data = response.getJSONObject("data");
            return data;

        }
        return new JSONObject();
    }

    @Override
    public String getIamDeviceId(String deviceId, String deviceType) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/device/id";
        AthenaBasicHttpRequestHead athenaBasicHttpRequestHead = new AthenaBasicHttpRequestHead();
        athenaBasicHttpRequestHead.setDeviceId(deviceId);
        athenaBasicHttpRequestHead.setDeviceType(deviceType);
        athenaBasicHttpRequestHead.setSourceFrom("app");
        // 使用ParameterizedTypeReference对返回带泛型参数的包装
        ParameterizedTypeReference<DigiwinIamResult> responseType =
                new ParameterizedTypeReference<DigiwinIamResult>() {
                };
        ResponseEntity<DigiwinIamResult> responseEntity = this.httpService.doGet(url, athenaBasicHttpRequestHead,
                CollectionUtils.newHashMap(2), responseType);
        if (responseEntity.getStatusCode().is2xxSuccessful()) {
            return String.valueOf(responseEntity.getBody().getData());
        }
        return null;
    }

    @Override
    public JSONObject getUserInfo(String iamUserToken, String userId) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/user";
        // 使用ParameterizedTypeReference对返回带泛型参数的包装
        ParameterizedTypeReference<JSONObject> responseType =
                new ParameterizedTypeReference<JSONObject>() {
                };
        Map<String, String> userInfo = new HashMap<>();
        userInfo.put("id", userId);
        ResponseEntity<JSONObject> responseEntity = this.httpService.doPost(url, new AthenaBasicHttpRequestHead(iamUserToken),
                JsonUtil.javaObjectToJsonString(userInfo), responseType);
        if (responseEntity.getStatusCode().is2xxSuccessful()) {
            return responseEntity.getBody();
        }
        return null;
    }

    @Override
    public DigiwinIamResult deleteUserMapping(String iamUserToken, Map<String, Object> params) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/mapping/delete";
        // 使用ParameterizedTypeReference对返回带泛型参数的包装
        ParameterizedTypeReference<DigiwinIamResult> responseType =
                new ParameterizedTypeReference<DigiwinIamResult>() {
                };
        ResponseEntity<DigiwinIamResult> responseEntity = this.httpService.doPost(url, new AthenaBasicHttpRequestHead(iamUserToken),
                JsonUtil.javaObjectToJsonString(params), responseType);
        if (responseEntity.getStatusCode().is2xxSuccessful()) {
            return responseEntity.getBody();
        }
        return null;
    }

    @Override
    public DigiwinIamResult getUserInfoByEmail(String iamUserToken, Map<String, String> params) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/user/email";
        // 使用ParameterizedTypeReference对返回带泛型参数的包装
        ParameterizedTypeReference<DigiwinIamResult> responseType =
                new ParameterizedTypeReference<DigiwinIamResult>() {
                };
        ResponseEntity<DigiwinIamResult> responseEntity = this.httpService.doPost(url, new AthenaBasicHttpRequestHead(iamUserToken),
                JsonUtil.javaObjectToJsonString(params), responseType);
        if (responseEntity.getStatusCode().is2xxSuccessful()) {
            return responseEntity.getBody();
        }
        return null;
    }

    @Override
    public DigiwinIamResult userAddJoinTenantCustom(String iamUserToken, Map<String, String> params) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/user/add/join/tenant/custom";
        // 使用ParameterizedTypeReference对返回带泛型参数的包装
        ParameterizedTypeReference<DigiwinIamResult> responseType =
                new ParameterizedTypeReference<DigiwinIamResult>() {
                };
        ResponseEntity<DigiwinIamResult> responseEntity = this.httpService.doPost(url, new AthenaBasicHttpRequestHead(iamUserToken),
                JsonUtil.javaObjectToJsonString(params), responseType);
        if (responseEntity.getStatusCode().is2xxSuccessful()) {
            return responseEntity.getBody();
        }
        return null;
    }

    @Override
    public ResponseEntity<JSONObject> registerIamUser(String iamUserToken, Map<String, Object> params) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/user/register/without/captcha";
        // 使用ParameterizedTypeReference对返回带泛型参数的包装
        ParameterizedTypeReference<JSONObject> responseType =
                new ParameterizedTypeReference<JSONObject>() {
                };
        ResponseEntity<JSONObject> responseEntity = this.httpService.doPost(url, new AthenaBasicHttpRequestHead(iamUserToken),
                JsonUtil.javaObjectToJsonString(params), responseType);
        return responseEntity;

    }

    @Override
    public ResponseEntity<DigiwhaleLoginResponse> loginInternal(String tenantId) {
        return this.loginInternal(tenantId, LOGIN_INTERNAL_USERNAME, LOGIN_INTERNAL_PASSWORD);
    }

    private ResponseEntity<DigiwhaleLoginResponse> loginInternal(String tenantId, String userId, String password) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/identity/login/internal";
        // 使用ParameterizedTypeReference对返回带泛型参数的包装
        ParameterizedTypeReference<DigiwhaleLoginResponse> responseType =
                new ParameterizedTypeReference<DigiwhaleLoginResponse>() {
                };
        Map<String, String> params = new HashMap<>();
        params.put("tenantId", tenantId);
        params.put("userId", userId);
        params.put("passwordHash", password);
        ResponseEntity<DigiwhaleLoginResponse> responseEntity = this.httpService.doPost(url, new AthenaBasicHttpRequestHead(),
                JsonUtil.javaObjectToJsonString(params), responseType);
        if (responseEntity.getStatusCode().is2xxSuccessful()) {
            return responseEntity;
        }
        return null;
    }

    @Override
    public JSONObject queryGlobalLine(String iamUserToken, Map<String, String> params) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/mapping/query/global/single";
        // 使用ParameterizedTypeReference对返回带泛型参数的包装
        ParameterizedTypeReference<JSONObject> responseType =
                new ParameterizedTypeReference<JSONObject>() {
                };
        ResponseEntity<JSONObject> responseEntity = this.httpService.doPost(url, new AthenaBasicHttpRequestHead(iamUserToken),
                JsonUtil.javaObjectToJsonString(params), responseType);
        if (responseEntity.getStatusCode().is2xxSuccessful()) {
            return responseEntity.getBody();
        }
        return null;
    }

    @Override
    public Boolean setUserDefaultTenant(String iamUserToken, String tenantId, String tenantSid) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/user/setting/default/tenant";

        Map<String, Object> bodyPayload = new HashMap<>(1);
        if (StringUtils.hasLength(tenantSid)) {
            bodyPayload.put("sid", NumberUtils.parseNumber(tenantSid, Long.class));
        }
        if (StringUtils.hasLength(tenantId)) {
            bodyPayload.put("id", tenantId);
        }

        ResponseEntity<String> responseEntity =
                this.httpService.doPost(url, new AthenaBasicHttpRequestHead(iamUserToken), JsonUtil.javaObjectToJsonString(bodyPayload),
                        String.class);

        return responseEntity.getStatusCode().is2xxSuccessful();
    }

    @Override
    public Boolean setUserDefaultAcceptLanguage(String iamUserToken, Long tenantSid, Long userSid, String locale) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/usermetadata/user/language";

        Map<String, Object> bodyPayload = new HashMap<>(1);
        if (null != tenantSid) {
            bodyPayload.put("tenantSid", tenantSid);
        } else {
            bodyPayload.put("tenantId", 0L);
        }
        bodyPayload.put("catalogId", "basic");
        bodyPayload.put("key", "dapAcceptLanguage");
        bodyPayload.put("userSid", userSid);
        bodyPayload.put("value", locale);

        ResponseEntity<String> responseEntity =
                this.httpService.doPost(url, new AthenaBasicHttpRequestHead(iamUserToken), JsonUtil.javaObjectToJsonString(bodyPayload),
                        String.class);

        return responseEntity.getStatusCode().is2xxSuccessful();
    }

    @Override
    public DigiwinIamTenant currentTenant(String iamUserToken) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/tenant/current/simple";
        // 使用ParameterizedTypeReference对返回带泛型参数的包装
        ParameterizedTypeReference<DigiwinIamResult<DigiwinIamTenant>> responseType =
                new ParameterizedTypeReference<DigiwinIamResult<DigiwinIamTenant>>() {
                };
        ResponseEntity<DigiwinIamResult<DigiwinIamTenant>> responseEntity =
                this.httpService.doGet(url, new AthenaBasicHttpRequestHead(iamUserToken),
                        new HashMap<>(), responseType);
        if (responseEntity.getStatusCode().is2xxSuccessful()) {
            return responseEntity.getBody().getData();
        }
        return null;
    }

    @Override
    public DigiwinIamUserInfo queryUserInfo(String iamUserToken, String userId) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/user";
        // 使用ParameterizedTypeReference对返回带泛型参数的包装
        ParameterizedTypeReference<DigiwinIamUserInfo> responseType =
                new ParameterizedTypeReference<DigiwinIamUserInfo>() {
                };
        Map<String, String> params = new HashMap<>();
        params.put("id", userId);
        ResponseEntity<DigiwinIamUserInfo> responseEntity = this.httpService.doPost(url, new AthenaBasicHttpRequestHead(iamUserToken),
                JsonUtil.javaObjectToJsonString(params), responseType);
        if (!responseEntity.getStatusCode().is2xxSuccessful()) {
            throw new ServiceException(responseEntity.getBody().getMessage());
        }
        return responseEntity.getBody();
    }

    @Override
    public Boolean doubleCheckUpdate(String iamUserToken, Map<String, Object> params) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/user/update";
        // 使用ParameterizedTypeReference对返回带泛型参数的包装
        ParameterizedTypeReference<DigiwinIamBaseResponse> responseType =
                new ParameterizedTypeReference<DigiwinIamBaseResponse>() {
                };
        // 此接口响应为null，需要注意下
        ResponseEntity<DigiwinIamBaseResponse> responseEntity = this.httpService.doPost(url, new AthenaBasicHttpRequestHead(iamUserToken),
                JsonUtil.javaObjectToJsonString(params), responseType);
        if (!responseEntity.getStatusCode().is2xxSuccessful()) {
            throw new ServiceException(responseEntity.getBody().getMessage());
        }
        return true;
    }

    @Override
    public DigiwinIamAccountRegisterResponse emailExistCheck(String iamUserToken, String account) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/user/email/exist";
        // 使用ParameterizedTypeReference对返回带泛型参数的包装
        ParameterizedTypeReference<DigiwinIamAccountRegisterResponse> responseType =
                new ParameterizedTypeReference<DigiwinIamAccountRegisterResponse>() {
                };
        Map<String, String> params = new HashMap<>();
        params.put("email", account);
        ResponseEntity<DigiwinIamAccountRegisterResponse> responseEntity =
                this.httpService.doPost(url, new AthenaBasicHttpRequestHead(iamUserToken),
                        JsonUtil.javaObjectToJsonString(params), responseType);
        if (!responseEntity.getStatusCode().is2xxSuccessful()) {
            throw new ServiceException(responseEntity.getBody().getMessage());
        }
        return responseEntity.getBody();
    }

    @Override
    public DigiwinIamAccountRegisterResponse mobilephoneExistCheck(String iamUserToken, String account) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/user/mobilephone/exist";
        // 使用ParameterizedTypeReference对返回带泛型参数的包装
        ParameterizedTypeReference<DigiwinIamAccountRegisterResponse> responseType =
                new ParameterizedTypeReference<DigiwinIamAccountRegisterResponse>() {
                };
        Map<String, String> params = new HashMap<>();
        params.put("telephone", account);
        ResponseEntity<DigiwinIamAccountRegisterResponse> responseEntity =
                this.httpService.doPost(url, new AthenaBasicHttpRequestHead(iamUserToken),
                        JsonUtil.javaObjectToJsonString(params), responseType);
        if (!responseEntity.getStatusCode().is2xxSuccessful()) {
            throw new ServiceException(responseEntity.getBody().getMessage());
        }
        return responseEntity.getBody();
    }

    @Override
    public Boolean passwordUpdate(String iamUserToken, Map<String, Object> params) {
        String url = AppContext.getApiUrlSetting().getIamUrl() + DigiwinIamProxyService.API_PREFIX
                + "/user/password/update";
        // 使用ParameterizedTypeReference对返回带泛型参数的包装
        ParameterizedTypeReference<DigiwinIamBaseResponse> responseType =
                new ParameterizedTypeReference<DigiwinIamBaseResponse>() {
                };
        // 此接口响应为null，需要注意下
        ResponseEntity<DigiwinIamBaseResponse> responseEntity = this.httpService.doPost(url, new AthenaBasicHttpRequestHead(iamUserToken),
                JsonUtil.javaObjectToJsonString(params), responseType);
        if (!responseEntity.getStatusCode().is2xxSuccessful()) {
            throw new ServiceException(responseEntity.getBody().getMessage());
        }
        return true;
    }
}