package com.digiwin.mobile.mobileuibot.proxy.uibot.model;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.stream.Collectors;


/**
 * 多个数据源组成的集合
 */
public class QueryResultSet {

    private List<QueryResult> queryResults;

    private QueryResult mainQueryResult;

    public List<QueryResult> getQueryResults() {
        return queryResults;
    }

    public void setMainQueryResult(QueryResult mainQueryResult) {
        this.mainQueryResult = mainQueryResult;
    }

    public QueryResult getMainQueryResult() {
        if (mainQueryResult == null && this.queryResults.size() > 0) {
            mainQueryResult = this.queryResults.get(0);
        }
        return mainQueryResult;
    }

    /**
     * 获取api的response的单头字段，可以作为数据源名称使用
     *
     * @return 单头字段名，默认是data
     */
    public String getDataSourceNameFromApiResp() {
        if (getMainQueryResult() == null) {
            return "";
        }
        return getMainQueryResult().getDataSourceNameFromApiResp();
    }

    /**
     * 获取api的response的单头字段
     *
     * @return 单头字段
     */
    public PcUiBotMetadataField getMasterField() {
        if (getMainQueryResult() == null) {
            return null;
        }
        return getMainQueryResult().getMasterField();
    }

    private QueryResultSet() {
    }

    public static QueryResultSet empty() {
        QueryResultSet queryResultSet = new QueryResultSet();
        queryResultSet.queryResults = new ArrayList<>();
        queryResultSet.mainQueryResult = QueryResult.empty();
        return queryResultSet;
    }

    public static QueryResultSet withData(String mainDatasource, QueryResult queryResult) {
        if (queryResult == null) {
            throw new IllegalArgumentException("queryResult");
        }
        List<QueryResult> queryResultList = new ArrayList<>();
        queryResultList.add(queryResult);
        return withData(mainDatasource, queryResultList);
    }

    public static QueryResultSet withData(String mainDatasource, List<QueryResult> queryResultList) {
        if (queryResultList == null) {
            throw new IllegalArgumentException("queryResultList");
        }

        QueryResultSet queryResultSet = new QueryResultSet();
        queryResultSet.queryResults = queryResultList;
        if (StringUtils.hasText(mainDatasource)) {
            for (QueryResult queryResult : queryResultList) {
                if (mainDatasource.equals(queryResult.getDataSourceName())) {
                    queryResultSet.mainQueryResult = queryResult;
                    break;
                }
            }
        } else {
            if (queryResultList.size() > 0) {
                queryResultSet.mainQueryResult = queryResultList.get(0);
            } else {
                queryResultSet.mainQueryResult = QueryResult.empty();
            }
        }
        return queryResultSet;
    }

    public QueryResultSet appendData(QueryResult queryResult) {
        if (queryResult == null) {
            throw new IllegalArgumentException("queryResult");
        }
        this.queryResults.add(queryResult);
        return this;
    }


    public boolean contains(String name) {
        return this.queryResults.stream().anyMatch((queryResult -> name.equals(queryResult.getDataSourceName())));
    }

    public QueryResult get(String name) {
        Optional<QueryResult> optionalQueryResult = this.queryResults.stream()
                .filter((queryResult -> name.equals(queryResult.getDataSourceName()))).findFirst();
        if (optionalQueryResult.isPresent()) {
            return optionalQueryResult.get();
        } else {
            return null;
        }
    }

    /**
     * 构建pageData，key为数据源名称，value为map或list<map>
     *
     * @return
     */
    public Map<String, Object> getPageData() {
        Map<String, Object> pageData = new HashMap<>();
        for (QueryResult queryResult : queryResults) {
            if (queryResult.isSingle()) {
                if (!CollectionUtils.isEmpty(queryResult.getData())) {
                    pageData.put(queryResult.getDataSourceName(), queryResult.getData().get(0));
                } else {
                    pageData.put(queryResult.getDataSourceName(), new HashMap<>());
                }
            } else {
                pageData.put(queryResult.getDataSourceName(), queryResult.getData());
            }
        }
        return pageData;
    }

    /**
     * 构建空数据集，包含全部元数据
     *
     * @param allFields
     * @return
     */
    public Map<String, Object> getPageData(List<PcUiBotAllFields> allFields) {
        Map pageData = new HashMap<>();
        for (QueryResult queryResult : queryResults) {
            if (queryResult.isSingle()) {
                if (!CollectionUtils.isEmpty(queryResult.getData())) {
                    pageData.put(queryResult.getDataSourceName(), queryResult.getData().get(0));
                } else {

                    pageData.put(queryResult.getDataSourceName(), bindEmptyPageData(allFields));
                }
            } else {
                pageData.put(queryResult.getDataSourceName(), queryResult.getData());
            }
        }
        return pageData;
    }

    /**
     * 获取数据索引字段
     *
     * @return
     */
    public Map<String, Map<String, Integer>> getPageDataIndex() {
        Map pageData = new HashMap<>();
        for (QueryResult queryResult : queryResults) {
//            if(queryResult.isSingle() && CollectionUtils.isNotEmpty(queryResult.getData())){
//                pageData.put(queryResult.getDataSourceName(),queryResult.getDataKeyIndex());
//            }else{
//                pageData.put(queryResult.getDataSourceName(),queryResult.getDataKeyIndex());
//            }
            pageData.put(queryResult.getDataSourceName(), queryResult.getDataKeyIndex());
        }
        return pageData;
    }

    /**
     * 获取数据索引字段集合
     *
     * @return
     */
    public Map<String, List<String>> getPageDataKeys() {
        Map<String, List<String>> pageDataKeys = new HashMap<>();
        pageDataKeys.put(mainQueryResult.getDataSourceName(), mainQueryResult.getDataKeys());
        /*for (QueryResult queryResult : queryResults) {
            if (queryResult.isSingle() && CollectionUtils.isNotEmpty(queryResult.getDataKeys())) {
                pageDataKeys.put(queryResult.getDataSourceName(), queryResult.getDataKeys());
            } else {
                pageDataKeys.put(queryResult.getDataSourceName(), queryResult.getDataKeys());
            }
        }*/
        return pageDataKeys;
    }

    /**
     * 构建空数据集
     */
    @JsonIgnore
    private Map bindEmptyPageData(List<PcUiBotAllFields> allFields) {
        if (CollectionUtils.isEmpty(allFields)) {
            return Maps.newHashMapWithExpectedSize(0);
        }
        Set<String> keys = allFields.stream().map(PcUiBotAllFields::getName).collect(Collectors.toSet());
        if (keys.isEmpty()) {
            return Maps.newHashMapWithExpectedSize(0);
        }
        Map<String, Object> pageDataMap = Maps.newHashMapWithExpectedSize(keys.size());
        List<String> typeToNull = Lists.newArrayList("number", "numeric", "boolean", "date", "datetime");
        //只获取第一层数据结构
        for (String key : keys) {
            Optional<String> dataType = allFields.stream()
                    .filter(field -> key.equals(field.getName()))
                    .map(PcUiBotAllFields::getDataType)
                    .findAny();
            if (dataType.isPresent()) {
                String type = dataType.get();
                if (typeToNull.contains(type)) {
                    pageDataMap.put(key, null);
                } else if ("array".equals(type)) {
                    pageDataMap.put(key, Lists.newArrayListWithExpectedSize(0));
                } else if ("object".equals(type)) {
                    pageDataMap.put(key, Maps.newHashMapWithExpectedSize(0));
                } else {
                    pageDataMap.put(key, "");
                }
            }
        }
        return pageDataMap;
    }

    /**
     * 所有数据
     *
     * @Author：SYQ
     * @Date：2022/7/19 14:31
     */
    public List<Map<String, Object>> getQueryResultData() {
        List<Map<String, Object>> dataList = new ArrayList<>();
        for (QueryResult queryResult : queryResults) {
            if (!CollectionUtils.isEmpty(queryResult.getData())) {
                dataList.addAll(queryResult.getData());
            }
        }
        return dataList;
    }

    /**
     * 主查询结构是否有下一页
     *
     * @return
     */
    public boolean isMainQueryResultHasNext() {
        return this.getMainQueryResult() != null && this.getMainQueryResult().isHasNext();
    }

}
