package com.digiwin.mobile.mobileuibot.proxy.uibot.model;

import com.alibaba.fastjson.annotation.JSONField;
import com.digiwin.mobile.mobileuibot.common.calculate.UUIDUtil;
import com.digiwin.mobile.mobileuibot.common.context.AppContext;
import com.digiwin.mobile.mobileuibot.common.context.SpringContextHolder;
import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.common.localization.LocaleUtil;
import com.digiwin.mobile.mobileuibot.core.appec.UiBotSlip;
import com.digiwin.mobile.mobileuibot.core.component.BaseMobileComponent;
import com.digiwin.mobile.mobileuibot.core.component.BaseMobileComponentWrapper;
import com.digiwin.mobile.mobileuibot.core.component.MobileLayoutTypeEnum;
import com.digiwin.mobile.mobileuibot.core.component.action.Action;
import com.digiwin.mobile.mobileuibot.core.component.action.ActionSubmitParam;
import com.digiwin.mobile.mobileuibot.core.component.action.ActionSubmitParamJsonDeserializer;
import com.digiwin.mobile.mobileuibot.core.component.basic.BlankArea;
import com.digiwin.mobile.mobileuibot.core.component.basic.Empty;
import com.digiwin.mobile.mobileuibot.core.component.basic.HiddenConfig;
import com.digiwin.mobile.mobileuibot.core.component.button.BottomButtonDigiwinAthena;
import com.digiwin.mobile.mobileuibot.core.component.button.BottomButtonList;
import com.digiwin.mobile.mobileuibot.core.component.button.Button;
import com.digiwin.mobile.mobileuibot.core.component.card.card.Card;
import com.digiwin.mobile.mobileuibot.core.component.group.CustomGroup;
import com.digiwin.mobile.mobileuibot.core.component.group.CustomGroupContent;
import com.digiwin.mobile.mobileuibot.core.component.group.CustomGroupStyleTypeEnum;
import com.digiwin.mobile.mobileuibot.core.component.title.titlebody.TitleBody;
import com.digiwin.mobile.mobileuibot.core.component.webview.WebviewPanel;
import com.digiwin.mobile.mobileuibot.core.layout.LayoutGeneratorServiceResult;
import com.digiwin.mobile.mobileuibot.core.layout.doublepattern.bean.PcLayoutEnum;
import com.digiwin.mobile.mobileuibot.core.layout.doublepattern.bean.PcModuleEnum;
import com.digiwin.mobile.mobileuibot.designer.uibot.service.UiBotDesignerRenderService;
import com.digiwin.mobile.mobileuibot.locale.service.LocaleService;
import com.digiwin.mobile.mobileuibot.locale.service.impl.LocaleServiceImpl;
import com.digiwin.mobile.mobileuibot.proxy.uibot.PcUiBotLayoutJsonDeserializer;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.content.UiBotContent;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.layout.UiBotLayout;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.rule.UiBotRule;
import com.digiwin.mobile.mobileuibot.task.model.TbdsParameter;
import com.digiwin.mobile.mobileuibot.task.model.TbdsQueryInfo;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import lombok.AllArgsConstructor;
import lombok.Data;
import org.apache.commons.lang3.ObjectUtils;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.io.Serializable;
import java.util.*;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

/**
 * <p>功能描述：UI Bot数据完整模型</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: UiBotModel
 * @Author: Zaregoto
 * @Date: 2021/4/20 16:24
 */
@Data
@AllArgsConstructor
public class UiBotModel implements Serializable {
    private static final long serialVersionUID = -8228407989245090791L;

    private List<UiBotAction> actions;
    // 转派按钮
    private Button reassignButton;
    private List<Button> appbarButtonList = new ArrayList<>(3);
    private UiBotContent content;
    private UiBotExecuteContext executeContext;

    @JsonDeserialize(contentUsing = PcUiBotLayoutJsonDeserializer.class)
    private List<UiBotLayout> layout;

    private UiBotPageData pageData;
    private UiBotPageData pageDataIndex;
    private UiBotPageData pageDataKeys;
    private List<UiBotRule> rules;

    private UiBotStyle style;
    private Boolean finished;

    private String finishedTitle;

    /*********************************************************************************************
     * ******************************** 移动端特有属性 BEGIN ****************************************
     *********************************************************************************************/
    private String dataId;
    /**
     * 调用searchPcTargetModel方法时才会有值，值为找到layout的父节点schema,如果没有父节点则为null
     */
    private String parentSchema;

    /**
     * 页面类型；目前可选值：
     * INPUT_FROM：录入表单类型页面，为该类型页面时需要使用组件默认隐藏逻辑
     */
    private String pageType;
    private UiBotSlip slip;

    //目前只支持3。1-内容左对齐，2-内容居中对齐，3-内容右对齐
    private Integer alignmentType;
    //1-全屏时的尺寸类型，2-嵌套时的尺寸类型
    private Integer sizeType;

    /**
     * 收起部分组件，展示查看更多，无值时按false处理，为true时
     * 页面自动隐藏相关组件，底部出现查看更多功能（隐藏的组件在页面layout下，或者CUSTOM_GROUP组件内）
     */
    private Boolean showMore;
    private HiddenConfig hiddenConfig;
    /**
     * 是否有下一页（目前仅用于消息列表分页）
     */
    private Boolean hasMore;
    /**
     * 用于多层布局详情返给前端
     */
    private String editId;
    /**
     * 目前用于多层布局已处理数据的回填
     */
    @JsonDeserialize(contentUsing = ActionSubmitParamJsonDeserializer.class)
    private List<ActionSubmitParam> params;
    /**
     * 公共参数（当前页面所有按钮跳转都会把该参数里面的数据塞到rawData里面）
     */
    private Map<String, Object> commonRawData = new HashMap<>(4);

    /**
     * 页面背景色
     */
    private String pageBackgroudColor;
    private String name;

    /**
     * 前端app使用的渲染引擎版本号。
     * 若该字段没有值，或版本号不为2，则前端app会使用旧渲染引擎与相关配套组件（无BaseComponent版本）
     * 若该字段有值，且版本号为2，则前端app会使用新渲染引擎与相关配套组件（有BaseComponent版本）
     * add by mowj 20230208
     *
     * @see com.digiwin.mobile.mobileuibot.core.component.MobileRenderMetaData
     */
    private Integer renderVersion;

    /**
     * 前端app使用的渲染数据DSL
     * 从renderVersion为2时开始使用
     * add by mowj 20230209
     */
    private UiBotRenderData renderData;
    private Map<String, Object> variableMaps = new HashMap<>(8);

    /**
     * 是否应用了设计器的字段范围与顺序配置
     */
    @JsonProperty(access = JsonProperty.Access.WRITE_ONLY)
    @JSONField(serialize = false)
    private Boolean appliedDesignerFieldAndOrder;

    /**
     * 是否应用了设计器的字段标签配置
     */
    @JsonProperty(access = JsonProperty.Access.WRITE_ONLY)
    @JSONField(serialize = false)
    private Boolean appliedDesignerFieldTag;

    /**
     * 是否应用了设计器的字段规则配置
     */
    @JsonProperty(access = JsonProperty.Access.WRITE_ONLY)
    @JSONField(serialize = false)
    private Boolean appliedDesignerFieldRule;

    /**
     * 是否应用了设计器的字段操作配置
     */
    @JsonProperty(access = JsonProperty.Access.WRITE_ONLY)
    @JSONField(serialize = false)
    private Boolean appliedDesignerFieldOperation;

    /**
     * 布局生成服务的处理结果
     */
    @JsonProperty(access = JsonProperty.Access.WRITE_ONLY)
    @JSONField(serialize = false)
    private LayoutGeneratorServiceResult layoutGeneratorServiceResult;

//    /**
//     * PC下的任务卡详情页面是否定制
//     * true:是，false:否
//     */
//    @JsonProperty(access = JsonProperty.Access.WRITE_ONLY)
//    @JSONField(serialize = false)
//    private Boolean isPcCustomize;

    /**
     * 当前页面渲染完毕后，前端立刻执行该action
     */
    private Action beforeAction;

    //ztb运输在途--定制字段
    /**
     * 是否提醒开启定位
     */
    private Boolean showStartPosition;

    /**
     * 只有v2版本的DSL时增加,前端使用
     */
    private UiBotPageData rawUiBotPageData;

    /**
     * 移动端布局类型
     * 提供给前端判断是走单层、多层、其它逻辑处理
     *
     * @see com.digiwin.mobile.mobileuibot.core.component.MobileLayoutTypeEnum
     */
    private String mobileLayoutType;
    /**
     * 跳转页面时，返回给前端的页面标题
     */
    private String pageTitle;
    /**
     * 标题是否显示，true：显示，false：隐藏 -- 设计器新增配置
     */
    private Boolean titleVisible = false;

    /**
     * 手机灭屏，亮屏后，是否刷新
     */
    private Boolean isRefreshBrightScreen;

    /**
     * app退到后台，再回到前台，是否刷新
     */
    private Boolean isRefreshSwitchAppToForeground;


    /**
     * app退到后台，再回到前台，是否刷新
     */
    private Map<String, Object> hooks;

    /**
     * 局部规则
     */
    private Map<String, List<UiBotRule>> localRules;

    /**
     * 判断是不是任务卡界面(埋点需求),
     * 0：默认类型
     * 1:任务卡类型的model
     * 后续再拓展
     */
    private Integer modelType = UibotModelTypeEnum.DEFAULT_PAGE_MODEL.getValue();

    /*********************************************************************************************
     * ******************************** 移动端特有属性 END *****************************************
     *********************************************************************************************/

    protected UiBotModel() {
        this.actions = new ArrayList<>();
        this.content = new UiBotContent();
        this.executeContext = new UiBotExecuteContext();
        this.layout = new ArrayList<>();
        this.pageData = new UiBotPageData();
        this.rules = new ArrayList<>();
        this.style = new UiBotStyle();
        this.slip = new UiBotSlip();
    }

    public UiBotModel(List<UiBotLayout> layout) {
        this();
        this.layout = layout;
    }

    public static UiBotModel emptyUiBotModel() {
        return new UiBotModel();
    }

    public static UiBotModel emptyAbiReportUiBotModel(String locale, String emptyMsg, String buttonName, String dataId) {
        UiBotModel uiBotModel = new UiBotModel();
        UiBotPageData emptyDataModelPageData = new UiBotPageData();

        String blankAreaSchema = UUIDUtil.getUuid();
        UiBotLayout blankAreaLayout = new UiBotLayout(blankAreaSchema, BlankArea.COMPONENT_TYPE);
        uiBotModel.addLayout(blankAreaLayout);
        emptyDataModelPageData.put(blankAreaSchema, BlankArea.create(326, 1.0));

        String emptySchema = UUIDUtil.getUuid();
        UiBotLayout emptyDataModelLayout = new UiBotLayout(emptySchema, Empty.COMPONENT_TYPE);
        uiBotModel.addLayout(emptyDataModelLayout);
        emptyDataModelPageData.put(emptySchema, Empty.createByAbiReport(LocaleUtil.getMobileTextByKey(locale, emptyMsg), true, 364, 197, "#D2D2E6"));

        String buttonSchema = UUIDUtil.getUuid();
        UiBotLayout buttonLayout = new UiBotLayout(buttonSchema, Button.COMPONENT_TYPE);
        uiBotModel.addLayout(buttonLayout);
        emptyDataModelPageData.put(buttonSchema, Button.createAbiReport(buttonName, dataId,
                null, "", "", Collections.emptyMap(),
                false, true));

        uiBotModel.setPageData(emptyDataModelPageData);
        return uiBotModel;
    }

    /**
     * 创建一个具有空内容提示的页面模型
     *
     * @param locale   语言别
     * @param emptyMsg 提示文字，会经过多语言处理
     * @return
     */
    public void createEmptyContentModel(String locale, String emptyMsg) {
        String schema = "emptyData";
        UiBotLayout emptyDataModelLayout = new UiBotLayout(schema, Empty.COMPONENT_TYPE);
        UiBotPageData emptyDataModelPageData = new UiBotPageData();
        emptyDataModelPageData.put(schema, Empty.create(LocaleUtil.getMobileTextByKey(locale, emptyMsg)));

        this.addLayout(emptyDataModelLayout);
        this.setPageData(emptyDataModelPageData);
    }

    /**
     * 创建一个具有空内容提示的页面模型
     *
     * @param emptyMsg 提示文字（不做多语言处理）
     * @return
     */
    public UiBotModel createEmptyContentModel(String emptyMsg) {
        String schema = "emptyData";
        UiBotLayout emptyDataModelLayout = new UiBotLayout(schema, Empty.COMPONENT_TYPE);
        UiBotPageData emptyDataModelPageData = new UiBotPageData();
        emptyDataModelPageData.put(schema, Empty.create(emptyMsg));

        this.addLayout(emptyDataModelLayout);
        this.setPageData(emptyDataModelPageData);

        return this;
    }

    public void addLayout(UiBotLayout uiBotLayout) {
        if (null != this.layout) {
            this.layout.add(uiBotLayout);
        }
    }

    /**
     * 创建展示固定欢迎提示短语的 自定义内容组件
     *
     * @param locale 语言别
     * @return
     */
    public static UiBotModel createCustomContentWelcomeModel(String locale) {
        String schema = "panel";
        UiBotLayout layout = new UiBotLayout();
        layout.setSchema(schema);
        layout.setType(WebviewPanel.COMPONENT_TYPE);

        List<UiBotLayout> layoutList = new ArrayList<>();
        layoutList.add(layout);

        WebviewPanel webviewPanel = new WebviewPanel();
        webviewPanel.setTitle(LocaleUtil.getMobileTextByKey(locale, "敬请期待") + "!");
        webviewPanel.setData(new ArrayList<>());

        UiBotModel uiBotModel = new UiBotModel(layoutList);
        uiBotModel.addPageData(schema, webviewPanel);

        return uiBotModel;
    }

    /**
     * 创建单个WEBVIEW_PANEL的页面Model
     *
     * @param webviewPanel WebViewPanel的一个实例对象
     * @return
     */
    public static UiBotModel createCustomContentSingleWebviewPanelModel(WebviewPanel webviewPanel) {
        String schema = "panel";
        UiBotLayout layout = new UiBotLayout();
        layout.setSchema(schema);
        layout.setType(WebviewPanel.COMPONENT_TYPE);

        List<UiBotLayout> layoutList = new ArrayList<>();
        layoutList.add(layout);

        UiBotModel uiBotModel = new UiBotModel(layoutList);
        uiBotModel.addPageData(schema, webviewPanel);

        return uiBotModel;
    }

    public static UiBotModel createSingleButton(List<BottomButtonDigiwinAthena> buttonList) {
        String schema = "bottomButtons";
        UiBotLayout layout = new UiBotLayout();
        layout.setSchema(schema);
        layout.setType(BottomButtonDigiwinAthena.COMPONENT_TYPE);

        List<UiBotLayout> layoutList = new ArrayList<>();
        layoutList.add(layout);

        UiBotModel uiBotModel = new UiBotModel(layoutList);
        List<BottomButtonDigiwinAthena> bottomButtons = new ArrayList<>();
        for (BottomButtonDigiwinAthena button : buttonList) {
            bottomButtons.add(button);
        }
        uiBotModel.addPageData(schema, bottomButtons);
        //TODO 先用魔法值写。后续改成枚举类值
        uiBotModel.setAlignmentType(3);
        uiBotModel.setSizeType(2);

        return uiBotModel;
    }

    public UiBotModel addPageData(String dataKey, Object data) {
        this.pageData.put(dataKey, data);
        return this;
    }

    /**
     * 找到实际明细数据的layout.
     * schema="customData"的，表示是pc端的定制页面；需要按定制的写法做；其他情况则是标准页面。
     *
     * @return
     */
    public UiBotTargetLayoutSearchResult searchPcTargetLayout() {
        return this.searchPcTargetLayout(this.getLayout());
    }

    /**
     * 找到实际明细数据的layout.
     * schema="customData"的，表示是pc端的定制页面；需要按定制的写法做；其他情况则是标准页面。
     *
     * @param layoutList
     * @return
     */
    public UiBotTargetLayoutSearchResult searchPcTargetLayout(List<UiBotLayout> layoutList) {
        boolean hasCustomData = false;
        int targetLayoutIdx = -1;
        UiBotLayout targetLayout = null;
        for (int i = 0; i < layoutList.size(); i++) {
            UiBotLayout currentLayout = layoutList.get(i);
            if ("customData".equalsIgnoreCase(currentLayout.getSchema())) {
                hasCustomData = true;
                targetLayoutIdx = i;
                targetLayout = currentLayout;
                break;
            } else if (currentLayout.isTypeOfTable()
                    || "FORM_LIST".equalsIgnoreCase(currentLayout.getType())
                    || "SPLIT_LAYOUT".equalsIgnoreCase(currentLayout.getType())) {
                targetLayoutIdx = i;
                targetLayout = currentLayout;
                break;
            } else {
                if (null != currentLayout.getId() && currentLayout.getId()
                        .equalsIgnoreCase(currentLayout.getSchema())) {
                    targetLayoutIdx = i;
                    targetLayout = currentLayout;
                    break;
                }
            }
        }
        if (null == targetLayout) {
            return UiBotTargetLayoutSearchResult.create(targetLayoutIdx, targetLayout, hasCustomData);
        }
        // FIXME 因产品设计还未考虑多表格情况，故遇到多表格数据时，只取主表数据
        if ("SPLIT_LAYOUT".equalsIgnoreCase(targetLayout.getType())) {
            Optional<UiBotLayout> optTargetLayout = targetLayout.getGroup().stream()
                    .filter(UiBotLayout::getIsMainTable).findFirst();
            if (optTargetLayout.isPresent()) {
                targetLayout = optTargetLayout.get();
            } else if (targetLayout.getGroup().size() > 0) {
                targetLayout = targetLayout.getGroup().get(0);
            }
        }
        return UiBotTargetLayoutSearchResult.create(targetLayoutIdx, targetLayout, hasCustomData);
    }

    /**
     * 搜索业务数据在pageData对象中的属性名称
     *
     * @return
     */
    public String searchBizDataSchema() {
        UiBotTargetLayoutSearchResult searchResult = this.searchPcTargetLayout();
        if (!searchResult.found()) {
            return "";
        }

        UiBotLayout targetLayout = searchResult.getTargetUiBotLayout();
        String bizDataSchema = "";
        List<UiBotAction> uiBotActions = null;
        if (searchResult.getHasCustomData()) {
            if (null == targetLayout) {
                return "";
            }
            if (null != targetLayout.getFinished() && targetLayout.getFinished()) {
                return "";
            }
            uiBotActions = targetLayout.getActions();
            bizDataSchema = CollectionUtils.isEmpty(uiBotActions) ?
                    "" :
                    Optional.ofNullable(uiBotActions.get(0)).map(UiBotAction::getSubmitType)
                            .map(UiBotActionSubmitType::getSchema).orElse("");
        } else {
            bizDataSchema = targetLayout.getSchema();
        }

        return bizDataSchema;
    }

    /**
     * 从pc端uiBot中找出对应的layout、匹配的pageData、以及上一层的schema(parentSchema)
     * <p>
     * 查找逻辑说明：
     * 1、先从跟节点下查找，找到直接返回；
     * 2、没找到从parentLayoutTypeList下的group查找，找到返回；
     * 3、没找到则继续从group下的group下查找，直到找到或遍历结束。
     * <p>
     * 注：此方法只会返回同一层级符合条件的layout集合。
     * 注：如果没有父节点，则parentSchema为null
     * 注：如果找到的多个layout在不同父节点下，则parentSchema值为最后一个layout的父节点schema
     *
     * @param targetLayoutTypeList 要查找的layout的type集合
     * @param parentLayoutTypeList 目标layout可在的父layout的type集合，若只在跟节点下查找，则传空List
     * @param accurate             是否为精确查找，为false表示目标layout的type包含targetLayoutType即可
     * @return 若遍历结束没找到则返回的UiBotModel中layout为空数组
     */
    public UiBotModel searchPcTargetModel(List<String> targetLayoutTypeList, List<String> parentLayoutTypeList, boolean accurate) {
        UiBotModel resultModel = new UiBotModel();
        resultModel.setParentSchema(this.getParentSchema());
        List<UiBotLayout> resultLayoutList = new ArrayList<>();
        UiBotPageData resultPageData = new UiBotPageData();
        resultModel.setLayout(resultLayoutList);
        resultModel.setPageData(resultPageData);
        List<UiBotLayout> layoutList = this.getLayout();
        UiBotPageData pageData = this.getPageData();

        List<UiBotLayout> targetLayoutList = findLayout(layoutList, targetLayoutTypeList, accurate);
        if (!targetLayoutList.isEmpty()) {
            resultModel.setLayout(targetLayoutList);
            resultModel.setPageData(pageData);
            resultModel.setParentSchema(this.getParentSchema());
            return resultModel;
        }
        if (parentLayoutTypeList.isEmpty()) {
            return resultModel;
        }
        layoutList = layoutList.stream().filter(layout -> parentLayoutTypeList.contains(layout.getType()))
                .collect(Collectors.toList());
        if (layoutList.isEmpty()) {
            return resultModel;
        }
        layoutList.forEach(layout -> {
            if ("TABS".equals(layout.getType())) {
                if (!CollectionUtils.isEmpty(layout.getTabs())) {
                    layout.getTabs().forEach(tab -> {
                        if (!CollectionUtils.isEmpty(tab.getGroup())) {
                            tab.getGroup().forEach(group -> group.setHeaderName(tab.getTitle()));
                            if (CollectionUtils.isEmpty(layout.getGroup())) {
                                layout.setGroup(new ArrayList<>());
                            }
                            layout.getGroup().addAll(tab.getGroup());
                        }
                    });
                }
            }
            if (layout.getGroup() != null) {
                resultLayoutList.addAll(layout.getGroup());
                if (StringUtils.hasLength(layout.getSchema())) {
                    if (StringUtils.hasLength(layout.getSchema())) {
                        resultModel.setParentSchema(layout.getSchema());
                    }
                    resultPageData.putAll(JsonUtil.objectToJavaObject(pageData.get(layout.getSchema()), UiBotPageData.class));
                }
            }
        });
        if (resultLayoutList.isEmpty()) {
            return resultModel;
        }
        resultModel.setLayout(resultLayoutList);
        resultModel.setPageData(resultPageData);
        if (resultPageData.isEmpty()) {
            resultModel.setPageData(pageData);
        }
        return resultModel.searchPcTargetModel(targetLayoutTypeList, parentLayoutTypeList, accurate);
    }

    /**
     * 判断layoutList包不包含指定type的layout,有则返回目标layout，没有则返回空数组
     *
     * @param layoutList
     * @param targetLayoutTypeList
     * @param accurate
     * @return
     */
    private List<UiBotLayout> findLayout(List<UiBotLayout> layoutList, List<String> targetLayoutTypeList, boolean accurate) {
        List<UiBotLayout> targetLayoutList = new ArrayList<>();
        if (accurate) {
            for (UiBotLayout layout : layoutList) {
                for (String type : targetLayoutTypeList) {
                    if (StringUtils.hasLength(type) && type.equalsIgnoreCase(layout.getType())) {
                        targetLayoutList.add(layout);
                    }
                }

            }
        } else {
            for (UiBotLayout layout : layoutList) {
                for (String type : targetLayoutTypeList) {
                    if (StringUtils.hasLength(type) && layout.getType().toLowerCase().contains(type.toLowerCase())) {
                        targetLayoutList.add(layout);
                    }
                }
            }
        }
        return targetLayoutList;
    }

    /**
     * 录入页面卡片化设置
     */
    public void setCustomGroup() {
        // 已经外部有设置过的不动它，否则会给默认的true
        if (null == this.getShowMore()) {
            this.setShowMore(true);
        }
        List<UiBotLayout> layoutList = this.getLayout();
        UiBotPageData uiBotPageData = this.getPageData();
        if (CollectionUtils.isEmpty(layoutList)) {
            return;
        }
        List<UiBotLayout> groupLayoutList = layoutList.stream().filter(layout ->
                CustomGroup.INSIDE_COMPONENT_LIST.contains(layout.getType())).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(groupLayoutList)) {
            return;
        }
        int position = layoutList.indexOf(groupLayoutList.get(0));
        layoutList.removeAll(groupLayoutList);
        CustomGroup customGroup = new CustomGroup();
        String schema = UUIDUtil.getUuid();
        customGroup.setStyleType(CustomGroupStyleTypeEnum.STYLE_INPUT_FORM.getValue());
        customGroup.setCmptUuid(schema);
        CustomGroupContent customGroupContent = new CustomGroupContent();
        customGroup.setContent(customGroupContent);
        customGroupContent.setLayout(groupLayoutList);
        UiBotPageData groupPageData = new UiBotPageData();
        customGroupContent.setPageData(groupPageData);
        groupLayoutList.forEach(layout -> {
            if (!Card.COMPONENT_TYPE.equals(layout.getType())
                    || !CustomGroup.COMPONENT_TYPE.equals(layout.getType())) {
                groupPageData.put(layout.getSchema(), uiBotPageData.get(layout.getSchema()));
                uiBotPageData.remove(layout.getSchema());
            }
        });
        UiBotLayout customGroupLayout = new UiBotLayout();
        customGroupLayout.setSchema(schema);
        customGroupLayout.setType(CustomGroup.COMPONENT_TYPE);
        layoutList.add(position, customGroupLayout);
        uiBotPageData.put(schema, customGroup);
    }

    /**
     * 创建空的uibotModel
     *
     * @param locale
     * @return
     */
    public static UiBotModel emptyUibotModel(String locale) {
        List<UiBotLayout> uiBotLayoutList = new ArrayList<>(1);
        UiBotLayout uiBotLayout1 = new UiBotLayout();
        uiBotLayout1.setSchema("emptyData");
        uiBotLayout1.setType(Empty.COMPONENT_TYPE);
        uiBotLayoutList.add(uiBotLayout1);
        UiBotModel uiBotModel1 = new UiBotModel(uiBotLayoutList);
        UiBotPageData pageData = uiBotModel1.getPageData();
        pageData.put("emptyData", Empty.create(LocaleUtil.getMobileTextByKey(locale, "暂无数据"), true));
        return uiBotModel1;
    }

    /**
     * 创建空的任务卡uibotModel
     *
     * @param locale
     * @return
     */
    public static UiBotModel emptyTaskUibotModel(String locale) {
        return buildEmptyUibotModel(locale, "暂无相关任务卡");
    }

    /**
     * 创建空的uibotModel
     *
     * @param locale
     * @return
     */
    public static UiBotModel buildEmptyUibotModel(String locale, String name) {
        List<UiBotLayout> uiBotLayoutList = new ArrayList<>(1);
        UiBotLayout uiBotLayout1 = new UiBotLayout();
        uiBotLayout1.setSchema("emptyData");
        uiBotLayout1.setType(Empty.COMPONENT_TYPE);
        uiBotLayoutList.add(uiBotLayout1);
        UiBotModel uiBotModel1 = new UiBotModel(uiBotLayoutList);
        UiBotPageData pageData = uiBotModel1.getPageData();
        LocaleService localeService = SpringContextHolder.getBean(LocaleServiceImpl.class);
        pageData.put("emptyData", Empty.create(localeService.getLanguageValue(locale, name), true));
        return uiBotModel1;
    }

    /**
     * 创建空带图片的UiBotModel
     *
     * @param locale
     * @param phrase
     * @param imageWidth
     * @param imageHeight
     * @param image
     * @return UiBotModel
     * @author yanfeng
     */
    public static UiBotModel emptyImageUibotModel(String locale, String phrase, Integer imageWidth, Integer imageHeight, String image) {
        List<UiBotLayout> uiBotLayoutList = new ArrayList<>(1);
        UiBotLayout uiBotLayout1 = new UiBotLayout();
        uiBotLayout1.setSchema("emptyData");
        uiBotLayout1.setType(Empty.COMPONENT_TYPE);
        uiBotLayoutList.add(uiBotLayout1);
        UiBotModel uiBotModel1 = new UiBotModel(uiBotLayoutList);
        UiBotPageData pageData = uiBotModel1.getPageData();
        pageData.put("emptyData", Empty.create(LocaleUtil.getMobileTextByKey(locale, phrase), true, imageWidth, imageHeight, image));
        return uiBotModel1;
    }

    /**
     * 创建空的uibotModel包含取消按钮
     *
     * @param locale
     * @return
     */
    public static UiBotModel emptyUibotModelWithCancelButton(String locale) {
        List<UiBotLayout> uiBotLayoutList = new ArrayList<>(1);
        uiBotLayoutList.add(new UiBotLayout("emptyData", Empty.COMPONENT_TYPE));
        uiBotLayoutList.add(new UiBotLayout("buttons", BottomButtonDigiwinAthena.COMPONENT_TYPE));
        UiBotModel uiBotModel = new UiBotModel(uiBotLayoutList);
        UiBotPageData pageData = uiBotModel.getPageData();
        pageData.put("emptyData", Empty.create(LocaleUtil.getMobileTextByKey(locale, "暂无数据"), true));
        BottomButtonList bottomButtonList = new BottomButtonList();
        bottomButtonList.add(BottomButtonDigiwinAthena.createNormalCancelButton(locale));
        pageData.put("buttons", bottomButtonList);
        return uiBotModel;
    }

    /**
     * 获取所属公司-工厂
     *
     * @return
     */
    public static List<String> getEocNames(UiBotPageData pageData) {
        if (pageData.get("eocName") != null) {
            List<String> eocNames = (List<String>) pageData.get("eocName");
            if (eocNames != null && eocNames.size() > 0) {
                return eocNames;
            }
        }
        return null;
    }

    /**
     * 获取queryButton的layout
     *
     * @param uiBotLayouts
     * @return
     */
    public static UiBotLayout getContentQueryButtonLayout(List<UiBotLayout> uiBotLayouts) {
        for (UiBotLayout uiBotLayout : uiBotLayouts) {
            if (uiBotLayout.getType().equals(PcModuleEnum.CONTENT_QUERY_BUTTON.getValue())) {
                return uiBotLayout;
            }
        }
        return null;
    }

    public static UiBotLayout getApprovalDesriptionLayout(List<UiBotLayout> uiBotLayouts) {
        for (UiBotLayout uiBotLayout : uiBotLayouts) {
            if (uiBotLayout.getType().equals(PcModuleEnum.APPROVAL_DESCRIPTION.getValue())) {
                return uiBotLayout;
            }
        }
        return null;
    }

    public static UiBotLayout getCustomUibotLayout(List<UiBotLayout> uiBotLayouts, String customType) {
        for (UiBotLayout uiBotLayout : uiBotLayouts) {
            if (uiBotLayout.getType().equals(customType)) {
                return uiBotLayout;
            }
        }
        return null;
    }

    /**
     * 获取AthenaTable的schema
     *
     * @param uiBotLayouts
     * @return
     */
    public static String getAthenaTableSchema(List<UiBotLayout> uiBotLayouts) {
        for (UiBotLayout uiBotLayout : uiBotLayouts) {
            if (uiBotLayout.getType().equals(PcLayoutEnum.ATHENA_TABLE.getValue())) {
                return uiBotLayout.getSchema();
            }
        }
        return "";
    }

    public static UiBotAction getSubmitAction(UiBotModel pcUiBotModel, String actionId) {
        List<UiBotLayout> layoutList = pcUiBotModel.getLayout();
        if (CollectionUtils.isEmpty(layoutList)) {
            return null;
        }
        AtomicReference<UiBotAction> uiBotAction = new AtomicReference<>();
        outer:
        for (UiBotLayout layout : layoutList) {
            List<UiBotAction> actions = layout.getActions();
            if (CollectionUtils.isEmpty(actions)) {
                continue;
            }
            for (UiBotAction action : actions) {
                if (actionId.equalsIgnoreCase(action.getActionId())) {
                    uiBotAction.set(action);
                    break outer;
                }
            }
        }
        return uiBotAction.get();
    }

    public void initSetting(LocaleService localeService, String locale) {
        if (!CollectionUtils.isEmpty(this.appbarButtonList)) {
            for (Button button : appbarButtonList) {
                if (button.getName() != null) {
                    button.setName(localeService.getLanguageValue(locale, button.getName()));
                }
                if (button.getSubName() != null) {
                    button.setSubName(localeService.getLanguageValue(locale, button.getSubName()));
                }
                if (button.getAction() != null) {
                    Action action = button.getAction();
                    if (action.getJumpPageTitle() != null) {
                        action.setJumpPageTitle(localeService.getLanguageValue(locale, action.getJumpPageTitle()));
                    }
                    if (StringUtils.hasLength(action.getToastMsg())) {
                        action.setToastMsg(localeService.getLanguageValue(locale, action.getToastMsg()));
                    }
                }
                this.setRequestUrl(button);
            }
        }
    }

    private void setRequestUrl(Button button) {
        Action action = button.getAction();
        if (action == null || action.getRequestUrl() == null || action.getRequestUrl().length() == 0) {
            return;
        }
        action.setRequestUrl(AppContext.getBaseUrl() + action.getRequestUrl());
    }

    /**
     * 创建空组件
     *
     * @param msg            提示信息
     * @param isRefreshRetry 是否有刷新当前页面 重试按钮
     * @param image          空组件图片
     * @return
     */
    public static UiBotModel createEmptyComponent(String msg, boolean isRefreshRetry, String image) {
        UiBotModel uiBotModel = UiBotModel.emptyUiBotModel();
        List<UiBotLayout> emptyLayout = uiBotModel.getLayout();
        UiBotPageData emptyPageData = uiBotModel.getPageData();

        String blankAreaSchema = UUIDUtil.getUuid();
        emptyLayout.add(new UiBotLayout(blankAreaSchema, BlankArea.COMPONENT_TYPE));
        emptyPageData.put(blankAreaSchema, BlankArea.create(326, 1.0));

        String emptySchema = UUIDUtil.getUuid();
        emptyLayout.add(new UiBotLayout(emptySchema, Empty.COMPONENT_TYPE));
        emptyPageData.put(emptySchema, Empty.create(msg,
                true,
                400,
                180,
                image,
                isRefreshRetry));
        return uiBotModel;
    }

    /**
     * 展示任务标题，以及空组件提示。提示文案：『APP端暂不支持该定制任务，请到Web端查看』
     *
     * @param localeService
     * @param locale
     */
    public void buildShowTitleAndEmpty(LocaleService localeService, String locale) {
        List<UiBotLayout> layouts = this.getLayout().stream()
                .filter(layout -> !TitleBody.COMPONENT_TYPE.equals(layout.getType()))
                .collect(Collectors.toList());
        for (UiBotLayout layout : layouts) {
            this.getPageData().remove(layout.getSchema());
        }
        this.getLayout().removeAll(layouts);
        // 创建空组件
        UiBotModel empty = UiBotModel.createEmptyComponent(
                localeService.getLanguageValue(locale, "此作业未配置移动端界面，暂不支持查看\n请前往Web端查阅处理"),
                false,
                null);
        this.getLayout().addAll(empty.getLayout());
        this.getPageData().putAll(empty.getPageData());
    }

    public void initMobileLayoutType() {
        this.mobileLayoutType = this.buildMobileLayoutType(this);
    }

    /**
     * 解析构建 布局类型，提供前端，判断走不同逻辑
     * <p>
     * 判断逻辑：
     * 1、是否存在底部按钮
     * 1-1、否：其它布局
     * 1-2、是：是否存在cardList组件
     * 1-2-1、是：双层布局
     * 1-2-2、否：单层布局
     * </p>
     *
     * @param uiBotModel 构建完成后的DSL
     * @return java.lang.String
     */
    private String buildMobileLayoutType(UiBotModel uiBotModel) {
        UiBotRenderData renderData = uiBotModel.getRenderData();
        if (UiBotDesignerRenderService.isExistBottomButton(renderData)) {
            // 存在底部按钮
            if (UiBotDesignerRenderService.isContainListComponent(renderData.getBody())) {
                // 存在底部按钮，存在列表组件，多层布局
                return MobileLayoutTypeEnum.MULTI_PATTERN_LAYOUT.getValue();
            } else {
                // 存在底部按钮，不存在列表组件，单层布局
                return MobileLayoutTypeEnum.SINGLE_PATTERN_LAYOUT.getValue();
            }
        } else {
            // 不存在底部按钮，其它布局
            return MobileLayoutTypeEnum.OTHER_PATTERN_LAYOUT.getValue();
        }
    }

    /**
     * 当前渲染DSL是否有应用设计器配置
     *
     * @return
     */
    public boolean hasAppliedDesignerConfig() {
        // FIXME 先仅使用是否处理过规则来判断：因设计器版本的规则是JS，只有新render的组件库支持，可用于区分是否需要使用MobileRenderDataUtil进行数据升级
        return Boolean.TRUE.equals(this.appliedDesignerFieldRule);
//                Boolean.TRUE.equals(this.appliedDesignerFieldAndOrder)
//                || Boolean.TRUE.equals(this.appliedDesignerFieldTag)
//                || Boolean.TRUE.equals(this.appliedDesignerFieldOperation)
//                || Boolean.TRUE.equals(this.appliedDesignerFieldRule);
    }

    public void buildCommonRawData(Map<String, Object> commonRawData, Map<String, Object> opData, UiBotModel pcUiBotModel) {
        commonRawData.putAll(opData);
//        commonRawData.remove("parameter_data");
//        commonRawData.remove("operator_data");
        commonRawData.put("executeContext", pcUiBotModel.getExecuteContext());
        commonRawData.put("parameter", buildTbdsParameter(opData));
    }

    /**
     * 构建queryInfo
     *
     * @param opData
     * @return
     */
    public TbdsParameter buildTbdsParameter(Map<String, Object> opData) {
        TbdsParameter tbdsParameter = new TbdsParameter();
        TbdsQueryInfo tbdsQueryInfo = new TbdsQueryInfo();
        tbdsQueryInfo.setEocCompanyId((String) opData.get("eoc_company_id"));
        tbdsQueryInfo.setEocSiteId((String) opData.get("eoc_site_id"));
        tbdsQueryInfo.setOpShowStatus((String) opData.get("op_show_status"));
        if (StringUtils.hasLength((String) opData.get("project_no"))) {
            tbdsQueryInfo.setProjectNo((String) opData.get("project_no"));
        }
        if (StringUtils.hasLength((String) opData.get("parent_seq"))) {
            tbdsQueryInfo.setProjectNo((String) opData.get("parent_seq"));
        }
        tbdsQueryInfo.setTrialWoNo((String) opData.get("trial_wo_no"));
        tbdsParameter.getQueryInfo().add(tbdsQueryInfo);
        return tbdsParameter;
    }

    /**
     * 工作提醒点击查看更多，如果任务卡已完成：返回带图片的空 UiBotModel
     *
     * @param locale
     * @return UiBotModel
     * @author yanfeng
     */
    public static UiBotModel workReminderEmptyImageUibotModel(String locale) {
        LocaleService localeService = SpringContextHolder.getBean(LocaleServiceImpl.class);
        List<UiBotLayout> uiBotLayoutList = new ArrayList<>(1);
        UiBotLayout uiBotLayout1 = new UiBotLayout();
        uiBotLayout1.setSchema("emptyData");
        uiBotLayout1.setType(Empty.COMPONENT_TYPE);
        uiBotLayoutList.add(uiBotLayout1);
        UiBotModel uiBotModel1 = new UiBotModel(uiBotLayoutList);
        UiBotPageData pageData = uiBotModel1.getPageData();
        pageData.put("emptyData", Empty.create(localeService.getLanguageValue(locale, "当前任务已完成，移动端暂不支持查看\n" +
                "可前往web端查看"), true, 10, 1000, 240, 240, "#D2D2E6", "IMAGE_TASK_FINISH"));
        return uiBotModel1;
    }

    /**
     * V2-工作提醒点击查看更多，如果任务卡已完成：返回带图片的空 UiBotModel
     *
     * @param locale
     * @return UiBotModel
     * @author yanfeng
     */
    public static UiBotModel workReminderEmptyImageUibotModelV2(String locale) {
        UiBotModel uiBotModel = UiBotModel.emptyUiBotModel();
        if (ObjectUtils.isEmpty(uiBotModel.getRenderData())) {
            uiBotModel.setRenderData(UiBotRenderData.createEmptyRenderData());
        }
        BaseMobileComponentWrapper<BaseMobileComponent> wrapper = new BaseMobileComponentWrapper<>(
                Empty.create(
                        SpringContextHolder.getBean(LocaleServiceImpl.class)
                                .getLanguageValue(locale, "当前任务已完成，移动端暂不支持查看\n" +
                                        "可前往web端查看"), true, 10, 1000, 240, 240, "#D2D2E6", "IMAGE_TASK_FINISH"),
                "DW_" + Empty.COMPONENT_TYPE);
        uiBotModel.getRenderData().addWrapperedComponentToBody(wrapper);
        return uiBotModel;
    }

    public List<UiBotLayout> getTargetLayoutByType(String targetType, List<UiBotLayout> uiBotLayouts) {
        List<UiBotLayout> targetLayouts = new ArrayList<>(1);
        for (UiBotLayout uiBotLayout : uiBotLayouts) {
            if (targetType.equalsIgnoreCase(uiBotLayout.getType())) {
                targetLayouts.add(uiBotLayout);
            } else {
                targetLayouts.addAll(getTargetLayoutByType(targetType, uiBotLayout.getGroup()));
            }
        }
        return targetLayouts;
    }

    public List<UiBotLayout> getTargetLayoutsByTypes(String targetType1, String targetType2, List<UiBotLayout> uiBotLayouts) {
        List<UiBotLayout> targetLayouts = new ArrayList<>(4);
        for (UiBotLayout uiBotLayout : uiBotLayouts) {
            if (targetType1.equalsIgnoreCase(uiBotLayout.getType())) {
                targetLayouts.add(uiBotLayout);
            } else if (targetType2.equalsIgnoreCase(uiBotLayout.getType())) {
                targetLayouts.add(uiBotLayout);
            } else {
                targetLayouts.addAll(getTargetLayoutsByTypes(targetType1, targetType2, uiBotLayout.getGroup()));
            }
        }
        return targetLayouts;
    }

    public static UiBotModel emptyCmptUibotModelV2() {
        UiBotModel uiBotModel = UiBotModel.emptyUiBotModel();
        if (ObjectUtils.isEmpty(uiBotModel.getRenderData())) {
            uiBotModel.setRenderData(UiBotRenderData.createEmptyRenderData());
        }
        uiBotModel.getRenderData().addEmptyComponentToBody();
        return uiBotModel;
    }

    public UiBotPageData getRawUiBotPageData() {
        return this.rawUiBotPageData = Optional.ofNullable(this.rawUiBotPageData).orElse(new UiBotPageData());
    }
}
