package com.digiwin.mobile.mobileuibot.proxy.uibot.model.agiledata;

import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.annotation.JSONField;
import com.digiwin.mobile.mobileuibot.common.math.MathUtil;
import com.digiwin.mobile.mobileuibot.core.component.chart.ChartTypeEnum;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.agiledata.chart.AgileDataChartFieldDisplayFormat;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.agiledata.chart.AgileDataChartValueField;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.agiledata.chart.ChartRenderSetting;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.agiledata.statsindex.IndexRenderSetting;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.agiledata.table.TableRenderSetting;
import com.fasterxml.jackson.annotation.JsonIgnore;
import lombok.Data;
import org.apache.commons.math3.stat.descriptive.DescriptiveStatistics;
import org.springframework.util.CollectionUtils;

import java.io.Serializable;
import java.util.*;
import java.util.stream.Collectors;

/**
 * <p>功能描述：敏捷数据——中间态DSL的数据项对象</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: AgileDataIntermediateDataItem
 * @Author: zaregoto
 * @Date: 2023/11/20 19:18
 */
@Data
public class AgileDataIntermediateDataItem implements Serializable {
    private static final long serialVersionUID = 7061403906084212134L;
    /**
     * 数据呈现分类
     *
     * @see AgileDataShowCategoryEnum
     */
    private Integer showCategory;

    /**
     * 数据呈现名称
     */
    private String showName;

    /**
     * 数据唯一标识
     */
    private String bizDataId;

    /**
     * 数据的元数据定义
     */
    private List<AgileDataMetadataField> metadataFields;

    /**
     * 实际数据（因实际数据由API不同而不同，故这里使用支持动态结构的JSONObject）
     */
    private List<JSONObject> pageData;

    /**
     * 表格类呈现设置
     */
    private TableRenderSetting tableRenderSetting;

    /**
     * 图表类呈现设置
     */
    private ChartRenderSetting chartRenderSetting;

    /**
     * 统计指标卡类呈现设置
     */
    private IndexRenderSetting mobileDSLIndexRender;

    //敏捷数据使用，数据提示信息
    private String dataTipMessage;

//    /**
//     * 数据观察的结果。
//     * key为统计字段对象，value为对该统计字段对象观察的结果
//     */
//    @JsonIgnore
//    @JSONField(serialize = false, deserialize = false)
//    private Map<AgileDataChartValueField, DataObserveResult> dataObserveResultMap = new HashMap<>(2);

    /**
     * FIXME
     * 数据观察的结果。
     * key为图表类型枚举，value为对该统计字段对象观察的结果
     */
    @JsonIgnore
    @JSONField(serialize = false, deserialize = false)
    private Map<Integer, DataObserveResult> dataObserveResultMap = new LinkedHashMap<>(2);

    /**
     * Y轴的度量字段的分组关系
     */
    private Map<Integer, List<AgileDataChartValueField>> groupedValueFields = new LinkedHashMap<>(2);

    /**
     * 根据指定字段ID，从pageData中获取一系列的数据（如获取X轴坐标点的清单等）
     *
     * @param fieldId 字段ID
     * @return
     */
    public List<String> getSeriesDataWithFieldId(String fieldId) {
        return this.pageData.stream().map(j -> j.getString(fieldId)).collect(Collectors.toList());
    }

    /**
     * 根据统计值字段的的格式获取图表展示的文字Label。
     *
     * @param locale
     * @param valueField
     * @param value
     * @return
     */
    public String getLabelWithValueField(String locale, AgileDataChartValueField valueField, Double value) {
        if (null == valueField.getFormat()) {
            return String.format("%,." + AgileDataChartFieldDisplayFormat.DEFAULT_DECIMAL + "f", value);
        } else {
            return valueField.getFormat().getFormattedString(value);
        }
    }

    /**
     * 数据集是否可被观察
     *
     * @return
     */
    private boolean dataObservable() {
        return null != this.chartRenderSetting
                && !CollectionUtils.isEmpty(this.chartRenderSetting.getValues())
                || (null != this.chartRenderSetting && !(Objects.equals(this.chartRenderSetting.getChartType(), ChartTypeEnum.LINE.getType())
                || Objects.equals(this.chartRenderSetting.getChartType(), ChartTypeEnum.BAR.getType())
                || Objects.equals(this.chartRenderSetting.getChartType(), ChartTypeEnum.LINE_BAR.getType())
                || Objects.equals(this.chartRenderSetting.getChartType(), ChartTypeEnum.SCATTER.getType())
                || Objects.equals(this.chartRenderSetting.getChartType(), ChartTypeEnum.BUBBLE.getType())
                || Objects.equals(this.chartRenderSetting.getChartType(), ChartTypeEnum.MIXED.getType())
        ));
    }

    /**
     * 观察数据集，得到数据的观察结果，用于后续图表处理。<br/>
     * 比如最小值、最大值、平均值、建议的坐标轴信息等。<br/>
     * 只有特定的图表类型查看后会有结果。
     *
     * @param locale 语言别
     * @return
     */
    public void observeDataForChart(String locale) {
        if (!this.dataObservable()) {
            return;
        }
        this.normalizeChartRenderSetting();
        // 使用统计字段的group来明确Y轴的分组
        this.groupedValueFields = this.chartRenderSetting.getValues()
                .stream().collect(
                        // 指定使用有序Map和Collection，保存分组内的结果，保证分组顺序和原始数据中的顺序相同
                        Collectors.groupingBy(field -> field.getGroup(),
                                LinkedHashMap::new,
                                Collectors.toCollection(ArrayList::new)));

        for (Integer groupNo : groupedValueFields.keySet()) {
            List<AgileDataChartValueField> valueFieldList = groupedValueFields.get(groupNo);
            DataObserveResult dataObserveResult = new DataObserveResult(locale, groupNo, valueFieldList.get(0).getFormat());
            // 挂载到Map中
            this.dataObserveResultMap.put(groupNo, dataObserveResult);

            // 根据统计值字段，取出数据集，存入统计对象
            DescriptiveStatistics stats = new DescriptiveStatistics();
            for (AgileDataChartValueField valueField : valueFieldList) {
                for (JSONObject pageDatum : this.pageData) {
                    stats.addValue(pageDatum.getDoubleValue(valueField.getName()));
                }
            }

            // 得到最小值、最大值、中位数、平均值、标准差
            dataObserveResult.setMinValue(stats.getMin());
            dataObserveResult.setMaxValue(stats.getMax());
            dataObserveResult.setMedianValue(stats.getPercentile(50));
            dataObserveResult.setMeanValue(stats.getMean());
            dataObserveResult.setStdDeviation(stats.getStandardDeviation());

            // 找出数据异常值

            /**
             * 获取最佳的数轴最大、最小值。
             * 20231204更新：先不使用平均值与三分原则，直接使用数据最小与最大值填入dataObserveResult。因为在填入dataObserveResult时，
             * 内部还会有一层向上向下取整的计算，已经会把数据区间再放大。
             *
             * 1204新规则如下：
             * 1. 如果数据都大于等于零，则坐标轴最小值直接为0，最大值为数据最大值
             * 2. 如果数据都小于等于零，则坐标轴最大值直接为0，最小值为数据最大值
             * 3. 如果数据是正负数混合，则坐标轴最小与最大值，直接使用数据的最小与最大值
             *
             * 旧规则如下（已不使用）：
             * 1. 如果数据都大于等于零，则坐标轴最小值直接为0，最大值为数据最大值+平均值的绝对值
             * 2. 如果数据都小于等于零，则坐标轴最大值直接为0，最小值为数据最大值+平均值的绝对值
             * 3. 如果数据是正负数混合，则坐标轴最小与最大值，直接使用数据的最小与最大值
             *
             * 三分原则：数据的最大值max与最小值min之差正好要是一份的值，
             * ---- yMax = max + (max-min)
             * ---- yMin = min - (max-min)
             * -----参考：https://www.jianshu.com/p/ffc2d6a3b4de
             */
            if (stats.getMin() == stats.getMax()) {
                if (stats.getMin() == 0.0) {
                    dataObserveResult.setPreferredAxisMinValue(0.0);
                    dataObserveResult.setPreferredAxisMaxValue(1.0);
                } else if (stats.getMin() > 0.0) {
                    dataObserveResult.setPreferredAxisMinValue(0.0);
                    dataObserveResult.setPreferredAxisMaxValue(MathUtil.getFiveMultiplesRoundedValueByDigits(stats.getMin(), 1));
                } else {
                    dataObserveResult.setPreferredAxisMinValue(MathUtil.getFiveMultiplesRoundedValueByDigits(stats.getMin(), -1));
                    dataObserveResult.setPreferredAxisMaxValue(0.0);
                }
            } else if (stats.getMin() >= 0.0 && stats.getMin() != stats.getMax()) {
                dataObserveResult.setPreferredAxisMinValue(0.0);
//            // 坐标轴最小值为零时，最大值设置为数据最大值+平均值，测试下来整体观感效果较好
//            double upScale = dataObserveResult.getMaxValue() + Math.abs(dataObserveResult.getMeanValue());
//            dataObserveResult.setPreferredAxisMaxValue(upScale);
                dataObserveResult.setPreferredAxisMaxValue(MathUtil.getFiveMultiplesRoundedValueByDigits(stats.getMax(), 1));
            } else if (stats.getMax() <= 0.0 && stats.getMin() != stats.getMax()) {
                dataObserveResult.setPreferredAxisMaxValue(0.0);
//            // 坐标轴最大值为零时，最小值设置为数据最小值-平均值，测试下来整体观感效果较好
//            double downScale = dataObserveResult.getMinValue() - Math.abs(dataObserveResult.getMeanValue());
//            dataObserveResult.setPreferredAxisMinValue(downScale);
                dataObserveResult.setPreferredAxisMinValue(MathUtil.getFiveMultiplesRoundedValueByDigits(stats.getMin(), -1));
            } else {
//            double diff = dataObserveResult.getMaxValue() - dataObserveResult.getMinValue();
//            dataObserveResult.setPreferredAxisMinValue(dataObserveResult.getMinValue() - diff);
//            dataObserveResult.setPreferredAxisMaxValue(dataObserveResult.getMaxValue() + diff);
                dataObserveResult.setPreferredAxisMinValue(MathUtil.getFiveMultiplesRoundedValueByDigits(stats.getMin(), -1));
                dataObserveResult.setPreferredAxisMaxValue(MathUtil.getFiveMultiplesRoundedValueByDigits(stats.getMax(), 1));
            }

            // TODO 计算坐标轴刻度合适的间隔，目前默认分成五等分。但后面要依据数据集中程度或变化率，或其他因素决定：
            int equalDivisionCount = DataObserveResult.DEFAULT_VALUE_AXIS_EQUAL_PARTS_COUNT;
            if (dataObserveResult.dataDistributionIsBetweenPlusAndMinusOne()) {
                Double interval = (dataObserveResult.getPreferredAxisMaxValue()
                        - dataObserveResult.getPreferredAxisMinValue()) / equalDivisionCount;
                dataObserveResult.setPreferredAxisInterval(interval);
            } else if (dataObserveResult.dataDistributionIsBetweenPlusAndMinusTen()) {
                Double interval = (dataObserveResult.getPreferredAxisMaxValue()
                        - dataObserveResult.getPreferredAxisMinValue()) / equalDivisionCount;
                if (interval > 1) {
                    dataObserveResult.setPreferredAxisInterval(Math.ceil(interval));
                } else {
                    dataObserveResult.setPreferredAxisInterval(interval);
                }
            } else {
                Double interval = Math.ceil((dataObserveResult.getPreferredAxisMaxValue()
                        - dataObserveResult.getPreferredAxisMinValue()) / equalDivisionCount);
                dataObserveResult.setPreferredAxisInterval(interval);
            }
//            // 前端显示时增加安全区间距
//            dataObserveResult.addSafeAreaToAxisScale(stats.getMin(), stats.getMax());

            // 微调坐标轴刻度间距
            dataObserveResult.tunePreferredAxisInterval();
        }
    }

    /**
     * 规范化图表设置数据
     */
    private void normalizeChartRenderSetting() {
        ChartTypeEnum chartTypeEnum = ChartTypeEnum.getEnumByValue(this.chartRenderSetting.getChartType());
        if (ChartTypeEnum.LINE == chartTypeEnum || ChartTypeEnum.BAR == chartTypeEnum || ChartTypeEnum.PIE == chartTypeEnum) {
            this.chartRenderSetting.getValues().forEach(v -> v.setChartType(chartTypeEnum.getType()));
        }
    }

    /**
     * 对齐图表中所有统计字段的刻度数量.<br/>
     * 具体做法参考语雀文档中的《双Y轴的刻度算法》：https://mobile-digiwin.yuque.com/cog7oa/gbmk5r/pxny2g9qs6vdtum0
     */
    public void alignChartValueAxesScaleCount() {
        if (!this.dataObservable()) {
            return;
        }

        // 扫描统计字段的刻度信息，得到待呈现的刻度集合的类型
        ScaleSetScanResultType scanResultType = this.getScaleScanResultType();
        switch (scanResultType) {
            case EXIST_NEGATIVE_AND_POSITIVE:
                // 更新所有数据观察结果
                for (DataObserveResult observeResult : this.dataObserveResultMap.values()) {
                    Double interval = observeResult.getPreferredAxisInterval();
                    // 只有正数的情况
                    if (observeResult.getNegativeValueAxisScaleCount() == 0 && observeResult.getPositiveValueAxisScaleCount() > 0) {
                        observeResult.setPreferredAxisMaxValue(observeResult.getPreferredAxisMaxValue() + interval);
                    }
                    // 只有负数的情况
                    if (observeResult.getNegativeValueAxisScaleCount() > 0 && observeResult.getPositiveValueAxisScaleCount() == 0) {
                        observeResult.setPreferredAxisMinValue(observeResult.getPreferredAxisMinValue() - interval);
                    }
                }
                break;
            case ONLY_NEGATIVE:
            case ONLY_POSITIVE:
            case ONLY_ONE_DIRECTION_AND_EXIST_OPPOSITE_DIRECTION:
            case NONE:
            default:
                // 其他任何情况则不对刻度做任何对齐处理
                break;
        }

    }

    /**
     * 刻度集合扫描结果类型
     */
    enum ScaleSetScanResultType {
        /**
         * 无结果
         */
        NONE,

        /**
         * 存在一个统计字段既有负数刻度，又有正数刻度
         */
        EXIST_NEGATIVE_AND_POSITIVE,

        /**
         * 任意一个统计字段只存在负数刻度
         */
        ONLY_NEGATIVE,

        /**
         * 任意一个统计字段只存在正数刻度
         */
        ONLY_POSITIVE,

        /**
         * 所有统计字段仅存在一种方向（正or负）的刻度，且刻度位置存在相反情况
         */
        ONLY_ONE_DIRECTION_AND_EXIST_OPPOSITE_DIRECTION
    }

    /**
     * 扫描所有统计字段的刻度信息，得到待呈现的刻度集合的类型
     *
     * @return
     */
    private ScaleSetScanResultType getScaleScanResultType() {
        ScaleSetScanResultType scanResultType = null;
        int negativeScaleCount = 0;
        int positiveScaleCount = 0;
        // 仅存在正数刻度的统计字段数量
        int onlyPositiveFieldScaleCount = 0;
        // 仅存在负数刻度的统计字段数量
        int onlyNegativeFieldScaleCount = 0;
        for (DataObserveResult observeResult : this.dataObserveResultMap.values()) {
            negativeScaleCount += observeResult.getNegativeValueAxisScaleCount();
            positiveScaleCount += observeResult.getPositiveValueAxisScaleCount();
            if (observeResult.getNegativeValueAxisScaleCount() > 0
                    && observeResult.getPositiveValueAxisScaleCount() > 0) {
                scanResultType = ScaleSetScanResultType.EXIST_NEGATIVE_AND_POSITIVE;
                break;
            }
            if (observeResult.getNegativeValueAxisScaleCount() == 0 && observeResult.getPositiveValueAxisScaleCount() > 0) {
                onlyPositiveFieldScaleCount++;
            }
            if (observeResult.getNegativeValueAxisScaleCount() > 0 && observeResult.getPositiveValueAxisScaleCount() == 0) {
                onlyNegativeFieldScaleCount++;
            }
        }
        if (null != scanResultType) {
            return scanResultType;
        }

        if (negativeScaleCount > 0 && positiveScaleCount == 0) {
            scanResultType = ScaleSetScanResultType.ONLY_NEGATIVE;
        } else if (negativeScaleCount == 0 && positiveScaleCount > 0) {
            scanResultType = ScaleSetScanResultType.ONLY_POSITIVE;
        } else {
            // 如果只有正数刻度的字段数量+只有负数刻度的字段数量=所有字段的数量，则说明所有字段都只有一种方向且存在两种相反方向
            if (onlyPositiveFieldScaleCount + onlyNegativeFieldScaleCount == this.dataObserveResultMap.size()) {
                scanResultType = ScaleSetScanResultType.ONLY_ONE_DIRECTION_AND_EXIST_OPPOSITE_DIRECTION;
            } else {
                scanResultType = ScaleSetScanResultType.NONE;
            }
        }
        return scanResultType;
    }

    /**
     * 获取所有统计字段正数刻度个数的最大值
     *
     * @return
     */
    private Integer getMaxPositiveScaleCount() {
        Integer maxPositiveScaleCount = 0;
        for (DataObserveResult observeResult : this.dataObserveResultMap.values()) {
            maxPositiveScaleCount = Math.max(maxPositiveScaleCount, observeResult.getPositiveValueAxisScaleCount());
        }
        return maxPositiveScaleCount;
    }

    /**
     * 获取所有统计字段负数刻度个数的最大值
     *
     * @return
     */
    private Integer getMaxNegativeScaleCount() {
        Integer maxNegativeScaleCount = 0;
        for (DataObserveResult observeResult : this.dataObserveResultMap.values()) {
            maxNegativeScaleCount = Math.max(maxNegativeScaleCount, observeResult.getNegativeValueAxisScaleCount());
        }
        return maxNegativeScaleCount;
    }

//    /**
//     * 根据统计字段对象，获取该字段的数据观察结果
//     *
//     * @param valueField
//     * @return
//     */
//    public DataObserveResult getExactDataObserveResult(AgileDataChartValueField valueField) {
//        return this.getExactDataObserveResult(valueField.getName());
//    }

//    /**
//     * 根据字段id，获取该字段的数据观察结果
//     *
//     * @param fieldId
//     * @return
//     */
//    public DataObserveResult getExactDataObserveResult(String fieldId) {
//        if (this.dataObserveResultMap.isEmpty()) {
//            return null;
//        }
//        for (AgileDataChartValueField valueField : this.dataObserveResultMap.keySet()) {
//            if (Objects.equals(valueField.getName(), fieldId)) {
//                return this.dataObserveResultMap.get(valueField);
//            }
//        }
//        return null;
//    }

    /**
     * 根据数据分组情况，获取该字段的数据观察结果
     *
     * @param groupNo
     * @return
     */
    public DataObserveResult getExactDataObserveResult(Integer groupNo) {
        if (this.dataObserveResultMap.isEmpty()) {
            return null;
        }
        for (Integer group : this.dataObserveResultMap.keySet()) {
            if (Objects.equals(groupNo, group)) {
                return this.dataObserveResultMap.get(group);
            }
        }
        return null;
    }
}