package com.digiwin.mobile.mobileuibot.proxy.uibot.model.agiledata;

import cn.hutool.core.collection.CollectionUtil;
import com.alibaba.fastjson.annotation.JSONField;
import com.digiwin.mobile.mobileuibot.core.component.chart.ChartTypeEnum;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.agiledata.chart.ChartRenderSetting;
import com.fasterxml.jackson.annotation.JsonIgnore;
import lombok.Data;
import org.springframework.util.CollectionUtils;

import java.io.Serializable;
import java.util.*;
import java.util.stream.Collectors;

/**
 * <p>功能描述：敏捷数据组件提供的中间态DSL——Layout对象</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: AgileDataIntermediateLayout
 * @Author: zaregoto
 * @Date: 2023/11/20 19:18
 */
@Data
public class AgileDataIntermediateLayout implements Serializable {
    private static final long serialVersionUID = 1719944742293640765L;

    /**
     * 类型名。
     * 当前先有固定值：DSL_MOBILEPLATFORM_PAGEDEFINE
     */
    private String type;

    /**
     * 中间态DSL生成时间
     */
    private String generateTime;

    /**
     * 敏捷数据
     */
    private List<AgileDataIntermediateDataItem> data;

    /**
     * 敏捷数据所需的数据结论
     */
    private String dataConclusion;

    /**
     * 敏捷数据所使用的呈现经验图谱对象
     */
    private AgileDataRenderGraph renderGraph;

    /**
     * Web端使用的动态布局参考（移动端未来有平板，可用于参考）
     */
    private List<WebDynamicLayout> dynamicLayout;

    @JsonIgnore
    @JSONField(serialize = false)
    private List<AgileDataIntermediateDataItem> sortedDataItems;

    /**
     * 初始化方法。
     * 在第一次获取当前layout后，须首先调用。
     */
    public void init() {
        this.sort();
        this.checkChartType();
    }

    /**
     * 对敏捷数据项清单做排序，并赋值给内部属性。
     * 排序依据Web端的动态布局结果，先看行，再看列。
     *
     * @return
     */
    private void sort() {
        if (CollectionUtil.isEmpty(this.data)) {
            this.sortedDataItems = Collections.emptyList();
        }
        if (CollectionUtil.isEmpty(this.dynamicLayout)) {
            this.sortedDataItems = this.data;
            return;
        }
        WebDynamicLayout dynamicLayout = this.dynamicLayout.get(0);
        if (!Objects.equals(dynamicLayout.getType(), "GRIDSTER")) {
            this.sortedDataItems = this.data;
        }
        if (CollectionUtil.isEmpty(dynamicLayout.getItems())) {
            this.sortedDataItems = this.data;
        }
        List<String> dashboardLayoutBizIds = dynamicLayout.getItems()
                .stream().sorted((o1, o2) -> {
                    Integer x1 = o1.getX(), y1 = o1.getY();
                    Integer x2 = o2.getX(), y2 = o2.getY();
                    if (Objects.equals(y1, y2)) {
                        return x1.compareTo(x2);
                    } else {
                        return y1.compareTo(y2);
                    }
                })
                .map(WebGridsterLayoutDashboard::getId)
                .collect(Collectors.toList());

        Map<String, AgileDataIntermediateDataItem> mappedDataItem = this.data.stream()
                .collect(Collectors.toMap(AgileDataIntermediateDataItem::getBizDataId, dataItem -> dataItem));

        List<AgileDataIntermediateDataItem> s = new ArrayList<>(this.data.size());
        dashboardLayoutBizIds.forEach(bizId -> {
            if (mappedDataItem.containsKey(bizId)) {
                s.add(mappedDataItem.get(bizId));
            }
        });
        // 防止dashboard和数据中的id不一致
        this.sortedDataItems = CollectionUtils.isEmpty(s) ? this.data : s;
    }

    /**
     * 检查中间态数据的图表类型
     * 1. 如果图表中的所有统计字段均使用line，则整个图表type重新设置成line，以兼容前端图表组件逻辑
     */
    private void checkChartType() {
        for (AgileDataIntermediateDataItem datum : this.data) {
            if (Objects.equals(AgileDataShowCategoryEnum.CHART.getValue(), datum.getShowCategory())) {
                ChartRenderSetting chartRenderSetting = datum.getChartRenderSetting();
                if (ChartTypeEnum.SCATTER.getType().equals(chartRenderSetting.getChartType()) || ChartTypeEnum.BUBBLE.getType().equals(chartRenderSetting.getChartType())) {
                    return;
                }
                boolean isAllLine = chartRenderSetting.getValues().stream()
                        .allMatch(value -> Objects.equals(ChartTypeEnum.LINE.getType(), value.getChartType()));
                if (isAllLine) {
                    chartRenderSetting.setChartType(ChartTypeEnum.LINE.getType());
                }
            }
        }
    }
}