package com.digiwin.mobile.mobileuibot.proxy.uibot.model.agiledata;

import com.digiwin.mobile.mobileuibot.common.currency.CurrencyUtil;
import com.digiwin.mobile.mobileuibot.common.currency.MagnitudeUnit;
import com.digiwin.mobile.mobileuibot.common.math.MathUtil;
import com.digiwin.mobile.mobileuibot.core.component.chart.ChartTypeEnum;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.agiledata.chart.AgileDataChartFieldDisplayFormat;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.agiledata.chart.AgileDataChartValueField;
import lombok.Getter;
import org.jetbrains.annotations.NotNull;

import java.io.Serializable;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

/**
 * <p>功能描述：数据集观察结果</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 * <p>
 * 数据集观察结果，用于自动计算出统计轴的最小值、最大值、刻度等信息。<br/>
 * <b>目前只使用了第1点的数据范围来确定</b><br/>
 * 以下来自文心大模型3.5：<br/>
 * 1.数据范围：观察数据集的最大值和最小值，以及数据分布情况。这些信息可以用来确定Y轴的最大值和最小值。<br/>
 * 2.数据集中度：观察数据的集中趋势和分布情况，以确定合适的刻度。如果数据分布较为集中，可以使用较小的刻度；如果数据分布较为分散，则可以使用较大的刻度。<br/>
 * 观察数据集的数据集中度，可以使用以下指标来判断：<br/>
 * - 均值：计算数据集的均值，均值反映了数据的平均水平，可以反映数据的集中趋势。如果均值较小，说明数据集中趋势较明显，数据较为集中；反之则说明数据较为分散。<br/>
 * - 中位数：将数据集按照大小顺序排列后，位于中间位置的数值即为中位数。中位数能够直接表示数据集的中心位置，对于存在极端值或离群点的数据集，中位数比均值更能反映数据的集中趋势。<br/>
 * - 众数：数据集中出现次数最多的数值即为众数。众数反映了数据集中最常见的数值，也可以反映数据的集中趋势。<br/>
 * - 标准差：标准差反映了数据集中数值的离散程度，标准差越小说明数据越集中；反之则说明数据越分散。<br/>
 * 3.数据变化率：观察数据的变化率和变化趋势，以确定是否需要使用次坐标轴或者对数刻度等特殊刻度。如果数据变化较大或变化率较快，可以考虑使用次坐标轴或对数刻度。<br/>
 * 4.数据异常值：观察数据中的异常值，如离群点或极端值。这些异常值可能会对最大值和最小值的计算产生影响，需要特别处理。<br/>
 * - 观察数据集的数据异常值，可以设置异常值的标准来进行判定。通常有以下一些方法可以设置异常值的标准：<br/>
 * - 基于统计的方法：可以使用均值、标准差等指标来判断异常值。例如，可以使用3σ原则，将距离均值超过3倍标准差的数值判定为异常值。<br/>
 * - 基于四分位数的方法：可以通过计算数据的四分位数（Q1，Q2，Q3）和四分位距（IQR）来判断异常值。例如，将小于Q1-1.5IQR或大于Q3+1.5IQR的数值判定为异常值。<br/>
 * - 基于箱线图的方法：箱线图是一种直观展示数据分布情况的图表，可以通过观察箱线图的异常值来判断数据的异常值。通常，箱线图将距离箱体上下边缘超过1.5倍IQR的数值判定为异常值。<br/>
 * - 基于业务规则的方法：根据具体业务规则，可以设定一些特定的阈值或条件来判断异常值。例如，在某个具体业务场景中，如果某个数据项的数值明显超出正常范围或不符合业务逻辑，则可以将其判定为异常值。<br/>
 * 5.数据集的单位：观察数据集的单位，以确定刻度的合适单位。如果数据的单位不同或跨度较大，可以考虑使用科学记数法或其他适当的表示方法。<br/>
 *
 * @FileName: DataObserveResult
 * @Author: zaregoto
 * @Date: 2023/11/30 13:58
 */
@Getter
public class DataObserveResult implements Serializable {
    private static final long serialVersionUID = 2019013015209035321L;

    /**
     * 默认统计轴刻度等分数量
     */
    public static final int DEFAULT_VALUE_AXIS_EQUAL_PARTS_COUNT = 5;

    /**
     * 当前语言别
     */
    private String locale;

    /**
     * 图表类型
     */
    private ChartTypeEnum chartTypeEnum;

    /**
     * 统计字段（或叫度量字段）对应的分组编号
     */
    private Integer group;

    /**
     * 数据范围-最小值
     */
    private Double minValue;

    /**
     * 数据范围-最大值
     */
    private Double maxValue;

    /**
     * 数据集中度-平均值。均值反映了数据的平均水平，可以反映数据的集中趋势。
     * 如果均值较小，说明数据集中趋势较明显，数据较为集中；反之则说明数据较为分散。
     */
    private Double meanValue;

    /**
     * 数据集中度-中位数。将数据集按照大小顺序排列后，位于中间位置的数值即为中位数。
     * 中位数能够直接表示数据集的中心位置，对于存在极端值或离群点的数据集，中位数比均值更能反映数据的集中趋势。
     */
    private Double medianValue;

    /**
     * 数据集中度-标准差。越小说明数据越集中；反之则说明数据越分散。
     */
    private Double stdDeviation;

    /**
     * 异常数据值
     */
    private Double[] irregularValues;

    /**
     * 建议的数轴最小值
     */
    private Double preferredAxisMinValue;

    /**
     * 建议的数轴最大值
     */
    private Double preferredAxisMaxValue;

    /**
     * 建议的数轴刻度间隔
     */
    private Double preferredAxisInterval;

    /**
     * 数轴刻度展示格式
     * TODO
     */
    private AgileDataChartFieldDisplayFormat format;

    /**
     * 负轴刻度数
     */
    private Integer negativeValueAxisScaleCount;

    /**
     * 正轴刻度数
     */
    private Integer positiveValueAxisScaleCount;

    private DataObserveResult() {
    }

    public DataObserveResult(AgileDataChartValueField valueField) {
        this();
        this.format = valueField.getFormat();
    }

    public DataObserveResult(String locale, Integer groupNo, AgileDataChartFieldDisplayFormat format) {
        this();
        this.locale = locale;
        this.group = groupNo;
        this.format = format;
    }

    public void setMinValue(Double minValue) {
        this.minValue = minValue;
    }

    public void setMaxValue(Double maxValue) {
        this.maxValue = maxValue;
    }

    public void setMeanValue(Double meanValue) {
        this.meanValue = meanValue;
    }

    public void setMedianValue(Double medianValue) {
        this.medianValue = medianValue;
    }

    public void setStdDeviation(Double stdDeviation) {
        this.stdDeviation = stdDeviation;
    }

    public void setPreferredAxisMinValue(Double preferredAxisMinValue) {
        this.preferredAxisMinValue = preferredAxisMinValue;
    }

    public void setPreferredAxisMaxValue(Double preferredAxisMaxValue) {
        this.preferredAxisMaxValue = preferredAxisMaxValue;
    }

    public void setPreferredAxisInterval(Double preferredAxisInterval) {
        this.preferredAxisInterval = Math.abs(preferredAxisInterval);
    }

    public void setNegativeValueAxisScaleCount(Integer negativeValueAxisScaleCount) {
        this.negativeValueAxisScaleCount = negativeValueAxisScaleCount;
    }

    public void setPositiveValueAxisScaleCount(Integer positiveValueAxisScaleCount) {
        this.positiveValueAxisScaleCount = positiveValueAxisScaleCount;
    }

    /**
     * 判断数据分布是否集中，通过标准差与平均值来判断：
     * 标准差在多少范围内可以认为数据是比较集中的，并没有一个固定的标准。
     * 一般来说，标准差在10%以内通常被认为是比较集中的；而如果标准差超过20%，则说明数据分布比较离散
     *
     * @return
     */
    public boolean dataDistributionIsCentralized() {
        if (null == this.meanValue && null == this.stdDeviation) {
            return false;
        }
        double diff = this.stdDeviation / this.meanValue;
        return Math.abs(diff) <= 0.1;
    }

    /**
     * 判断数据分布是否离散，通过标准差与平均值来判断：
     * 标准差在多少范围内可以认为数据是比较集中的，并没有一个固定的标准。
     * 一般来说，标准差在10%以内通常被认为是比较集中的；而如果标准差超过20%，则说明数据分布比较离散
     *
     * @return
     */
    public boolean dataDistributionIsDiscrete() {
        if (null == this.meanValue && null == this.stdDeviation) {
            return false;
        }
        double diff = this.stdDeviation / this.meanValue;
        return Math.abs(diff) >= 0.2;
    }

    /**
     * 判断数据分布是否适中，即：既不集中，也不离散
     *
     * @return
     */
    public boolean dataDistributionIsModerate() {
        return !this.dataDistributionIsCentralized() && !this.dataDistributionIsDiscrete();
    }

    /**
     * 判断数据分布是否在±1之间
     *
     * @return
     */
    public boolean dataDistributionIsBetweenPlusAndMinusOne() {
        return this.getMinValue() >= -1.0 && this.getMaxValue() <= 1.0;
    }

    /**
     * 判断数据分布是否在±10之间
     *
     * @return
     */
    public boolean dataDistributionIsBetweenPlusAndMinusTen() {
        return this.getMinValue() >= -10.0 && this.getMaxValue() <= 10.0;
    }

    /**
     * 获得最佳的数轴刻度信息（根据preferredAxisMinValue、preferredAxisMaxValue和preferredAxisInterval计算得到）。
     * 刻度中会自动计算零刻度线。<br/>
     * 刻度是有千分位的整数，没有小数。<br/>
     *
     * @return 计算后的最佳数轴刻度信息
     */
    public List<String> calculatePreferredAxisDatas() {
        if (null == this.getPreferredAxisMinValue() || null == this.getPreferredAxisMaxValue()
                || null == this.getPreferredAxisInterval()) {
            return Collections.emptyList();
        }
        if (null != this.format) {
            return calculateValueAxisScalesByFormat(this.format);
        } else {
            return calculateValueAxisScalesByDefault();
        }
    }

    /**
     * 微调建议的坐标轴刻度间隔，以取整5的倍数来计算，并重新计算对应的最小、最大值
     */
    public void tunePreferredAxisInterval() {
        if (null == this.getPreferredAxisInterval()
                || null == this.getPreferredAxisMinValue()
                || null == this.getPreferredAxisMaxValue()
                || null == this.getMinValue()
                || null == this.getMaxValue()) {
            return;
        }

        Double oldInterval = this.getPreferredAxisInterval();

        // 间隔大于1表示数据变化较大，需要重新微调间距；间隔小于1表示数据变化较小，不需要再微调间距
        if (oldInterval < 1) {
            // 求解该式子中，n的最小整数值，获得负数区间最小所需的刻度间隔：0 - min(n) * new_interval<=old_min_axis_value
            int minNegativeIntervalCnt = (int) Math.ceil((0.0 - this.getPreferredAxisMinValue()) / this.getPreferredAxisInterval());
            // 求解该式子中，n的最小整数值，获得正数区间最小所需的刻度间隔：0 + min(n) * new_interval>=old_max_axis_value
            int minPositiveIntervalCnt = (int) Math.ceil(this.getPreferredAxisMaxValue() / this.getPreferredAxisInterval());

            // 重新更新坐标轴的最小、最大值
            this.setPreferredAxisMinValue(0 - minNegativeIntervalCnt * oldInterval);
            this.setPreferredAxisMaxValue(0 + minPositiveIntervalCnt * oldInterval);

            // 更新刻度数量
            this.setNegativeValueAxisScaleCount(minNegativeIntervalCnt);
            this.setPositiveValueAxisScaleCount(minPositiveIntervalCnt);

            return;
        }

        // 坐标轴刻度间距取整（自动根据上方间距决定向上或向下取整）
        Double newInterval = MathUtil.getFiveMultiplesRoundedValueByDigits(oldInterval, 0);
        // 设置新的刻度间距
        if (newInterval > 0) {
            this.setPreferredAxisInterval(newInterval);
        }

        // 重新设置坐标轴的最小、最大值
        if (Objects.equals(this.getMinValue(), this.getMaxValue())) {
            // 求解该式子中，n的最小整数值，获得负数区间最小所需的刻度间隔：0 - min(n) * new_interval<=old_min_axis_value
            int minNegativeIntervalCnt = (int) Math.ceil((0.0 - this.getPreferredAxisMinValue()) / this.getPreferredAxisInterval());
            // 求解该式子中，n的最小整数值，获得正数区间最小所需的刻度间隔：0 + min(n) * new_interval>=old_max_axis_value
            int minPositiveIntervalCnt = (int) Math.ceil(this.getPreferredAxisMaxValue() / this.getPreferredAxisInterval());

            // 更新刻度数量
            this.setNegativeValueAxisScaleCount(minNegativeIntervalCnt);
            this.setPositiveValueAxisScaleCount(minPositiveIntervalCnt);
        } else if (this.getMaxValue() <= 0) {
            // 重新用预计坐标最大值，和上方新的坐标刻度间距，计算出预计坐标的最小值
            Double oldPreferredAxisMinValue = this.getPreferredAxisMinValue();
            // 求解该式子中，n的最大整数值：max_axis_value - max(n) * new_interval<=old_min_axis_value
            int minNegativeIntervalCnt = (int) Math.ceil((this.getPreferredAxisMaxValue() - oldPreferredAxisMinValue) / this.getPreferredAxisInterval());
            Double newPreferredAxisMinValue = this.getPreferredAxisMaxValue() - minNegativeIntervalCnt * this.getPreferredAxisInterval();
            this.setPreferredAxisMinValue(MathUtil.getFiveMultiplesRoundedValueByDigits(newPreferredAxisMinValue, -1));

            // 更新刻度数量
            this.setNegativeValueAxisScaleCount(minNegativeIntervalCnt);
            this.setPositiveValueAxisScaleCount(0);
        } else if (this.getMinValue() >= 0) {
            // 重新用预计坐标最小值，和上方新的坐标刻度间距，计算出预计坐标的最大值
            Double oldPreferredAxisMaxValue = this.getPreferredAxisMaxValue();
            // 求解该式子中，n的最小整数值：min_axis_value + min(n) * new_interval>=old_max_axis_value
            int minPositiveIntervalCnt = (int) Math.ceil((oldPreferredAxisMaxValue - this.getPreferredAxisMinValue()) / this.getPreferredAxisInterval());
            Double newPreferredAxisMaxValue = this.getPreferredAxisMinValue() + minPositiveIntervalCnt * this.getPreferredAxisInterval();
            this.setPreferredAxisMaxValue(MathUtil.getFiveMultiplesRoundedValueByDigits(newPreferredAxisMaxValue, 1));

            // 更新刻度数量
            this.setNegativeValueAxisScaleCount(0);
            this.setPositiveValueAxisScaleCount(minPositiveIntervalCnt);
        } else {
            // 重新用上方新的坐标刻度间距，从原点计算出预计坐标的最小值与最大值
            Double oldPreferredAxisMinValue = this.getPreferredAxisMinValue();
            Double oldPreferredAxisMaxValue = this.getPreferredAxisMaxValue();
            // 求解该式子中，n的最小整数值，获得负数区间最小所需的刻度间隔：0 - min(n) * new_interval<=old_min_axis_value
            int minNegativeIntervalCnt = (int) Math.ceil((0.0 - oldPreferredAxisMinValue) / this.getPreferredAxisInterval());
            // 求解该式子中，n的最小整数值，获得正数区间最小所需的刻度间隔：0 + min(n) * new_interval>=old_max_axis_value
            int minPositiveIntervalCnt = (int) Math.ceil(oldPreferredAxisMaxValue / this.getPreferredAxisInterval());
            Double newPreferredAxisMinValue = 0 - minNegativeIntervalCnt * this.getPreferredAxisInterval();
            Double newPreferredAxisMaxValue = 0 + minPositiveIntervalCnt * this.getPreferredAxisInterval();
            /**
             * 2024-08-22 去除坐标轴刻度间距取整逻辑，只保留计算完的数值，以防在后面计算刻度值时，因存在零刻度，导致间隔不正确。
             * FIXME 待观察测试效果
             * 举例：
             * - 经过上方逻辑处理后，最小值是-270000000，最大值是270000000，负值、正值刻度各3个，间隔90000000，上下基于零刻度对称，后面
             *   计算刻度时，刻度是-270000000，-180000000，-90000000，0，90000000，180000000，270000000，刻度是基于零刻度对称的
             * - 如果增加了依5倍数取整的逻辑，最小值变为-300000000，最大值变为300000000，负值、正值刻度依然是各3个，间隔90000000，后面
             *   计算刻度时，刻度是-300000000，-210000000，-120000000，-30000000，0，60000000，150000000，240000000，
             *   刻度不是基于零刻度对称的，前端APP呈现会错乱。
             */
//            this.setPreferredAxisMinValue(MathUtil.getFiveMultiplesRoundedValueByDigits(newPreferredAxisMinValue, -1));
//            this.setPreferredAxisMaxValue(MathUtil.getFiveMultiplesRoundedValueByDigits(newPreferredAxisMaxValue, 1));
            this.setPreferredAxisMinValue(newPreferredAxisMinValue);
            this.setPreferredAxisMaxValue(newPreferredAxisMaxValue);

            // 设置刻度数量
            this.setNegativeValueAxisScaleCount(minNegativeIntervalCnt);
            this.setPositiveValueAxisScaleCount(minPositiveIntervalCnt);
        }
    }


    /**
     * 显示时增加安全间距，用于支持图形点击时的label正常显示：
     * 若数据最大值与建议坐标轴最大值的差值小于1个间距，则建议坐标轴最大值增加1个间距
     *
     * @param dataSetMin
     * @param dataSetMax
     */
    public void addSafeAreaToAxisScale(Double dataSetMin, Double dataSetMax) {
        if (Math.abs(dataSetMax - this.getPreferredAxisMaxValue()) < this.getPreferredAxisInterval()) {
            Double oldAxisMaxValue = this.getPreferredAxisMaxValue();
            Double axisInterval = this.getPreferredAxisInterval();
            this.setPreferredAxisMaxValue(oldAxisMaxValue + axisInterval);
        }
        if (Math.abs(dataSetMin - this.getPreferredAxisMinValue()) < this.getPreferredAxisInterval()) {
            Double oldAxisMinValue = this.getPreferredAxisMinValue();
            Double axisInterval = this.getPreferredAxisInterval();
            this.setPreferredAxisMinValue(oldAxisMinValue - axisInterval);
        }
    }

    /**
     * 根据统计轴数据最小值和最大值，以及默认的格式，获取刻度清单。
     * 缺点：无法出现百分比符号
     *
     * @return 坐标轴刻度字符串清单
     */
    @NotNull
    private List<String> calculateValueAxisScalesByDefault() {
        /**
         * 动态计算刻度要保留的小数位数
         * 计算依据：若数据分布在±1之间，则数据间隔也小于1，此时需要计算待保留的小数位数，否则当数据过小时坐标轴可能都会显示成0；
         * 如果保留位数还为0，则使用默认小数位数参数。
         */
        int n = 0;
        if (this.getPreferredAxisInterval() < 1.0) {
            n = MathUtil.getMinimumTimesToGreaterThanOne(this.getPreferredAxisInterval());
        }
        // 防止意外情况，使用默认小数位数。如为零则String.format时会变成四舍五入，坐标轴会奇怪（如-0，0；或者0，2，3，5，6，8，9，不连续）
        if (n == 0) {
            n = AgileDataChartFieldDisplayFormat.DEFAULT_DECIMAL;
        }
        String axisScaleFormat = "%,." + n + "f";

        // 根据绝对值的最大值，获取坐标轴的数量级单位和文字
        MagnitudeUnit magnitudeUnit = CurrencyUtil.getUnitOfMagnitudeByLocale(this.locale,
                MathUtil.getMaxValueByAbs(this.getPreferredAxisMinValue(), this.getPreferredAxisMaxValue()));

        List<String> axisData = new ArrayList<>();

        // 零刻度线处理
        if (this.getPreferredAxisMinValue() > 0.0) {
            axisData.add(String.format(axisScaleFormat, 0.0));
        }

        double prevValue = 0.0;
        for (double i = this.getPreferredAxisMinValue(); i <= this.getPreferredAxisMaxValue(); i += this.getPreferredAxisInterval()) {
            // 上一个值小于零，且当前值大于零时，在坐标轴中间加一个零刻度线
            if (prevValue < 0.0 && i > 0.0) {
                axisData.add(String.format(axisScaleFormat, 0.0));
            }
           /* String scaleString = String.format(axisScaleFormat, i / magnitudeUnit.getLevel());
            if (i != 0) {
                scaleString += magnitudeUnit.getUnitText();
            }*/
            //2024-06-27 因需求32676,改为由前端计算，返回原始值大小，不带单位
            String scaleString = String.valueOf(new BigDecimal(String.valueOf(i)));
            axisData.add(scaleString);
            prevValue = i;
        }

        // 零刻度线处理
        if (this.getPreferredAxisMaxValue() < 0.0) {
            axisData.add(String.format(axisScaleFormat, 0.0));
        }
        return axisData;
    }

    /**
     * 根据统计轴数据最小值和最大值，以及传入的格式，获取刻度清单。
     *
     * @return 坐标轴刻度字符串清单
     */
    private List<String> calculateValueAxisScalesByFormat(AgileDataChartFieldDisplayFormat format) {
        // 根据绝对值的最大值，获取坐标轴的数量级单位和文字
        MagnitudeUnit magnitudeUnit = CurrencyUtil.getUnitOfMagnitudeByLocale(this.locale,
                MathUtil.getMaxValueByAbs(this.getPreferredAxisMinValue(), this.getPreferredAxisMaxValue()));

        List<String> axisData = new ArrayList<>();

        // 零刻度线处理
        if (this.getPreferredAxisMinValue() > 0.0) {
            axisData.add(format.getFormattedString(0.0));
        }
        // 定义误差值epsilon。如果传入值的绝对值小于误差值，则认为是零
        double epsilon = 1e-9;
        double prevValue = 0.0;
        for (double i = this.getPreferredAxisMinValue(); i <= this.getPreferredAxisMaxValue(); i += this.getPreferredAxisInterval()) {
            // 上一个值小于零且绝对值大于误差值时，表示上一个值并未被当做零刻度处理；且当前值大于零时，需在坐标轴中间加一个零刻度线
            if (prevValue < 0.0 && Math.abs(prevValue) > epsilon && i > 0.0) {
                axisData.add(format.getFormattedString(0.0));
            }
            // 因为小于设定的误差值后，会认为是0，所以不需要再加单位
            /*String scaleString = format.getFormattedString(i / magnitudeUnit.getLevel());
            if (Math.abs(i) > epsilon) {
                scaleString += magnitudeUnit.getUnitText();
            }*/
            //2024-06-27 因需求32676,改为由前端计算，返回原始值大小，不带单位
            String scaleString = format.getFormattedString(new BigDecimal(String.valueOf(i)).doubleValue());
            axisData.add(scaleString);
            prevValue = i;
        }

        // 零刻度线处理
        if (this.getPreferredAxisMaxValue() < 0.0) {
            axisData.add(format.getFormattedString(0.0));
        }
        return axisData;
    }
}