package com.digiwin.mobile.mobileuibot.proxy.uibot.model.layout;

import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.core.component.BaseMobileComponent;
import com.digiwin.mobile.mobileuibot.core.component.BaseMobileComponentWrapper;
import com.digiwin.mobile.mobileuibot.core.component.BaseMobileComponentWrapperJsonDeserializer;
import com.digiwin.mobile.mobileuibot.core.component.DataSource;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.google.common.collect.Lists;
import lombok.Data;
import org.springframework.util.CollectionUtils;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * <p>功能描述：页面布局类</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: UiBotPageLayout.java
 * @Author: wangjwc
 * @Date: created at 2024/9/9 15:29
 */
@Data
public class UiBotPageLayout implements Serializable {
    private static final long serialVersionUID = -7678136368997085052L;

    /**
     * 页面背景色
     *
     * @see com.digiwin.mobile.mobileuibot.core.common.BackgroundColorEnum
     */
    private String pageBackgroudColor;
    /**
     * 收起部分组件，展示查看更多，无值时按false处理，为true时
     * 页面自动隐藏相关组件，底部出现查看更多功能（隐藏的组件在页面layout下，或者CUSTOM_GROUP组件内）
     */
    private Boolean showMore;
    private String title;
    /**
     * 标题是否显示，true：显示，false：隐藏
     * 默认false
     */
    private Boolean titleVisible = false;
    private String pageId;
    private String schema;
    private String componentType;
    private String componentId;

    /**
     * "title": {
     * "zh_CN": "主表单",
     * "zh_TW": "主表單",
     * "en_US": "Main"
     * }
     */
    private Map<String, Object> lang;

    /**
     * 页面/弹窗 --多数据源配置
     */
    private DataSource dataSource;

    /**
     * 页面/弹窗 --节点信息
     */
    private String node;

    private String nodePath;


    /**
     * 画面头部组件list（固定不动）
     */
    @JsonDeserialize(contentUsing = BaseMobileComponentWrapperJsonDeserializer.class)
    private List<BaseMobileComponentWrapper<BaseMobileComponent>> header;

    /**
     * 画面身体部分组件list（可动）
     */
    @JsonDeserialize(contentUsing = BaseMobileComponentWrapperJsonDeserializer.class)
    private List<BaseMobileComponentWrapper<BaseMobileComponent>> body;

    /**
     * 画面底部组件list（固定不动）
     */
    @JsonDeserialize(contentUsing = BaseMobileComponentWrapperJsonDeserializer.class)
    private List<BaseMobileComponentWrapper<BaseMobileComponent>> bottom;

    /**
     * 页面下控件和布局 -- 未加工数据
     */
    List<Map<String, Object>> data;

    public void convert(String componentType) {
        if (CollectionUtils.isEmpty(this.data)) {
            return;
        }
        // 设计器，默认第一个元素为body布局
        this.buildBody(this.data.get(0), componentType);
        if (this.data.size() > 1) {
            // 设计器，默认第二个元素为bottom布局
            this.buildBottom(this.data.get(1));
        }
        // clear
        this.data = null;
    }

    private void buildBody(Map<String, Object> body, String componentType) {
        if (CollectionUtils.isEmpty(body)) {
            return;
        }
        if (LayoutComponentTypeEnum.SINGLE_HEAD.getValue().equalsIgnoreCase(componentType)) {
            Map<String, Object> map = new HashMap<>(1);
            map.put("data", Lists.newArrayList(body));
            Layout layout = JsonUtil.objectToJavaObject(map,
                    new TypeReference<Layout>() {
                    });
            if (layout == null) {
                return;
            }
            this.body = layout.getData();
        } else {
            MobileCrudBlock mobileCrudBlock = JsonUtil.objectToJavaObject(body,
                    new TypeReference<MobileCrudBlock>() {
                    });
            if (null == mobileCrudBlock) {
                return;
            }
            Layout layout = mobileCrudBlock.getData();
            if (null == layout) {
                return;
            }
            this.body = layout.getData();
        }

    }

    private void buildBottom(Map<String, Object> bottom) {
        if (CollectionUtils.isEmpty(bottom)) {
            return;
        }
        Map<String, Object> map = new HashMap<>(1);
        map.put("data", Lists.newArrayList(bottom));
        Layout layout = JsonUtil.objectToJavaObject(map,
                new TypeReference<Layout>() {
                });
        if (layout == null) {
            return;
        }
        this.bottom = layout.getData();
    }

    public List<BaseMobileComponentWrapper<BaseMobileComponent>> getHeader() {
        if (header == null) {
            header = new ArrayList<>();
        }
        return header;
    }

    public List<BaseMobileComponentWrapper<BaseMobileComponent>> getBody() {
        if (body == null) {
            body = new ArrayList<>();
        }
        return body;
    }

    public List<BaseMobileComponentWrapper<BaseMobileComponent>> getBottom() {
        if (bottom == null) {
            bottom = new ArrayList<>();
        }
        return bottom;
    }

    @Data
    private static class MobileCrudBlock {
        private String componentType;
        private Layout data;
    }

    @Data
    private static class Layout {
        @JsonDeserialize(contentUsing = BaseMobileComponentWrapperJsonDeserializer.class)
        private List<BaseMobileComponentWrapper<BaseMobileComponent>> data;
    }
}
