package com.digiwin.mobile.mobileuibot.proxy.uibot.model.table;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.annotation.JSONField;
import com.digiwin.mobile.mobileuibot.common.datetime.DateTimeUtil;
import com.digiwin.mobile.mobileuibot.common.math.MathUtil;
import com.digiwin.mobile.mobileuibot.proxy.uibot.PcUiBotLayoutJsonDeserializer;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.UiBotBizFieldBase;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.UiBotExecuteContext;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.layout.UiBotLayout;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.tag.PcUiBotTagDefinition;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.NoArgsConstructor;
import org.springframework.util.StringUtils;

import java.math.BigDecimal;
import java.util.*;

/**
 * <p>功能描述：</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: UiBotTableColumn
 * @Author: Zaregoto
 * @Date: 2021/5/24 2:10
 */
@Data
@NoArgsConstructor
@AllArgsConstructor
public class UiBotTableColumn extends UiBotBizFieldBase {
    private static final long serialVersionUID = 6293604605440647803L;

    public static final String VALUE_EMPTY_SYMBOL = "-";
    public static final String VALUE_EMPTY_STRING_SYMBOL = "";

    /**
     * 敏捷数据默认小数精度
     */
    public static final Integer AGILE_DATA_DEFAULT_DECIMAL = 2;

    private List<UiBotTableColumn> columns;

    private Boolean filterable;
    /**
     * 组件标题
     */
    private String headerName;
    /**
     * 字段操作定义配置
     */
    private JSONArray operations;

    /**
     * 字段所属pageData内业务数据的路径名称，类似XPath的概念
     * 示例值：project_info。表示当前字段属于pageData.project_info对象
     */
    private String path;
    private Boolean showIcon;
    /**
     * 是否可编辑
     */
    private Boolean editable;
    private Boolean sortable;
    private Boolean rowGroupable;

    /**
     * 数据精度
     * 目前在super.type=PERCENT_INPUT时见过
     */
    private JSONObject dataPrecision;
    private UiBotTableColumnEditor editor;
    private List<PcUiBotTagDefinition> tagDefinitions;
    private UiBotTableAttachmentColumnAttribute attribute;
    private Boolean thousandthPercentile;
    private String buckets;
    private Map<String, Object> extendParas;

    /**
     * 保留小数位数。若Web端uibot没有给，则默认为-1，表示不处理；下方代码判断小于等于-1是为了防止在元数据中配置成负数导致不正确
     * 通常在super.dataType=numeric的时候才有该字段
     */
    private Integer decimal = -1;

    /**
     * 下一层级的 group 现作用于签核套件子单身（操作抽屉）
     */
    @JsonDeserialize(contentUsing = PcUiBotLayoutJsonDeserializer.class)
    private List<UiBotLayout> group;

    /* 敏捷数据使用 BEGIN */

    /**
     * 敏捷数据:指标单位
     */
    private String businessType;


    /**
     * 字段值是否需要使用百分号，或千分位显示
     * 当使用百分号时，值为percent；
     * 当使用千分位时，值为thousand
     */
    private String percent;

    /**
     * 舍入方式
     */
    private UiBotTableDecimalRule decimalRule;

    /**
     * 计算单位
     */
    private UiBotTableUnit unit;

    /* 敏捷数据使用 END */

    /*********************************************************************************************
     * ******************************** 移动端特有属性 BEGIN ****************************************
     *********************************************************************************************/
    /**
     * 0代表身份证；1代表银行卡
     */
    private Integer ocrType;
    /**
     * 内容的组件类型
     */
    @JsonProperty(access = JsonProperty.Access.WRITE_ONLY)
    @JSONField(serialize = false)
    private String mContentType;

    /**
     * 字段是否必须（移动端特有，web端没有）
     */
    @JsonProperty(access = JsonProperty.Access.WRITE_ONLY)
    @JSONField(serialize = false)
    private Boolean mRequired;

    /**
     * 最多字数
     */
    @JsonProperty(access = JsonProperty.Access.WRITE_ONLY)
    @JSONField(serialize = false)
    private Integer maxLength = 255;

    /**
     * 数据字典key，用来查询数据字典，以便转换映射值
     */
    private String enumKey;

    /*********************************************************************************************
     * ******************************** 移动端特有属性 END *****************************************
     *********************************************************************************************/

    public static UiBotTableColumn create(String headerName, String schema) {
        UiBotTableColumn column = new UiBotTableColumn();
        column.setHeaderName(headerName);
        column.setSchema(schema);
        return column;
    }

    public static void main(String[] args) {
        System.out.println(getMaxPublicSubNameBetweenColumns("new_issue_qty", "old_issue_qty"));
    }

    public static String getMaxPublicSubNameBetweenColumns(String columnHeaderName1, String columnHeaderName2) {
        // 参考链接：https://www.cnblogs.com/ningvsban/p/3967231.html
        if (null == columnHeaderName1 || null == columnHeaderName2) {
            return "";
        }
        if (columnHeaderName1.isEmpty() || columnHeaderName2.isEmpty()) {
            return "";
        }

        columnHeaderName1 = columnHeaderName1.toLowerCase();
        columnHeaderName2 = columnHeaderName2.toLowerCase();
        int len1 = columnHeaderName1.length();
        int len2 = columnHeaderName2.length();
        String min = null;
        String max = null;
        String target = null;
        min = len1 <= len2 ? columnHeaderName1 : columnHeaderName2;
        max = len1 > len2 ? columnHeaderName1 : columnHeaderName2;
        //最外层：min子串的长度，从最大长度开始
        for (int i = min.length(); i >= 1; i--) {
            //遍历长度为i的min子串，从0开始
            for (int j = 0; j <= min.length() - i; j++) {
                target = min.substring(j, j + i);
                //遍历长度为i的max子串，判断是否与target子串相同，从0开始
                for (int k = 0; k <= max.length() - i; k++) {
                    if (max.substring(k, k + i).equals(target)) {
                        return target;
                    }
                }
            }
        }
        return "";

    }

    /**
     * 判断本栏位是否可编辑
     *
     * @return
     */
    @Override
    public boolean canEdit() {
        boolean nonFileCanEdit =
                Optional.ofNullable(this.getEditable()).orElse(false) ||
                        Optional.ofNullable(this.getEditor()).map(UiBotTableColumnEditor::getEditable).orElse(false);
        boolean fileCanEdit = this.getType() != null &&
                this.getType().toUpperCase().contains("UPLOAD") &&
                this.getAttribute().getEditable();

        return nonFileCanEdit || fileCanEdit;
    }

    /**
     * 强制设置为可编辑状态
     */
    @Override
    public void doSetCanEdit() {
        // NOT FILE
        this.setEditable(true);
        Optional.ofNullable(this.getEditor()).ifPresent(editor -> editor.setEditable(true));

        // FILE
        if (null != this.getType() && this.getType().toUpperCase().contains("UPLOAD")) {
            Optional.ofNullable(this.getAttribute()).ifPresent(attribute -> attribute.setEditable(true));
        }
    }

    /**
     * 强制设置为不可编辑状态
     */
    @Override
    public void doSetCanNotEdit() {
        // NOT FILE
        this.setEditable(false);
        Optional.ofNullable(this.getEditor()).ifPresent(editor -> editor.setEditable(false));

        // FILE
        if (null != this.getType() && this.getType().toUpperCase().contains("UPLOAD")) {
            Optional.ofNullable(this.getAttribute()).ifPresent(attribute -> attribute.setEditable(false));
        }
    }

    /**
     * 反转栏位的可编辑状态
     */
    @Override
    public void reverseEditableState() {
        if (this.canEdit()) {
            this.doSetCanNotEdit();
        } else {
            this.doSetCanEdit();
        }
    }

    /**
     * 判断本栏位是否可上传文件
     *
     * @return
     */
    public boolean canUpload() {
        return this.getType() != null &&
                this.getType().toUpperCase().contains("UPLOAD") &&
                this.getAttribute().getUploadEnable();
    }

    /**
     * 判断本栏位是否可编辑且是通过选择来输入值
     *
     * @return
     */
    public boolean canEditWithSelect() {
        return this.canEdit()
                && "SELECT".equalsIgnoreCase(
                Optional.ofNullable(this.getEditor()).map(UiBotTableColumnEditor::getType).orElse("")
        );
    }

    /**
     * 判断本栏位是否可编辑且是通过开窗来输入值
     *
     * @return
     */
    public boolean canEditWithOpenWindow() {
        return this.canEdit()
                && "OPERATION_EDITOR".equalsIgnoreCase(Optional.ofNullable(this.getType()).orElse("")
        );
    }

    /**
     * 判断本栏位值是否数值类型
     *
     * @return
     */
    public boolean isNumericData() {
        return null != this.getDataType() && "numeric".equalsIgnoreCase(this.getDataType());
    }

    /**
     * 是否使用百分比类别的组件渲染
     *
     * @return
     */
    public boolean isRenderInPercentType() {
        return null != this.getPercent() && this.getPercent().contains("percent");
    }

    /**
     * 根据字段的数据类型及展示类型，返回字段的值的文字展示内容
     *
     * @param bizData
     * @return
     */
    @Override
    public String getValueForDisplay(Map<String, Object> bizData) {
        return getValueForDisplay(bizData, false);
    }

    /**
     * 根据字段的数据类型及展示类型，返回字段的值的文字展示内容（增加一个参数区分）
     *
     * @param bizData
     * @param isAgileData 是否是敏捷数据中使用
     * @return
     */
    public String getValueForDisplay(Map<String, Object> bizData, boolean isAgileData) {
        String bizDataSchema = this.getSchema();
        String type = Optional.ofNullable(this.getType()).orElse("");
        String dataType = this.getDataType();
        String editorType = Optional.ofNullable(this.getEditor()).map(UiBotTableColumnEditor::getType).orElse("");
        String value = "";

        boolean isPercentType = this.isRenderInPercentType();
        // 敏捷数据场景下，设置精度为敏捷数据的默认精度
        /*if (isAgileData && this.decimal <= -1) {
            this.decimal = AGILE_DATA_DEFAULT_DECIMAL;
        }*/
        if (!bizData.containsKey(bizDataSchema) || null == bizData.get(bizDataSchema)) {
            value = VALUE_EMPTY_STRING_SYMBOL;
        } else if ("boolean".equalsIgnoreCase(dataType) || "string".equalsIgnoreCase(dataType)) {
            Object rawValue = bizData.get(bizDataSchema);
            value = String.valueOf(rawValue);

            if ("".equalsIgnoreCase(editorType) && "LABEL".equalsIgnoreCase(type) && "null".equalsIgnoreCase(value)) {
                value = VALUE_EMPTY_SYMBOL;
            }

            if ("SELECT".equalsIgnoreCase(editorType) || "SELECT".equalsIgnoreCase(type)) {
                String bizValue = String.valueOf(bizData.get(bizDataSchema));
                value = this.getSelectFieldOptionTitle(this, bizValue);
            }

            // 新增敏捷数据中web给的中间态DSL里遇到的百分比组件
            if (isPercentType) {
                // 小数位数
                Integer percentDecimal = this.decimal;
                // FIXME 千分位没有，默认移动端处理时加上
                BigDecimal bigDecimalValue = new BigDecimal(value);
                if (null == percentDecimal) {
                    value = String.format("%,f%%", bigDecimalValue.movePointRight(2).doubleValue());
                } else {
                    value = String.format("%,." + percentDecimal + "f%%", bigDecimalValue.movePointRight(2).doubleValue());
                }
            }

            if (rawValue instanceof Double && this.decimal <= -1) {
                value = MathUtil.stripTrailingZerosInDoubleString(value);
            }
        } else if ("date".equalsIgnoreCase(dataType)) {
            value = String.valueOf(bizData.get(bizDataSchema));
            //目前时间只要年月日
            value = null != value && value.length() > 10 ? value.substring(0, 10) : value;
            value = DateTimeUtil.getDateTextInMobileShowDefaultPattern(value);
        } else if ("numeric".equalsIgnoreCase(dataType)) {
            BigDecimal bigDecimalValue = new BigDecimal(String.valueOf(bizData.get(bizDataSchema)));
            double dValue = bigDecimalValue.doubleValue();
            boolean hasThousandthPercentile = Optional.ofNullable(this.getThousandthPercentile()).orElse(false);
            if (hasThousandthPercentile && isPercentType) {
                if (this.decimal <= -1) {
                    value = bigDecimalValue.toPlainString();
                } else {
                    value = String.format("%,." + this.decimal + "f%%", bigDecimalValue.movePointRight(2).doubleValue());
                }
            } else if (hasThousandthPercentile && !isPercentType) {
                if (this.decimal <= -1) {
                    value = bigDecimalValue.toPlainString();
                } else {
                    value = String.format("%,." + this.decimal + "f", dValue);
                }
            } else if (!hasThousandthPercentile && isPercentType) {
                if (this.decimal <= -1) {
                    value = bigDecimalValue.toPlainString();
                } else {
                    value = String.format("%." + this.decimal + "f%%", bigDecimalValue.movePointRight(2).doubleValue());
                }
            } else {
                if (this.decimal <= -1) {
                    value = bigDecimalValue.toPlainString();
                } else {
                    value = String.format("%." + this.decimal + "f", dValue);
                }
            }

            /**
             * 如果不是敏捷数据场景使用，才执行以下逻辑，用于向下兼容以前场景：
             * 1. 去掉尾零，向下兼容之前PCC应用的报工任务卡要求
             * 2. FIXME 录入型签核暂时不支持，故出现的百分比录入型字段被隐藏，仅展示字符。将列的类型设置成LABEL
             */
            if (!isAgileData) {
                value = MathUtil.stripTrailingZerosInDoubleString(value);
                // FIXME 录入型签核暂时不支持，故出现的百分比录入型字段被隐藏，仅展示字符
                this.setType("LABEL");
            }
        } else if ("object".equalsIgnoreCase(dataType)) {
            if ("FORM_OPERATION_EDITOR".equalsIgnoreCase(type)) {
                if ("task_member_info".equals(bizDataSchema)) {
                    List<Map<String, Object>> opertaionValueList = (List<Map<String, Object>>) bizData.get(bizDataSchema);
                    for (Object map : opertaionValueList) {
                        if (map instanceof HashMap) {
                            value = value + ((HashMap<?, ?>) map).get("executor_name") + ",";
                        }
                    }
                    if (StringUtils.hasLength(value)) {
                        value = value.substring(0, value.length() - 1);
                    } else {
                        value = "-";
                    }
                } else {
                    List<Map<String, Object>> opertaionValueList = (List<Map<String, Object>>) bizData.get(bizDataSchema);
                    List<String> values = new ArrayList<>();
                    String commonKeyName = bizDataSchema.substring(0, bizDataSchema.indexOf("info"));
                    opertaionValueList.forEach(opeMap -> {
                        String noKey = commonKeyName + "no";
                        String nameKey = commonKeyName + "name";
                        String displayValue = (String) opeMap.get(noKey) + " " + (String) opeMap.get(nameKey);
                        values.add(displayValue);
                    });
                    value = String.join(";", values);
                }
            } else {
                value = VALUE_EMPTY_STRING_SYMBOL;
            }
            this.setType("LABEL");
        } else if (type.toUpperCase().contains("UPLOAD")) {
            // FIXME 录入型签核暂时不支持，故出现的附件上传字段被隐藏，仅展示字符
            value = VALUE_EMPTY_STRING_SYMBOL;
            this.setType("LABEL");
        } else if ("DYNAMIC_GRAPH_VIEWER".equalsIgnoreCase(type)) {
            Object dataObj = bizData.get(bizDataSchema);
            Map<String, Object> data;
            if (dataObj instanceof List) {
                List tempList = (List) dataObj;
                if (tempList.isEmpty()) {
                    data = Collections.emptyMap();
                } else {
                    // FIXME 先以第一条数据来做，因为数据可能是列表。后续若出问题再改做法。
                    data = (Map<String, Object>) tempList.get(0);
                }
            } else if (dataObj instanceof Map) {
                data = (Map<String, Object>) dataObj;
            } else {
                data = Collections.emptyMap();
            }

            // 对图纸字段值的判断
            // FIXME 应该要知道哪些字段的值组合的情况才认为字段为空，所以先以graph_no来判断。后续若出问题再改做法。
            /**
             * 示例JSON数据如下：
             * graph_info: [{
             * cad_url: ""
             * cad_url_3d: ""
             * graph_no: ""
             * graph_no_version: ""
             * online_drawing_format: "1"
             * thumbnail: ""}]
             */
            String graphNo = String.valueOf(data.getOrDefault("graph_no", ""));
            if (StringUtils.hasLength(graphNo)) {
                value = graphNo;
            } else {
                value = VALUE_EMPTY_STRING_SYMBOL;
            }
        } else {
            value = String.valueOf(bizData.get(bizDataSchema));
        }

        return null == value || value.trim().isEmpty() ? VALUE_EMPTY_STRING_SYMBOL : value;
    }

    /**
     * 根据栏位编辑为SELECT的类型，从栏位定义的editor中获取对应值的外显标题。
     * 栏位编辑时的类型。本方法只对SELECT生效，其他类型时返回空字符串
     * FIXME 临时做法，先这样解析，但应该有漏洞
     * 示例JSON数据如下：
     * "editor": {
     * "id": "a05d3f16-6f13-4372-807d-f8ecb1b833e3",
     * "type": "SELECT",
     * "schema": "is_approve",
     * "options": [
     * {
     * "title": "需签核",
     * "value": "true"
     * },
     * {
     * "title": "无需签核",
     * "value": "false"
     * }
     * ]
     * }
     *
     * @param column       栏位对象
     * @param bizDataValue 业务数据值
     * @return
     */
    private String getSelectFieldOptionTitle(UiBotTableColumn column, String bizDataValue) {
        UiBotTableColumnEditor editor = column.getEditor();
        if (null == editor || !editor.getSchema().equalsIgnoreCase(column.getSchema()) ||
                !"SELECT".equalsIgnoreCase(editor.getType())) {
            return VALUE_EMPTY_SYMBOL;
        }

        if (null == editor.getOptions()) {
            return VALUE_EMPTY_SYMBOL;
        }

        for (UiBotTableColumnEditorOption option : editor.getOptions()) {
            String optionValue = String.valueOf(option.getValue());
            if (optionValue.equalsIgnoreCase(bizDataValue)) {
                return String.valueOf(option.getTitle());
            }
        }
        return VALUE_EMPTY_SYMBOL;
    }

    public UiBotTableColumn(String headerName) {
        this.headerName = headerName;
    }

    public static UiBotTableColumn createEmpty() {
        return new UiBotTableColumn();
    }

    public Map<String, Object> parseExecuteContextToRawData(UiBotTableColumn column, UiBotExecuteContext executeContext) {
        Map<String, Object> rawData = new HashMap<>(8);
        if (column.getOperations() != null && column.getOperations().size() > 0) {
            JSONObject operation = column.getOperations().getJSONObject(0);
            rawData.put("dataSourceSet",
                    operation.getJSONObject("openWindowDefine").getJSONObject("allAction").getJSONObject("dataSourceSet"));
            rawData.put("defaultShow",
                    operation.getJSONObject("openWindowDefine").getJSONObject("allAction").getBooleanValue("defaultShow"));
            JSONObject executeContextObj =
                    operation.getJSONObject("openWindowDefine").getJSONObject("allAction").getJSONObject("executeContext");
            if (executeContextObj != null) {
                rawData.put("executeContext", executeContextObj);
            } else {
                rawData.put("executeContext", executeContext);
            }
            rawData.put("multipleSelect",
                    operation.getJSONObject("openWindowDefine").getBooleanValue("multipleSelect"));
            rawData.put("selectedFirstRow",
                    operation.getJSONObject("openWindowDefine").getBooleanValue("selectedFirstRow"));
            rawData.put("queryTagSuffix", "DIALOG");
            rawData.put("buttons",
                    operation.getJSONObject("openWindowDefine").getJSONArray("buttons"));
        }
        Map<String, Object> parameter = new HashMap<>(8);
        parameter.put("", "");
        rawData.put("parameter", parameter);
        return rawData;
    }

}