package com.digiwin.mobile.mobileuibot.proxy.uibot.pcservice.activity.analyzer;

import com.digiwin.mobile.mobileuibot.api.ApiRequest;
import com.digiwin.mobile.mobileuibot.common.context.AppRequestContext;
import com.digiwin.mobile.mobileuibot.common.exception.ServiceException;
import com.digiwin.mobile.mobileuibot.common.http.model.AthenaBasicHttpRequestHead;
import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.core.strategy.modelbuild.pattern.PatternTaskInfo;
import com.digiwin.mobile.mobileuibot.designer.uibot.UiBotDesignerConfig;
import com.digiwin.mobile.mobileuibot.proxy.DigiwinAthenaApiRequest;
import com.digiwin.mobile.mobileuibot.proxy.atmc.model.DigiwinAtmcActivityData;
import com.digiwin.mobile.mobileuibot.proxy.atmc.model.DigiwinAtmcProjectData;
import com.digiwin.mobile.mobileuibot.proxy.atmc.model.DigiwinAtmcTaskWithBacklogData;
import com.digiwin.mobile.mobileuibot.proxy.atmc.service.DigiwinAtmcProxyService;
import com.digiwin.mobile.mobileuibot.proxy.knowledgemaps.model.response.KnowledgeMapsActivityDefinition;
import com.digiwin.mobile.mobileuibot.proxy.knowledgemaps.service.DigiwinKnowledgeMapsProxyService;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.*;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.operation.PcUiBotTmOperation;
import com.digiwin.mobile.mobileuibot.proxy.uibot.pcservice.activity.service.NewTaskSubmitService;
import com.digiwin.mobile.mobileuibot.proxy.uibot.pcservice.activity.service.ProjectDataSourceAnalysisService;
import com.digiwin.mobile.mobileuibot.proxy.uibot.pcservice.activity.service.TmOtherInfoService;
import com.digiwin.mobile.mobileuibot.proxy.uibot.pcservice.activity.service.TmPageOperationService;
import com.digiwin.mobile.mobileuibot.proxy.uibot.pcservice.util.PcUiBotAttachmentUtil;
import com.digiwin.mobile.mobileuibot.proxy.uibot.pcservice.util.PcUiBotConvertTmActivityUtils;
import com.digiwin.mobile.mobileuibot.proxy.uibot.pcservice.util.PcUiBotSubmitActionUtil;
import com.fasterxml.jackson.core.type.TypeReference;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;

import java.text.MessageFormat;
import java.util.*;


/**
 * <p>功能描述：项目页面定义分析器</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: ProjectPageDefineAnalyzer.java
 * @Author: wangjwc
 * @Date: created at 2024/12/3 19:57
 */
@Service
public class ProjectPageDefineAnalyzer extends AbstractPageDefineAnalyzer<PcUiBotTaskPageDefine> {

    @Autowired
    private DigiwinKnowledgeMapsProxyService digiwinKnowledgeMapsProxyService;

    @Autowired
    private DigiwinAtmcProxyService digiwinAtmcProxyService;

    @Autowired
    private ProjectDataSourceAnalysisService tmProjectDataSourceService;

    @Autowired
    private TmPageOperationService tmPageOperationService;

    @Autowired
    private TmOtherInfoService tmOtherInfoService;

    @Autowired
    private NewTaskSubmitService newTaskSubmitService;

    public List<PcUiBotTaskPageDefine> analysis(PcUiBotExecuteContext executeContext, String projectId, ApiRequest apiRequest) {
        DigiwinAtmcActivityData activityData = digiwinAtmcProxyService.getProjectData(projectId, executeContext);
        if (activityData != null
                && CollectionUtils.isNotEmpty(activityData.getTasks())
                && StringUtils.hasText(activityData.getTasks().get(0).getProxyToken())) {
            executeContext.setProxyToken(activityData.getTasks().get(0).getProxyToken());
            AppRequestContext.getContextEntity().setProxyToken(executeContext.getProxyToken());
        }
        if (activityData == null || CollectionUtils.isEmpty(activityData.getTasks())) {
            return new ArrayList<>();
        }
        executeContext.appendTaskData(activityData);

        //1.获取到活动定义
        PcUiBotTmActivity tmActivity = this.getActivityAction(apiRequest, executeContext);
        if (tmActivity.getDataSources() == null) {
            String message = MessageFormat.format("任务定义不完整，没有相关的数据源定义,tmActivity：{0}", tmActivity.getActivityId());
            throw new ServiceException(message);
        }
        //2、完善上下文信息
        executeContext.appendTmActivityInfo(tmActivity);
        //3、
        List<PcUiBotTaskPageDefine> pageDefineList = super.createPageDefines(executeContext, tmActivity);
        if (null == pageDefineList) {
            pageDefineList = new ArrayList<>();
        }
        // 添加“已终止”，"封存中"页签
        appendInternalPageDefine(executeContext, tmActivity, pageDefineList);
        return this.filterNoResultQuery(pageDefineList);
    }

    public PcUiBotTmActivity getActivityAction(ApiRequest apiRequest, PcUiBotExecuteContext executeContext) {
        KnowledgeMapsActivityDefinition rawActivityDefinition = Optional.ofNullable(apiRequest).map(ApiRequest::getPatternTaskInfo).map(PatternTaskInfo::getUiBotDesignerConfig).map(UiBotDesignerConfig::getRawTaskActivityDefinition).orElse(null);
        Map<String, String> uriVariables = new HashMap<>();
        uriVariables.put("taskId", executeContext.getTmProjectId());
        uriVariables.put("activityId", executeContext.getTmActivityId());
        uriVariables.put("pageCode", executeContext.getPageCode());
        DigiwinAthenaApiRequest athenaApiRequest =
                DigiwinAthenaApiRequest.builder()
                        .requestHead(new AthenaBasicHttpRequestHead("",
                                executeContext.getLocale(), executeContext.getAuthoredUser().getToken(),
                                executeContext.getTenantId()))
                        .uriVariables(uriVariables)
                        .build();
        rawActivityDefinition = rawActivityDefinition != null ? rawActivityDefinition :
                this.digiwinKnowledgeMapsProxyService.getActivityDefinition(athenaApiRequest);
        PcUiBotTmActivity tmActivity = JsonUtil.objectToJavaObject(rawActivityDefinition, PcUiBotTmActivity.class);
        // 重新处理数据状态数据
        this.handleDataState(tmActivity);
        // 数据状态dataState下layout组件中的sumitAction属性提取到外层，统一构建
        this.buildDataStateSubmitAction(tmActivity);
        PcUiBotConvertTmActivityUtils.convert(executeContext.getPageCode(), executeContext.getLocale(), tmActivity);
        return tmActivity;
    }

    /**
     * 添加标题的汇总数据
     *
     * @param actionVOList
     */
    private List<PcUiBotTaskPageDefine> filterNoResultQuery(List<PcUiBotTaskPageDefine> actionVOList) {
        if (actionVOList == null) {
            return actionVOList;
        }
        List<PcUiBotTaskPageDefine> filter = new ArrayList<>();
        for (PcUiBotTaskPageDefine pageDefine : actionVOList) {
            if (pageDefine.getDataSourceSet() != null) {
                pageDefine.setType("async");
            }
            filter.add(pageDefine);
        }
        return filter;
    }

    /**
     * 创建一个新的页面
     *
     * @return
     */
    @Override
    protected PcUiBotTaskPageDefine createNewPageDefine() {
        return new PcUiBotTaskPageDefine();
    }

    /**
     * 重新为每个查询定义独立的上下文
     *
     * @param oldExecuteContext
     * @return
     */
    protected PcUiBotExecuteContext createNewContext(PcUiBotExecuteContext oldExecuteContext) {
        //任务上的每个查询(待处理、已完成等)都需要是新的上下文
        return oldExecuteContext.clone();
    }

    @Override
    protected void addDefaultConfigInfo(PcUiBotExecuteContext executeContext, PcUiBotTmActivity tmActivity) {
        PcUiBotAttachmentUtil.addDefaultAttachmentDataSourceProcess(executeContext, tmActivity);
        PcUiBotAttachmentUtil.appendEffectAfterSubmitAction(executeContext, tmActivity);
    }

    /**
     * 根据任务上的变量来解析基本信息
     *
     * @param pageDefine
     * @param executeContext
     * @param tmPage
     */
    @Override
    protected void analysisBaseInfo(PcUiBotExecuteContext executeContext, PcUiBotTaskPageDefine pageDefine, PcUiBotTmActivity tmActivity, PcUiBotTmPage tmPage) {
        if (executeContext.getTaskWithBacklogDataList() == null
                || executeContext.getTaskWithBacklogDataList().size() == 0) {
            return;
        }
        pageDefine.setPrePerformer(tmPage.getPrePerformer());
        pageDefine.setSubjectMatter(tmPage.getSubjectMatter());
        pageDefine.setSubjectMatterTargets(tmPage.getSubjectMatterTargets());
        pageDefine.setSubjectMatterProperties(tmPage.getSubjectMatterProperties());
        pageDefine.setRoleAttention(tmPage.getRoleAttention());
        pageDefine.setStartApproveActivityName(tmActivity.getStartApproveActivityName());
        pageDefine.setStartApproveActivity(tmActivity.getStartApproveActivity());
        pageDefine.setVersion(tmActivity.isFlowEngine() ? "1.0" : "2.0");

        //xuxx add 增加任务tags，用于存放任务所涉及到的领域、客制的tag
        pageDefine.setDomain(tmPage.getDomain());

        //如果任务已完成，设置相关的信息
        initFinishedInfo(executeContext, pageDefine, tmPage);

        //xuxx add 标准任务详情 补充字段
        pageDefine.setDataKeyProperties(tmPage.getDataKeyProperties());
        pageDefine.setSubjectExpect(tmPage.getSubjectExpect());
        pageDefine.setSubjectShortfall(tmPage.getSubjectShortfall());
        pageDefine.setDataStateCount(tmPage.getDataStates() == null ? 0 : tmPage.getDataStates().size());

        tmOtherInfoService.analysis(pageDefine, tmPage, executeContext.getTaskWithBacklogDataList().get(0).getBpmData());

        // 模型驱动的任务类型
        pageDefine.setBindForm(tmActivity.getBindForm());
        pageDefine.setFieldConfig(tmActivity.getFieldConfig());
    }

    private void initFinishedInfo(PcUiBotExecuteContext executeContext, PcUiBotTaskPageDefine pageDefine, PcUiBotTmPage tmPage) {
        //判断任务已经提交
        if (!org.springframework.util.CollectionUtils.isEmpty(executeContext.getTaskWithBacklogDataList()) && !org.springframework.util.CollectionUtils.isEmpty(executeContext.getTaskWithBacklogDataList().get(0).getBacklog())) {
            String finishedActionId = executeContext.getTaskWithBacklogDataList().get(0).getBacklog().get(0).getFinishedActionId();
            if (!org.springframework.util.StringUtils.isEmpty(finishedActionId)) {
                //任务合并时，如果有多个 taskWithBacklogData ,优先使用未完成的
                for (DigiwinAtmcTaskWithBacklogData withBacklogData : executeContext.getTaskWithBacklogDataList()) {
                    if (org.apache.commons.lang3.StringUtils.isEmpty(withBacklogData.getBacklog().get(0).getFinishedActionId())) {
                        finishedActionId = null;
                        break;
                    }
                }
            }
            //判断是否为空
            if (!org.apache.commons.lang3.StringUtils.isEmpty(finishedActionId)) {
                pageDefine.setFinished(true);
                pageDefine.setFinishedActionId(finishedActionId);
                List<PcUiBotTmAction> submitActions = null;
                if (!org.springframework.util.CollectionUtils.isEmpty(tmPage.getDataStates())) {
                    for (PcUiBotTmDataState dataState : tmPage.getDataStates()) {
                        if (!org.springframework.util.CollectionUtils.isEmpty(dataState.getSubmitActions())) {
                            submitActions = dataState.getSubmitActions();
                            break;
                        }
                    }
                }
                if (submitActions != null) {
                    for (PcUiBotTmAction submitAction : submitActions) {
                        if (Objects.equals(submitAction.getId(), finishedActionId)
                                || Objects.equals(submitAction.getActionId(), finishedActionId)) {
                            if (org.apache.commons.lang3.StringUtils.isEmpty(submitAction.getReturnText())) {
                                pageDefine.setFinishedTitle(submitAction.getTitle());
                            } else {
                                pageDefine.setFinishedTitle(getMessageWithLanguage(submitAction.getReturnText(), executeContext.getLocale()));
                            }
                            break;
                        }
                    }
                }
            } else {
                boolean isFinished = !(executeContext.getTaskWithBacklogDataList().stream().filter(x -> Objects.equals(x.getBacklog().get(0).isClosed(), false)).count() > 0);
                pageDefine.setFinished(isFinished);
            }


            if (org.apache.commons.lang3.StringUtils.isEmpty(pageDefine.getFinishedTitle())) {
                if (Objects.equals(finishedActionId, "athena_bpm_reassign")) {
                    pageDefine.setFinishedTitle("转派");
                } else if (Objects.equals(finishedActionId, "athena_bpm_cancel")) {
                    pageDefine.setFinishedTitle("撤销");
                } else if (Objects.equals(finishedActionId, PcUiBotConstants.UIBOT_RETRIEVE)) {
                    pageDefine.setFinishedTitle("已撤回");
                } else if (Objects.equals(finishedActionId, "uibot_action_id_finished")) {
                    pageDefine.setFinishedTitle("已结束");
                } else if (Objects.equals(finishedActionId, "athena_overdue_task_terminate")) {
                    pageDefine.setFinishedTitle("逾期终止");
                }
            }
        }
    }

    public static String getMessageWithLanguage(String message, String language) {
        if (org.apache.commons.lang3.StringUtils.isBlank(message)) {
            return message;
        }

        try {
            Map<String, String> map = JsonUtil.jsonStringToObject(message, new TypeReference<Map<String, String>>() {
            });
            return map.get(language);
        } catch (Exception ex) {
            return message;
        }
    }

    /**
     * 处理操作的定义
     *
     * @param ？pageDefine
     * @param operations
     */
    @Override
    protected void analysisOperators(PcUiBotExecuteContext executeContext, PcUiBotTaskPageDefine pageDefine, PcUiBotTmActivity tmActivity, List<PcUiBotTmOperation> operations) {
        //4.1 解析 operations
        tmPageOperationService.createOperations(executeContext, pageDefine, tmActivity, operations);
    }

    /**
     * 处理查询的逻辑
     *
     * @param executeContext
     * @param pageDefine
     * @param tmActivity
     * @param tmDataState
     * @param dataSources
     * @param tmDataFilter
     * @return
     */
    @Override
    protected void analysisDataSource(PcUiBotExecuteContext executeContext, PcUiBotTaskPageDefine pageDefine,
                                      PcUiBotTmActivity tmActivity, PcUiBotTmDataState tmDataState,
                                      Map<String, PcUiBotTmQueryAction> dataSources, PcUiBotTmDataFilter tmDataFilter) {
        //如果指定来数据源，则解析指定数据源，否则获取全部数据源
        PcUiBotDataSourceSetDTO dataSourceSet = tmProjectDataSourceService.analysis(executeContext, tmActivity, dataSources, tmDataFilter);
        pageDefine.setDataSourceSet(dataSourceSet);
        if (tmDataFilter != null) {
            pageDefine.setQueryTitle(tmDataFilter.getTitle());
            pageDefine.setRowSizeType(tmDataFilter.getRowSizeType());
        }
    }

    @Override
    protected void analysisAllDataSource(PcUiBotExecuteContext executeContext, PcUiBotTaskPageDefine pageDefine, PcUiBotTmActivity tmActivity, PcUiBotTmDataState tmDataState, PcUiBotTmDataFilter tmDataFilter) {
        PcUiBotDataSourceSetDTO dataSourceSet = tmProjectDataSourceService.analysis(executeContext, tmActivity, tmActivity.getDataSources(), tmDataFilter);
        pageDefine.setAllDataSourceSet(dataSourceSet);
    }

    /**
     *  项目SubmitActions的分析
     * @param executeContext
     * @param pageDefine
     * @param tmActivity
     * @param dataState
     * @param submitActions
     */
    @Override
    protected void analysisSubmitActions(PcUiBotExecuteContext executeContext, PcUiBotTaskPageDefine pageDefine, PcUiBotTmActivity tmActivity, PcUiBotTmDataState dataState, List<PcUiBotTmAction> submitActions) {
        String executeType = tmActivity.getExecuteType();
        DigiwinAtmcProjectData projectData = executeContext.getProjectData();
        // 项目数据和项目卡片状态为3的不处理
        if (projectData == null || Objects.equals(projectData.getProjectCardState(), 3)) {
            return;
        }

        // 自动类型项目新增处理“退回”、“跨BK撤回”
        if ("auto".equals(executeType)) {
            if (executeContext.getTaskWithBacklogData() == null || CollectionUtils.isEmpty(executeContext.getTaskWithBacklogData().getBacklog())) {
                //兼容旧的没有backlog的自动型任务，旧的不渲染submitAction
                return;
            }
            for (PcUiBotTmAction tmAction : submitActions) {
                if (!PcUiBotSubmitActionUtil.isReturnOrTaskWithdrawAction(tmAction)) {
                    return;
                }
            }
        }
        // 创建按钮权限的判断在action/show
        newTaskSubmitService.createTaskSubmit(executeContext, tmActivity, dataState, pageDefine);
        // 设置所有的SubmitType的isBatch为true
        setSubmitTypeBatch(pageDefine.getSubmitActions());
    }

    /**
     *  设置项目的SubmitType中的isBatch为true
     * @param actionList
     */
    private void setSubmitTypeBatch(List<PcUiBotSubmitAction> actionList) {
        if (CollectionUtils.isNotEmpty(actionList)) {
            for (PcUiBotSubmitAction submitAction : actionList) {
                PcUiBotSubmitType submitType = submitAction.getSubmitType();
                if (submitType != null) {
                    submitType.setIsBatch(true);
                }
                // 按钮中attachAction也要判断
                setSubmitTypeBatch(submitAction.getAttachActions());
            }
        }
    }
}
