package com.digiwin.mobile.mobileuibot.proxy.uibot.pcservice.activity.parser;

import com.digiwin.mobile.mobileuibot.common.exception.ServiceException;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.*;
import com.digiwin.mobile.mobileuibot.proxy.uibot.pcservice.activity.service.*;
import com.google.common.collect.Lists;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.Collections;
import java.util.List;
import java.util.Optional;

/**
 * <p>功能描述：组合按钮解析器；</p>
 * <p>
 * 抽象类用于递归解析combine按钮；
 * 通过init方法初始化必要参数；
 * 通过继承的方式重写buildSubmitAction或buildCombineSubmitAction方法，实现不同逻辑；
 * <p>
 * 注意：每个子类需定义为非单例模式@Scope(ConfigurableBeanFactory.SCOPE_PROTOTYPE)
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: AbstractCombineSubmitActionParser.java
 * @Author: wangjwc
 * @Date: created at 2023/7/13 13:04
 */
public abstract class AbstractCombineSubmitActionParser {

    @Lazy
    @Autowired
    protected SubmitActionCreateService submitActionCreateService;

    @Lazy
    @Autowired
    protected TaskSubmitActionCreateService taskSubmitActionCreateService;

    @Lazy
    @Autowired
    protected NewTaskSubmitActionCreateService newTaskSubmitActionCreateService;

    @Lazy
    @Autowired
    protected TmApprovalService tmApprovalService;

    @Lazy
    @Autowired
    protected ActionDataSourceAnalysisService tmDataSourceService;

    protected PcUiBotTmDataState tmDataState;

    protected PcUiBotPageDefine pageDefine;

    /**
     * 初始化参数
     *
     * @param tmDataState
     */
    protected void init(PcUiBotTmDataState tmDataState) {
        this.tmDataState = tmDataState;
    }

    /**
     * 初始化参数
     *
     * @param tmDataState
     * @param pageDefine
     */
    protected void init(PcUiBotTmDataState tmDataState, PcUiBotPageDefine pageDefine) {
        this.tmDataState = tmDataState;
        this.pageDefine = pageDefine;
    }

    /**
     * 解析
     *
     * @param tmActionList
     * @return
     */
    public List<PcUiBotSubmitAction> parse(List<PcUiBotTmAction> tmActionList) {
        tmActionList = Optional.ofNullable(tmActionList).orElse(Lists.newArrayList());
        List<PcUiBotSubmitAction> submitActionList = Lists.newArrayList();
        for (PcUiBotTmAction tmAction : tmActionList) {
            PcUiBotSubmitAction submitAction;
            // 当按钮类型为COMBINE时，combineActions若为空，则抛出异常
            if (PcUiBotActivityConstants.TmActionType.COMBINE.equals(tmAction.getType())
                    || !CollectionUtils.isEmpty(tmAction.getCombineActions())) {
                List<PcUiBotTmAction> combineActions = tmAction.getCombineActions();
                if (CollectionUtils.isEmpty(combineActions)) {
                    throw new ServiceException("组合按钮缺少必要配置");
                }
                // 递归
                List<PcUiBotSubmitAction> combineSubmitActionList = this.parse(combineActions);
                submitAction = buildCombineSubmitAction(tmAction, combineSubmitActionList);
                tmAction.setType(PcUiBotActivityConstants.TmActionType.COMBINE);
            } else {
                submitAction = buildSubmitAction(tmAction);
            }
            // 回填按钮类型，用于判断是否填入edit_type到pageData，详见GeneralPageTemplate#appendEditTypeParam
            if (submitAction != null) {
                submitAction.setType(tmAction.getType());
            }
            submitActionList.add(submitAction);
        }
        // 移除所有空元素
        submitActionList.removeAll(Collections.singletonList(null));
        return submitActionList;
    }

    /**
     * 构建非组合操作按钮
     *
     * @param tmAction
     * @return
     */
    protected abstract PcUiBotSubmitAction buildSubmitAction(PcUiBotTmAction tmAction);

    /**
     * 构建组合操作按钮
     *
     * @param tmAction
     * @param combineSubmitActionList
     * @return
     */
    protected PcUiBotSubmitAction buildCombineSubmitAction(PcUiBotTmAction tmAction,
                                                           List<PcUiBotSubmitAction> combineSubmitActionList) {
        if (CollectionUtils.isEmpty(combineSubmitActionList)) {
            return null;
        }
        PcUiBotSubmitAction submitAction = new PcUiBotSubmitAction();
        submitAction.setCombineActions(combineSubmitActionList);
        submitAction.setActionType(tmAction.getActionType());
        submitAction.setDefaultAction(tmAction.isDefaultAction());
        submitAction.setUuid(tmAction.getUuid());
        submitAction.setBtnId(tmAction.getBtnId());
        submitAction.setBtnType(tmAction.getBtnType());
        submitAction.setJumpAction(tmAction.getJumpAction());
        submitAction.setTitle(tmAction.getTitle());
        submitAction.setId(tmAction.getId());
        submitAction.setSubmitType(tmAction.getSubmitType());
        submitAction.setCategory(tmAction.getType());
        if (tmAction.getIgnoreRule() != null) {
            submitAction.setIgnoreRule(tmAction.getIgnoreRule());
        }
        return submitAction;
    }

    /**
     * 设置submitType
     *
     * @param submitAction
     * @param mainDataSourceName
     */
    protected void setSubmitType(PcUiBotSubmitAction submitAction, String mainDataSourceName) {
        if (submitAction == null) {
            return;
        }
        if (submitAction.getSubmitType() == null) {
            if (tmDataState.getSubmitType() != null) {
                submitAction.setSubmitType(tmDataState.getSubmitType());
            } else {
                PcUiBotSubmitType submitType = new PcUiBotSubmitType();
                submitType.setIsBatch(Boolean.FALSE);
                submitType.setSchema(mainDataSourceName);
                submitAction.setSubmitType(submitType);
                tmDataState.setSubmitType(submitType);
            }
        }
    }

}
