package com.digiwin.mobile.mobileuibot.proxy.uibot.pcservice.activity.service;

import cn.hutool.core.map.MapUtil;
import com.digiwin.mobile.mobileuibot.common.context.AppContext;
import com.digiwin.mobile.mobileuibot.common.context.AppRequestContext;
import com.digiwin.mobile.mobileuibot.common.http.model.AthenaBasicHttpRequestHead;
import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.core.pagesetting.PageSettingIdPresetEnum;
import com.digiwin.mobile.mobileuibot.designer.uibot.rule.DesignerTrigger;
import com.digiwin.mobile.mobileuibot.designer.uibot.service.UiBotDesignerService;
import com.digiwin.mobile.mobileuibot.mongodb.designerconfig.MobileDesignerConfigRepository;
import com.digiwin.mobile.mobileuibot.proxy.DigiwinAthenaApiRequest;
import com.digiwin.mobile.mobileuibot.proxy.knowledgemaps.model.response.KnowledgeMapsRuleQueryResult;
import com.digiwin.mobile.mobileuibot.proxy.knowledgemaps.service.DigiwinKnowledgeMapsProxyService;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.*;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.rule.UiBotRule;
import com.digiwin.mobile.mobileuibot.proxy.uibot.pcservice.util.PcUiBotSubmitActionUtil;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.Lists;
import lombok.AllArgsConstructor;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 规则的统一处理
 */
@Service
public class PcRuleService {

    private static final String PAGE_SUFFIX = "page1";

    @Autowired
    private ShowFieldService showFieldService;

    @Autowired
    protected DigiwinKnowledgeMapsProxyService digiwinKnowledgeMapsProxyService;

    @Autowired
    private MobileDesignerConfigRepository mobileDesignerConfigRepository;

    /**
     * 规则的后置处理
     *
     * @param processType
     */
    public void rulePostProcessor(ProcessType processType, UiBotModel dynamicForm, QueryResultSet queryResultSet) {
        // 目前PAGE_BUILDER和DSL_BUILDER都会调用
        showFieldService.removeWordHiddenRule(dynamicForm.getRules());
        dealHooksRule(dynamicForm);
        // 手动添加项目卡checkbox按钮规则的禁用
        addProjectDetailRule(dynamicForm.getExecuteContext(), queryResultSet, dynamicForm.getActions(), dynamicForm.getRules());
    }

    /**
     * 删除可编辑规则
     *
     * @param dynamicForm
     */
    public void removeEditRule(UiBotModel dynamicForm) {
        String pageCode = Optional.of(dynamicForm)
                .map(UiBotModel::getExecuteContext)
                .map(ex -> ex.getString("pageCode"))
                .orElse(null);
        if (UiBotDesignerService.PAGECODE_BROWSE_PAGE.equals(pageCode)) {
            List<UiBotRule> rules = dynamicForm.getRules();
            removeEditRule(rules);
        }
    }

    public void removeEditRule(List<UiBotRule> rules) {
        rules.removeIf(rule -> Objects.equals("EDIT", rule.getScope()));
    }


    /**
     * 将hooks规则放入mechnismConfig中并在rules中删除
     *
     * @param dynamicForm
     */
    private void dealHooksRule(UiBotModel dynamicForm) {
//        List<Map<String, Object>> rules = dynamicForm.getRules();
//        if (CollectionUtils.isEmpty(rules)) {
//            return;
//        }
//        // 复制规则
//        List<Map<String, Object>> ruleCopyList = JsonUtils.jsonToObject(JsonUtils.objectToString(rules), List.class);
//        // 机制规则
//        List<Map<String, Object>> apiTriggerRules = new ArrayList<>();
//        // 处理特殊规则
//        for (Map<String, Object> rule : ruleCopyList) {
//            if (UiBotConstants.RuleKeyType.API_TRIGGER.equals(rule.get("key"))) {
//                apiTriggerRules.add(rule);
//                // 在原来的rules中删除apiTrigger
//                rules.remove(rule);
//            }
//        }
//        // 已有的mechnismConfig
//        Map<String, Object> mechnismConfigMap = new HashMap<>();
//        if (null != dynamicForm.getMechnismConfig() && dynamicForm.getMechnismConfig().containsKey("rules")) {
//            mechnismConfigMap = dynamicForm.getMechnismConfig();
//        }
//        // 已有的机制rules
//        List<Map<String, Object>> mechnismConfigRules = new ArrayList<>();
//        if (mechnismConfigMap.containsKey("rules") && null != mechnismConfigMap.get("rules")) {
//            mechnismConfigRules = JsonUtils.jsonToObject(JsonUtils.objectToString(mechnismConfigMap.get("rules")), List.class);
//        }
//        mechnismConfigRules.addAll(apiTriggerRules);
//
//        mechnismConfigMap.put("rules", mechnismConfigRules);
//        dynamicForm.setMechnismConfig(mechnismConfigMap);
    }

    /**
     * 添加项目详情的手动规则,checkbox设置禁选规则
     */
    private void addProjectDetailRule(UiBotExecuteContext executeContext, QueryResultSet queryResultSet, List<UiBotAction> actions, List<UiBotRule> rules) {
        if (executeContext == null) {
            return;
        }
        String pageCode = executeContext.getString("pageCode");
        List<PcUiBotSubmitAction> submitActions = JsonUtil.objectToJavaObject(actions, new TypeReference<List<PcUiBotSubmitAction>>() {
        });
        // 是项目卡且判断提交按钮是有批量提交参数
        if (UiBotDesignerService.PAGECODE_PROJECT_DETAIL.equals(pageCode) && PcUiBotSubmitActionUtil.hasSubmitTypeBatch(submitActions)) {
            // 撤回按钮才手动添加规则
            List<PcUiBotSubmitAction> actionList = PcUiBotSubmitActionUtil.onlyKeepWithdrawAction(submitActions);
            if (!CollectionUtils.isEmpty(actionList)) {
                UiBotRule rule = new UiBotRule();
                rule.setSchema("uibot_checked");
                // 获取数据源的第一个名称
                rule.setPath(queryResultSet.getDataSourceNameFromApiResp());
                rule.setKey("disabled");
                DesignerTrigger trigger = new DesignerTrigger();
                trigger.setCondition("data.activity__backLog__data.projectState == 3");
                rule.setTrigger(trigger);
                rules.add(rule);
            }
        }
    }

    /**
     * 调用处理器类型
     */
    @AllArgsConstructor
    public enum ProcessType {

        PAGE_BUILDER("PageBuilderBase", "构造页面基础类"),
        DSL_BUILDER("DslPageBuilderBase", "DSL构造页面基础类");

        private String type;
        private String desc;
    }

    /**
     * 是否存在指定规则
     *
     * @param rules  所有规则
     * @param schema 作用字段
     * @param key    规则类型
     * @return
     */
    public static boolean exist(List<Map<String, Object>> rules, String schema, String key) {
        for (Map<String, Object> rule : rules) {
            String keyFromRule = MapUtil.getStr(rule, "key", "");
            String schemaFromRule = MapUtil.getStr(rule, "schema", "");
            if (keyFromRule.equals(key) && schemaFromRule.equals(schema)) {
                return true;
            }
        }
        return false;
    }

    /**
     * 获取平台配置规则
     *
     * @param tmActivityId
     * @param locale
     * @param iamUserToken
     * @param tenantId
     * @return
     */
    public List<UiBotRule> buildRuleOfKm(String tmActivityId, String locale, String iamUserToken, String tenantId) {
        // 调用知识图谱的API，获取任务规则配置
        Map<String, Object> map = new HashMap<>();
        map.put("include", Lists.newArrayList("contentMobile"));
        DigiwinAthenaApiRequest athenaApiRequest =
                DigiwinAthenaApiRequest.builder()
                        .requestHead(new AthenaBasicHttpRequestHead("", locale, iamUserToken, tenantId))
                        .bodyPayloadMap("activityId", tmActivityId)
                        .bodyPayloadMap("count", -1)
                        .bodyPayloadMap("features", map)
                        .build();
        return this.filterRules(this.buildRule(this.digiwinKnowledgeMapsProxyService.getRuleQueryResult(athenaApiRequest)));
    }

    public List<UiBotRule> buildRule(KnowledgeMapsRuleQueryResult ruleQueryResult) {
        if (null == ruleQueryResult || CollectionUtils.isEmpty(ruleQueryResult.getRules())) {
            return new ArrayList<>();
        }
        return ruleQueryResult.getRules().stream()
                .filter(rule -> !CollectionUtils.isEmpty(rule.getContentMobile()))
                .map(rule -> {
                    UiBotRule uiBotRule = JsonUtil.objectToJavaObject(rule.getContentMobile(), UiBotRule.class);
                    uiBotRule.setName(rule.getName());

                    // 规则trigger 中如果apiPrefixType有值，apiUrl需要拼接上对应前缀
                    Optional<String> apiPrefixType = Optional.ofNullable(uiBotRule.getTrigger()).map(DesignerTrigger::getApiPrefixType);
                    if (apiPrefixType.isPresent() && StringUtils.isNotEmpty(apiPrefixType.get().trim())) {
                        String prefixUrl = AppContext.getUrl(apiPrefixType.get());
                        Optional<String> apiUrl = Optional.ofNullable(uiBotRule.getTrigger()).map(DesignerTrigger::getApiUrl);
                        if (apiUrl.isPresent() && StringUtils.isNotEmpty(apiUrl.get().trim())) {
                            uiBotRule.getTrigger().setApiUrl(prefixUrl + apiUrl.get());
                        }
                    }
                    return uiBotRule;
                })
                .collect(Collectors.toList());
    }

    /**
     * 获取本地配置规则
     *
     * @param tmActivityId
     * @return
     */
    public List<UiBotRule> buildRuleOfLocal(String tmActivityId, String type) {
        return this.filterRules(this.buildRule(JsonUtil.objectToJavaObject(mobileDesignerConfigRepository.selectAllRule(tmActivityId, type), KnowledgeMapsRuleQueryResult.class)));
    }

    /**
     * 根据pageId过滤规则
     *
     * @param uiBotRules
     * @return List<UiBotRule>
     * @author yanfeng
     */
    private List<UiBotRule> filterRules(List<UiBotRule> uiBotRules) {
        List<UiBotRule> rules = new ArrayList<>();
        if (CollectionUtils.isEmpty(uiBotRules)) {
            return rules;
        }
        //获取当前页面ID
        String currentPageId = AppRequestContext.getContextEntity().getPageId();
        /*
         *
         * 因为主页面的pageId，我们是配置在MongoDB中的,还有的任务详情页的pageId是我们写死的，和设计器提供的主页面pageId不一致
         * 导致上下文中存的pageId和rules返回这里面的pageId不一致,故需要这样处理:
         *                                        当前currentPageId是否在 PageSettingIdPresetEnum 枚举中，
         *                                        如果空，走currentPageId和pageId精确匹配
         *                                        如果不为空，走pageId后缀匹配 endsWith("page1")
         */
        PageSettingIdPresetEnum enumPageId = PageSettingIdPresetEnum.getEnumByName(currentPageId);
        uiBotRules.forEach(rule -> {
            String pageId = rule.getPageId();
            //pageId为空,规则也要保留
            if (StringUtils.isEmpty(pageId)) {
                rules.add(rule);
                return;
            }
            //pageId后缀匹配 endsWith("page1")
            if (enumPageId != null) {
                if (pageId.endsWith(PAGE_SUFFIX)) {
                    rules.add(rule);
                }
            } else {
                if (StringUtils.isEmpty(currentPageId) && pageId.endsWith("page1")) {
                    // 主页面多tab异步调用时currentPageId为空
                    rules.add(rule);
                    return;
                }
                //如果不为空，走currentPageId和pageId精确匹配
                if (StringUtils.isNotEmpty(currentPageId) && currentPageId.equalsIgnoreCase(pageId)) {
                    rules.add(rule);
                }
            }
        });
        return rules;
    }
}