package com.digiwin.mobile.mobileuibot.proxy.uibot.pcservice.impl;

import cn.hutool.core.map.MapUtil;
import com.digiwin.mobile.mobileuibot.api.ApiRequest;
import com.digiwin.mobile.mobileuibot.common.http.model.AthenaBasicHttpRequestHead;
import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.proxy.DigiwinAthenaApiRequest;
import com.digiwin.mobile.mobileuibot.proxy.atdm.service.DigiwinAtdmProxyService;
import com.digiwin.mobile.mobileuibot.proxy.knowledgemaps.model.response.KnowledgeMapsDataViewQueryResult;
import com.digiwin.mobile.mobileuibot.proxy.knowledgemaps.service.DigiwinKnowledgeMapsProxyService;
import com.digiwin.mobile.mobileuibot.proxy.uibot.PcUiBotTmPageUIElements;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.*;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.layout.UiBotPageLayout;
import com.digiwin.mobile.mobileuibot.proxy.uibot.pcservice.DataViewService;
import com.digiwin.mobile.mobileuibot.proxy.uibot.pcservice.activity.analyzer.DataViewPageDefineAnalyzer;
import com.digiwin.mobile.mobileuibot.proxy.uibot.pcservice.activity.service.GeneralDataSourceService;
import com.digiwin.mobile.mobileuibot.proxy.uibot.pcservice.builder.DataViewPageBuilder;
import com.digiwin.mobile.mobileuibot.proxy.uibot.pcservice.util.PcUiBotApiMetadataUtil;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * <p>功能描述：数据视图服务实现类</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: DataViewServiceImpl.java
 * @Author: wangjwc
 * @Date: created at 2023/12/26 17:15
 */
@Slf4j
@Service
public class DataViewServiceImpl implements DataViewService {

    private static final String DATA_VIEW_SERVICE_NAME = "modeldriven.commons.eai.dataview.get";

    @Autowired
    private DigiwinKnowledgeMapsProxyService digiwinKnowledgeMapsProxyService;

    @Autowired
    private GeneralDataSourceService generalDataSourceService;

    @Autowired
    private DataViewRawDslRenderService dataViewRawDslRenderService;

    @Autowired
    private DataViewPageDefineAnalyzer dataViewPageDefineAnalyzer;

    @Autowired
    private DigiwinAtdmProxyService digiwinAtdmProxyService;

    @Autowired
    private DataViewPageBuilder dataViewPageBuilder;

    @Override
    public UiBotModel createDynamicForm(PcUiBotDataViewPageDefine dataViewPageDefine, ApiRequest apiRequest) {
        // 解析页面定义
        List<PcUiBotDataViewPageDefine> pageDefines = dataViewPageDefineAnalyzer.analysis(dataViewPageDefine);
        if (CollectionUtils.isEmpty(pageDefines)) {
            return UiBotModel.emptyUiBotModel();
        }
        PcUiBotDataViewPageDefine dataViewPageDefineInner = pageDefines.get(0);
        PcUiBotExecuteContext executeContextInner = dataViewPageDefineInner.getExecuteContext();
        // 获取业务数据
        QueryResultSet queryResultSet = this.queryByPagedefineWithMetaData(executeContextInner, dataViewPageDefineInner);
        // 根据业务数据设置"是否启用分页"
        dataViewPageDefineInner.resetUseHasNext(queryResultSet);
        // 构建当前数据视图下的dataViewQueryList
        this.buildCurrentDataViewQueryList(dataViewPageDefineInner);
        // 创建页面
        return dataViewPageBuilder.createPage(executeContextInner, queryResultSet, dataViewPageDefineInner, apiRequest);
    }

    private void buildCurrentDataViewQueryList(PcUiBotDataViewPageDefine dataViewPageDefineInner) {
        // 构建同步tab时，dataViewQueryDTO使用了什么属性，则构建dataViewQueryDTO中就存放什么属性，为了异步加载中返回当前tab信息
        PcUiBotDataViewQueryDTO dataViewQueryDTO = new PcUiBotDataViewQueryDTO();
        dataViewQueryDTO.setViewCode(dataViewPageDefineInner.getViewCode());
        dataViewQueryDTO.setViewName(dataViewPageDefineInner.getViewName());
        dataViewPageDefineInner.setDataViewQueryList(Lists.newArrayList(dataViewQueryDTO));
    }

    private QueryResultSet queryByPagedefineWithMetaData(PcUiBotExecuteContext executeContext, PcUiBotDataViewPageDefine pageDefine) {
        if (pageDefine.getDataSourceSet() == null) {
            return QueryResultSet.empty();
        }
        Map<String, Object> body = new HashMap<>();
        body.put("dataSourceSetDTO", pageDefine.getDataSourceSet());
        body.put("executeContext", executeContext);
        body.put("parameter", pageDefine.getParameter());
        body.put("settings", pageDefine.getSettings());
        if (pageDefine.getPageInfo() != null) {
            body.put("pageInfo", pageDefine.getPageInfo());
        }
        if (pageDefine.getSortInfo() != null && !pageDefine.getSortInfo().isEmpty()) {
            body.put("sortInfo", pageDefine.getSortInfo());
        }
        if (pageDefine.getSearchInfo() != null && !pageDefine.getSearchInfo().isEmpty()) {
            body.put("searchInfo", pageDefine.getSearchInfo());
        }
        return digiwinAtdmProxyService.queryWithMetaData(executeContext.getLocale(),
                executeContext.getAuthoredUser().getToken(),
                JsonUtil.javaObjectToJsonString(body),
                executeContext.getAuthoredUser().getTenantId());
    }

    @Override
    public void appendNecessaryData(String viewCode, PcUiBotDoubleDocumentPageDefine pageDefine, PcUiBotExecuteContext executeContext) {
        if (StringUtils.isBlank(viewCode) || !this.needAppendData(pageDefine)) {
            return;
        }
        PcUiBotTmDataViewQuery tmDataViewQuery = this.queryDataView(viewCode, executeContext);
        this.checkDataViewConfig(tmDataViewQuery);
        this.appendDataSourceSet(viewCode, pageDefine, tmDataViewQuery);
        this.appendLayout(executeContext, pageDefine, tmDataViewQuery);
    }

    /**
     * 是否需要添加参数
     *
     * @param pageDefine
     * @return
     */
    private boolean needAppendData(PcUiBotPageDefine pageDefine) {
        // layout为空 或 数据查询条件为空
        return CollectionUtils.isEmpty(pageDefine.getPageLayout()) || pageDefine.dataSourceSetEmpty();
    }

    public PcUiBotTmDataViewQuery queryDataView(String viewCode, PcUiBotExecuteContext executeContext) {
        DigiwinAthenaApiRequest athenaApiRequest =
                DigiwinAthenaApiRequest.builder()
                        .requestHead(new AthenaBasicHttpRequestHead("",
                                executeContext.getLocale(), executeContext.getAuthoredUser().getToken(),
                                executeContext.getTenantId()))
                        .bodyPayloadMap("viewCodes", Lists.newArrayList(viewCode))
                        .build();
        KnowledgeMapsDataViewQueryResult dataViewQueryResult = digiwinKnowledgeMapsProxyService.queryDataViewList(athenaApiRequest);

        if (dataViewQueryResult == null || CollectionUtils.isEmpty(dataViewQueryResult.getDataViews())) {
            throw new IllegalStateException("Data view obtained from KM is empty, dataViewCode: " + viewCode);
        }
        return dataViewQueryResult.getDataViews().get(0);
    }

    @Override
    public PcUiBotTmQueryAction buildTmQueryAction(String viewCode, PcUiBotExecuteContext executeContext) {
        PcUiBotTmDataViewQuery tmDataViewQueryDefault = this.queryDataView(viewCode, executeContext);
        return this.buildTmQueryAction(tmDataViewQueryDefault);
    }

    @Override
    public PcUiBotTmQueryAction buildTmQueryAction(PcUiBotTmQueryAction tmQueryAction, PcUiBotTmDataViewQuery tmDataViewQueryDefault) {
        PcUiBotTmQueryAction tmQueryActionFromDataView = this.buildTmQueryAction(tmDataViewQueryDefault);
        if (tmQueryAction != null) {
            tmQueryActionFromDataView.setActionParams(tmQueryAction.getActionParams());
            tmQueryActionFromDataView.setDataKeys(tmQueryAction.getDataKeys());
            tmQueryActionFromDataView.setNotArray(tmQueryAction.getNotArray());
            tmQueryActionFromDataView.setViewCode(tmQueryAction.getViewCode());
        }
        return tmQueryActionFromDataView;
    }

    public PcUiBotTmQueryAction buildTmQueryAction(PcUiBotTmDataViewQuery tmDataViewQueryDefault) {
        PcUiBotDataViewQuerySetDTO dataViewQuerySet = PcUiBotDataViewQuerySetDTO.builder()
                .code(tmDataViewQueryDefault.getCode())
                .orderList(tmDataViewQueryDefault.getOrderList())
                .tables(tmDataViewQueryDefault.getTables())
                .returnFields(tmDataViewQueryDefault.getReturnFields())
                .queryConditions(tmDataViewQueryDefault.getQueryConditions())
                .build();
        Map<String, Object> paras = new HashMap<>();
        paras.put("view_info", dataViewQuerySet);

        PcUiBotTmViewShowFields viewShowFields = tmDataViewQueryDefault.getViewShowFields();
        // 数据源名称为单头名称
        String dataSourceName = viewShowFields.getDataName();

        PcUiBotTmQueryAction tmQueryAction = new PcUiBotTmQueryAction();
        tmQueryAction.setName(dataSourceName);
        // 该接口没有元数据，所以不设置actionId（atdm会根据actionId查元数据）
        tmQueryAction.setServiceName(DATA_VIEW_SERVICE_NAME);
        tmQueryAction.setType(PcUiBotConstants.ACTION_CATEGORY_ESP);
        tmQueryAction.setProductName(tmDataViewQueryDefault.getProductCode());
        tmQueryAction.setParas(paras);
        // 通过虚拟字段透传元数据；因为DATA_VIEW_SERVICE_NAME没有元数据；
        PcUiBotMetadataField metadataField = DataViewRawDslRenderService.convertToMetadataField(viewShowFields);
        tmQueryAction.setMetadataFields(Lists.newArrayList(metadataField));
        // 根据单身字段，设置主键或业务主键
        List<PcUiBotMetadataField> subFields = metadataField.getSubFields();
        tmQueryAction.setDataKeys(PcUiBotApiMetadataUtil.getDataKeys(subFields));
        return tmQueryAction;
    }

    /**
     * 数据视图配置校验
     *
     * @param tmDataViewQuery
     */
    private void checkDataViewConfig(PcUiBotTmDataViewQuery tmDataViewQuery) {
        PcUiBotTmViewShowFields viewShowFields = tmDataViewQuery.getViewShowFields();
        if (viewShowFields == null) {
            throw new IllegalStateException("Data view missing configuration [viewShowFields]");
        }
    }

    /**
     * 添加数据查询条件
     *
     * @param viewCode
     * @param pageDefine
     * @param tmDataViewQuery
     */
    private void appendDataSourceSet(String viewCode, PcUiBotDoubleDocumentPageDefine pageDefine, PcUiBotTmDataViewQuery tmDataViewQuery) {
        if (!pageDefine.dataSourceSetEmpty()) {
            return;
        }
        // 构建dataSourceSet
        PcUiBotDataSourceSetDTO dataViewDataSourceSet = this.buildDataViewDataSourceSet(pageDefine.getExecuteContext(), tmDataViewQuery);
        // 回填至页面定义
        pageDefine.setDataSourceSet(dataViewDataSourceSet);
        // 回填至数据视图；dataView/show调用时，不需要回填
        Optional.ofNullable(pageDefine.getDataViewQueryList())
                .orElse(Lists.newArrayList())
                .stream()
                .filter(dataView -> viewCode.equals(dataView.getViewCode()))
                .findFirst()
                .ifPresent(dataView -> dataView.setDataSourceSet(dataViewDataSourceSet));
    }

    /**
     * 添加布局
     *
     * @param executeContext
     * @param pageDefine
     * @param tmDataViewQuery
     */
    private void appendLayout(PcUiBotExecuteContext executeContext, PcUiBotDoubleDocumentPageDefine pageDefine, PcUiBotTmDataViewQuery tmDataViewQuery) {
        if (!CollectionUtils.isEmpty(pageDefine.getPageLayout())) {
            return;
        }
        pageDefine.setPageLayout(this.buildDataViewLayout(executeContext, pageDefine, tmDataViewQuery));
    }

    /**
     * 构建数据视图页面布局
     *
     * @param executeContext
     * @param tmDataViewQueryDefault
     * @return
     */
    private List<UiBotPageLayout> buildDataViewLayout(PcUiBotExecuteContext executeContext, PcUiBotDoubleDocumentPageDefine pageDefine, PcUiBotTmDataViewQuery tmDataViewQueryDefault) {
        PcUiBotTmViewShowFieldsMobile viewShowFields = tmDataViewQueryDefault.getViewShowFieldsMobile();
        if (viewShowFields == null) {
            log.error("ViewShowFields is null, view code: {}", tmDataViewQueryDefault.getCode());
            return Lists.newArrayList(new UiBotPageLayout());
        }
        return dataViewRawDslRenderService.bulidPageLayout(executeContext, pageDefine, viewShowFields);
    }

    /**
     * 构建数据视图页面数据
     *
     * @param executeContext
     * @param tmDataViewQueryDefault
     * @return
     */
    private PcUiBotDataSourceSetDTO buildDataViewDataSourceSet(PcUiBotExecuteContext executeContext, PcUiBotTmDataViewQuery tmDataViewQueryDefault) {
        PcUiBotTmQueryAction tmQueryAction = this.buildTmQueryAction(tmDataViewQueryDefault);
        Map<String, PcUiBotTmQueryAction> dataSources = MapUtil.of(tmQueryAction.getName(), tmQueryAction);
        return generalDataSourceService.analysis(executeContext, dataSources);
    }

    @Override
    public void buildDslByViewCode(String viewCode, String activityId, String pageCode, PcUiBotTmDataState dataState, PcUiBotExecuteContext executeContext) {
        PcUiBotPageUIElement elements = this.getDslByViewCode(viewCode, pageCode, executeContext);
        if (elements != null) {
            List<PcUiBotTmDataState.Layout> layout = JsonUtil.objectToJavaObject(elements.getLayout(), new TypeReference<List<PcUiBotTmDataState.Layout>>() {
            });
            dataState.setLayout(layout);
            dataState.setOperations(elements.getOperations());
            dataState.setHooks(elements.getHooks());
        }
    }

    @Override
    public PcUiBotPageUIElement getDslByViewCode(String viewCode, String pageCode, PcUiBotExecuteContext executeContext) {
        if (StringUtils.isBlank(viewCode)) {
            log.warn("根据viewCode获取Dsl_入参中viewCode为空_不继续执行");
            return null;
        }
        DigiwinAthenaApiRequest athenaApiRequest =
                DigiwinAthenaApiRequest.builder()
                        .requestHead(new AthenaBasicHttpRequestHead("",
                                executeContext.getLocale(), executeContext.getAuthoredUser().getToken(),
                                executeContext.getTenantId()))
                        .bodyPayloadMap("code", viewCode)
                        .bodyPayloadMap("activityId", executeContext.getTmActivityId())
                        .bodyPayloadMap("pageCode", pageCode)
                        .build();
        return Optional.ofNullable(digiwinKnowledgeMapsProxyService.getPageUIElement(athenaApiRequest)).map(PcUiBotTmPageUIElements::getElements).orElse(null);
    }
}
