package com.digiwin.mobile.mobileuibot.proxy.uibot.pcservice.util;

import com.digiwin.mobile.mobileuibot.common.context.AppRequestContext;
import com.digiwin.mobile.mobileuibot.common.context.SpringContextHolder;
import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.locale.service.impl.LocaleServiceImpl;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.*;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * <p>功能描述：${请输入当前Class的用途}</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: PcUiBotSubmitActionUtil.java
 * @Author: wangjwc
 * @Date: created at 2023/7/11 19:57
 */
public class PcUiBotSubmitActionUtil {

    /**
     * 已完成可出现的按钮
     */
    private final static String[] COMPLETED_SUBMIT_ACTION = new String[]{PcUiBotConstants.UIBOT_TASK_WITHDRAW, PcUiBotConstants.TaskReturn};

    private static final Map<String, String> FINISHED_ACTION_ID_TITLE_MAPPING = new HashMap<>();

    static {
        // 转派
        FINISHED_ACTION_ID_TITLE_MAPPING.put("athena_bpm_reassign", "backlog.activity.reassign");
        // 撤销
        FINISHED_ACTION_ID_TITLE_MAPPING.put("athena_bpm_cancel", "backlog.activity.cancel");
        // 已撤回
        FINISHED_ACTION_ID_TITLE_MAPPING.put(PcUiBotConstants.UIBOT_RETRIEVE, "uibot.activity.retrieved");
        // 已结束
        FINISHED_ACTION_ID_TITLE_MAPPING.put("uibot_action_id_finished", "uibot_action_id_finished");
        // 逾期终止
        FINISHED_ACTION_ID_TITLE_MAPPING.put("athena_overdue_task_terminate", "athena.overdue.task.terminate");
        // 已交接
        FINISHED_ACTION_ID_TITLE_MAPPING.put("uibot_action_id_handover", "backlog.activity.handover");
        // 转派重复
        FINISHED_ACTION_ID_TITLE_MAPPING.put("athena_bpm_reassign_repeat", "backlog.activity.reassign.repeat");
        // 交接重复
        FINISHED_ACTION_ID_TITLE_MAPPING.put("athena_bpm_handover_repeat", "backlog.activity.handover.repeat");
    }

    /**
     * 是否为“任务撤回”按钮
     *
     * @param action
     * @return
     */
    public static boolean isTaskWithdrawAction(PcUiBotTmAction action) {
        if (null == action) {
            return false;
        }
        if (PcUiBotConstants.UIBOT_TASK_WITHDRAW.equals(action.getActionId())) {
            return true;
        }

        List<PcUiBotTmAction> attachActions = action.getAttachActions();
        if (!CollectionUtils.isEmpty(attachActions)) {
            for (PcUiBotTmAction attachAction : attachActions) {
                if (isTaskWithdrawAction(attachAction)) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * 是否为“任务撤回”按钮
     *
     * @param action
     * @return
     */
    public static boolean isTaskWithdrawAction(PcUiBotSubmitAction action) {
        if (null == action) {
            return false;
        }
        if (PcUiBotConstants.UIBOT_TASK_WITHDRAW.equals(action.getActionId())) {
            return true;
        }

        List<PcUiBotSubmitAction> attachActions = action.getAttachActions();
        if (!CollectionUtils.isEmpty(attachActions)) {
            for (PcUiBotSubmitAction attachAction : attachActions) {
                if (isTaskWithdrawAction(attachAction)) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * 只保留 新“撤回”按钮
     *
     * @param submitActions
     * @return
     */
    public static List<PcUiBotSubmitAction> onlyKeepWithdrawAction(List<PcUiBotSubmitAction> submitActions) {
        if (CollectionUtils.isEmpty(submitActions)) {
            return new ArrayList<>();
        }
        return submitActions.stream()
                .filter(PcUiBotSubmitActionUtil::isTaskWithdrawAction)
                .collect(Collectors.toList());
    }

    public static List<PcUiBotSubmitAction> onlyKeepAction(List<PcUiBotSubmitAction> submitActions) {
        if (CollectionUtils.isEmpty(submitActions)) {
            return new ArrayList<>();
        }
        return submitActions.stream()
                .filter(submitAction -> {
                    for (String s : COMPLETED_SUBMIT_ACTION) {
                        if (isTaskActionByActionId(submitAction, s)) {
                            return true;
                        }
                    }
                    return false;
                })
                .collect(Collectors.toList());
    }

    public static boolean isTaskActionByActionId(PcUiBotSubmitAction action, String actionId) {
        if (null == action) {
            return false;
        }
        if (actionId.equals(action.getActionId())) {
            return true;
        }

        List<PcUiBotSubmitAction> attachActions = action.getAttachActions();
        if (!CollectionUtils.isEmpty(attachActions)) {
            for (PcUiBotSubmitAction attachAction : attachActions) {
                if (isTaskActionByActionId(attachAction, actionId)) {
                    return true;
                }
            }
        }
        return false;
    }

    public static boolean isTaskActionByActionId(PcUiBotTmAction action, String actionId) {
        if (null == action) {
            return false;
        }
        if (actionId.equals(action.getActionId())) {
            return true;
        }

        List<PcUiBotTmAction> attachActions = action.getAttachActions();
        if (!CollectionUtils.isEmpty(attachActions)) {
            for (PcUiBotTmAction attachAction : attachActions) {
                if (isTaskActionByActionId(attachAction, actionId)) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * 设置条件相关参数：包含禁用条件和隐藏条件
     *
     * @param tmAction
     * @param submitAction
     */
    public static void setCondition(PcUiBotTmAction tmAction, PcUiBotSubmitAction submitAction) {
        submitAction.setApplyAlthoughFinished(tmAction.getApplyAlthoughFinished());
        if (StringUtils.isNotEmpty(tmAction.getCondition())) {
            PcUiBotCondition condition = new PcUiBotCondition();
            submitAction.setCondition(condition);

            condition.setScript(tmAction.getCondition());
            condition.setTrigger(Lists.newArrayList(PcUiBotConstants.TriggerPoint.INIT));
            List<String> trigger = tmAction.getTrigger();
            // 移除空白参数
            if (!CollectionUtils.isEmpty(trigger)) {
                trigger.removeIf(StringUtils::isBlank);
            }
            if (!CollectionUtils.isEmpty(trigger)) {
                condition.setTrigger(trigger);
            }
        }
        submitAction.setHidden(JsonUtil.objectToJavaObject(tmAction.getHidden(), new TypeReference<PcUiBotCondition>() {
        }));
    }

    /**
     * 是否为“退回”、“跨BK撤回”按钮
     *
     * @param action
     * @return
     */
    public static boolean isReturnOrTaskWithdrawAction(PcUiBotTmAction action) {
        if (null == action) {
            return false;
        }
        if (PcUiBotConstants.TaskReturn.equals(action.getActionId()) || PcUiBotConstants.UIBOT_TASK_WITHDRAW_CROSS_BK.equals(action.getActionId())) {
            return true;
        }

        List<PcUiBotTmAction> attachActions = action.getAttachActions();
        if (!CollectionUtils.isEmpty(attachActions)) {
            for (PcUiBotTmAction attachAction : attachActions) {
                if (isReturnOrTaskWithdrawAction(attachAction)) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * 根据finishedActionId解析出finishedTitle
     *
     * @param finishedActionId
     * @return
     */
    public static String finishedActionIdToTitle(String finishedActionId) {
        if (StringUtils.isBlank(finishedActionId)) {
            return StringUtils.EMPTY;
        }

        for (Map.Entry<String, String> entry : FINISHED_ACTION_ID_TITLE_MAPPING.entrySet()) {
            if (StringUtils.equals(entry.getKey(), finishedActionId)) {
                return entry.getValue();
            }
        }
        return StringUtils.EMPTY;
    }

    /**
     * 将finishedTitle翻译成前端盖章内容
     *
     * @param finishedTitle
     * @return
     */
    public static String translateFinishedTitle(String finishedTitle) {
        if (StringUtils.isNotBlank(finishedTitle)) {
            return SpringContextHolder.getBean(LocaleServiceImpl.class)
                    .getLanguageValue(AppRequestContext.getContextEntity().getLocale(), finishedTitle);
        }
        return StringUtils.EMPTY;
    }

    /**
     * 将finishedActionId翻译成前端的盖章内容
     *
     * @param finishedActionId
     * @return
     */
    public static String translateFinishedActionIdToTitle(String finishedActionId) {
        return translateFinishedTitle(finishedActionIdToTitle(finishedActionId));
    }

    /**
     * SubmitType中的isBatch是否是true
     * 判断提交按钮是否有批量提交参数
     *
     * @return
     */
    public static boolean hasSubmitTypeBatch(List<PcUiBotSubmitAction> submitActions) {
        return getFirstBatchSubmitType(submitActions) != null;
    }

    /**
     * 获取含有第一个有SubmitType中的isBatch为true
     *
     * @param submitActions
     * @return
     */
    public static PcUiBotSubmitType getFirstBatchSubmitType(List<PcUiBotSubmitAction> submitActions) {
        if (!CollectionUtils.isEmpty(submitActions)) {
            for (PcUiBotSubmitAction submitAction : submitActions) {
                PcUiBotSubmitType submitType = submitAction.getSubmitType();
                if (submitType != null && submitType.getIsBatch()) {
                    return submitType;
                }
                PcUiBotSubmitType b = getFirstBatchSubmitType(submitAction.getAttachActions());
                if (b != null) {
                    return b;
                }
            }
        }
        return null;
    }

    public static boolean showWhenBacklogFinished(PcUiBotTmAction action) {
        if (null == action) {
            return false;
        }
        return BooleanUtils.isTrue(action.getApplyAlthoughFinished());
    }
}
