package com.digiwin.mobile.mobileuibot.sso.dingding;

import com.digiwin.mobile.mobileuibot.api.ApiResponse;
import com.digiwin.mobile.mobileuibot.common.context.AppContext;
import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.common.string.StringUtil;
import com.digiwin.mobile.mobileuibot.proxy.cam.model.DigiwinCamDingDingMapperResult;
import com.digiwin.mobile.mobileuibot.proxy.cam.model.DigiwinCamDingDingQuery;
import com.digiwin.mobile.mobileuibot.proxy.cam.model.DigiwinCamDingDingResult;
import com.digiwin.mobile.mobileuibot.proxy.cam.service.DigIwinCamProxyService;
import com.digiwin.mobile.mobileuibot.proxy.iam.model.DigiwinIamAnalyzedToken;
import com.digiwin.mobile.mobileuibot.proxy.iam.service.DigiwinIamProxyService;
import com.digiwin.mobile.mobileuibot.sso.SsoController;
import com.digiwin.mobile.mobileuibot.sso.enums.SsoMongoKeyEnum;
import com.digiwin.mobile.mobileuibot.sso.enums.SsoSourceTypeEnum;
import com.digiwin.mobile.mobileuibot.sso.model.EnvironmentalConfiguration;
import com.digiwin.mobile.mobileuibot.sso.model.SsoConfigModel;
import com.digiwin.mobile.mobileuibot.sso.model.ThirdPartyForParticipationRequestModel;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.web.bind.annotation.*;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 钉钉相关接口
 *
 * @author yanfeng
 * @date 2023/09/18
 */
@Slf4j
@RestController
@RequestMapping("/mobile/v1/ding_talk")
public class DingDingController {

    @Autowired
    private MongoTemplate mongoTemplate;

    @Autowired
    private DigIwinCamProxyService camProxyService;

    @Autowired
    private DigiwinIamProxyService iamProxyService;

    /**
     * 钉钉应用首页/待办/消息地址
     *
     * @param appId 鼎捷云应用Id
     * @author yanfeng
     */
    @GetMapping("/getClientKey")
    public ApiResponse<Map<String, String>> getClientKey(@RequestParam("appId") String appId) {
        try {
            Map<String, String> map = new HashMap<>(2);
            List<DigiwinCamDingDingMapperResult.MapperData> goodsMappings = camProxyService.getGoodsMappings(appId);
            goodsMappings.forEach(data -> {
                if ("clientKey".equalsIgnoreCase(data.getKey())) {
                    map.put("clientKey", data.getValue());
                }
                if ("appId".equalsIgnoreCase(data.getKey())) {
                    map.put("dingTalkAppId", data.getValue());
                }
            });
            return ApiResponse.buildOK().setData(map);
        } catch (Exception e) {
            log.error("getClientKey error:{}", e.getMessage());
            return ApiResponse.buildError("DingTalk getClientKey异常: " + e.getMessage());
        }
    }

    /**
     * Cam钉钉单点登录
     *
     * @param query
     * @return ApiResponse<DigiwinCamDingDingResult>
     * @link <a href="https://console-docs.apipost.cn/preview/e7358d62fa5809f7/97ecc6524d3aabb9?target_id=46930876-cee8-4bbe-a666-41d9cfd2f932">...</a>
     * @author yanfeng
     */
    @PostMapping("/ssoLogin")
    public ApiResponse<DigiwinCamDingDingResult> handleDingTalkSsoLogin(@RequestBody DigiwinCamDingDingQuery query) {
        try {
            DigiwinCamDingDingResult data = camProxyService.dingDingSsoLogin(query);
            log.error("钉钉cam 返回:{}", JsonUtil.javaObjectToJsonString(data));
            if (ObjectUtils.isEmpty(data)) {
                return ApiResponse.buildError("code expire");
            }
            DigiwinIamAnalyzedToken digiwinIamAnalyzedToken = iamProxyService.tokenAnalyze(data.getToken());
            if (ObjectUtils.isNotEmpty(digiwinIamAnalyzedToken)) {
                data.setUserName(digiwinIamAnalyzedToken.getName());
                data.setTenantName(digiwinIamAnalyzedToken.getTenantName());
            }
            // 任务id不为空，则需要跳转到任务详情页面，返还对应的url
            if (StringUtil.isNotEmpty(query.getTaskId())) {
                handelTaskDeatilUrl(data, query);
            }
            log.error("钉钉cam 最终返回:{}", JsonUtil.javaObjectToJsonString(data));
            return ApiResponse.buildOK().setData(data);
        } catch (Exception e) {
            log.error("ding_talk,ssoLogin error:{}", e.getMessage());
            return ApiResponse.buildError(e.getMessage());
        }
    }

    /**
     * 处理详情url
     *
     * @param data
     * @param camDingDingQuery
     * @author yanfeng
     */
    private void handelTaskDeatilUrl(DigiwinCamDingDingResult data, DigiwinCamDingDingQuery camDingDingQuery) throws Exception {
        Query query = new Query();
        query.addCriteria(Criteria.where("key").is(SsoMongoKeyEnum.MOBILE.getSource()));
        SsoConfigModel mobileModel = mongoTemplate.findOne(query, SsoConfigModel.class);
        if (ObjectUtils.isEmpty(mobileModel)) {
            return;
        }
        ApiResponse response = SsoController.getRedirectUrl(mobileModel, false, handelModelParams(data, camDingDingQuery), true);
        if (response != null) {
            data.setUrl(String.valueOf(response.getData()));
        }
    }

    @NotNull
    private static ThirdPartyForParticipationRequestModel handelModelParams(DigiwinCamDingDingResult data, DigiwinCamDingDingQuery camDingDingQuery) {
        ThirdPartyForParticipationRequestModel model = new ThirdPartyForParticipationRequestModel();
        model.setIamUserToken(data.getToken());
        model.setUserId(data.getUserId());
        model.setTenantId(data.getTenantId());
        model.setIsHideAppBar(true);
        model.setTaskId(camDingDingQuery.getTaskId());
        model.setTargetTenantId(camDingDingQuery.getTargetTenantId());
        model.setLocal("zh_CN");
        model.setSourceType(SsoSourceTypeEnum.DING_TALK.getSource());
        return model;
    }

    /**
     * 处理重定向前端地址
     *
     * @param appId
     * @param corpId
     * @param taskId
     * @param targetTenantId
     * @param mobileModel
     * @param suiteKey
     * @return
     */
    @Nullable
    private String handelUrl(String appId, String corpId, String taskId, String targetTenantId, SsoConfigModel mobileModel, String suiteKey) {
        String parameter = null;
        for (EnvironmentalConfiguration configuration : mobileModel.getParameterMap().get("CN")) {
            if (AppContext.getEnvType().equalsIgnoreCase(configuration.getEnvironment())) {
                parameter = configuration.getParameter().replace("{muiUrl}", AppContext.getApiUrlSetting().getWebUrl());
            }
        }
        if (parameter != null) {
            String url = parameter.replace("{corpId}", corpId)
                    .replace("{clientKey}", suiteKey).replace("{appId}", appId);
            if (StringUtil.isNotEmpty(taskId)) {
                url = url + "&taskId=" + taskId;
            }
            if (StringUtil.isNotEmpty(targetTenantId)) {
                url = url + "&targetTenantId=" + targetTenantId;
            }
            return url;
        } else {
            return null;
        }
    }

}