package com.digiwin.mobile.mobileuibot.sso.model;

import cn.hutool.json.JSONUtil;
import com.digiwin.mobile.mobileuibot.common.crypto.aes.AESUtil;
import com.digiwin.mobile.mobileuibot.common.crypto.aes.AESUtils;
import com.digiwin.mobile.mobileuibot.sso.SsoController;
import com.digiwin.mobile.mobileuibot.sso.enums.SsoChannelSourceEnum;
import com.digiwin.mobile.mobileuibot.sso.enums.SsoSourceTypeEnum;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.util.Assert;

import javax.servlet.http.HttpServletRequest;
import java.util.Map;

import static com.digiwin.mobile.mobileuibot.proxy.ProxyController.SECRET_KEY;

/**
 * 鼎捷移动sso 第三方请求入参
 *
 * @author yanfeng
 * @date 2023/7/24
 */
@Slf4j
@Data
public class ThirdPartyForParticipationRequestModel {

    /**
     * 代理token
     */
    private String proxyToken;

    /**
     * 租户ID
     */
    private String targetTenantId;

    /**
     * 任务卡ID
     */
    private String taskId;

    /**
     * 任务卡分类
     */
    private String taskCategory;

    /**
     * 语言
     */
    private String local;

    /**
     * 第三方系统合法应用id
     */
    private String appId;

    /**
     * 第三方系统token
     */
    private String accessToken;

    /**
     * 第三方系统unionId
     */
    private String unionid;

    /**
     * 鼎捷用户Id
     */
    private String userId;

    /**
     * 鼎捷用户token
     */
    private String iamUserToken;

    /**
     * 鼎捷用户租户Id
     */
    private String tenantId;

    /**
     * @see SsoSourceTypeEnum
     */
    private String sourceType;

    /**
     * h5 web页面是否隐藏返回箭头
     */
    private Boolean isHideAppBar;

    /**
     * 页面Id
     */
    private String pageId;

    /**
     * 企微授权码
     */
    private String code;

    /**
     * line状态码
     */
    private String state;

    /**
     * 业务渠道来源
     *
     * @see SsoChannelSourceEnum
     */
    private String channelSource;

    /**
     * 业务渠道来源
     *
     * @see SsoChannelSourceEnum
     */
    private String queryString;

    /**
     * 处理 sso 登录参数
     *
     * @param request
     * @return
     * @link <a href="http://172.16.2.141:22694/html/web/controller/share/share.html#6441feefeae133071b1a7c17">...</a>
     */
    public static ThirdPartyForParticipationRequestModel handleParams(HttpServletRequest request) throws Exception {
        Map<String, String[]> parameterMap = request.getParameterMap();
        log.info("ThirdPartyForParticipationRequestModel.sso.param:{}", JSONUtil.toJsonStr(parameterMap));
        ThirdPartyForParticipationRequestModel requestDTO = new ThirdPartyForParticipationRequestModel();

        requestDTO.setSourceType(request.getParameter("sourceType"));
        requestDTO.setLocal(String.valueOf(request.getLocale()));
        String dtoSourceType = requestDTO.getSourceType();

        if (StringUtils.isNotEmpty(request.getParameter("isHideAppBar"))) {
            String isHideAppBar = request.getParameter("isHideAppBar");
            requestDTO.setIsHideAppBar(Boolean.parseBoolean(isHideAppBar));
        }
        requestDTO.setProxyToken(request.getParameter("proxyToken"));
        //租户ID
        requestDTO.setTargetTenantId(request.getParameter("targetTenantId"));
        //任务卡ID
        requestDTO.setTaskId(request.getParameter("taskId"));

        handleEnterpriseWechat(request, dtoSourceType, requestDTO);

        handleLine(request, dtoSourceType, requestDTO);

        handleNanaAppletCodeAndAppId(request, dtoSourceType, requestDTO);

        handleWeChatPhaseOne(request, requestDTO);

        return requestDTO;
    }

    /**
     * 处理Line参数
     *
     * @param request
     * @param dtoSourceType
     * @param requestDTO
     * @author yanfeng
     */
    private static void handleLine(HttpServletRequest request, String dtoSourceType, ThirdPartyForParticipationRequestModel requestDTO) throws Exception {
        if (SsoSourceTypeEnum.LINE.getSource().equalsIgnoreCase(dtoSourceType)) {
            String channelSource = request.getParameter("channelSource");
            if(!SsoChannelSourceEnum.NANA.getChannelSource().equals(channelSource)){
                //第三方系统合法应用id
                requestDTO.setAppId(AESUtil.decrypt(SECRET_KEY, request.getParameter("appId")));
                //第三方系统token
                requestDTO.setAccessToken(AESUtil.decrypt(SECRET_KEY, request.getParameter("accessToken")));
                //第三方系统unionId
                String unionId = AESUtil.decrypt(SECRET_KEY, request.getParameter("unionId"));
                //调用iam sso第三方登录：unionId需要加密
                requestDTO.setUnionid(AESUtils.aesEncrypt(unionId, SsoController.THIRD_PARTY_SSO_SECRET_KEY));
                //鼎捷userId
                requestDTO.setUserId(AESUtil.decrypt(SECRET_KEY, request.getParameter("userId")));
            }
            //渠道来源
            requestDTO.setChannelSource(channelSource);
            //line授权码
            requestDTO.setCode(request.getParameter("code"));
            requestDTO.setState(request.getParameter("state"));
            //line转发所有参数
            requestDTO.setQueryString(request.getQueryString());
        }
    }

    private static void handleNanaAppletCodeAndAppId(HttpServletRequest request, String dtoSourceType, ThirdPartyForParticipationRequestModel requestDTO) {
        if (SsoSourceTypeEnum.NANA_APPLET.getSource().equalsIgnoreCase(dtoSourceType)) {
            Assert.notNull(request.getParameter("code"), "code must not null!");
            Assert.notNull(request.getParameter("appId"), "appId must not null!");
            requestDTO.setCode(request.getParameter("code"));
            requestDTO.setAppId(request.getParameter("appId"));
        }
    }

    /**
     * 处理企微2期参数
     *
     * @param request
     * @param dtoSourceType
     * @param requestDTO
     */
    private static void handleEnterpriseWechat(HttpServletRequest request, String dtoSourceType, ThirdPartyForParticipationRequestModel requestDTO) {
        if (SsoSourceTypeEnum.ENTERPRISE_WECHAT.getSource().equalsIgnoreCase(dtoSourceType)) {
            Assert.notNull(request.getParameter("code"), "code must not null!");
            Assert.notNull(request.getParameter("tenantId"), "tenantId must not null!");
            requestDTO.setCode(request.getParameter("code"));
            requestDTO.setTenantId(request.getParameter("tenantId"));
            requestDTO.setAppId(request.getParameter("appId"));
        }
    }

    /**
     * 处理企微1期参数
     *
     * @param request
     * @param requestDTO
     * @author yanfeng
     */
    private static void handleWeChatPhaseOne(HttpServletRequest request, ThirdPartyForParticipationRequestModel requestDTO) throws Exception {
        if (SsoSourceTypeEnum.MOBILE_WECHAT.getSource().equalsIgnoreCase(requestDTO.getSourceType())) {
            requestDTO.setIamUserToken(AESUtil.decrypt(SECRET_KEY, request.getParameter("iamUserToken")));
            requestDTO.setUserId(AESUtil.decrypt(SECRET_KEY, request.getParameter("userId")));
            requestDTO.setTenantId(AESUtil.decrypt(SECRET_KEY, request.getParameter("tenantId")));
        }
    }
}