package com.digiwin.mobile.mobileuibot.sysmanage.cache;

import com.digiwin.mobile.mobileuibot.api.ApiResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.cache.Cache;
import org.springframework.cache.CacheManager;
import org.springframework.cache.caffeine.CaffeineCache;
import org.springframework.cache.caffeine.CaffeineCacheManager;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

/**
 * <p>功能描述：缓存控制器</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: CacheController
 * @Author: Zaregoto
 * @Date: 2021/11/24 23:22
 */
@RestController
@RequestMapping("/mobile/v1/cache")
public class CacheController {

    private static final Logger logger = LoggerFactory.getLogger(CacheController.class);

    @Resource(name = "systemCacheManager")
    private CaffeineCacheManager systemCacheManager;

    @Resource(name = "projectCacheManager")
    private CaffeineCacheManager projectCacheManager;

    private CaffeineCacheManager getTargetCacheManagerByCategory(String category) {
        if ("system".equals(category)) {
            return this.systemCacheManager;
        } else if ("proxy".equals(category)) {
            return this.projectCacheManager;
        } else {
            return null;
        }
    }

    @RequestMapping(value = "/", method = RequestMethod.GET)
    public ApiResponse<Collection<String>> getCacheCategories() {
        List<String> cacheCategories = new ArrayList<>();
        cacheCategories.add("system");
        cacheCategories.add("proxy");
        return ApiResponse.buildOK().setData(cacheCategories);
    }

    @RequestMapping(value = "/{category}", method = RequestMethod.GET)
    public ApiResponse<Collection<String>> getCachesNamesByCategory(
            @PathVariable(value = "category", required = true) String category) {
        CacheManager cacheManager = this.getTargetCacheManagerByCategory(category);
        Collection<String> cacheNames;
        if (null != cacheManager) {
            cacheNames = cacheManager.getCacheNames();
        } else {
            cacheNames = Collections.emptyList();
        }
        return ApiResponse.buildOK().setData(cacheNames);
    }

    @RequestMapping(value = "/{category}/{cacheName}", method = RequestMethod.GET)
    public ApiResponse<Object> getCacheDetailByCategoryAndCacheName(
            @PathVariable(value = "category", required = true) String category,
            @PathVariable(value = "cacheName", required = true) String cacheName) {
        CacheManager cacheManager = this.getTargetCacheManagerByCategory(category);
        Cache cache = null;
        if (null != cacheManager) {
            cache = cacheManager.getCache(cacheName);
        }
        if (null != cache) {
            if (cache instanceof CaffeineCache) {
                CaffeineCache loadingCache = (CaffeineCache) cache;
                return ApiResponse.buildOK().setData(loadingCache.getNativeCache().asMap());
            } else {
                return ApiResponse.buildOK().setData(cache);
            }

        } else {
            return ApiResponse.buildError("There is no cache named " + cacheName);
        }
    }

    @RequestMapping(value = "/stats/{category}/{cacheName}", method = RequestMethod.GET)
    public ApiResponse<Boolean> getCacheStatsByCategoryAndCacheName(
            @PathVariable(value = "category", required = true) String category,
            @PathVariable(value = "cacheName", required = true) String cacheName) {
        CacheManager cacheManager = this.getTargetCacheManagerByCategory(category);
        Cache cache = null;
        if (null != cacheManager) {
            cache = cacheManager.getCache(cacheName);
        }
        if (null != cache) {
            if (cache instanceof CaffeineCache) {
                CaffeineCache loadingCache = (CaffeineCache) cache;
                return ApiResponse.buildOK().setData(loadingCache.getNativeCache().stats().toString());
            } else {
                return ApiResponse.buildOK().setData(cache);
            }

        } else {
            return ApiResponse.buildError("There is no cache named " + cacheName);
        }
    }

    @RequestMapping(value = "/invalidate/{category}/{cacheName}", method = RequestMethod.POST)
    public ApiResponse<Boolean> invalidateCache(
            @PathVariable(value = "category", required = true) String category,
            @PathVariable(value = "cacheName", required = true) String cacheName) {
        CacheManager cacheManager = this.getTargetCacheManagerByCategory(category);
        Cache cache = null;
        if (null != cacheManager) {
            cache = cacheManager.getCache(cacheName);
        }
        boolean invalidateOperateResult = false;
        if (null != cache) {
            invalidateOperateResult = cache.invalidate();
            return ApiResponse.buildOK().setData(invalidateOperateResult);
        } else {
            return ApiResponse.buildError("There is no cache named " + cacheName);
        }
    }
}
