package com.digiwin.mobile.mobileuibot.task;

import com.digiwin.mobile.mobileuibot.api.ApiRequest;
import com.digiwin.mobile.mobileuibot.api.ApiResponse;
import com.digiwin.mobile.mobileuibot.common.context.AppContext;
import com.digiwin.mobile.mobileuibot.common.request.RequestParameterUtil;
import com.digiwin.mobile.mobileuibot.core.component.MobileComponent;
import com.digiwin.mobile.mobileuibot.core.component.action.Action;
import com.digiwin.mobile.mobileuibot.core.component.action.ActionTypeEnum;
import com.digiwin.mobile.mobileuibot.core.component.index.TodoList;
import com.digiwin.mobile.mobileuibot.core.strategy.modelbuild.UiBotModelBuildMyTaskListStrategy;
import com.digiwin.mobile.mobileuibot.environment.EnvironmentService;
import com.digiwin.mobile.mobileuibot.locale.service.LocaleService;
import com.digiwin.mobile.mobileuibot.model.db1.UserTaskFilter;
import com.digiwin.mobile.mobileuibot.mysql.model.UserTaskFilterRequest;
import com.digiwin.mobile.mobileuibot.mysql.service.UserTaskFilterService;
import com.digiwin.mobile.mobileuibot.proxy.atmc.service.DigiwinAtmcProxyService;
import com.digiwin.mobile.mobileuibot.task.model.ApiRequestTaskDetail;
import com.digiwin.mobile.mobileuibot.task.model.SaveCardSwitchRequest;
import com.digiwin.mobile.mobileuibot.task.model.SaveCommonrFilterRecordRequest;
import com.digiwin.mobile.mobileuibot.task.model.SaveGroupRecordRequest;
import com.digiwin.mobile.mobileuibot.task.model.SaveModuleInfoRequest;
import com.digiwin.mobile.mobileuibot.task.model.SaveSeniorFilterRecordRequest;
import com.digiwin.mobile.mobileuibot.task.model.TaskContentResult;
import com.digiwin.mobile.mobileuibot.task.service.TaskService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.BooleanUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import java.io.IOException;
import java.util.*;

/**
 * <p>功能描述：</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: TaskController
 * @Author: Zaregoto
 * @Date: 2021/5/13 16:10
 */
@Slf4j
@RestController
@RequestMapping("/mobile/v1/task")
public class TaskController {

    @Autowired
    private EnvironmentService environmentService;

    @Autowired
    private TaskService taskService;

    @Autowired
    private DigiwinAtmcProxyService digiwinAtmcProxyService;

    @Autowired
    private UserTaskFilterService userTaskFilterService;

    @Autowired
    private UiBotModelBuildMyTaskListStrategy uiBotModelBuildMyTaskListStrategy;

    @Autowired
    private LocaleService localeService;

    @RequestMapping(value = "/card/content", method = RequestMethod.GET)
    public ApiResponse<TaskContentResult> getTaskCardContent(
            @RequestParam(name = "dataId") String dataId,
            @RequestParam(name = "iamUserToken") String iamUserToken,
            @RequestParam(name = "tenantId") String tenantId,
            @RequestParam(name = "locale") String locale,
            @RequestParam(name = "tmCategory") String tmCategory,
            @RequestParam(name = "tmPattern") String tmPattern,
            @RequestParam(name = "type") String type,
            @RequestParam(name = "tmTaskId") String tmTaskId,
            @RequestParam(name = "tmActivityId") String tmActivityId,
            @RequestParam(name = "proxyToken") String proxyToken
    ) {
        TaskContentResult result = this.taskService.getTaskContentResult(
                dataId, iamUserToken, tenantId, locale, tmCategory, tmPattern, type, tmTaskId, tmActivityId, proxyToken);
        if (null == result) {
            return ApiResponse.buildError();
        }
        return ApiResponse.buildOK().setData(result);
    }

    /**
     * 根据关键字去搜索相关任务
     *
     * @param request
     * @return
     * @throws IOException
     */
    @PostMapping(value = "/search")
    public ApiResponse<Map> taskSearch(HttpServletRequest request) throws IOException {
        Map<String, Object> params = RequestParameterUtil.getPostDataMap(request, false, "UTF-8");
        String iamUserToken = (String) params.get("iamUserToken");
        String locale = (String) params.get("locale");
        String tenantId = (String) params.get("tenantId");
        String queryParam = (String) params.get("queryParam");
        String clientId = (String) params.get("clientId");
        String userId = (String) params.get("userId");
        // 是否不保存查询条件（扫码目前不需要保存，其他场景需要）
        boolean isNotNeedSaveCondition = MapUtils.getBooleanValue(params, "isNotNeedSaveCondition");
        UserTaskFilter userTaskFilter = new UserTaskFilter();
        userTaskFilter.setTenantId(tenantId);
        userTaskFilter.setUserId(userId);
        if (!isNotNeedSaveCondition) {
            userTaskFilterService.insertUserTaskFilterByQueryParam(userTaskFilter, queryParam);
        }
        Set<String> ids = digiwinAtmcProxyService.taskSearch(queryParam, iamUserToken, tenantId, locale, clientId);
        Map<String, Object> map = new HashMap<>(1);
        map.put("idList", ids);
        return ApiResponse.buildOK().setData(map);
    }

    /**
     * 保存相关筛选记录
     *
     * @param userTaskFilterRequest
     * @return
     * @throws IOException
     */
    @RequestMapping(value = "/save/filterIds", method = RequestMethod.POST)
    public ApiResponse<Boolean> saveFilterIds(@RequestBody UserTaskFilterRequest userTaskFilterRequest) throws IOException {
        UserTaskFilter userTaskFilter = new UserTaskFilter();
        BeanUtils.copyProperties(userTaskFilterRequest, userTaskFilter);
        List<String> filterIds = userTaskFilterRequest.getFilterIds();
        if (CollectionUtils.isEmpty(filterIds)) {
            return ApiResponse.buildOK().setData(userTaskFilterService.insertUserTaskFilterByFilterIds(userTaskFilter, ""));
        }
        StringBuffer filterId = new StringBuffer();
        for (int i = 0; i < filterIds.size(); i++) {
            filterId.append(filterIds.get(i));
            if (i != filterIds.size() - 1) {
                filterId.append(";");
            }
        }
        return ApiResponse.buildOK().setData(userTaskFilterService.insertUserTaskFilterByFilterIds(userTaskFilter, filterId.toString()));
    }

    /**
     * 通过mqtt接收到的任务消息，构建移动任务列表返回的结构。
     * 返回结构与首页任务列表响应中data数据结构一样
     *
     * @param apiRequest
     * @return
     */
    @RequestMapping(value = "/detailBacklog", method = RequestMethod.POST)
    public ApiResponse<MobileComponent> taskDetailBacklog(@RequestBody ApiRequestTaskDetail apiRequest) {
        TodoList result = new TodoList().setTodoList(new ArrayList<>()).setEnableMqtt(false);
        try {
            uiBotModelBuildMyTaskListStrategy
                    .paddingMobileTaskListForm(apiRequest, apiRequest.getMyBacklogs(), result);
            Map<String, Object> appConfig = environmentService.getAppConfig();
            boolean enableMqtt = false;
            if (Boolean.TRUE.equals(appConfig.get("enableMqtt"))) {
                enableMqtt = "true".equalsIgnoreCase(AppContext.getApiUrlSetting().getEnableMqtt());
            }
            result.setEnableMqtt(enableMqtt);
        } catch (Exception e) {
            log.error("ProxyController[taskDetailBacklog] get data error:{}", e);
        }
        return ApiResponse.buildOK().setData(result);
    }

    /**
     * 关注
     *
     * @param apiRequest
     * @return
     */
    @PostMapping(value = "/care")
    public ApiResponse<Action> care(@RequestBody ApiRequest apiRequest) {
        Boolean isSuccess = this.taskService.taskImportance(apiRequest, true);
        Action action = new Action();
        action.setType(ActionTypeEnum.SHOW_TOAST.getValue());
        String toastMsg = "设置特别关注" + (BooleanUtils.isTrue(isSuccess) ? "成功" : "失败");
        action.setToastMsg(localeService.getLanguageValue(apiRequest.getLocale(), toastMsg));
        return ApiResponse.buildOK().setData(action);
    }

    /**
     * 取消关注
     *
     * @param apiRequest
     * @return
     */
    @PostMapping(value = "/uncare")
    public ApiResponse<Action> uncare(@RequestBody ApiRequest apiRequest) {
        Boolean isSuccess = this.taskService.taskImportance(apiRequest, false);
        Action action = new Action();
        action.setType(ActionTypeEnum.SHOW_TOAST.getValue());
        String toastMsg = "取消特别关注" + (BooleanUtils.isTrue(isSuccess) ? "成功" : "失败");
        action.setToastMsg(localeService.getLanguageValue(apiRequest.getLocale(), toastMsg));
        return ApiResponse.buildOK().setData(action);
    }

    /**
     * 取消查看
     *
     * @param apiRequest
     * @return
     */
    @PostMapping(value = "/cancelPinning")
    public ApiResponse<Action> cancelPinning(@RequestBody ApiRequest apiRequest) {
        Boolean isSuccess = this.taskService.cancelPinning(apiRequest);
        Action action = new Action();
        action.setType(ActionTypeEnum.SHOW_TOAST.getValue());
        String toastMsg = "取消查看" + (BooleanUtils.isTrue(isSuccess) ? "成功" : "失败");
        action.setToastMsg(localeService.getLanguageValue(apiRequest.getLocale(), toastMsg));
        return ApiResponse.buildOK().setData(action);
    }

    /**
     * 保存分组信息
     *
     * @param apiRequest
     * @return
     */
    @PostMapping(value = "/saveSwitchGroupRecord")
    public ApiResponse<Action> saveSwitchGroupRecord(@RequestBody SaveGroupRecordRequest apiRequest) {
        Boolean isSuccess = this.taskService.saveSwitchGroupRecord(apiRequest);
        Action action = new Action();
        action.setType(ActionTypeEnum.SHOW_TOAST.getValue());
        String toastMsg = "保存" + (BooleanUtils.isTrue(isSuccess) ? "成功" : "失败");
        action.setToastMsg(localeService.getLanguageValue(apiRequest.getLocale(), toastMsg));
        return ApiResponse.buildOK().setData(action);
    }

    /**
     * 保存排序信息
     *
     * @param apiRequest
     * @return
     */
    @PostMapping(value = "/saveModuleInfo")
    public ApiResponse<Action> saveModuleInfo(@RequestBody SaveModuleInfoRequest apiRequest) {
        Boolean isSuccess = this.taskService.saveModuleInfo(apiRequest);
        Action action = new Action();
        action.setType(ActionTypeEnum.SHOW_TOAST.getValue());
        String toastMsg = "保存" + (BooleanUtils.isTrue(isSuccess) ? "成功" : "失败");
        action.setToastMsg(localeService.getLanguageValue(apiRequest.getLocale(), toastMsg));
        return ApiResponse.buildOK().setData(action);
    }

    /**
     * 保存开关信息
     *
     * @param apiRequest
     * @return
     */
    @PostMapping(value = "/saveCardSwitch")
    public ApiResponse<Action> saveCardSwitch(@RequestBody SaveCardSwitchRequest apiRequest) {
        Boolean isSuccess = this.taskService.saveCardSwitch(apiRequest);
        Action action = new Action();
        action.setType(ActionTypeEnum.SHOW_TOAST.getValue());
        String toastMsg = "保存" + (BooleanUtils.isTrue(isSuccess) ? "成功" : "失败");
        action.setToastMsg(localeService.getLanguageValue(apiRequest.getLocale(), toastMsg));
        return ApiResponse.buildOK().setData(action);
    }

    /**
     * 保存普通筛选信息
     *
     * @param apiRequest
     * @return
     */
    @PostMapping(value = "/saveCommonFilterRecord")
    public ApiResponse<Action> saveSeniorFilterRecord(@RequestBody SaveCommonrFilterRecordRequest apiRequest) {
        Boolean isSuccess = this.taskService.saveCommonFilterRecord(apiRequest);
        Action action = new Action();
        action.setType(ActionTypeEnum.SHOW_TOAST.getValue());
        String toastMsg = "保存" + (BooleanUtils.isTrue(isSuccess) ? "成功" : "失败");
        action.setToastMsg(localeService.getLanguageValue(apiRequest.getLocale(), toastMsg));
        return ApiResponse.buildOK().setData(action);
    }

    /**
     * 保存高级筛选信息
     *
     * @param apiRequest
     * @return
     */
    @PostMapping(value = "/saveSeniorFilterRecord")
    public ApiResponse<Action> saveSeniorFilterRecord(@RequestBody SaveSeniorFilterRecordRequest apiRequest) {
        Boolean isSuccess = this.taskService.saveSeniorFilterRecord(apiRequest);
        Action action = new Action();
        action.setType(ActionTypeEnum.SHOW_TOAST.getValue());
        String toastMsg = "保存" + (BooleanUtils.isTrue(isSuccess) ? "成功" : "失败");
        action.setToastMsg(localeService.getLanguageValue(apiRequest.getLocale(), toastMsg));
        return ApiResponse.buildOK().setData(action);
    }
}
