package com.digiwin.mobile.mobileuibot.task.common;

import com.digiwin.mobile.mobileuibot.common.datetime.DateTimeUtil;
import com.digiwin.mobile.mobileuibot.common.localization.LocaleUtil;
import com.digiwin.mobile.mobileuibot.locale.service.LocaleService;
import com.digiwin.mobile.mobileuibot.proxy.atmc.model.DigiwinAtmcBacklog;
import com.digiwin.mobile.mobileuibot.proxy.atmc.model.DigiwinAtmcBacklogDetail;
import org.apache.commons.lang3.BooleanUtils;
import org.springframework.util.Assert;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import java.time.Duration;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.Optional;

/**
 * <p>功能描述：和任务相关字符串的工具类</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: TaskStringUtil
 * @Author: Zaregoto
 * @Date: 2021/5/17 11:15
 */
public final class TaskStringUtil {

    /*********************************************************************************************
     * *********************************以下为定制任务类型标识****************************************
     *********************************************************************************************/
    /**
     * 员工入职第一天项目编号、任务编号，以及组合的任务类型
     */
    public static final String CUSTOM_PROJECT_CODE_EMPLOYEE_INDUCTION = "task_induction_firstDay";
    public static final String CUSTOM_TASK_CODE_EMPLOYEE_INDUCTION = "employeeInduction";
    /**
     * 员工考勤确认任务编号
     */
    public static final String CUSTOM_PROJECT_CODE_ATTENDANCE_CONFIRMATION =
            "attendance_confirmation";
    public static final String CUSTOM_TASK_CODE_EMPLOYEE_ATTENDANCE_CONFIRMATION =
            "employeeAttendanceConfirmation";

    /**
     * 智互联打印任务编号
     */
    public static final String CUSTOM_PROJECT_CODE_ZHILINK_PRINT = "print_task_userProject";
    public static final String CUSTOM_TASK_CODE_ZHILINK_PRINT = "snmPrintTask";
    /**
     * 智互联粘贴任务编号
     */
    public static final String CUSTOM_PROJECT_CODE_ZHILINK_PASTE = "print_task_userProject";
    public static final String CUSTOM_TASK_CODE_ZHILINK_PASTE = "snmPasteTask";
    /**
     * 智互联扫码出厂任务编号
     */
    public static final String CUSTOM_PROJECT_CODE_ZHILINK_CHECKOUT = "factory_task_userProject";
    public static final String CUSTOM_TASK_CODE_ZHILINK_CHECKOUT = "snmFactoryTask";
    public static final String LINK_SYMBOL = "_";

    /**
     * 工时回报任务编号
     */
    public static final String CUSTOM_PROJECT_CODE_WORKING_HOUR_REPORT = "workshop_plan_syn_userproject";
    public static final String CUSTOM_TASK_CODE_WORKING_HOUR_REPORT = "apc_working_hour";

    /**
     * 工时支援任务编号
     */
    public static final String CUSTOM_TASK_CODE_SUPPORT_HOUR_REPORT = "apc_hour_support";

    /**
     * 进度回报任务编号
     */
    public static final String CUSTOM_PROJECT_CODE_PROJECT_SCHEDULE_REPORT = "workshop_plan_syn_userproject";
    public static final String CUSTOM_TASK_CODE_PROJECT_SCHEDULE_REPORT = "apc_project_schedule";

    /**
     * 请求支援任务编号
     */
    public static final String CUSTOM_TASK_CODE_SUPPORT_NEEDS = "apc_support_needs";
    public static final String CUSTOM_PROJECT_CODE_SUPPORT_NEEDS = "workshop_plan_syn_userproject";

    /**
     * 异常回报任务编号
     */
    public static final String CUSTOM_TASK_CODE_ABNORMAL_REPORT = "apc_abnormalReport";
    public static final String CUSTOM_PROJECT_CODE_ABNORMAL_REPORT = "workshop_plan_syn_userproject";

    /**
     * 委外报工任务编号
     */
    public static final String CUSTOM_TASK_CODE_OUTSOURCE_REPORT = "outSourcingWorkReport";
    public static final String CUSTOM_PROJECT_CODE_OUTSOURCE_REPORT = "caCimAp_outSourcing_purchaseReport_userProject";

    /**
     * 体系云管家报工任务编号
     */
    public static final String CUSTOM_TASK_CODE_CLOUD_STEWARD_REPORT = "sch_manualAssignment";
    public static final String CUSTOM_PROJECT_CODE_CLOUD_STEWARD_REPORT = "systemCloudHouseKeeper_userProject";

    /**
     * 领料通知任务编号
     */
    public static final String CUSTOM_TASK_CODE_MATERIAL_NOTICE = "ItemPickingNotice";
    public static final String CUSTOM_PROJECT_CODE_MATERIAL_NOTICE = "caCimAp_outSourcing_purchaseReport_userProject";

    /**
     * 合并工时任务编号
     */
    public static final String CUSTOM_TASK_CODE_MERGE_WOK_HOUR = "apc_mergeWork";
    public static final String CUSTOM_PROJECT_CODE_MERGE_WOK_HOUR = "mergeWorkApplication_userproject";

    /**
     * 试验记录任务编号
     */
    public static final String CUSTOM_TASK_CODE_PRODUCETION_PROGRESS_REPORT = "TBDS_ProductionProcessRecord";
    public static final String CUSTOM_PROJECT_CODE_PRODUCETION_PROGRESS_REPORT = "TBDS_sampleTrialProduction_userProject";

    /**
     * 对账周报任务编号
     */
    public static final String CUSTOM_TASK_CODE_RECONILIATION_PROGRESS_NOTICE = "aacr_AccountReconciliationReport_Project";
    public static final String CUSTOM_PROJECT_CODE_RECONILIATION_PROGRESS_NOTICE = "aacr_AccountReconciliationReport_TaskCard";

    // 随心控(人工型：code+_task_manual;审核型：code+_task_approve)
    public static final String CUSTOM_TASK_CODE_SMART_ADJUST = "custom_task_code_smart_adjust";

    public static final String CUSTOM_TASK_TYPE_WORKING_HOUR_REPORT =
            CUSTOM_PROJECT_CODE_WORKING_HOUR_REPORT + LINK_SYMBOL + CUSTOM_TASK_CODE_WORKING_HOUR_REPORT;
    public static final String CUSTOM_TASK_TYPE_SUPPORT_HOUR_REPORT =
            CUSTOM_PROJECT_CODE_WORKING_HOUR_REPORT + LINK_SYMBOL + CUSTOM_TASK_CODE_SUPPORT_HOUR_REPORT;
    public static final String CUSTOM_TASK_TYPE_PROJECT_SCHEDULE_REPORT =
            CUSTOM_PROJECT_CODE_PROJECT_SCHEDULE_REPORT + LINK_SYMBOL + CUSTOM_TASK_CODE_PROJECT_SCHEDULE_REPORT;
    public static final String CUSTOM_TASK_TYPE_SUPPORT_NEEDS =
            CUSTOM_TASK_CODE_SUPPORT_NEEDS + LINK_SYMBOL + CUSTOM_PROJECT_CODE_SUPPORT_NEEDS;
    public static final String CUSTOM_TASK_TYPE_ABNORMAL_REPORT =
            CUSTOM_TASK_CODE_ABNORMAL_REPORT + LINK_SYMBOL + CUSTOM_PROJECT_CODE_ABNORMAL_REPORT;

    public static final String CUSTOM_TASK_TYPE_OUTSOURCE_REPORT =
            CUSTOM_TASK_CODE_OUTSOURCE_REPORT + LINK_SYMBOL + CUSTOM_PROJECT_CODE_OUTSOURCE_REPORT;

    public static final String CUSTOM_TASK_TYPE_MATERIAL_NOTICE =
            CUSTOM_TASK_CODE_MATERIAL_NOTICE + LINK_SYMBOL + CUSTOM_PROJECT_CODE_MATERIAL_NOTICE;

    public static final String CUSTOM_TASK_TYPE_MERGE_WOK_HOUR =
            CUSTOM_TASK_CODE_MERGE_WOK_HOUR + LINK_SYMBOL + CUSTOM_PROJECT_CODE_MERGE_WOK_HOUR;

    public static final String CUSTOM_TASK_TYPE_PRODUCETION_PROGRESS_REPORT =
            CUSTOM_TASK_CODE_PRODUCETION_PROGRESS_REPORT + LINK_SYMBOL + CUSTOM_PROJECT_CODE_PRODUCETION_PROGRESS_REPORT;

    public static final String CUSTOM_TASK_TYPE_RECONCILIATION_PROGRESS_NOTICE =
            CUSTOM_TASK_CODE_RECONILIATION_PROGRESS_NOTICE + LINK_SYMBOL + CUSTOM_PROJECT_CODE_RECONILIATION_PROGRESS_NOTICE;

    /*********************************************************************************************
     * *********************************以下为标准任务类型标识****************************************
     *********************************************************************************************/

    public static final String TASK_TYPE_BIZ_APPROVAL = "BUSINESS" + LINK_SYMBOL + "APPROVAL";
    public static final String TASK_TYPE_BIZ_REPORT = "BUSINESS" + LINK_SYMBOL + "REPORT";
    public static final String TASK_TYPE_CUSTOM_REPORT = "CUSTOM" + LINK_SYMBOL + "REPORT";
    public static final String TASK_TYPE_CUSTOM_PROCESS = "CUSTOM" + LINK_SYMBOL + "PROCESS";
    public static final String TASK_TYPE_BIZ_ASSISTAN = "BUSINESS_ASSISTAN" + LINK_SYMBOL + "ASSISTAN";
    public static final String TASK_TYPE_BIZ_PROCESS = "BUSINESS" + LINK_SYMBOL + "PROCESS";
    public static final String TASK_TYPE_BUSINESS_REPLY = "BUSINESS" + LINK_SYMBOL + "REPLY";
    public static final String TASK_TYPE_BIZ_DATA_ENTRY_DRIVEN = "BUSINESS" + LINK_SYMBOL + "DATA_ENTRY_DRIVEN";

    // 定制 PATTERN==CUSTOM 类型任务卡详情
    public static final String CUSTOM_PATTERN_TASK = "CUSTOM";


    /*********************************************************************************************
     * ******************************以下为移动自定义的任务类型标识************************************
     *********************************************************************************************/


    public static boolean isBizApproval(String tmPattern, String tmCategory) {
        Assert.notNull(tmPattern, "tmPattern must not null!");
        Assert.notNull(tmCategory, "tmCategory must not null!");
        return (tmPattern + LINK_SYMBOL + tmCategory).equalsIgnoreCase(TASK_TYPE_BIZ_APPROVAL);
    }

    public static boolean isBizReport(String tmPattern, String tmCategory) {
        Assert.notNull(tmPattern, "tmPattern must not null!");
        Assert.notNull(tmCategory, "tmCategory must not null!");
        return (tmPattern + LINK_SYMBOL + tmCategory).equalsIgnoreCase(TASK_TYPE_BIZ_REPORT);
    }

    public static boolean isBizProcess(String tmPattern, String tmCategory) {
        Assert.notNull(tmPattern, "tmPattern must not null!");
        Assert.notNull(tmCategory, "tmCategory must not null!");
        return (tmPattern + LINK_SYMBOL + tmCategory).equalsIgnoreCase(TASK_TYPE_BIZ_PROCESS);
    }

    public static boolean isCustomReport(String tmPattern, String tmCategory) {
        Assert.notNull(tmPattern, "tmPattern must not null!");
        Assert.notNull(tmCategory, "tmCategory must not null!");
        return (tmPattern + LINK_SYMBOL + tmCategory).equalsIgnoreCase(TASK_TYPE_CUSTOM_REPORT);
    }

    public static boolean isCustomProcess(String tmPattern, String tmCategory) {
        Assert.notNull(tmPattern, "tmPattern must not null!");
        Assert.notNull(tmCategory, "tmCategory must not null!");
        return (tmPattern + LINK_SYMBOL + tmCategory).equalsIgnoreCase(TASK_TYPE_CUSTOM_PROCESS);
    }

    public static boolean isBusinessReply(String tmPattern, String tmCategory) {
        Assert.notNull(tmPattern, "tmPattern must not null!");
        Assert.notNull(tmCategory, "tmCategory must not null!");
        return (tmPattern + LINK_SYMBOL + tmCategory).equalsIgnoreCase(TASK_TYPE_BUSINESS_REPLY);
    }

    public static boolean isBizDataEntryDriven(String tmPattern, String tmCategory) {
        if (!StringUtils.hasLength(tmPattern) || !StringUtils.hasLength(tmCategory)) {
            return false;
        }
        return (tmPattern + LINK_SYMBOL + tmCategory).equalsIgnoreCase(TASK_TYPE_BIZ_DATA_ENTRY_DRIVEN);
    }

    /**
     * 判断是否是 随心控(人工型：code+_task_manual;审核型：code+_task_approve)
     */
    public static boolean isBizSmartAdjust(String tmActivityId) {
        if (!StringUtils.hasLength(tmActivityId)) {
            return false;
        }
        return tmActivityId.endsWith("_task_manual") || tmActivityId.endsWith("_task_approve");
    }

    /**
     * 判断是否是 模型驱动手动发起项目(tmActivityId=StartNode_*******)
     */
    public static boolean isProjectCreateOfModelDriven(String tmActivityId) {
        if (!StringUtils.hasLength(tmActivityId)) {
            return false;
        }
        return tmActivityId.startsWith("StartNode_");
    }

    /**
     * 判断任务是否是SSC员工入职第一天
     *
     * @param tmTaskId
     * @param tmActivityId
     * @return
     */
    public static boolean isSscEmployeeInduction(String tmTaskId, String tmActivityId) {
        if (!StringUtils.hasLength(tmTaskId) || !StringUtils.hasLength(tmActivityId)) {
            return false;
        }
//        Assert.notNull(tmTaskId, "tmTaskId must not null!");
//        Assert.notNull(tmActivityId, "tmActivityId must not null!");
        return tmTaskId.contains(CUSTOM_PROJECT_CODE_EMPLOYEE_INDUCTION) && tmActivityId.contains(
                CUSTOM_TASK_CODE_EMPLOYEE_INDUCTION);
    }

    /**
     * 判断任务是否是APC请求支援
     *
     * @param tmTaskId
     * @param tmActivityId
     * @return
     */
    public static boolean isApcSupportNeedsStrategy(String tmTaskId, String tmActivityId) {
        if (!StringUtils.hasLength(tmTaskId) || !StringUtils.hasLength(tmActivityId)) {
            return false;
        }
        return tmTaskId.contains(CUSTOM_PROJECT_CODE_SUPPORT_NEEDS) && tmActivityId.contains(
                CUSTOM_TASK_CODE_SUPPORT_NEEDS);
    }

    /**
     * 判断任务是否是APC请求支援
     *
     * @param tmTaskId
     * @param tmActivityId
     * @return
     */
    public static boolean isApcAbnoralReport(String tmTaskId, String tmActivityId) {
        if (!StringUtils.hasLength(tmTaskId) || !StringUtils.hasLength(tmActivityId)) {
            return false;
        }
        return tmTaskId.contains(CUSTOM_PROJECT_CODE_ABNORMAL_REPORT) && tmActivityId.contains(
                CUSTOM_TASK_CODE_ABNORMAL_REPORT);
    }

    /**
     * 判断任务是否是委外报工
     *
     * @param tmTaskId
     * @param tmActivityId
     * @return
     */
    public static boolean isOutSourceReport(String tmTaskId, String tmActivityId) {
        if (!StringUtils.hasLength(tmTaskId) || !StringUtils.hasLength(tmActivityId)) {
            return false;
        }
        return tmTaskId.contains(CUSTOM_PROJECT_CODE_OUTSOURCE_REPORT) && tmActivityId.contains(
                CUSTOM_TASK_CODE_OUTSOURCE_REPORT);
    }

    /**
     * 判断任务是否是领料通知
     *
     * @param tmTaskId
     * @param tmActivityId
     * @return
     */
    public static boolean isMaterialNotice(String tmTaskId, String tmActivityId) {
        if (!StringUtils.hasLength(tmTaskId) || !StringUtils.hasLength(tmActivityId)) {
            return false;
        }
        return tmTaskId.contains(CUSTOM_PROJECT_CODE_MATERIAL_NOTICE) && tmActivityId.contains(
                CUSTOM_TASK_CODE_MATERIAL_NOTICE);
    }

    /**
     * 判断任务是否是合并工时
     *
     * @param tmTaskId
     * @param tmActivityId
     * @return
     */
    public static boolean isMergeWorkHour(String tmTaskId, String tmActivityId) {
        if (!StringUtils.hasLength(tmTaskId) || !StringUtils.hasLength(tmActivityId)) {
            return false;
        }
        return tmTaskId.contains(CUSTOM_PROJECT_CODE_MERGE_WOK_HOUR) && tmActivityId.contains(
                CUSTOM_TASK_CODE_MERGE_WOK_HOUR);
    }

    /**
     * 判断任务是否是对账周报
     *
     * @param tmTaskId
     * @param tmActivityId
     * @return
     */
    public static boolean isReconiliationProgressNotice(String tmTaskId, String tmActivityId) {
        if (!StringUtils.hasLength(tmTaskId) || !StringUtils.hasLength(tmActivityId)) {
            return false;
        }
        return tmTaskId.contains(CUSTOM_TASK_CODE_RECONILIATION_PROGRESS_NOTICE) && tmActivityId.contains(
                CUSTOM_PROJECT_CODE_RECONILIATION_PROGRESS_NOTICE);
    }

    /**
     * 判断任务是否是SSC考勤确认
     *
     * @param tmTaskId
     * @param tmActivityId
     * @return
     */
    public static boolean isSscAttendanceConfirmation(String tmTaskId, String tmActivityId) {
        Assert.notNull(tmTaskId, "tmTaskId must not null!");
        Assert.notNull(tmActivityId, "tmActivityId must not null!");
        return tmTaskId.contains(CUSTOM_PROJECT_CODE_ATTENDANCE_CONFIRMATION)
                && tmActivityId.contains(CUSTOM_TASK_CODE_EMPLOYEE_ATTENDANCE_CONFIRMATION);
    }

    /**
     * 判断任务是否是智互联打印任务
     *
     * @param tmTaskId
     * @param tmActivityId
     * @return
     */
    public static boolean isZhilinkBarCodePrint(String tmTaskId, String tmActivityId) {
        if (ObjectUtils.isEmpty(tmTaskId) || ObjectUtils.isEmpty(tmActivityId)) {
            return false;
        }
        Assert.notNull(tmTaskId, "tmTaskId must not null!");
        Assert.notNull(tmActivityId, "tmActivityId must not null!");
        return tmTaskId.contains(CUSTOM_PROJECT_CODE_ZHILINK_PRINT) && tmActivityId.contains(
                CUSTOM_TASK_CODE_ZHILINK_PRINT);
    }

    /**
     * 判断任务是否是智互联扫码粘贴任务
     *
     * @param tmTaskId
     * @param tmActivityId
     * @return
     */
    public static boolean isZhilinkBarCodePaste(String tmTaskId, String tmActivityId) {
        if (ObjectUtils.isEmpty(tmTaskId) || ObjectUtils.isEmpty(tmActivityId)) {
            return false;
        }
        Assert.notNull(tmTaskId, "tmTaskId must not null!");
        Assert.notNull(tmActivityId, "tmActivityId must not null!");
        return tmTaskId.contains(CUSTOM_PROJECT_CODE_ZHILINK_PASTE) && tmActivityId.contains(
                CUSTOM_TASK_CODE_ZHILINK_PASTE);
    }

    /**
     * 判断任务是否是智互联扫码出厂任务
     *
     * @param tmTaskId
     * @param tmActivityId
     * @return
     */
    public static boolean isZhilinkBarCodeCheckout(String tmTaskId, String tmActivityId) {
        if (ObjectUtils.isEmpty(tmTaskId) || ObjectUtils.isEmpty(tmActivityId)) {
            return false;
        }
        Assert.notNull(tmTaskId, "tmTaskId must not null!");
        Assert.notNull(tmActivityId, "tmActivityId must not null!");
        return tmTaskId.contains(CUSTOM_PROJECT_CODE_ZHILINK_CHECKOUT) && tmActivityId.contains(
                CUSTOM_TASK_CODE_ZHILINK_CHECKOUT);
    }

    /**
     * 判断任务是否是工时回报任务
     *
     * @param tmTaskId
     * @param tmActivityId
     * @return
     */
    public static boolean isWorkingHourReport(String tmTaskId, String tmActivityId) {
        Assert.notNull(tmTaskId, "tmTaskId must not null!");
        Assert.notNull(tmActivityId, "tmActivityId must not null!");
        return tmTaskId.contains(CUSTOM_PROJECT_CODE_WORKING_HOUR_REPORT) && tmActivityId.contains(
                CUSTOM_TASK_CODE_WORKING_HOUR_REPORT);
    }

    /**
     * 判断任务是否是工时支援任务
     *
     * @param tmTaskId
     * @param tmActivityId
     * @return
     */
    public static boolean isSupportHourReport(String tmTaskId, String tmActivityId) {
        Assert.notNull(tmTaskId, "tmTaskId must not null!");
        Assert.notNull(tmActivityId, "tmActivityId must not null!");
        return tmTaskId.contains(CUSTOM_PROJECT_CODE_WORKING_HOUR_REPORT) && tmActivityId.contains(
                CUSTOM_TASK_CODE_SUPPORT_HOUR_REPORT);
    }

    /**
     * 判断任务是否是进度回报任务
     *
     * @param tmTaskId
     * @param tmActivityId
     * @return
     */
    public static boolean isProjectScheduleReport(String tmTaskId, String tmActivityId) {
        Assert.notNull(tmTaskId, "tmTaskId must not null!");
        Assert.notNull(tmActivityId, "tmActivityId must not null!");
        return tmTaskId.contains(CUSTOM_PROJECT_CODE_PROJECT_SCHEDULE_REPORT) && tmActivityId.contains(
                CUSTOM_TASK_CODE_PROJECT_SCHEDULE_REPORT);
    }

    /**
     * 判断是否是在途宝 发车任务卡
     *
     * @param tmTaskId
     * @param tmActivityId
     * @return
     */
    public static boolean isZtbDepartureTaskCard(String tmTaskId, String tmActivityId) {
        return " freightage_start_user".contains(tmTaskId) && "freightage_departure".contains(tmActivityId);
    }

    /**
     * 判断是否是在途宝 在途卡
     *
     * @param tmTaskId
     * @param tmActivityId
     * @return
     */
    public static boolean isZtbTransportTaskCard(String tmTaskId, String tmActivityId) {
        return " freightage_start_user".contains(tmTaskId) && "freightage_transport_loading".contains(tmActivityId);
    }

    public static boolean isZtbDepartureOfTransportTaskCard(String tmTaskId, String tmActivityId) {
        return isZtbDepartureTaskCard(tmTaskId, tmActivityId) || isZtbTransportTaskCard(tmTaskId, tmActivityId);
    }

    /**
     * 判断任务是否是PCC任务
     *
     * @param tmTaskId
     * @param tmActivityId
     * @return
     */
    public static boolean isPccTask(String tmTaskId, String tmActivityId) {
        Assert.notNull(tmTaskId, "tmTaskId must not null!");
        Assert.notNull(tmActivityId, "tmActivityId must not null!");
        boolean isPccFlowEngineManualReportTask = "manualAssignment".equals(tmActivityId)
                && "task_Assignment".equals(tmTaskId);
        boolean isPccTaskEngineManualReportTask = ("manual_DTD_Assignment".equals(tmActivityId)
                || "uc_manual_DTD_Assignment".equals(tmActivityId))
                && "projectCenterConsole_userProject".equals(tmTaskId);
        return isPccFlowEngineManualReportTask || isPccTaskEngineManualReportTask;
    }

    /**
     * 判断是否是签核任务
     */
    public static boolean isNeedSubName(DigiwinAtmcBacklog digiwinAtmcBacklog) {
        return "APPROVAL".equals(digiwinAtmcBacklog.getTmCategory());
    }

    public static String getTaskStateText(DigiwinAtmcBacklog digiwinAtmcBacklog) {
        /**
         * 只展示加签，退回，撤回状态 (条件不是转派且状态有值)
         */
        if (BooleanUtils.isNotTrue(digiwinAtmcBacklog.getReassign()) && StringUtils.hasLength(digiwinAtmcBacklog.getApprovalState())) {
            return digiwinAtmcBacklog.getApprovalState();
        } else {
            return "";
        }
    }

    public static String getDataChangeStatusText(DigiwinAtmcBacklog digiwinAtmcBacklog, LocaleService localeService, String locale) {
        /**
         * 展示已变更/变更中
         */
        if (BooleanUtils.isTrue(digiwinAtmcBacklog.getDataChangeRead())) {
            return localeService.getLanguageValue(locale, "已变更");
        } else if (BooleanUtils.isTrue(digiwinAtmcBacklog.getChanged())) {
            return localeService.getLanguageValue(locale, "变更中");
        } else {
            return "";
        }
    }

    public static String getTaskBuildTypeByDetailPatternCategory(
            DigiwinAtmcBacklogDetail digiwinAtmcBacklogDetail) {
        /** 定制应用名：员工共享服务中心，简称：SSC。 **/
        // 任务内容：入职办理任务
        if (TaskStringUtil.isSscEmployeeInduction(digiwinAtmcBacklogDetail.getTmTaskId(),
                digiwinAtmcBacklogDetail.getTmActivityId())) {
            return TaskStringUtil.CUSTOM_PROJECT_CODE_EMPLOYEE_INDUCTION
                    + TaskStringUtil.LINK_SYMBOL + TaskStringUtil.CUSTOM_TASK_CODE_EMPLOYEE_INDUCTION;
        }
        if (TaskStringUtil.isSscAttendanceConfirmation(digiwinAtmcBacklogDetail.getTmTaskId(),
                digiwinAtmcBacklogDetail.getTmActivityId())) {
            return TaskStringUtil.CUSTOM_PROJECT_CODE_ATTENDANCE_CONFIRMATION
                    + TaskStringUtil.LINK_SYMBOL + TaskStringUtil.CUSTOM_TASK_CODE_EMPLOYEE_ATTENDANCE_CONFIRMATION;
        }

        /** 定制应用名：智互联条码管理， **/
        // 打印任务
        if (TaskStringUtil.isZhilinkBarCodePrint(digiwinAtmcBacklogDetail.getTmTaskId(),
                digiwinAtmcBacklogDetail.getTmActivityId())) {
            return TaskStringUtil.CUSTOM_PROJECT_CODE_ZHILINK_PRINT
                    + TaskStringUtil.LINK_SYMBOL + TaskStringUtil.CUSTOM_TASK_CODE_ZHILINK_PRINT;
        }
        // 粘贴任务
        if (TaskStringUtil.isZhilinkBarCodePaste(digiwinAtmcBacklogDetail.getTmTaskId(),
                digiwinAtmcBacklogDetail.getTmActivityId())) {
            return TaskStringUtil.CUSTOM_PROJECT_CODE_ZHILINK_PASTE
                    + TaskStringUtil.LINK_SYMBOL + TaskStringUtil.CUSTOM_TASK_CODE_ZHILINK_PASTE;
        }
        // 出厂任务
        if (TaskStringUtil.isZhilinkBarCodeCheckout(digiwinAtmcBacklogDetail.getTmTaskId(),
                digiwinAtmcBacklogDetail.getTmActivityId())) {
            return TaskStringUtil.CUSTOM_PROJECT_CODE_ZHILINK_CHECKOUT
                    + TaskStringUtil.LINK_SYMBOL + TaskStringUtil.CUSTOM_TASK_CODE_ZHILINK_CHECKOUT;
        }

        /** 定制应用名：装配进度控制（APC) **/
        //工时回报
        if (TaskStringUtil.isWorkingHourReport(digiwinAtmcBacklogDetail.getTmTaskId(),
                digiwinAtmcBacklogDetail.getTmActivityId())) {
            return TaskStringUtil.CUSTOM_TASK_TYPE_WORKING_HOUR_REPORT;
        }
        //工时支援
        if (TaskStringUtil.isSupportHourReport(digiwinAtmcBacklogDetail.getTmTaskId(),
                digiwinAtmcBacklogDetail.getTmActivityId())) {
            return TaskStringUtil.CUSTOM_TASK_TYPE_SUPPORT_HOUR_REPORT;
        }
        //进度回报
        if (TaskStringUtil.isProjectScheduleReport(digiwinAtmcBacklogDetail.getTmTaskId(),
                digiwinAtmcBacklogDetail.getTmActivityId())) {
            return TaskStringUtil.CUSTOM_TASK_TYPE_PROJECT_SCHEDULE_REPORT;
        }
        //请求支援
        if (TaskStringUtil.isApcSupportNeedsStrategy(digiwinAtmcBacklogDetail.getTmTaskId(),
                digiwinAtmcBacklogDetail.getTmActivityId())) {
            return TaskStringUtil.CUSTOM_TASK_TYPE_SUPPORT_NEEDS;
        }
        //异常回报
        if (TaskStringUtil.isApcAbnoralReport(digiwinAtmcBacklogDetail.getTmTaskId(),
                digiwinAtmcBacklogDetail.getTmActivityId())) {
            return TaskStringUtil.CUSTOM_TASK_TYPE_ABNORMAL_REPORT;
        }
        //委外报工
        if (TaskStringUtil.isOutSourceReport(digiwinAtmcBacklogDetail.getTmTaskId(),
                digiwinAtmcBacklogDetail.getTmActivityId())) {
            return TaskStringUtil.CUSTOM_TASK_TYPE_OUTSOURCE_REPORT;
        }
        //领料通知
        if (TaskStringUtil.isMaterialNotice(digiwinAtmcBacklogDetail.getTmTaskId(),
                digiwinAtmcBacklogDetail.getTmActivityId())) {
            return TaskStringUtil.CUSTOM_TASK_TYPE_MATERIAL_NOTICE;
        }
        //合并工时
        if (TaskStringUtil.isMergeWorkHour(digiwinAtmcBacklogDetail.getTmTaskId(),
                digiwinAtmcBacklogDetail.getTmActivityId())) {
            return TaskStringUtil.CUSTOM_TASK_TYPE_MERGE_WOK_HOUR;
        }
        //试研北斗-试验过程记录
        if (TaskStringUtil.isProductionProgressReport(digiwinAtmcBacklogDetail.getTmTaskId(),
                digiwinAtmcBacklogDetail.getTmActivityId())) {
            return TaskStringUtil.CUSTOM_TASK_TYPE_PRODUCETION_PROGRESS_REPORT;
        }
        //对账周报
        if (TaskStringUtil.isReconiliationProgressNotice(digiwinAtmcBacklogDetail.getTmTaskId(),
                digiwinAtmcBacklogDetail.getTmActivityId())) {
            return TaskStringUtil.CUSTOM_TASK_TYPE_RECONCILIATION_PROGRESS_NOTICE;
        }
        // 随心控(人工型：code+_task_manual;审核型：code+_task_approve)
        if (TaskStringUtil.isBizSmartAdjust(digiwinAtmcBacklogDetail.getTmActivityId())) {
            return TaskStringUtil.CUSTOM_TASK_CODE_SMART_ADJUST;
        }

        // FIXME PCC项目中控台 进度回报任务（手动报工任务）特殊硬编码处理，后续除了通过pattern+category的自动判断，也应支持由用户配置界面生成策略
        // FIXME 有乾冶客户客制，tmActivityId由manual_DTD_Assignment变成了uc_manual_DTD_Assignment。后期需要同上面备注一样修改
        boolean isPccFlowEngineManualReportTask = "manualAssignment".equals(digiwinAtmcBacklogDetail.getTmActivityId())
                && "task_Assignment".equals(digiwinAtmcBacklogDetail.getTmTaskId());
        boolean isPccTaskEngineManualReportTask = ("manual_DTD_Assignment".equals(digiwinAtmcBacklogDetail.getTmActivityId())
                || "uc_manual_DTD_Assignment".equals(digiwinAtmcBacklogDetail.getTmActivityId()))
                && "projectCenterConsole_userProject".equals(digiwinAtmcBacklogDetail.getTmTaskId());
        if (isPccFlowEngineManualReportTask || isPccTaskEngineManualReportTask) {
            return TASK_TYPE_BIZ_REPORT;
        }

        String tmPattern = Optional.ofNullable(digiwinAtmcBacklogDetail).map(DigiwinAtmcBacklogDetail::getTmPattern).orElse("");
        String tmCategory = Optional.ofNullable(digiwinAtmcBacklogDetail).map(DigiwinAtmcBacklogDetail::getTmCategory).orElse("");

        // 定制 PATTERN==CUSTOM 类型任务卡详情
        if (TaskStringUtil.matchPatternIsCustom(tmPattern)) {
            return TaskStringUtil.CUSTOM_PATTERN_TASK;
        }
        return tmPattern + "_" + tmCategory;
    }

    private static boolean isProductionProgressReport(String tmTaskId, String tmActivityId) {
        if (!StringUtils.hasLength(tmTaskId) || !StringUtils.hasLength(tmActivityId)) {
            return false;
        }
        return tmTaskId.contains(CUSTOM_PROJECT_CODE_PRODUCETION_PROGRESS_REPORT) && tmActivityId.contains(
                CUSTOM_TASK_CODE_PRODUCETION_PROGRESS_REPORT);
    }

    /**
     * 判断是否是 ：定制 PATTERN==CUSTOM 类型任务卡详情
     *
     * @return
     */
    private static boolean matchPatternIsCustom(String tmPattern) {
        return TaskStringUtil.CUSTOM_PATTERN_TASK.equals(tmPattern);
    }

    public static String getTaskInNotificationSummaryContent(String locale, long count) {
        String content = "";
        if (count > 0L) {
            content = String.format(LocaleUtil.getMobileTextByKey(locale, "首页任务提示逾期"), count);
        } else {
            content = LocaleUtil.getMobileTextByKey(locale, "首页任务提示");
        }
        return content;
    }

    /**
     * 获取任务时间提示文字。
     * locale= zh_CN 时，
     * 18天=>18天
     * 1小时=>1小时
     * 3分钟=>3分钟
     * locale= zh_TW 时，
     * 18天=>18天
     * 1小时=>1小時
     * 3分钟=>3分鐘
     * locale= en_US 时，
     * 18天=>18 Days
     * 1小时=>1 Hour
     * 3分钟=>3 Minutes
     * <p>
     * 计算任务的超时规则是这样的：
     * 1.超时时间=现在的时间-任务结束时间，单位是毫秒；
     * 2.如果超时<=0分钟，则认为不超时；
     * 如果0分钟<超时时间<60分钟，则展示超时的分钟数；
     * 如果60分钟<=超时时间<24小时，则展示超时的小时数；
     * 如果24小时<超时时间，则展示超时的天数。
     * 3.另外，以上数字的结果都是整数，不做四舍五入，直接向下取整。比如超时1.5小时，展示超时1小时
     *
     * @param locale
     * @param taskEndTimeString 任务预计完成时间，格式为：YYYY-MM-DD HH:MM:SS
     * @return 倒计时xxx天/分钟 或者 逾期xxx天/分钟 的两类文字
     */
    public static String getTaskTimeHintText(String locale, String taskEndTimeString) {
        //        String regExp = "[^0-9]";
        //        Pattern pattern = Pattern.compile(regExp);
        //        String hintData = pattern.matcher(rawHintText).replaceAll("").trim();
        //        Integer hintDataNumber = Integer.parseInt(hintData);

        String hintType;
        long hintNumber;
        String hintUnit;
        LocalDateTime taskEndTime = DateTimeUtil.parseUseDefaultPattern(taskEndTimeString);
        LocalDateTime now = LocalDateTime.now();
        Duration duration = Duration.between(now, taskEndTime);
        // 未逾期的话，倒计时
        if (!duration.isNegative() || duration.isZero()) {
            hintType = LocaleUtil.getMobileTextByKey(locale, "倒计时");
        } else {
            hintType = LocaleUtil.getMobileTextByKey(locale, "逾期");
        }
        long durationSeconds = duration.abs().getSeconds();
        if (durationSeconds >= 60 && durationSeconds < 60 * 60) {
            hintUnit = LocaleUtil.getMobileTextByKey(locale, "分钟");
            hintNumber = durationSeconds / 60;
        } else if (durationSeconds >= 60 * 60 && durationSeconds < 24 * 60 * 60) {
            hintUnit = LocaleUtil.getMobileTextByKey(locale, "小时");
            hintNumber = duration.abs().toHours();
        } else {
            hintUnit = LocaleUtil.getMobileTextByKey(locale, "天");
            LocalDate nowDate = now.toLocalDate();
            LocalDate taskEndDate = taskEndTime.toLocalDate();
            hintNumber = Math.abs(nowDate.toEpochDay() - taskEndDate.toEpochDay());
        }
        return LocaleUtil.getMobileTextWithUnit(locale, hintType, String.valueOf(hintNumber),
                hintUnit, false, "");
    }

    /**
     * 获取任务时距的文字描述。
     *
     * @param startDateInDefaultPattern
     * @param endDateInDefaultPattern
     * @return
     * @see DateTimeUtil#getDateDistanceFromStartToEnd(String, String)
     */
    public static String getTaskDateDistanceText(String startDateInDefaultPattern,
                                                 String endDateInDefaultPattern) {
        if (!DateTimeUtil.isDateUseDefaultPattern(startDateInDefaultPattern)
                || !DateTimeUtil.isDateUseDefaultPattern(endDateInDefaultPattern)) {
            return "";
        }
        return DateTimeUtil.getDateDistanceFromStartToEnd(startDateInDefaultPattern,
                endDateInDefaultPattern);
    }
}
