package com.digiwin.mobile.mobileuibot.task.filter;

import com.digiwin.mobile.mobileuibot.common.string.StringUtil;
import com.digiwin.mobile.mobileuibot.core.common.EnvScope;
import com.digiwin.mobile.mobileuibot.openapi.designer.enums.DataVersionEnum;
import lombok.Data;
import org.springframework.data.annotation.Id;
import org.springframework.data.mongodb.core.mapping.Document;
import org.springframework.util.StringUtils;

import java.io.Serializable;
import java.util.List;

/**
 * <p>功能描述：任务过滤设置数据</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: TaskFilter
 * @Author: Zaregoto
 * @Date: 2021/11/20 18:58
 */
@Data
@Document(collection = "taskFilter")
public class TaskFilter implements Serializable {
    private static final long serialVersionUID = -4347695742053143376L;

    @Id
    private String id;
    private String source;
    /**
     * 任务过滤设置项类型。
     *
     * @see TaskFilterCategoryEnum
     */
    private String category;
    /**
     * 任务过滤使用的参数
     */
    private TaskFilterParam param;
    /**
     * 参数操作符
     *
     * @see TaskFilterParamOpEnum
     */
    private String paramOp;
    /**
     * 任务过滤设置项是否有效。
     * 本设置为true时，则本设置是开启有效的；为false时，则本设置是关闭无效的。
     * 它的优先级最高。
     */
    private Boolean valid;

    private List<EnvScope> envScopes;

    //***********从设计器来源时存储的相关信息******begin******************
    /**
     * 从哪来
     *
     * @see com.digiwin.mobile.mobileuibot.task.filter.TaskFilterFromEnum
     */
    private String from;
    private String createTime;
    private String updateTime;
    //************从设计器来源时存储的相关信息*******end****************

    /**
    * 应用code,注意：为空表示通用配置
    */
    private String appCode;
    /**
    * 数据版本
     * @see DataVersionEnum
    */
    private String version;

    public TaskFilter() {
    }

    /**
     * 根据当前启动的环境类型，判断当前页面过滤设置是否生效。
     *
     * @param appEnvType 系统当前启动环境类型。参考AppContext.envType
     * @return true-生效，false-不生效
     */
    public Boolean validWithAppEnvType(final String appEnvType) {
        // 如果没有配置页面可使用的环境，则表示不知道在哪个环境可使用，故直接返回false，表示不匹配
        if (this.getEnvScopes() == null || this.getEnvScopes().isEmpty()) {
            return false;
        }
        boolean result = false;
        for (EnvScope envScope : this.getEnvScopes()) {
            String type = envScope.getType();
            boolean envScopeValid = envScope.getValid();
            if (appEnvType.equals(type)) {
                result = envScopeValid;
                break;
            }
        }
        return result;
    }

    /**
     * 判断是否和当前页面过滤设置的pattern及category匹配
     *
     * @param tmPattern
     * @param tmCategory
     * @return
     */
    public Boolean matchInPatternAndCategory(final String tmPattern, final String tmCategory) {
        if (null == tmPattern || null == tmCategory) {
            return false;
        }
        if (this.param == null) {
            return false;
        }
        String paramTmPattern = StringUtil.valueOf(this.param.getTmPattern()).trim();
        String paramTmCategory = StringUtil.valueOf(this.param.getTmCategory()).trim();
        if (!StringUtils.hasLength(paramTmCategory) && !StringUtils.hasLength(paramTmPattern)) {
            return false;
        }
        return (!StringUtils.hasLength(paramTmPattern) || paramTmPattern.equals(tmPattern))
                && (!StringUtils.hasLength(paramTmCategory) || paramTmCategory.equals(tmCategory));
    }

    /**
     * 判断是否和当前页面过滤设置的taskId匹配
     *
     * @param tmTaskId
     * @return
     */
    public Boolean matchInTaskId(final String tmTaskId) {
        if (null == tmTaskId) {
            return false;
        }
        if (this.param == null) {
            return false;
        }
        String paramTmTaskId = StringUtil.valueOf(this.param.getTmTaskId()).trim();
        return TaskFilter.match(tmTaskId, paramTmTaskId, this.param.getTmTaskIdMatch());
    }

    /**
     * 判断是否和当前页面过滤设置的taskId及activityId匹配
     *
     * @param tmTaskId
     * @param tmActivityId
     * @return
     */
    public Boolean matchInTaskIdAndActivityId(final String tmTaskId, final String tmActivityId) {
        if (null == tmTaskId || null == tmActivityId) {
            return false;
        }
        if (this.param == null) {
            return false;
        }
        String paramTmTaskId = StringUtil.valueOf(this.param.getTmTaskId()).trim();
        String paramTmActivityId = StringUtil.valueOf(this.param.getTmActivityId()).trim();
        return (!StringUtils.hasLength(paramTmTaskId) || TaskFilter.match(tmTaskId, paramTmTaskId, this.param.getTmTaskIdMatch()))
                && TaskFilter.match(tmActivityId, paramTmActivityId, this.param.getTmActivityIdMatch());
    }

    /**
     * 参数是否匹配成功
     *
     * @param target   目标对象
     * @param original 原对象
     * @param match    匹配方式
     * @return
     */
    public static boolean match(String target, String original, String match) {
        boolean isMatch = false;
        if (original == null || target == null) {
            return isMatch;
        }
        TaskFilterMatchEnum matchEnum = TaskFilterMatchEnum.getEnumByValue(match);
        if (null == matchEnum) {
            matchEnum = TaskFilterMatchEnum.EQUALS;
        }
        switch (matchEnum) {
            case PREFIX:
                isMatch = target.startsWith(original);
                break;
            case SUFFIX:
                isMatch = target.endsWith(original);
                break;
            case CONTAIN:
                isMatch = target.contains(original);
                break;
            case EQUALS:
                isMatch = target.equals(original);
                break;
            default:
                break;
        }
        return isMatch;
    }
}
