package com.digiwin.mobile.mobileuibot.user;

import com.digiwin.mobile.mobileuibot.api.ApiResponse;
import com.digiwin.mobile.mobileuibot.api.BaseRequest;
import com.digiwin.mobile.mobileuibot.locale.service.LocaleService;
import com.digiwin.mobile.mobileuibot.proxy.iam.model.DigiwinIamAccountRegisterResponse;
import com.digiwin.mobile.mobileuibot.proxy.iam.model.DigiwinIamUserInfo;
import com.digiwin.mobile.mobileuibot.proxy.iam.service.DigiwinIamProxyService;
import com.digiwin.mobile.mobileuibot.user.enums.IdentityCodeAcceptTypeEnum;
import com.digiwin.mobile.mobileuibot.user.enums.VerificationTypeEnum;
import com.digiwin.mobile.mobileuibot.user.request.*;
import com.digiwin.mobile.mobileuibot.user.service.UserService;
import com.digiwin.mobile.mobileuibot.user.utils.EmailValidator;
import com.digiwin.mobile.mobileuibot.user.utils.PhoneValidator;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.validation.Valid;
import java.util.HashMap;
import java.util.Map;

/**
 * <p>功能描述：用户相关操作控制层</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: UserController.java
 * @Author: wangjwc
 * @Date: created at 2025/8/4 15:07
 */
@Slf4j
@RestController
@RequestMapping("/mobile/v1/user")
public class UserController {

    @Autowired
    private UserService userService;

    @Autowired
    private DigiwinIamProxyService digiwinIamProxyService;

    @Autowired
    private LocaleService localeService;

    /**
     * <p>功能描述：查询用户信息</p>
     *
     * @param request 请求参数
     * @return 用户信息
     */
    @PostMapping("/info")
    public ApiResponse<DigiwinIamUserInfo> queryUserInfo(@RequestBody BaseRequest request) {
        return ApiResponse.buildOK(digiwinIamProxyService.queryUserInfo(request.getIamUserToken(), request.getUserId()));
    }

    /**
     * <p>功能描述：更新用户双重验证状态</p>
     *
     * @param request 请求参数
     * @return 结果
     */
    @PostMapping("/doubleCheck/update")
    public ApiResponse<Boolean> doubleCheckUpdate(@RequestBody DoubleCheckUpdateReq request) {
        // 双重验证开启时，认证类型必须在枚举中存在
        if (BooleanUtils.isTrue(request.getDoubleCheck()) && IdentityCodeAcceptTypeEnum.notExist(request.getIdentityCodeAcceptType())) {
            return ApiResponse.buildError(localeService.getLanguageValue(request.getLocale(), "identityCodeAcceptType字段值不合法"));
        }

        // 查询用户信息
        DigiwinIamUserInfo user = digiwinIamProxyService.queryUserInfo(request.getIamUserToken(), request.getUserId());
        if (user == null) {
            return ApiResponse.buildError(localeService.getLanguageValue(request.getLocale(), "用户信息不存在"));
        }
        // 双重验证开启时，判断认证类型identityCodeAcceptType在用户信息中的值是否存在（邮箱、手机号）
        if (BooleanUtils.isTrue(request.getDoubleCheck())) {
            if (IdentityCodeAcceptTypeEnum.EMAIL.getValue().equals(request.getIdentityCodeAcceptType())) {
                if (StringUtils.isBlank(user.getEmail())) {
                    return ApiResponse.buildError(localeService.getLanguageValue(request.getLocale(), "当前账号尚未绑定邮箱，请选择其他验证方式"));
                }
            } else if (IdentityCodeAcceptTypeEnum.MOBILE.getValue().equals(request.getIdentityCodeAcceptType())) {
                if (StringUtils.isBlank(user.getTelephone())) {
                    return ApiResponse.buildError(localeService.getLanguageValue(request.getLocale(), "当前账号尚未绑定手机号，请选择其他验证方式"));
                }
            } else {
                return ApiResponse.buildError(localeService.getLanguageValue(request.getLocale(), "identityCodeAcceptType字段值不合法"));
            }
        }

        // 更新用户双重验证状态
        userService.doubleCheckUpdate(request, user);
        return ApiResponse.buildOK();
    }

    /**
     * <p>功能描述：检查邮箱、手机号是否存在</p>
     *
     * @param request 请求参数
     * @return 检查结果
     */
    @PostMapping("/account/exist")
    public ApiResponse<DigiwinIamAccountRegisterResponse> accountExistCheck(@Valid @RequestBody AccountExistCheckReq request) {
        if (VerificationTypeEnum.EMAIL.getValue().equals(request.getType())) {
            // 验证邮箱
            if (!EmailValidator.isValidBasic(request.getAccount())) {
                return ApiResponse.buildError(localeService.getLanguageValue(request.getLocale(), "邮箱格式不正确"));
            }
        } else if (VerificationTypeEnum.MOBILEPHONE.getValue().equals(request.getType())) {
            // 验证手机号
            if (!PhoneValidator.validatePhoneByRegion(request.getAccount())) {
                return ApiResponse.buildError(localeService.getLanguageValue(request.getLocale(), "手机号格式不正确"));
            }
        }
        return ApiResponse.buildOK(userService.accountExistCheck(request));
    }

    /**
     * <p>功能描述：获取手机、邮箱验证码</p>
     *
     * @param request 请求参数
     * @return 结果
     */
    @PostMapping("/verificationCode")
    public ApiResponse<Boolean> verificationCode(@Valid @RequestBody VerificationCodeReq request) {
        if (VerificationTypeEnum.EMAIL.getValue().equals(request.getType())) {
            // 验证邮箱
            if (!EmailValidator.isValidBasic(request.getAccount())) {
                return ApiResponse.buildError(localeService.getLanguageValue(request.getLocale(), "邮箱格式不正确"));
            }
        } else if (VerificationTypeEnum.MOBILEPHONE.getValue().equals(request.getType())) {
            // 验证手机号
            if (!PhoneValidator.validatePhoneByRegion(request.getAccount())) {
                return ApiResponse.buildError(localeService.getLanguageValue(request.getLocale(), "手机号格式不正确"));
            }
        }
        userService.verificationCode(request);
        return ApiResponse.buildOK();
    }

    /**
     * <p>功能描述：检查验证码并返回新验证码</p>
     *
     * @param request 请求参数
     * @return 结果
     */
    @PostMapping("/checkVerificationCode/renew")
    public ApiResponse<Map<String, Object>> checkVerificationCodeAndRenew(@Valid @RequestBody CheckVerificationCodeReq request) {
        if (!EmailValidator.isValidBasic(request.getAccount()) && !PhoneValidator.validatePhoneByRegion(request.getAccount())) {
            return ApiResponse.buildError(localeService.getLanguageValue(request.getLocale(), "手机号或邮箱格式不正确"));
        }
        Map<String, Object> map = new HashMap<>();
        map.put("code", userService.checkVerificationCodeAndRenew(request));
        return ApiResponse.buildOK(map);
    }

    /**
     * <p>功能描述：忘记/修改密码（不需要旧密码）</p>
     *
     * @param request 请求参数
     * @return 结果
     */
    @PostMapping("/password/update")
    public ApiResponse<Boolean> passwordUpdate(@Valid @RequestBody PasswordUpdateReq request) {
        if (!EmailValidator.isValidBasic(request.getAccount()) && !PhoneValidator.validatePhoneByRegion(request.getAccount())) {
            return ApiResponse.buildError(localeService.getLanguageValue(request.getLocale(), "手机号或邮箱格式不正确"));
        }
        userService.passwordUpdate(request);
        return ApiResponse.buildOK();
    }
}
