package com.digiwin.mobile.mobileuibot.user.service.impl;

import com.digiwin.mobile.mobileuibot.common.context.AppContext;
import com.digiwin.mobile.mobileuibot.common.context.AppRequestContext;
import com.digiwin.mobile.mobileuibot.common.crypto.aes.AESUtil;
import com.digiwin.mobile.mobileuibot.common.crypto.digest.DigestUtil;
import com.digiwin.mobile.mobileuibot.common.exception.ServiceException;
import com.digiwin.mobile.mobileuibot.common.string.StringUtil;
import com.digiwin.mobile.mobileuibot.locale.service.LocaleService;
import com.digiwin.mobile.mobileuibot.proxy.emc.model.CheckVerificationCodeResponse;
import com.digiwin.mobile.mobileuibot.proxy.emc.service.DigiwinEmcProxyService;
import com.digiwin.mobile.mobileuibot.proxy.iam.model.DigiwinIamAccountRegisterResponse;
import com.digiwin.mobile.mobileuibot.proxy.iam.model.DigiwinIamUserInfo;
import com.digiwin.mobile.mobileuibot.proxy.iam.service.DigiwinIamProxyService;
import com.digiwin.mobile.mobileuibot.user.enums.VerificationTypeEnum;
import com.digiwin.mobile.mobileuibot.user.request.*;
import com.digiwin.mobile.mobileuibot.user.service.UserService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.BooleanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

/**
 * <p>功能描述：用户服务实现类</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: UserServiceImpl.java
 * @Author: wangjwc
 * @Date: created at 2025/8/5 15:17
 */
@Component
@Slf4j
public class UserServiceImpl implements UserService {

    @Autowired
    private DigiwinIamProxyService digiwinIamProxyService;

    @Autowired
    private DigiwinEmcProxyService digiwinEmcProxyService;

    @Autowired
    private LocaleService localeService;

    @Override
    public void doubleCheckUpdate(DoubleCheckUpdateReq request, DigiwinIamUserInfo user) {
        // 构建更新入参
        Map<String, Object> params = new HashMap<>(4);
        params.put("id", user.getId());
        params.put("name", user.getName());
        if (BooleanUtils.isTrue(request.getDoubleCheck())) {
            params.put("doubleCheck", true);
            params.put("identityCodeAcceptType", request.getIdentityCodeAcceptType());
        } else {
            params.put("doubleCheck", false);
        }
        // 对应类型下变更双重验证状态
        digiwinIamProxyService.doubleCheckUpdate(request.getIamUserToken(), params);
    }

    @Override
    public DigiwinIamAccountRegisterResponse accountExistCheck(AccountExistCheckReq request) {
        if (VerificationTypeEnum.EMAIL.getValue().equals(request.getType())) {
            // 验证邮箱是否存在
            return digiwinIamProxyService.emailExistCheck(request.getIamUserToken(), request.getAccount());
        } else if (VerificationTypeEnum.MOBILEPHONE.getValue().equals(request.getType())) {
            // 验证手机号是否存在
            return digiwinIamProxyService.mobilephoneExistCheck(request.getIamUserToken(), request.getAccount());
        } else {
            // 其它，不存在
            return new DigiwinIamAccountRegisterResponse();
        }
    }

    @Override
    public void verificationCode(VerificationCodeReq request) {
        // 发送验证码
        digiwinEmcProxyService.sendVerificationCode(request.getIamUserToken(), request.getType(), request.getAccount(), request.getScene());
    }

    @Override
    public String checkVerificationCodeAndRenew(CheckVerificationCodeReq request) {
        // 构建更新入参
        Map<String, Object> params = new HashMap<>(3);
        params.put("account", request.getAccount());
        params.put("scene", request.getScene());
        params.put("identifyingCode", request.getIdentifyingCode());
        // 校验验证码
        CheckVerificationCodeResponse response = digiwinEmcProxyService.checkVerificationCodeAndRenew(request.getIamUserToken(), params);
        if (response == null || !response.isSuccess()) {
            // response = {"result":"failed:账号或验证码有误，请检查！！"}
            String result = Optional.ofNullable(response).map(CheckVerificationCodeResponse::getResult).map(s -> s.replace("failed:", "")).orElse("验证码有误，请重新输入");
            throw new ServiceException(localeService.getLanguageValue(AppRequestContext.getContextEntity().getLocale(), result));
        }
        return StringUtil.valueOf(response.getCode());

    }

    @Override
    public void passwordUpdate(PasswordUpdateReq request) {
        String encryptPwd = "";
        //解密
        try {
            encryptPwd = AESUtil.decrypt(AppContext.getApiUrlSetting().getAesKey(), request.getPassword());
        } catch (Exception e) {
            encryptPwd = request.getPassword();
        }
        Map<String, Object> params = new HashMap<>(3);
        params.put("account", request.getAccount());
        params.put("password", DigestUtil.getDigestedStringForDwIamLoginByAes(encryptPwd));
        params.put("verificationCode", request.getVerificationCode());
        // 忘记/修改密码（不需要旧密码）
        digiwinIamProxyService.passwordUpdate(request.getIamUserToken(), params);
    }
}
