package com.digiwin.mobile.mobileuibot.user.utils;

import java.util.regex.Pattern;

/**
 * <p>功能描述：邮箱验证工具类</p>
 * <p>提供多种邮箱验证方式，包括基本格式验证、严格格式验证等</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: EmailValidator.java
 * @Author: wangjwc
 * @Date: created at 2025/8/6 19:56
 */
public class EmailValidator {

    // 私有构造方法，防止工具类被实例化
    private EmailValidator() {
        throw new IllegalStateException("工具类无需实例化");
    }

    // 基本邮箱格式正则（RFC 5322标准）
    private static final String BASIC_EMAIL_REGEX = 
        "^[a-zA-Z0-9_!#$%&'*+/=?`{|}~^-]+(?:\\.[a-zA-Z0-9_!#$%&'*+/=?`{|}~^-]+)*@" + 
        "(?:[a-zA-Z0-9-]+\\.)+[a-zA-Z]{2,}$";
    
    // 严格邮箱格式正则（禁止连续点号，限制特殊字符位置）
    private static final String STRICT_EMAIL_REGEX = 
        "^(?!.*\\.{2})[a-zA-Z0-9]+[a-zA-Z0-9._!#$%&'*+/=?`{|}~^-]*[a-zA-Z0-9]+@" + 
        "(?!.*\\.{2})[a-zA-Z0-9-]+(?:\\.[a-zA-Z0-9-]+)*\\.[a-zA-Z]{2,}$";

    /**
     * 基本邮箱格式验证
     * 
     * <p>符合RFC 5322标准，允许大多数合法邮箱格式</p>
     * 
     * @param email 邮箱地址
     * @return true表示有效，false表示无效
     */
    public static boolean isValidBasic(String email) {
        if (email == null || email.trim().isEmpty()) {
            return false;
        }
        return Pattern.matches(BASIC_EMAIL_REGEX, email);
    }

    /**
     * 严格邮箱格式验证
     * 
     * <p>禁止本地部分和域名部分开头/结尾的特殊字符，禁止连续点号</p>
     * 
     * @param email 邮箱地址
     * @return true表示有效，false表示无效
     */
    public static boolean isValidStrict(String email) {
        if (email == null || email.trim().isEmpty()) {
            return false;
        }
        return Pattern.matches(STRICT_EMAIL_REGEX, email);
    }

    public static void main(String[] args) {
        // 测试用例
        String[] testEmails = {
            "user@example.com",             // 有效
            "first.last@sub.domain.co.uk",  // 有效
            "john.doe123@mail-server.com",  // 有效
            "user+tag@example.com",         // 有效
            "invalid.email@",               // 无效
            "no@tld.",                     // 无效
            "@domain.com",                 // 无效
            "user@.com",                   // 无效
            "user@inval..com",             // 无效
            "spaces are@invalid.com",      // 无效
            "john.doe@gmail.com",          // 常用服务商
            "test@mycompany.com"           // 非常用服务商
        };

        // 测试基本验证
        System.out.println("=== 基本邮箱验证 ===");
        for (String email : testEmails) {
            System.out.printf("%-30s -> %s%n", email, isValidBasic(email) ? "有效" : "无效");
        }

        // 测试严格验证
        System.out.println("\n=== 严格邮箱验证 ===");
        for (String email : testEmails) {
            System.out.printf("%-30s -> %s%n", email, isValidStrict(email) ? "有效" : "无效");
        }
    }
}