package com.digiwin.mobile.mobileuibot.user.utils;

import com.digiwin.mobile.mobileuibot.common.context.AppContext;
import com.digiwin.mobile.mobileuibot.common.context.AppEnvDeployAreaEnum;

import java.util.regex.Pattern;

/**
 * <p>功能描述：手机号验证工具类</p>
 * <p>提供中国大陆和台湾地区手机号的验证功能</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: PhoneValidator.java
 * @Author: wangjwc
 * @Date: created at 2025/8/6 18:00
 */
public class PhoneValidator {

    // 私有构造方法，防止工具类被实例化
    private PhoneValidator() {
        throw new IllegalStateException("工具类无需实例化");
    }

    /**
     * 验证中国大陆手机号
     *
     * <p>规则：以1开头，第二位为3-9，共11位数字</p>
     *
     * @param phone 手机号码字符串
     * @return true表示有效，false表示无效
     */
    public static boolean isValidMainlandPhone(String phone) {
        if (phone == null || phone.trim().isEmpty()) {
            return false;
        }

        // 去除所有空格和特殊字符
        String cleanedPhone = phone.replaceAll("[\\s-()]", "");

        // 中国大陆手机号正则表达式
        String regex = "^1[3-9]\\d{9}$";
        return Pattern.matches(regex, cleanedPhone);
    }

    /**
     * 验证台湾地区手机号
     *
     * <p>规则：以09开头，共10位数字</p>
     *
     * @param phone 手机号码字符串
     * @return true表示有效，false表示无效
     */
    public static boolean isValidTaiwanPhone(String phone) {
        if (phone == null || phone.trim().isEmpty()) {
            return false;
        }

        // 去除所有空格和特殊字符
        String cleanedPhone = phone.replaceAll("[\\s-()]", "");

        // 台湾手机号正则表达式
        String regex = "^09\\d{8}$";
        return Pattern.matches(regex, cleanedPhone);
    }

    /**
     * 根据地区自动验证手机号
     *
     * @param phone 手机号码
     * @return 验证结果
     */
    public static boolean validatePhoneByRegion(String phone) {
        // 环境部署的地区
        String envDeployArea = AppContext.getEnvDeployArea();
        if (AppEnvDeployAreaEnum.CN.name().equals(envDeployArea)) {
            // 大陆
            return isValidMainlandPhone(phone);
        } else if (AppEnvDeployAreaEnum.TW.name().equals(envDeployArea)) {
            // 台湾
            return isValidTaiwanPhone(phone);
        }
        return false;
    }
}