package com.digiwin.mobile.mobileuibot.versionManage;

import com.digiwin.mobile.mobileuibot.api.ApiResponse;
import com.digiwin.mobile.mobileuibot.api.PageResult;
import com.digiwin.mobile.mobileuibot.locale.service.LocaleService;
import com.digiwin.mobile.mobileuibot.model.db2.AppConfig;
import com.digiwin.mobile.mobileuibot.versionManage.request.AppConfigAddRequest;
import com.digiwin.mobile.mobileuibot.versionManage.request.AppConfigPageRequest;
import com.digiwin.mobile.mobileuibot.versionManage.request.AppConfigUpdateRequest;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.List;

/**
 * <p>功能描述：版本管理系统 -- 应用管理模块控制层</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: VersionManageApplicationController.java
 * @Author: wangjwc
 * @Date: created at 2025/8/8 15:14
 */
@Slf4j
@RestController
@RequestMapping("/mobile/v2/version/manage/application")
public class VersionManageApplicationController {

    @Autowired
    private LocaleService localeService;

    @Autowired
    private VersionManageApplicationService versionManageApplicationService;

    /**
     * <p>功能描述：获取应用列表</p>
     *
     * @param request 请求参数
     * @return ApiResponse<List < AppConfig>> 应用列表
     */
    @PostMapping(value = "/list")
    public ApiResponse<List<AppConfig>> list(@RequestBody AppConfigPageRequest request) {
        return ApiResponse.buildOK(versionManageApplicationService.list(request));
    }

    /**
     * <p>功能描述：获取应用列表 -- 分页</p>
     *
     * @param pageRequest  请求参数
     * @return ApiResponse<PageResult < AppConfig>> 应用列表
     */
    @PostMapping(value = "/list/page")
    public ApiResponse<PageResult<AppConfig>> listPage(@RequestBody AppConfigPageRequest pageRequest) {
        return ApiResponse.buildOK(versionManageApplicationService.listPage(pageRequest));
    }

    /**
     * <p>功能描述：根据ID获取应用详情</p>
     *
     * @param id 应用id
     * @return ApiResponse<AppConfig> 应用详情
     */
    @GetMapping(value = "/select/{id}")
    public ApiResponse<AppConfig> selectById(@PathVariable("id") Long id) {
        if (null == id) {
            return ApiResponse.buildError(localeService.getLanguageValue("查询数据不存在"));
        }
        AppConfig appConfig = versionManageApplicationService.selectById(id);
        if (null == appConfig) {
            return ApiResponse.buildError(localeService.getLanguageValue("查询数据不存在"));
        }
        return ApiResponse.buildOK(appConfig);
    }

    /**
     * <p>功能描述：新增应用</p>
     *
     * @param addRequest 新增应用请求参数
     * @return ApiResponse<Boolean> 新增结果
     */
    @PostMapping(value = "/add")
    public ApiResponse<Boolean> add(@Valid @RequestBody AppConfigAddRequest addRequest) {
        return versionManageApplicationService.add(addRequest);
    }

    /**
     * <p>功能描述：修改应用</p>
     *
     * @param updateRequest 修改应用请求参数
     * @return ApiResponse<Boolean> 修改结果
     */
    @PostMapping(value = "/update")
    public ApiResponse<Boolean> update(@Valid @RequestBody AppConfigUpdateRequest updateRequest) {
        return versionManageApplicationService.update(updateRequest);
    }

    /**
     * <p>功能描述：删除应用</p>
     *
     * @param id 应用id
     * @return ApiResponse<AppConfig> 删除结果
     */
    @GetMapping(value = "/delete/{id}")
    public ApiResponse<AppConfig> deleteById(@PathVariable("id") Long id) {
        if (null == id) {
            return ApiResponse.buildError(localeService.getLanguageValue("数据不存在"));
        }
        AppConfig appConfig = versionManageApplicationService.selectById(id);
        if (null == appConfig) {
            return ApiResponse.buildError(localeService.getLanguageValue("数据不存在"));
        }
        versionManageApplicationService.deleteById(id);
        return ApiResponse.buildOK(appConfig);
    }

    /**
     * <p>功能描述：更新Appid，内部接口，不可暴露</p>
     */
    @GetMapping(value = "/update/{appId}/{id}")
    public ApiResponse<Boolean> updateAppId(@PathVariable("id") Long id, @PathVariable("appId") String appId,
                                             @RequestParam("wordOfCommand") String wordOfCommand) {
        if (!"真嘟假嘟".equals(wordOfCommand)) {
            return ApiResponse.buildError("口令错误");
        }
        if (null == id) {
            return ApiResponse.buildError(localeService.getLanguageValue("数据不存在"));
        }
        if (StringUtils.isEmpty(appId)) {
            return ApiResponse.buildError(localeService.getLanguageValue("appid不能为空"));
        }
        AppConfig appConfig = versionManageApplicationService.selectById(id);
        if (null == appConfig) {
            return ApiResponse.buildError(localeService.getLanguageValue("数据不存在"));
        }
        versionManageApplicationService.updateAppId(id, appId);
        return ApiResponse.buildOK();
    }
}
