package com.digiwin.mobile.mobileuibot.versionManage;

import com.digiwin.mobile.mobileuibot.api.ApiResponse;
import com.digiwin.mobile.mobileuibot.api.PageResult;
import com.digiwin.mobile.mobileuibot.config.mysql.Db2Transactional;
import com.digiwin.mobile.mobileuibot.locale.service.LocaleService;
import com.digiwin.mobile.mobileuibot.model.db2.AppConfig;
import com.digiwin.mobile.mobileuibot.versionManage.request.AppConfigAddRequest;
import com.digiwin.mobile.mobileuibot.versionManage.request.AppConfigPageRequest;
import com.digiwin.mobile.mobileuibot.versionManage.request.AppConfigUpdateRequest;
import com.digiwin.mobile.mobileuibot.versionManage.service.AppConfigService;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.UUID;

/**
 * <p>功能描述：版本管理系统 -- 应用管理服务接口实现类</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: VersionManageApplicationServiceImpl.java
 * @Author: wangjwc
 * @Date: created at 2025/8/8 17:48
 */
@Service
public class VersionManageApplicationServiceImpl implements VersionManageApplicationService {

    @Autowired
    private AppConfigService appConfigService;

    @Autowired
    private LocaleService localeService;

    @Override
    public List<AppConfig> list(AppConfigPageRequest request) {
        return appConfigService.list(request);
    }

    @Override
    public PageResult<AppConfig> listPage(AppConfigPageRequest pageRequest) {
        return PageResult.createPageResult(appConfigService.listPage(pageRequest));
    }

    @Override
    public AppConfig selectById(Long id) {
        return appConfigService.selectById(id);
    }

    /**
     * 生成appId
     */
    private String generateAppId() {
        String appId = null;
        for (int i = 0; i < 50; i++) {
            // 生成标准UUID并移除连字符"-" 拼接 appid_{uuid}
            appId = "appid_" + UUID.randomUUID().toString().replace("-", "");
            // 检验appId是否存在
            if (appConfigService.getByCode(appId) == null) {
                break;
            }
            appId = null;
        }
        return appId;
    }

    @Override
    @Db2Transactional(rollbackFor = Exception.class)
    public ApiResponse<Boolean> add(AppConfigAddRequest addRequest) {
        // 生成appid
        String appId = this.generateAppId();
        if (StringUtils.isBlank(appId)) {
            return ApiResponse.buildError(localeService.getLanguageValue("生成appId失败"));
        }
        appConfigService.add(appId, addRequest);
        return ApiResponse.buildOK();
    }

    @Override
    @Db2Transactional(rollbackFor = Exception.class)
    public ApiResponse<Boolean> update(AppConfigUpdateRequest updateRequest) {
        // 根据id，code重查询数据是否存在
        AppConfig appConfig = appConfigService.selectByIdAndCode(updateRequest.getId(), updateRequest.getCode());
        if (appConfig == null) {
            return ApiResponse.buildError(localeService.getLanguageValue("应用数据不存在"));
        }
        // 更新数据
        appConfigService.update(appConfig, updateRequest);
        return ApiResponse.buildOK();
    }

    @Override
    @Db2Transactional(rollbackFor = Exception.class)
    public void deleteById(Long id) {
        appConfigService.deleteById(id);
    }

    @Override
    @Db2Transactional(rollbackFor = Exception.class)
    public void updateAppId(Long id, String appId) {
        appConfigService.updateAppId(id, appId);
    }
}