package com.digiwin.mobile.mobileuibot.versionManage;

import com.digiwin.mobile.mobileuibot.api.ApiResponse;
import com.digiwin.mobile.mobileuibot.api.PageResult;
import com.digiwin.mobile.mobileuibot.dto.AppClientPageDto;
import com.digiwin.mobile.mobileuibot.locale.service.LocaleService;
import com.digiwin.mobile.mobileuibot.versionManage.request.AppClientAddRequest;
import com.digiwin.mobile.mobileuibot.versionManage.request.AppClientUpdateRequest;
import com.digiwin.mobile.mobileuibot.versionManage.response.AppClientResponse;
import com.digiwin.mobile.mobileuibot.versionManage.response.AppUpdateResponse;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.List;

/**
 * <p>功能描述：版本管理系统 -- 版本管理控制器</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: VersionManageClientController.java
 * @Author: wangjwc
 * @Date: created at 2025/8/12 14:01
 */
@Slf4j
@RestController
@RequestMapping("/mobile/v2/version/manage/client")
public class VersionManageClientController {

    @Autowired
    private VersionManageClientService versionManageClientService;

    @Autowired
    private LocaleService localeService;

    /**
     * <p>功能描述：版本管理列表 -- 分页</p>
     *
     * @param pageRequest  请求参数
     * @return ApiResponse<PageResult < AppClientResponse>> 版本管理列表
     */
    @PostMapping(value = "/list/page")
    public ApiResponse<PageResult<AppClientResponse>> listPage(@RequestBody AppClientPageDto pageRequest) {
        return ApiResponse.buildOK(versionManageClientService.listPage(pageRequest));
    }

    /**
     * <p>功能描述：版本管理详情</p>
     *
     * @param id 版本管理id
     * @return ApiResponse<AppClientResponse> 版本管理详情
     */
    @GetMapping(value = "/select/{id}")
    public ApiResponse<AppClientResponse> selectById(@PathVariable("id") Long id) {
        if (null == id) {
            return ApiResponse.buildError(localeService.getLanguageValue("查询数据不存在"));
        }
        AppClientResponse appClientResponse = versionManageClientService.queryById(id);
        if (null == appClientResponse) {
            return ApiResponse.buildError(localeService.getLanguageValue("查询数据不存在"));
        }
        return ApiResponse.buildOK(appClientResponse);
    }

    /**
     * <p>功能描述：新增版本</p>
     *
     * @param addRequest 新增版本请求参数
     * @return ApiResponse<Boolean> 是否成功
     */
    @PostMapping(value = "/add")
    public ApiResponse<Boolean> add(/*@Valid*/ @RequestBody AppClientAddRequest addRequest) {
        if (null == addRequest.getAppId() || StringUtils.isBlank(addRequest.getVersion())) {
            return ApiResponse.buildError(localeService.getLanguageValue("应用、版本号不能为空"));
        }
        return versionManageClientService.add(addRequest);
    }

    /**
     * <p>功能描述：更新版本</p>
     *
     * @param updateRequest 更新版本请求参数
     * @return ApiResponse<Boolean> 是否成功
     */
    @PostMapping(value = "/update")
    public ApiResponse<Boolean> update(/*@Valid*/ @RequestBody AppClientUpdateRequest updateRequest) {
        if (null == updateRequest.getId()) {
            return ApiResponse.buildError(localeService.getLanguageValue("数据不存在"));
        }
        if (null == updateRequest.getAppId() || StringUtils.isBlank(updateRequest.getVersion())) {
            return ApiResponse.buildError(localeService.getLanguageValue("应用、版本号不能为空"));
        }
        return versionManageClientService.update(updateRequest);
    }

    /**
     * <p>功能描述：发布版本</p>
     *
     * @param releaseRequest 发布版本请求参数
     * @return ApiResponse<Boolean> 是否成功
     */
    @PostMapping(value = "/release")
    public ApiResponse<Boolean> release(@Valid @RequestBody AppClientUpdateRequest releaseRequest) {
        return versionManageClientService.release(releaseRequest);
    }

    /**
     * <p>功能描述：取消发布版本</p>
     *
     * @param id 版本id
     * @return ApiResponse<Boolean> 是否成功
     */
    @GetMapping(value = "/unrelease/{id}")
    public ApiResponse<Boolean> unrelease(@PathVariable("id") Long id) {
        if (null == id) {
            return ApiResponse.buildError(localeService.getLanguageValue("数据不存在"));
        }
        return versionManageClientService.unrelease(id);
    }

    /**
     * <p>功能描述：删除版本</p>
     *
     * @param id 删除版本id
     * @return ApiResponse<Boolean> 是否成功
     */
    @GetMapping(value = "/delete/{id}")
    public ApiResponse<Boolean> deleteById(@PathVariable("id") Long id) {
        if (null == id) {
            return ApiResponse.buildError(localeService.getLanguageValue("数据不存在"));
        }
        return versionManageClientService.deleteById(id);
    }

    /**
     * <p>功能描述：查询历史下发条件配置</p>
     *
     * @param id 版本id
     * @return ApiResponse<AppUpdateResponse> 历史下发条件配置
     */
    @GetMapping(value = "/update/history/{id}")
    public ApiResponse<List<AppUpdateResponse>> updateHistoryById(@PathVariable("id") Long id) {
        if (null == id) {
            return ApiResponse.buildError(localeService.getLanguageValue("数据不存在"));
        }
        return versionManageClientService.updateHistoryById(id);
    }


}
