package com.digiwin.mobile.mobileuibot.versionManage;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.digiwin.mobile.mobileuibot.api.ApiResponse;
import com.digiwin.mobile.mobileuibot.api.PageResult;
import com.digiwin.mobile.mobileuibot.config.mysql.Db2Transactional;
import com.digiwin.mobile.mobileuibot.dto.AppClientPageDto;
import com.digiwin.mobile.mobileuibot.dto.AppClientVo;
import com.digiwin.mobile.mobileuibot.locale.service.LocaleService;
import com.digiwin.mobile.mobileuibot.model.db2.AppClient;
import com.digiwin.mobile.mobileuibot.model.db2.AppConfig;
import com.digiwin.mobile.mobileuibot.versionManage.enums.*;
import com.digiwin.mobile.mobileuibot.versionManage.request.AppClientAddRequest;
import com.digiwin.mobile.mobileuibot.versionManage.request.AppClientUpdateRequest;
import com.digiwin.mobile.mobileuibot.versionManage.response.AppClientResponse;
import com.digiwin.mobile.mobileuibot.versionManage.response.AppUpdateResponse;
import com.digiwin.mobile.mobileuibot.versionManage.service.AppClientService;
import com.digiwin.mobile.mobileuibot.versionManage.service.AppConfigService;
import com.digiwin.mobile.mobileuibot.versionManage.service.AppUpdateService;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * <p>功能描述：版本管理系统 -- 版本管理服务接口实现类</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: VersionManageClientServiceImpl.java
 * @Author: wangjwc
 * @Date: created at 2025/8/12 14:01
 */
@Service
public class VersionManageClientServiceImpl implements VersionManageClientService {

    @Autowired
    private AppConfigService appConfigService;

    @Autowired
    private AppClientService appClientService;

    @Autowired
    private AppUpdateService appUpdateService;

    @Autowired
    private LocaleService localeService;

    @Override
    public PageResult<AppClientResponse> listPage(AppClientPageDto pageRequest) {
        Page<AppClientVo> page = appClientService.listPage(pageRequest);
        PageResult<AppClientResponse> pageResult = new PageResult<>();
        if (CollectionUtils.isEmpty(page.getRecords())) {
            pageResult.setData(new ArrayList<>());
        } else {
            pageResult.setData(page.getRecords().stream().map(AppClientResponse::convert).collect(Collectors.toList()));
        }
        pageResult.setTotal(page.getTotal());
        pageResult.setPageNum(page.getCurrent());
        pageResult.setPageSize(page.getSize());
        pageResult.setTotalPage(page.getPages());
        return pageResult;
    }

    @Override
    public AppClientResponse queryById(Long id) {
        AppClientVo appClientVo = appClientService.queryById(id);
        if (appClientVo == null) {
            return null;
        }
        return AppClientResponse.convert(appClientVo);
    }

    @Override
    @Db2Transactional(rollbackFor = Exception.class)
    public ApiResponse<Boolean> add(AppClientAddRequest addRequest) {
        // 校验入参
        // 应用id是否存在
        AppConfig appConfig = appConfigService.selectById(addRequest.getAppId());
        if (appConfig == null) {
            return ApiResponse.buildError(localeService.getLanguageValue("应用不存在"));
        }
//        ApiResponse<Boolean> checkResult = this.verifyAddParam(addRequest);
//        if (checkResult.isError()) {
//            return checkResult;
//        }

        // 添加数据
        AppClient appClient = appClientService.add(addRequest);
        appUpdateService.add(appClient.getId(), addRequest);
        return ApiResponse.buildOK(true);
    }

    private ApiResponse<Boolean> verifyAddParam(AppClientAddRequest request) {
        // 应用id是否存在
        AppConfig appConfig = appConfigService.selectById(request.getAppId());
        if (appConfig == null) {
            return ApiResponse.buildError(localeService.getLanguageValue("应用不存在"));
        }
        if ("ios".equalsIgnoreCase(appConfig.getOsType())) {
            // iOS只校验 url32 是否存在（因为下载链接存放在url32中）
            if (StringUtils.isBlank(request.getUrl32())) {
                return ApiResponse.buildError(localeService.getLanguageValue("下载地址不能为空"));
            }
        } else {
            // 校验 url32、md532、url64、md564 是否存在
            if (StringUtils.isBlank(request.getUrl32()) || StringUtils.isBlank(request.getMd532())
                    || StringUtils.isBlank(request.getUrl64()) || StringUtils.isBlank(request.getMd564())) {
                return ApiResponse.buildError(localeService.getLanguageValue("32位下载地址、64位下载地址、32位MD5、64位MD5不能为空"));
            }
        }
        // 校验brand值
        if (CollectionUtils.isNotEmpty(request.getBrand())) {
            for (String brand : request.getBrand()) {
                if (BrandEnum.exist(brand) == null) {
                    return ApiResponse.buildError(brand + localeService.getLanguageValue(" 商店不在选择范围内"));
                }
            }
        }
        // 校验updateRange值 -- release_type=1 && strong_update=1，update_range有值
        if (ReleaseTypeEnum.FULL.getValue().equals(request.getReleaseType())
                && StrongUpdateEnum.YES.getValue().equals(request.getStrongUpdate())) {
            if (UpdateRangeEnum.getByValue(request.getUpdateRange()) == null) {
                return ApiResponse.buildError(localeService.getLanguageValue("强制版更范围值不合法"));
            }
        }
        // 校验conditions值
        if (ReleaseTypeEnum.PARTIAL.getValue().equals(request.getReleaseType())) {
            // releaseType=0局部发布下(条件必须配置）
            if (CollectionUtils.isEmpty(request.getConditions())) {
                return ApiResponse.buildError(localeService.getLanguageValue("局部发布下，下发条件不能为空"));
            }
        }
        if (ReleaseTypeEnum.FULL.getValue().equals(request.getReleaseType())
                && StrongUpdateEnum.YES.getValue().equals(request.getStrongUpdate())
                && UpdateRangeEnum.PARTIAL.getValue().equals(request.getUpdateRange())) {
            // releaseType=1全量发布下 && strongUpdate=1强制更新下 && updateRange=1 局部范围下(条件必须配置）
            if (CollectionUtils.isEmpty(request.getConditions())) {
                return ApiResponse.buildError(localeService.getLanguageValue("全量发布下强制更新局部范围下，下发条件不能为空"));
            }
        }
        return ApiResponse.buildOK(true);
    }

    private ApiResponse<Boolean> verifyUpdateParam(AppClient appClient, AppClientUpdateRequest updateRequest) {
        if (null == appClient) {
            return ApiResponse.buildError(localeService.getLanguageValue("数据不存在"));
        }
        // 未发布状态才可以更新
        if (PublishEnum.PUBLISH.getValue().equals(appClient.getPublish()) || PublishEnum.CANCELPUBLISH.getValue().equals(appClient.getPublish())) {
            return ApiResponse.buildError(localeService.getLanguageValue("版本已发布或取消发布，不能修改"));
        }
        // appId、version不能修改
        if (!appClient.getAppId().equals(updateRequest.getAppId()) || !appClient.getVersion().equals(updateRequest.getVersion())) {
            return ApiResponse.buildError(localeService.getLanguageValue("应用与版本号不能修改"));
        }
        return ApiResponse.buildOK(true);
    }

    @Override
    @Db2Transactional(rollbackFor = Exception.class)
    public ApiResponse<Boolean> update(AppClientUpdateRequest updateRequest) {
        AppClient appClient = appClientService.selectById(updateRequest.getId());
        // 校验入参
        ApiResponse<Boolean> checkResult = this.verifyUpdateParam(appClient, updateRequest);
        if (checkResult.isError()) {
            return checkResult;
        }

        // 更新数据
        appClientService.update(appClient, updateRequest);
        appUpdateService.update(appClient.getId(), updateRequest);
        return ApiResponse.buildOK(true);
    }

    private ApiResponse<Boolean> verifyReleaseParam(AppClient appClient, AppClientUpdateRequest releaseRequest) {
        if (null == appClient) {
            return ApiResponse.buildError(localeService.getLanguageValue("数据不存在"));
        }
        // 取消发布状态不可发布
        if (PublishEnum.CANCELPUBLISH.getValue().equals(appClient.getPublish())) {
            return ApiResponse.buildError(localeService.getLanguageValue("取消发布下，不可再次发布"));
        }
        // appId、version不能修改
        if (!appClient.getAppId().equals(releaseRequest.getAppId()) || !appClient.getVersion().equals(releaseRequest.getVersion())) {
            return ApiResponse.buildError(localeService.getLanguageValue("应用与版本号不能修改"));
        }
        return this.verifyAddParam(releaseRequest);
    }

    @Override
    @Db2Transactional(rollbackFor = Exception.class)
    public ApiResponse<Boolean> release(AppClientUpdateRequest releaseRequest) {
        AppClient appClient = appClientService.selectById(releaseRequest.getId());
        // 校验入参
        ApiResponse<Boolean> checkResult = this.verifyReleaseParam(appClient, releaseRequest);
        if (checkResult.isError()) {
            return checkResult;
        }
        // 历史状态
        Integer publish = appClient.getPublish();
        // 发布 数据
        appClientService.release(appClient, releaseRequest);
        appUpdateService.release(appClient.getId(), publish, releaseRequest);
        return ApiResponse.buildOK(true);
    }

    @Override
    @Db2Transactional(rollbackFor = Exception.class)
    public ApiResponse<Boolean> unrelease(Long id) {
        AppClient appClient = appClientService.selectById(id);
        if (null == appClient) {
            return ApiResponse.buildError(localeService.getLanguageValue("数据不存在"));
        }
        // 发布状态才可以取消发布
        if (!PublishEnum.PUBLISH.getValue().equals(appClient.getPublish())) {
            return ApiResponse.buildError(localeService.getLanguageValue("发布状态才可以取消发布"));
        }
        appClientService.unrelease(appClient);
        return ApiResponse.buildOK(true);
    }

    @Override
    @Db2Transactional(rollbackFor = Exception.class)
    public ApiResponse<Boolean> deleteById(Long id) {
        AppClient appClient = appClientService.selectById(id);
        if (null == appClient) {
            return ApiResponse.buildError(localeService.getLanguageValue("数据不存在"));
        }
        // 发布状态不可删除
        if (PublishEnum.PUBLISH.getValue().equals(appClient.getPublish())) {
            return ApiResponse.buildError(localeService.getLanguageValue("发布状态下，不可删除"));
        }
        // 删除数据
        appClientService.deleteById(id);
        appUpdateService.deleteByAppClientId(id);
        return ApiResponse.buildOK(true);
    }

    @Override
    public ApiResponse<List<AppUpdateResponse>> updateHistoryById(Long id) {
        AppClient appClient = appClientService.selectById(id);
        if (null == appClient) {
            return ApiResponse.buildError(localeService.getLanguageValue("数据不存在"));
        }
        return ApiResponse.buildOK(appUpdateService.getBakByAppClientId(appClient.getId()).stream().map(AppUpdateResponse::convert).collect(Collectors.toList()));
    }

}