package com.digiwin.mobile.mobileuibot.versionManage;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.digiwin.mobile.mobileuibot.api.ApiResponse;
import com.digiwin.mobile.mobileuibot.api.PageResult;
import com.digiwin.mobile.mobileuibot.config.mysql.Db2Transactional;
import com.digiwin.mobile.mobileuibot.locale.service.LocaleService;
import com.digiwin.mobile.mobileuibot.model.db2.AppCondition;
import com.digiwin.mobile.mobileuibot.model.db2.AppConditionItem;
import com.digiwin.mobile.mobileuibot.versionManage.request.AppConditionAddRequest;
import com.digiwin.mobile.mobileuibot.versionManage.request.AppConditionItemRequest;
import com.digiwin.mobile.mobileuibot.versionManage.request.AppConditionPageRequest;
import com.digiwin.mobile.mobileuibot.versionManage.request.AppConditionUpdateRequest;
import com.digiwin.mobile.mobileuibot.versionManage.response.AppConditionItemResponse;
import com.digiwin.mobile.mobileuibot.versionManage.response.AppConditionResponse;
import com.digiwin.mobile.mobileuibot.versionManage.service.AppConditionItemService;
import com.digiwin.mobile.mobileuibot.versionManage.service.AppConditionService;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * <p>功能描述：版本管理系统 -- 条件管理服务接口实现类</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: VersionManageConditionServiceImpl.java
 * @Author: wangjwc
 * @Date: created at 2025/8/11 15:50
 */
@Service
public class VersionManageConditionServiceImpl implements VersionManageConditionService {

    @Autowired
    private AppConditionService appConditionService;

    @Autowired
    private AppConditionItemService appConditionItemService;

    @Autowired
    private LocaleService localeService;

    @Override
    public List<AppConditionResponse> list(AppConditionPageRequest request) {
        return this.convert(appConditionService.list(request));
    }

    private List<AppConditionResponse> convert(List<AppCondition> list) {
        if (CollectionUtils.isEmpty(list)) {
            return new ArrayList<>();
        }
        List<AppConditionResponse> conditionResponses = list.stream().map(AppConditionResponse::convert).collect(Collectors.toList());
        Set<Long> appConditionIds = conditionResponses.stream().map(AppConditionResponse::getId).collect(Collectors.toSet());
        List<AppConditionItem> appConditionItems = appConditionItemService.selectByAppConditionIds(appConditionIds);
        if (CollectionUtils.isNotEmpty(appConditionItems)) {
            Map<Long, List<AppConditionItemResponse>> itemMap = appConditionItems.stream()
                    .map(AppConditionItemResponse::convert)
                    .collect(Collectors.groupingBy(AppConditionItemResponse::getAppConditionId));
            conditionResponses.forEach(condition -> {
                List<AppConditionItemResponse> itemResponses = itemMap.get(condition.getId());
                if (CollectionUtils.isNotEmpty(itemResponses)) {
                    condition.setItem(itemResponses);
                }
            });
        }
        return conditionResponses;
    }

    @Override
    public PageResult<AppConditionResponse> listPage(AppConditionPageRequest pageRequest) {
        Page<AppCondition> page = appConditionService.listPage(pageRequest);
        PageResult<AppConditionResponse> pageResult = new PageResult<>();
        pageResult.setData(this.convert(page.getRecords()));
        pageResult.setTotal(page.getTotal());
        pageResult.setPageNum(page.getCurrent());
        pageResult.setPageSize(page.getSize());
        pageResult.setTotalPage(page.getPages());
        return pageResult;
    }

    @Override
    public AppConditionResponse selectById(Long id) {
        // 获取条件信息
        AppCondition appCondition = appConditionService.selectById(id);
        if (null == appCondition) {
            return null;
        }
        AppConditionResponse appConditionResponse = AppConditionResponse.convert(appCondition);
        // 获取条件项信息
        List<AppConditionItem> appConditionItem = appConditionItemService.selectByAppConditionId(id);
        if (CollectionUtils.isNotEmpty(appConditionItem)) {
            appConditionResponse.setItem(appConditionItem.stream().map(AppConditionItemResponse::convert).collect(Collectors.toList()));
        }
        return appConditionResponse;
    }

    @Override
    @Db2Transactional(rollbackFor = Exception.class)
    public ApiResponse<Boolean> add(AppConditionAddRequest addRequest) {
        // 校验入参
        ApiResponse<Boolean> checkResult = this.verifyAddOrUpdateParam(addRequest);
        if (checkResult.isError()) {
            return checkResult;
        }
        // 添加数据
        AppCondition appCondition = appConditionService.add(addRequest);
        appConditionItemService.add(appCondition.getId(), addRequest.getItem());
        return ApiResponse.buildOK();
    }

    private ApiResponse<Boolean> verifyAddOrUpdateParam(AppConditionAddRequest request) {
        if (CollectionUtils.isNotEmpty(request.getItem())) {
            // 校验 content、verify 字段
            for (AppConditionItemRequest item : request.getItem()) {
                if (!item.verifyContentParam()) {
                    return ApiResponse.buildError(localeService.getLanguageValue("条件值格式错误"));
                }
                if (!item.verifyVerifyParam()) {
                    return ApiResponse.buildError(localeService.getLanguageValue("条件值校验格式错误"));
                }
            }
        }
        return ApiResponse.buildOK();
    }

    @Override
    @Db2Transactional(rollbackFor = Exception.class)
    public ApiResponse<Boolean> update(AppConditionUpdateRequest updateRequest) {
        AppCondition appCondition = appConditionService.selectById(updateRequest.getId());
        if (null == appCondition) {
            return ApiResponse.buildError(localeService.getLanguageValue("数据不存在"));
        }
        // 校验入参
        ApiResponse<Boolean> checkResult = this.verifyAddOrUpdateParam(updateRequest);
        if (checkResult.isError()) {
            return checkResult;
        }
        // 更新数据
        appConditionService.update(appCondition, updateRequest);
        appConditionItemService.update(appCondition.getId(), updateRequest.getItem());
        return ApiResponse.buildOK();
    }

    @Override
    @Db2Transactional(rollbackFor = Exception.class)
    public ApiResponse<Boolean> deleteById(Long id) {
        AppCondition appCondition = appConditionService.selectById(id);
        if (null == appCondition) {
            return ApiResponse.buildError(localeService.getLanguageValue("数据不存在"));
        }
        // 删除数据
        appConditionService.deleteById(id);
        appConditionItemService.deleteByAppConditionId(id);
        return ApiResponse.buildOK();
    }
}